/*************************************************************************
  (c) Copyright.  Digital Equipment Corporation, 1995.  All Rights
  Reserved.

  Permission is hereby granted to use, copy, modify, or enhance this 
  software freely, as long as the foregoing copyright of Digital Equipment
  Corporation and this notice are retained on the software.  This 
  software may not be distributed or sublicensed for a fee.  Digital      
  makes this software available "AS IS" and without warranties of any
  kind.  
 *************************************************************************/
/*
 * Marko Kiiskila carnil@cs.tut.fi 
 * 
 * Tampere University of Technology - Telecommunications Laboratory
 *
 * Permission to use, copy, modify and distribute this
 * software and its documentation is hereby granted,
 * provided that both the copyright notice and this
 * permission notice appear in all copies of the software,
 * derivative works or modified versions, and any portions
 * thereof, that both notices appear in supporting
 * documentation, and that the use of this software is
 * acknowledged in any publications resulting from using
 * the software.
 * 
 * TUT ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION AND DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS
 * SOFTWARE.
 * 
 */

/*
* Module Name:
*   map_data.h
*   
* Overview:
*   This file defines the records stored in the address to VC mapping.  It is
*   concerned only with the VC and ATM lists.
*
* Authors:
*   TLR - Theodore L. Ross
*
* Modification History:
*   Date       Name  Description 
*   16-Feb-95  TLR   Created.
*   14-Oct-95  TLR   Updated VC_RECORD to handle buffer pools.
*
* Description:
*
*/

/*************************************************************************
 * Types that are specific to this interface.                            *
 *************************************************************************/

/* ESI_Status
 *
 *   This enumerated type lists the status values that are contained in
 *   48-bit address entries.
 *
 *      ESI_ARP_PENDING.....There is no ATM address associated with this
 *                          48-bit address.  The LE-ARP protocol is in
 *                          progress.
 *
 *      ESI_VC_PENDING......There is a valid ATM address associated with this
 *                          48-bit address but there is no VC set up to that
 *                          ATM address.  The signaling protocol is in process.
 *
 *      ESI_FLUSH_START.....There is a valid ATM address and VC associated
 *                          with this 48-bit address.  No packets have been
 *                          sent on this data direct VC for this address yet
 *                          so the LEC must initiate the flush protocol.
 *
 *      ESI_FLUSH_PENDING...The LEC has been notified of the FLUSH_START
 *                          status and it is assumed that the flush protocol is
 *                          in process.
 *
 *      ESI_FORWARD_DIRECT..Either the Path Switching Delay (C22) has elapsed
 *                          or the LEC has notified the Mapping that the flush
 *                          protocol has completed.  In either case, it is
 *                          safe to forward packets to this address via the
 *                          data direct VC.
 *
 *                          With regard to flushing, the LEC has the option of
 *                          participating in the flush protocol or simply
 *                          waiting until the Path Switching Delay elapses.
 *                          If the LEC chooses not to send a flush request,
 *                          the esi_status will transition from
 *                          ESI_FLUSH_PENDING to ESI_FORWARD_DIRECT
 *                          automatically after the delay has passed.
 */
#ifndef MAP_DATA_H
#define MAP_DATA_H
#include "utl.h"

typedef enum
   {
   ESI_UNKNOWN,
   ESI_ARP_PENDING,
   ESI_VC_PENDING,
   ESI_FLUSH_PENDING,
   ESI_FORWARD_DIRECT
   } ESI_STATUS;

/* ATM_Status
 *
 *   This enumerated type lists the ATM status values that are contained in
 *   ATM address entries.
 *
 *      ATM_NOVC........There is no VC associated with this ATM address and 
 *                      Signaling must be initiated in order to get a VC.
 *
 *      ATM_PENDING.....There is no VC associated with this ATM address and
 *                      Signaling is in the process of getting a VC for this
 *                      address.
 *
 *      ATM_VALID.......There is a valid and open VC associated with this
 *                      ATM address.
 *
 *      ATM_FAIL........Either the VC open failed or the SVC was released for
 *                      a reason other than "normal-unspecified".
 *
 */
typedef enum {
  ATM_NOVC,
  ATM_PENDING,
  ATM_VALID,
  ATM_FAIL
} ATM_STATUS;

/* VC_Flags definitions
 *
 * The vc_flags field of the MAP_VC_RECORD has the following structure:
 *
 *      15                              6   5          3   2          0
 *    +-----------------------------------+--------------+--------------+
 *    |             FLAGS                 |  PHY_STATE   |  SVC_STATE   |
 *    +-----------------------------------+--------------+--------------+
 *
 * Both PHY_STATE and SVC_STATE can have the following values:
 *
 *    VC_STATE_CLOSED   - PHY: The driver has not been notified of the VC,
 *                        SVC: The signaling protocol has not begun or has
 *                             completed with a release.
 *
 *    VC_STATE_OPENING  - PHY: The driver has been given a vc_setup request
 *                             and the request is pending (i.e. the driver
 *                             has not yet called the vc_setup_cfm routine).
 *                        SVC: The signaling protocol is in progress.
 *
 *    VC_STATE_OPEN     - PHY: The VC is open at the driver.
 *                        SVC: The signaling protocol completed successfully.
 *
 *    VC_STATE_CLOSING  - PHY: The driver has been given a vc_teardown request
 *                             and the request if pending (i.e. the driver
 *                             has not yet called the vc_teardown_cfm routine).
 *                        SVC: <not used>
 *
 * The following flags are in the FLAGS field:
 *
 *    VC_FLAG_NOTIFICATION_REQUIRED - The bound NSAP client requires
 *                                    notification of completed VC setup or
 *                                    teardown/setup-failure.
 *
 *    VC_FLAG_VC_ID_PRESENT         - The vc_id field is valid for an SVC.
 *                                    This is used during the signaling
 *                                    protocol because the vc_id becomes known
 *                                    at differing points in the protocol
 *                                    exchange depending on the direction of
 *                                    the call and the implementation of the
 *                                    switch.
 *
 *    VC_FLAG_CALLING_PARTY         - This end-station is the calling party
 *                                    for an SVC.
 *
 *    VC_FLAG_PVC                   - The VC is a PVC (Permanent VC).
 *
 *    VC_FLAG_RVC                   - The VC is an RVC (Digital Resilient VC).
 *
 */

#define VC_SVC_STATE_MASK  0x0007
#define VC_PHY_STATE_MASK  0x0038

#define VC_SVC_STATE(flags)   ((flags) & VC_SVC_STATE_MASK)
#define VC_PHY_STATE(flags)  (((flags) & VC_PHY_STATE_MASK) >> 3)

#define VC_SVC_STATE_SET(flags, state)                \
    {                                                 \
    (flags) = ((flags) & ~VC_SVC_STATE_MASK) | state; \
    }

#define VC_PHY_STATE_SET(flags, state)                       \
    {                                                        \
    (flags) = ((flags) & ~VC_PHY_STATE_MASK) | (state << 3); \
    }

#define VC_FLAGS_INIT      (0x0000 | VC_STATE_CLOSED | (VC_STATE_CLOSED << 3))

#define VC_STATE_CLOSED    1
#define VC_STATE_OPENING   2
#define VC_STATE_OPEN      3
#define VC_STATE_CLOSING   4
#define VC_STATE_FAIL      5

#define VC_FLAG_NOTIFICATION_REQUIRED  0x0040
#define VC_FLAG_VC_ID_PRESENT          0x0080
#define VC_FLAG_CALLING_PARTY          0x0100
#define VC_FLAG_PVC                    0x0200
#define VC_FLAG_RVC                    0x0400

#define VC_CALLING_PARTY(flags) \
   ((BOOLEAN) ((flags & VC_FLAG_CALLING_PARTY) ? TRUE : FALSE))

typedef struct _handle
   {
   struct _handle *p_next;
   HANDLE          handle;
   } HANDLE_RECORD;

SIMPLE_LIST (HANDLE_RECORD, HANDLE_LIST);

/* MAP_VC_RECORD
 *
 *   This structure is the user-visible part of a Virtual Channel entry.
 *   The VC table is composed of a list of these structures.
 *
 *   vc_id.............The VC_ID of the Virtual Channel.  This is an
 *                     opaque type which represents a VCI and a VPI
 *                     value to be used on the network.  VC_IDs are
 *                     supplied by the signaling module and are used
 *                     by the adapter hardware to derive a VCI and VPI.
 *
 *   conn_id...........The connection ID of the VC.  This is the handle
 *                     used by the Signaling stack to uniquely identify VCs.
 *                     This handle is generated by the Signaling module.
 *
 *   vc_status.........The current status of this VC.  See the above
 *                     definition of the VC_STATUS type for the possible
 *                     values.
 *
 *   xmt_queue.........The transmit queue identifier specifies which
 *                     transmit queue to use when transmitting data
 *                     on this VC.  It is assumed that ATM adapters
 *                     generally support multiple transmit and receive
 *                     queues.
 *
 *   rcv_queue.........The receive queue identifier specifies which
 *                     receive queue to use when data is received
 *                     on this VC.
 *
 *   sap_handle........Each VC must be bound to a Service Access Point
 *                     (SAP).  This value contains the binding for this
 *                     VC entry.  The sap_handle is generated by the
 *                     Connection Manager.
 *
 *                     When a packet is received, its VC_ID is looked
 *                     up in the VC tables.  Once the VC_ID is found,
 *                     this sap_handle is used to forward the packet
 *                     up to the next higher level.
 *
 *   map_lec_handle....This handle identifies the LEC entry in the mapping
 *                     that is associated with this VC.  This handle is
 *                     generated by the Mapping module.
 *
 *   conn_context......When a SAP creates a VC, it supplies a connection
 *                     context.  This context is then passed up to the SAP
 *                     along with every packet received on the VC.
 *
 *   requested_params..When a VC is opened, parameters are requested for
 *                     that VC.  This field contains the parameters that
 *                     were originally requested for this VC.
 *
 *   actual_params.....Because of numerical resolution issues and
 *                     variance in the availability of network
 *                     resources, the actual parameters granted to a VC
 *                     may be somewhat different from those requested.
 *                     This field contains the actual parameters that the
 *                     VC is operating with.
 *
 *   permanent_flag....This boolean flag is set if this is an SVC and it is
 *                     not to be removed to make room for opening SVCs.
 *
 *   xmt_packet_count..Number of Packets transmitted on this VC since it
 *                     was opened.
 *
 *   rcv_packet_count..Number of Packets received on this VC since it was
 *                     opened.
 */
typedef struct
   {
   MAP_LIST_HDR   hdr;
   UINT32         vc_id;
   CALL_ID        call_provider_id;
   CALL_ID        call_user_id;
   UINT16         vc_flags;
   HANDLE         xmt_pool_context;
   HANDLE         rcv_pool_context;
   HANDLE         sap_handle;
   HANDLE         conn_context;
   HANDLE         atm_link;
   HANDLE         spare;
   ADDR_ATM       calling_party;        /* See ATMF sec 8.1.11 */
   UINT32         xmt_packet_count;
   UINT32         rcv_packet_count;
   QOS            requested_qos;
   QOS            actual_qos;
   UINT32         idle_time;
   UINT32         age_limit;
   } MAP_VC_RECORD;

/* MAP_ATM_RECORD
 *
 *   This structure is the user-visible part of an ATM address entry.
 *   The ATM address table is composed of lists of these structures.
 */
typedef struct
   {
   MAP_LIST_HDR   hdr;
   ADDR_ATM       addr;
   ATM_STATUS     atm_status;
   HANDLE         active_vc_link;
   HANDLE_LIST    vc_link_list;
   HANDLE_LIST    esi_link_list;
   } MAP_ATM_RECORD;

/* AGE_REF
 *
 *  Age Reference.  This structure refers to a 48-bit address record, may be
 *  stored in a doubly linked list, and contains fields needed for the arp
 *  refresh and retry event queues.
 *
 *  p_event_queue   - Pointer to the event queue in which this strucure is
 *                    stored.  If this is not an element in an event queue,
 *                    this pointer MUST be NULL.  This pointer is used when
 *                    age references are unlinked from event queues.  It is
 *                    critical that the age reference be removed from the
 *                    queue of which it is a member.
 *
 *  delta_time      - Delta time used to schedule this age event in the event
 *                    queue.  It signifies how long after the previous event
 *                    this event is to be scheduled.
 *
 *  arp_retries     - Number of times an arp retry has been attempted on
 *                    behalf of this 48-bit address.  This is used to limit
 *                    the number of retries that will be attempted.
 */
typedef struct age_ref
   {
   struct age_ref      *p_next;
   struct age_ref      *p_prev;
   struct event_tag    *p_event_queue;
   HANDLE               esi_handle;
   UINT32               delta_time;
   UINT8                arp_retries;
   } AGE_REF;

typedef struct
   {
   MAP_LIST_HDR   hdr;
   ESI            esi;
   ESI_STATUS     esi_status;
   HANDLE         atm_link;
   HANDLE         vc_link;
   HANDLE         la_elan_handle;
   AGE_REF       *p_age_ref;
   BOOLEAN        remote_flag;
   BOOLEAN        permanent_entry;
   UINT32         arp_age;
   UINT32         use_age;
   UINT32         flush_tran_id;
   UINT16         tick_count;
   UINT16         packets_flooded;
   } MAP_ESI_RECORD;

#endif /* MAP_DATA_H */
