/* kernel.c - Processing of incoming kernel messages */
 
/* Written 1995 by Werner Almesberger, EPFL-LRC */
 

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <linux/atm.h>
#include <linux/atmsvc.h>
#include <linux/atmdev.h>

#include "q2931.h"
#include "qlib.h"

#include "diag.h"
#include "common.h"
#include "proto.h"
#include "io.h"
#include "timer.h"
#include "timeout.h"


#define COMPONENT "KSP"


extern int vci;


static int send_setup(SOCKET *sock)
{
    static unsigned long call_ref = 0;
    SOCKET *walk;
    Q_DSC dsc;
    int size;

    do {
	if (++call_ref == 0x800000) call_ref = 1;
	for (walk = sockets; walk; walk = walk->next)
	    if (walk->call_ref == call_ref) break;
    }
    while (walk);
    sock->call_ref = call_ref;
    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_call_ref,call_ref);
    q_assign(&dsc,QF_msg_type,QMSG_SETUP);
    q_assign(&dsc,QF_aal_type,5); /* AAL 5 */
    q_assign(&dsc,QF_fw_pcr_01,1000); /* Fore doesn't want ATM_OC3_PCR */
    q_assign(&dsc,QF_bw_pcr_01,1000); /* Fore doesn't want ATM_OC3_PCR */
    q_assign(&dsc,QF_best_effort,0);
    q_assign(&dsc,QF_bearer_class,3); /* BCOB-C */
    q_assign(&dsc,QF_upcc,0); /* p2p */
    q_assign(&dsc,QF_qos_fw,0); /* QOS 0 */
    q_assign(&dsc,QF_qos_bw,0);
    /* send local address if available */
    if (*sock->remote->sas_addr.prv)
	q_write(&dsc,QF_cdpn_nsap,(void *) sock->remote->sas_addr.prv,
	  ATM_ESA_LEN);
    if (net) {
	q_assign(&dsc,QF_vpi,0);
	q_assign(&dsc,QF_vci,vci); /* @@@ */
	sock->pvc.sap_addr.vpi = 0;
	sock->pvc.sap_addr.vci = vci++;
    }
    size = q_close(&dsc);
    to_signaling(q_buffer,size);
    return 0;
}


static int send_connect(SOCKET *sock)
{
    Q_DSC dsc;
    int size;

    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_msg_type,QMSG_CONNECT);
    q_assign(&dsc,QF_call_ref,sock->call_ref);
    /* lots of data */
    size = q_close(&dsc);
    to_signaling(q_buffer,size);
    return 0;
}


static void send_release_complete(SOCKET *sock,unsigned char reason)
{
    Q_DSC dsc;
    int size;
 
    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_msg_type,QMSG_REL_COMP);
    q_assign(&dsc,QF_call_ref,sock->call_ref);
    /* set cause */
    size = q_close(&dsc);
    to_signaling(q_buffer,size);
}


static void dispatch(SOCKET *sock,struct atmsvc_msg *msg)
{
    SOCKET *next;
    int error;

    switch (msg->type) {
	case as_bind: /* only in NULL state */
	    if (sock) break;
	    send_kernel(msg->vcc,0,as_okay,0,NULL);
	    /* send_kernel(msg->vcc,0,as_close,-EINVAL,NULL); ... or */
	    return;
	case as_establish: /* NULL or LISTENING */
	    if (sock && sock->state != ss_listening) break;
	    if (!sock) {
		sock = new_sock(msg->vcc);
		sock->local = alloc_sap(&msg->local);
		sock->remote = alloc_sap(&msg->remote);
		if (!sock->remote) {
		    free_sock(sock);
		    send_kernel(msg->vcc,0,as_close,-EINVAL,NULL);
		    return;
		}
		sock->state = ss_connecting;
		error = send_setup(sock);
		if (error) {
		    send_kernel(msg->vcc,0,as_close,error,NULL);
		    free_sock(sock);
		    return;
		}
		START_TIMER(sock,T303);
		new_state(sock,ss_connecting);
		return;
	    }
	    /* becomes INDICATED or ZOMBIE */
	    if (!sock->listen) {
		diag(COMPONENT,DIAG_WARN,
		  "socket 0x%lx got accept with empty listen queue",msg->vcc);
		return;
	    }
	    next = sock->listen;
	    sock->listen = next->listen;
	    next->listen = NULL;
	    if (next->state == ss_zombie) {
		send_kernel(msg->vcc,0,as_close,-ECONNRESET,NULL);
			/* -ERESTARTSYS ? */
		free_sock(next);
		return;
	    }
	    if (next->state != ss_indicated) {
		sock = next; /* for error reporting */
		break;
	    }
	    next->id = msg->vcc;
	    error = send_connect(next);
	    if (!error) {
		START_TIMER(next,T313);
		new_state(next,ss_accepting);
		return;
	    }
	    send_kernel(next->id,0,as_close,error,NULL);
	    send_release(next,0); /* @@@ */
	    START_TIMER(next,T308_1);
	    new_state(next,ss_wait_rel);
	    return;
	case as_listen: /* NULL */
	    if (sock) break;
	    /* check address @@@ */
	    error = 0;
	    if (error) {
		send_kernel(msg->vcc,0,as_close,error,NULL);
		return;
	    }
	    sock = new_sock(msg->vcc);
	    sock->local = alloc_sap(&msg->local);
	    send_kernel(sock->id,0,as_okay,error,NULL);
	    sock->state = ss_listening;
	    return;
	case as_close: /* all but ACCEPTING and WAIT_REL */
	    if (sock && (sock->state == ss_accepting || sock->state ==
	      ss_wait_rel)) break;
	    switch (sock ? sock->state : ss_null) {
		case ss_listening:
		case ss_zombie:
		    send_close(sock);
		    /* fall through */
		case ss_wait_close:
		    free_sock(sock);
		    /* fall through */
		case ss_null:
		case ss_rel_req:
		    return;
		case ss_connecting:
		    STOP_TIMER(sock);
		    /* fall through */
		case ss_connected:
		    if (sock->state == ss_connected)
			diag(COMPONENT,DIAG_INFO,"Active close (CR 0x%06X)",
			  sock->call_ref);
		    /* fall through */
		case ss_indicated:
		case ss_hold:
		    send_release(sock,0); /* @@@ */
		    START_TIMER(sock,T308_1);
		    new_state(sock,sock->state == ss_connected ? ss_rel_req :
		      ss_wait_rel);
		    return;
		case ss_rel_ind:
		    send_release_complete(sock,0); /* @@@ */
		    free_sock(sock);
		    return;
		default:
		    break;
	    }
	    /* fall through */
	default:
    	    diag(COMPONENT,DIAG_WARN,"invalid message %d",(int) msg->type);
	    return;
    }
    diag(COMPONENT,DIAG_WARN,"message %s is incompatible with state %s (%d)",
      as_name[msg->type],state_name[sock ? sock->state : ss_null],
      (int) (sock ? sock->state : ss_null));
}


void from_kernel(struct atmsvc_msg *msg,int size)
{
    SOCKET *curr;
    struct atm_blli **this;
    int bllis,i;

    for (curr = sockets; curr; curr = curr->next)
	if (msg->vcc == curr->id) break;
    if (!curr)
	for (curr = sockets; curr; curr = curr->next)
	    if (msg->listen_vcc == curr->id && curr->state == ss_listening)
		break;
    diag(COMPONENT,DIAG_DEBUG,"FROM KERNEL: %s for socket 0x%lx (0x%lx/0x%lx) "
      "in state %s",as_name[msg->type],(unsigned long) curr,msg->vcc,
      msg->listen_vcc,state_name[curr ? curr->state : ss_null]);
    msg->local.sas_addr.blli = NULL;
    bllis = msg2bllis(size);
    this = &msg->remote.sas_addr.blli;
    for (i = 0; i < bllis; i++) {
	*this = &msg->blli[i];
	this = &msg->blli[i].next;
    }
    *this = NULL;
    dispatch(curr,msg);
}
