/* q2931.c - Processing of incoming Q.2931 messages */
 
/* Written 1995 by Werner Almesberger, EPFL-LRC */
 

/*
 * TODO:
 * - concentrate sending of status
 */


#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#include "atm.h"
#include "atmd.h"
#include "q2931.h"
#include "qlib.h"

#include "proto.h"
#include "io.h"
#include "timeout.h"

#define __KERNEL__ /* since that's what we effectively are */
#include <linux/errno.h>


#define COMPONENT "Q2931"


static Q_DSC in_dsc;
int vci = 42; /* assigning VCIs from here on if acting as network */


static void send_call_proceeding(SOCKET *sock)
{
    Q_DSC dsc;
    int size;

    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_msg_type,QMSG_CALL_PROC);
    q_assign(&dsc,QF_call_ref,sock->call_ref);
    if (net) {
	q_assign(&dsc,QF_vpi,0);
	q_assign(&dsc,QF_vci,vci); /* @@@ */
	sock->pvc.sap_addr.itf = 0; /* @@@ use VPCI */
	sock->pvc.sap_addr.vpi = 0;
	sock->pvc.sap_addr.vci = vci++;
    }
    size = q_close(&dsc);
    to_signaling(q_buffer,size);
}


static void send_release_complete(unsigned long call_ref,unsigned char cause)
{
    Q_DSC dsc;
    int size;

    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_msg_type,QMSG_REL_COMP);
    q_assign(&dsc,QF_call_ref,call_ref);
    /* set cause (only if !!cause) */
    size = q_close(&dsc);
    to_signaling(q_buffer,size);
}


static struct sockaddr_atmsvc *build_sap(void)
{
    struct sockaddr_atmsvc *sap;
    struct atm_blli *blli;

    sap = alloc_t(struct sockaddr_atmsvc);
    memset(sap,0,sizeof(struct sockaddr_atmsvc));
    if (q_present(&in_dsc,QF_aal_type))
	if (q_fetch(&in_dsc,QF_aal_type) != 5)
	    diag(COMPONENT,DIAG_ERROR,"AAL type %d requested",
	      q_fetch(&in_dsc,QF_aal_type));
    if (q_present(&in_dsc,QF_best_effort))
	sap->sas_txtp.class = sap->sas_rxtp.class = ATM_UBR;
    else {
	sap->sas_txtp.class = sap->sas_rxtp.class = ATM_CBR;
	if (q_present(&in_dsc,QF_fw_pcr_01))
	    sap->sas_txtp.min_pcr = sap->sas_txtp.max_pcr =
	      q_fetch(&in_dsc,QF_fw_pcr_01);
	if (q_present(&in_dsc,QF_bw_pcr_01))
	    sap->sas_rxtp.min_pcr = sap->sas_rxtp.max_pcr =
	      q_fetch(&in_dsc,QF_bw_pcr_01);
	/* SHOULD ... fail call if anything is missing ... @@@ */
    }
    if (q_present(&in_dsc,QF_fw_max_sdu))
	sap->sas_txtp.max_sdu = q_fetch(&in_dsc,QF_fw_max_sdu);
    if (q_present(&in_dsc,QF_bw_max_sdu))
	sap->sas_rxtp.max_sdu = q_fetch(&in_dsc,QF_bw_max_sdu);
    if (q_present(&in_dsc,QG_bhli)) {
	sap->sas_addr.bhli.hl_type = q_fetch(&in_dsc,QF_hli_type);
	switch (sap->sas_addr.bhli.hl_type) {
	    case ATM_HL_ISO:
		sap->sas_addr.bhli.hl_length = q_length(&in_dsc,QF_iso_hli);
		q_read(&in_dsc,QF_iso_hli,sap->sas_addr.bhli.hl_info,
		  sap->sas_addr.bhli.hl_length);
		break;
	    case ATM_HL_USER:
		sap->sas_addr.bhli.hl_length = q_length(&in_dsc,QF_user_hli);
		q_read(&in_dsc,QF_user_hli,sap->sas_addr.bhli.hl_info,
		  sap->sas_addr.bhli.hl_length);
		break;
#ifdef UNI30
	    case ATM_HL_HLP:
		sap->sas_addr.bhli.hl_length = 4;
		q_read(&in_dsc,QF_hlp,sap->sas_addr.bhli.hl_info,4);
		break;
#endif
	    case ATM_HL_VENDOR:
		sap->sas_addr.bhli.hl_length = 7;
		q_read(&in_dsc,QF_hli_oui,sap->sas_addr.bhli.hl_info,3);
		q_read(&in_dsc,QF_app_id,sap->sas_addr.bhli.hl_info+3,4);
		break;
	    default:
		diag(COMPONENT,DIAG_FATAL,"unrecognized hl_type");
	}
    }

#define GET(var,field) \
   ({ if (q_present(&in_dsc,field)) blli->var = q_fetch(&in_dsc,field); })

    if (q_present(&in_dsc,QG_blli)) {
	sap->sas_addr.blli = blli = alloc_t(struct atm_blli);
	if (q_present(&in_dsc,QF_uil2_proto)) {
	    blli->l2_proto = q_fetch(&in_dsc,QF_uil2_proto);
	    GET(l2.itu.mode,QF_l2_mode);
	    GET(l2.itu.window,QF_window_size);
	    GET(l2.user,QF_user_l2);
	}
	if (q_present(&in_dsc,QF_uil3_proto)) {
	    blli->l3_proto = q_fetch(&in_dsc,QF_uil3_proto);
	    GET(l3.itu.mode,QF_l3_mode);
	    GET(l3.itu.def_size,QF_def_pck_size);
	    GET(l3.itu.window,QF_pck_win_size);
	    GET(l3.user,QF_user_l3);
	    if (q_present(&in_dsc,QF_ipi_high)) {
		blli->l3.tr9577.ipi = q_fetch(&in_dsc,QF_ipi_high) << 1;
		if (blli->l3.tr9577.ipi != NLPID_IEEE802_1_SNAP)
		    blli->l3.tr9577.ipi |= q_fetch(&in_dsc,QF_ipi_low);
		else if (!q_present(&in_dsc,QF_oui)) blli->l3.tr9577.ipi |= 1;
		    else {
			q_read(&in_dsc,QF_oui,blli->l3.tr9577.snap,3);
			q_read(&in_dsc,QF_pid,blli->l3.tr9577.snap+3,2);
		    }
	    }
	}
	blli->next = NULL;
    }
#undef GET
    return sap;
}


static void setup_call(unsigned long call_ref)
{
    SOCKET *sock,*this,**walk;
    struct sockaddr_atmsvc *sap;
    unsigned char buf[ATM_ESA_LEN];
    int len,i;

    sap = build_sap();
    sock = lookup_sap(sap);
    if (!sock) {
	send_release_complete(call_ref,ATM_CV_INCOMP_DEST); /* @@@ dunno */
	free_sap(sap);
	return;
    }
    this = new_sock(0);
    this->state = ss_indicated;
    this->q2931_state = qs_in_proc;
    this->call_ref = call_ref;
    send_call_proceeding(this);
    this->local = sap;
    this->pvc.sap_txtp = sap->sas_txtp;
    this->pvc.sap_rxtp = sap->sas_rxtp;
    /* if (sock->local) *this->local->sas_addr = sock->local->sas_addr; ??? */
    printf("AAL type %ld\n",q_fetch(&in_dsc,QF_aal_type));
    len = q_read(&in_dsc,QF_cdpn_esa,(void *) &buf,sizeof(buf));
    printf("Addr len %d:",len);
    for (i = 0; i < (len < 30 ? len : 30); i++) printf(" %02X",buf[i]);
    putchar('\n');
    if (q_present(&in_dsc,QF_vci)) {
	this->pvc.sap_addr.itf = 0; /* @@@ use VPCI */
	this->pvc.sap_addr.vpi = q_fetch(&in_dsc,QF_vpi);
	this->pvc.sap_addr.vci = q_fetch(&in_dsc,QF_vci);
printf("VPI/VCI: %d/%d\n",this->pvc.sap_addr.vpi,this->pvc.sap_addr.vci);
    }
    if (q_present(&in_dsc,QF_cgpn)) { /* should handle E.164 too */
	char buffer[MAX_ATM_ADDR_LEN+1];

	this->remote = alloc_t(struct sockaddr_atmsvc);
	memset(this->remote,0,sizeof(struct sockaddr_atmsvc));
	this->remote->sas_family = AF_ATMSVC;
	i = q_read(&in_dsc,QF_cgpn,(void *) this->remote->sas_addr.prv,
	  ATM_ESA_LEN);
	this->remote->sas_txtp = sap->sas_txtp;
	this->remote->sas_rxtp = sap->sas_rxtp;
	/*printf("addr_len = %d\n",i);*/
	if (atm2text(buffer,MAX_ATM_ADDR_LEN+1,(struct sockaddr *) this->remote,
	  A2T_PRETTY) < 0) strcpy(buffer,"<invalid address>");
	diag(COMPONENT,DIAG_DEBUG,"Incoming call from %s",buffer);
    }
    send_kernel(0,sock->id,as_indicate,0,this);
    for (walk = &sock->listen; *walk; walk = &(*walk)->listen);
    *walk = this;
    diag(COMPONENT,DIAG_DEBUG,"SE vpi.vci=%d.%d",this->pvc.sap_addr.vpi,
      this->pvc.sap_addr.vci);
}


static void send_status(SOCKET *sock,unsigned char cause,...)
{
    va_list ap;
    Q_DSC dsc;
    int size;

    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_msg_type,QMSG_STATUS);
    q_assign(&dsc,QF_call_ref,sock->call_ref);
    q_assign(&dsc,QF_call_state,(int) sock->q2931_state);
    q_assign(&dsc,QF_cause,cause);
    va_start(ap,cause);
    switch (cause) {
	case ATM_CV_UNKNOWN_MSG_TYPE:
	case ATM_CV_INCOMP_MSG:
	    q_assign(&dsc,QF_bad_msg_type,va_arg(ap,unsigned char));
	    break;
	default:
    }
    va_end(ap);
    size = q_close(&dsc);
    to_signaling(q_buffer,size);
}


static void send_connect_ack(SOCKET *sock)
{
    Q_DSC dsc;
    int size;

    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_msg_type,QMSG_CONN_ACK);
    q_assign(&dsc,QF_call_ref,sock->call_ref);
    size = q_close(&dsc);
    to_signaling(q_buffer,size);
}


static void q2931_call(SOCKET *sock,unsigned char mid)
{
    int error;

    switch (mid) {
	case QMSG_STATUS: /* 5.5.6.12 */
	    {
		Q2931_STATE state;

		if (sock->q2931_state == qs_rel_req || sock->q2931_state ==
		  qs_rel_ind) return;
		state = q_fetch(&in_dsc,QF_call_state);
		if (state == qs_null) break; /* clear call */
		if (state != sock->q2931_state)
		    diag(COMPONENT,DIAG_WARN,"STATUS %s received in state %s",
		      qs_name[state],qs_name[sock->q2931_state]);
	    }
	    return;
	case QMSG_STATUS_ENQ:
	    send_status(sock,ATM_CV_RESP_STAT_ENQ);
	    return;
	default:
	    ;
    }
    switch (mid) {
	case QMSG_CALL_PROC: /* CONNECTING, WAIT_REL, REL_REQ */
	    if (sock->state == ss_wait_rel || sock->state == ss_rel_req) {
		send_status(sock,ATM_CV_INCOMP_MSG,QMSG_CALL_PROC);
		return;
	    }
	    if (sock->state != ss_connecting) break;
	    /* check for 2nd CALL_PROC @@@ */
	    STOP_TIMER(sock);
	    if (q_present(&in_dsc,QG_conn_id)) {
		sock->pvc.sap_addr.itf = 0; /* @@@ should use VPCI */
		sock->pvc.sap_addr.vpi = q_fetch(&in_dsc,QF_vpi);
		sock->pvc.sap_addr.vci = q_fetch(&in_dsc,QF_vci);
printf("VPI/VCI: %d/%d\n",sock->pvc.sap_addr.vpi,sock->pvc.sap_addr.vci);
	    }
	    START_TIMER(sock,T310);
	    sock->q2931_state = qs_out_proc;
	    return;
	case QMSG_CONNECT: /* CONNECTING, REL_REQ */
	    if (sock->state == ss_rel_req) {
		send_status(sock,ATM_CV_INCOMP_MSG,QMSG_CONNECT);
		return;
	    }
	    if (sock->state != ss_connecting) break;
	    STOP_TIMER(sock);
	    if (q_present(&in_dsc,QG_conn_id)) {
		sock->pvc.sap_addr.itf = 0; /* @@@ should use VPCI */
		sock->pvc.sap_addr.vpi = q_fetch(&in_dsc,QF_vpi);
		sock->pvc.sap_addr.vci = q_fetch(&in_dsc,QF_vci);
printf("VPI/VCI: %d/%d\n",sock->pvc.sap_addr.vpi,sock->pvc.sap_addr.vci);
	    }
	    error = 0;
	    if (!sock->pvc.sap_addr.vpi && !sock->pvc.sap_addr.vci)
		error = -EPROTO;
	    /* more problems */
	    if (error) {
		set_error(sock,error);
		send_release(sock,0); /* @@@ cause follows reason ??? */
		START_TIMER(sock,T308_1);
		new_state(sock,ss_rel_req);
		return;
	    }
	    send_connect_ack(sock);
	    /* @@@ fill in sock->remote */
	    /* @@@ fill in traffic parameters */
	    send_kernel(sock->id,0,as_okay,0,sock);
	    new_state(sock,ss_connected);
	    diag(COMPONENT,DIAG_INFO,"Active open succeeded (CR 0x%06X)",
	      sock->call_ref);
	    return;
	case QMSG_CONN_ACK: /* ACCEPTING, WAIT_REL, REL_REQ */
	    diag(COMPONENT,DIAG_DEBUG,"CA vpi.vci=%d.%d",
	      sock->pvc.sap_addr.vpi,sock->pvc.sap_addr.vci);
	    if (sock->state == ss_wait_rel || sock->state == ss_rel_req) {
		send_status(sock,ATM_CV_INCOMP_MSG,QMSG_CONN_ACK);
		return;
	    }
	    if (sock->state != ss_accepting) break;
	    STOP_TIMER(sock);
	    send_kernel(sock->id,0,as_okay,0,sock);
		/* repeat remote information - just for fun */
	    new_state(sock,ss_connected);
	    diag(COMPONENT,DIAG_INFO,"Passive open succeeded (CR 0x%06X)",
	      sock->call_ref);
	    return;
	case QMSG_RELEASE: /* all states */
	    switch (sock->state) {
		case ss_connecting:
		    set_error(sock,-ECONNREFUSED);
		    /* fall through */
		case ss_accepting:
		    set_error(sock,-ECONNRESET); /* ERESTARTSYS ? */
		    send_release_complete(sock->call_ref,0);
		    /* fall through */
		case ss_rel_req:
		    send_close(sock);
		    /* fall through */
		case ss_wait_rel:
		    STOP_TIMER(sock);
		    free_sock(sock);
		    return;
		case ss_connected:
		    diag(COMPONENT,DIAG_INFO,"Passive close (CR 0x%06X)",
		      sock->call_ref);
		    send_close(sock);
		    /* fall through */
		case ss_hold:
		    new_state(sock,ss_rel_ind);
		    return;
		case ss_indicated:
		    new_state(sock,ss_zombie);
		    /* fall through */
		case ss_rel_ind:
		    /* send_release_complete(sock->call_ref,0); */
		    return;
		default:
		    send_release_complete(sock->call_ref,0); /* @@@ should
			be ATM_CV_INCOMP_MSG */
		    break;
	    }
	    break;
	case QMSG_STATUS: /* fall through when clearing */
	case QMSG_REL_COMP: /* basically any state (except LISTENING,
				  ZOMBIE, REL_IND) */
	    switch (sock->state) {
		case ss_connecting:
		    set_error(sock,-ECONNREFUSED);
		    /* fall through */
		case ss_accepting:
		    set_error(sock,-ECONNRESET); /* ERESTARTSYS ? */
		    /* fall through */
		case ss_rel_req:
		    send_close(sock);
		    /* fall through */
		case ss_wait_rel:
		    STOP_TIMER(sock);
		    free_sock(sock);
		    return;
		case ss_connected:
		    diag(COMPONENT,DIAG_INFO,"Passive clsoe (CR 0x%06X)",
		      sock->call_ref);
		    send_close(sock);
		    /* fall through */
		case ss_hold:
		    new_state(sock,ss_wait_close);
		    return;
		case ss_indicated:
		    new_state(sock,ss_zombie);
		    return;
		default:
		    break;
	    }
	default:
	    diag(COMPONENT,DIAG_WARN,"Bad Q.2931 message %d",mid);
	    send_status(sock,ATM_CV_UNKNOWN_MSG_TYPE,mid);
	    return;
    }
    diag(COMPONENT,DIAG_WARN,
      "Q.2931 message %s is incompatible with state %s/%s (%d?%d)",
      mid2name(mid),state_name[sock->state],qs_name[sock->q2931_state],
      (int) sock->state,(int) sock->q2931_state);
    send_status(sock,ATM_CV_INCOMP_MSG,mid);
}



void to_q2931(void *msg,int size)
{
    SOCKET *curr;
    unsigned long call_ref;
    unsigned char mid;

    q_open(&in_dsc,msg,size);
    call_ref = q_fetch(&in_dsc,QF_call_ref)^0x800000;
    if (!(call_ref & 0x7fffff)) {
	return; /* bad things happen ... @@@ */
    }
    for (curr = sockets; curr; curr = curr->next)
	if (curr->call_ref == call_ref) break;
    mid = q_fetch(&in_dsc,QF_msg_type);
    diag(COMPONENT,DIAG_DEBUG,"FROM NET: %s (0x%02X) CR 0x%06lx for 0x%lx",
      mid2name(((unsigned char *) msg)[5]),((unsigned char *)msg)[5],call_ref,
      curr ? curr->id : 0);
    if (mid == QMSG_SETUP) {
	if (!curr) setup_call(call_ref);
	return;
    }
    if (!curr || curr->q2931_state == qs_null) {
	if (mid == QMSG_REL_COMP)
	    send_release_complete(call_ref,ATM_CV_INV_CR);
	else if (mid != QMSG_STATUS)
		if (q_fetch(&in_dsc,QF_call_state) != (int) qs_null)
		    send_release_complete(call_ref,ATM_CV_INCOMP_MSG);
	return;
    }
    q2931_call(curr,mid);
    q_close(&in_dsc);
}
