/* proto.c - Common protocol functions and structures */
 
/* Written 1995 by Werner Almesberger, EPFL-LRC */
 

#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "atmd.h"
#include "q2931.h"
#include "qlib.h"

#include "io.h"
#include "proto.h"


#define COMPONENT "SIGD"


const char *state_name[] = { /* formatting aligned with STATE */
	"<invalid>",	"ss_null",	"ss_listening",	"ss_connecting",
	"ss_connected",	"ss_indicated",	"ss_accepting",	"ss_zombie",
	"ss_hold",	"ss_wait_rel",	"ss_wait_close","ss_rel_req",
	"ss_rel_ind" };

const char *qs_name[] = {
	"NULL",		"CALL_INIT",	"<invalid>",	"OUT_PROC",
	"<invalid>",	"<invalid>",	"CALL_PRES",	"<invalid>",
	"CONN_REQ",	"IN_PROC",	"ACTIVE",	"REL_REQ",
	"REL_IND" };

const char *as_name[] = { "<invalid>","as_bind","as_establish","as_listen",
  "as_okay","as_indicate","as_close" };

const Q2931_STATE state_map[] = { /* formatting aligned with STATE */
	qs_null,	qs_null,	qs_null,	qs_call_init,
	qs_active,	qs_in_proc,	qs_conn_req,	qs_null,
	qs_active,	qs_rel_req,	qs_null,	qs_rel_req,
	qs_rel_ind };


SOCKET *sockets = NULL;
unsigned char q_buffer[MAX_Q_MSG];


SOCKET *new_sock(unsigned long id)
{
    SOCKET *sock;
 
    sock = alloc_t(SOCKET);
    sock->state = ss_invalid;
    sock->pvc.sap_addr.vpi = sock->pvc.sap_addr.vci = 0;
    sock->pvc.sap_txtp.class = sock->pvc.sap_rxtp.class = ATM_UBR;
    sock->id = id;
    sock->local = sock->remote = NULL;
    sock->error = 0;
    sock->q2931_state = qs_null;
    sock->conn_timer = NULL;
    sock->listen = NULL;
    sock->next = sockets;
    sockets = sock;
    return sock;
}


void free_sap(struct sockaddr_atmsvc *sap)
{
    struct atm_blli *walk,*next;

    for (walk = sap->sas_addr.blli; walk; walk = next) {
	next = walk->next;
	free(walk);
    }
    free(sap);
}


void free_sock(SOCKET *sock)
{
    SOCKET **walk;

    diag(COMPONENT,DIAG_DEBUG,"freeing socket 0x%lx@0x%lx",sock->id,
      (unsigned long) sock);
    for (walk = &sockets; *walk != sock; walk = &(*walk)->next);
    if (!*walk)
	diag(COMPONENT,DIAG_FATAL,
	  "INTERNAL ERROR: freeing non-existing socket 0x%lx",sock->id);
    *walk = sock->next;
    if (sock->conn_timer) {
	diag(COMPONENT,DIAG_ERROR,"socket 0x%lx has timer (0x%lx) running",
	  sock->id,(unsigned long) sock->conn_timer->callback);
	stop_timer(sock->conn_timer);
    }
    if (sock->listen)
        diag(COMPONENT,DIAG_ERROR,"socket 0x%lx has non-empty listen queue",
	  sock->id);
    if (sock->local) free_sap(sock->local);
    if (sock->remote) free_sap(sock->remote);
    sock->state = ss_invalid;
    free(sock);
}


void new_state(SOCKET *sock,STATE state)
{
    diag(COMPONENT,DIAG_DEBUG,"socket 0x%lx enters state %s (Q.2931 %s)",
      sock->id,state_name[state],qs_name[state_map[state]]);
    sock->state = state;
    sock->q2931_state = state_map[state];
}


struct sockaddr_atmsvc *alloc_sap(const struct sockaddr_atmsvc *sap)
{
    struct sockaddr_atmsvc *new_sap;
    struct atm_blli *walk,**here;

    /*if (!*sap->sas_addr.prv && !*sap->sas_addr.pub) return NULL; @@@ */
    new_sap = alloc_t(struct sockaddr_atmsvc);
    *new_sap = *sap;
    here = &new_sap->sas_addr.blli;
    for (walk = sap->sas_addr.blli; walk; walk = walk->next) {
	*here = alloc_t(struct atm_blli);
	**here = *walk;
    }
    return new_sap;
}


static int match_blli(struct atm_blli *a,struct atm_blli *b)
{
    if (a->l2_proto != b->l2_proto || a->l3_proto != b->l3_proto) return 0;
    switch (a->l3_proto) {
	case ATM_L3_TR9577:
	    if (a->l3.tr9577.ipi != b->l3.tr9577.ipi) return 0;
	    if (a->l3.tr9577.ipi == NLPID_IEEE802_1_SNAP)
		if (memcmp(a->l3.tr9577.snap,b->l3.tr9577.snap,5)) return 0;
	    break;
	case ATM_L3_USER:
	    if (a->l3.user != b->l3.user) return 0;
	    break;
	default:
    }
    return 1;
}


static int match_sap(struct sockaddr_atmsvc *a,struct sockaddr_atmsvc *b)
{
    struct atm_blli *a_blli,*b_blli;
    int length;

/* @@@ check AAL */
    if (a->sas_addr.bhli.hl_type != b->sas_addr.bhli.hl_type) return 0;
    switch (a->sas_addr.bhli.hl_type) {
	case ATM_HL_ISO:
	case ATM_HL_USER:
	    length = a->sas_addr.bhli.hl_length;
	    if (length != b->sas_addr.bhli.hl_length) return 0;
	    break;
#ifdef UNI30
	case ATM_HL_HLP:
	    length = 4;
	    break;
#endif
	case ATM_HL_VENDOR:
	    length = 7;
	    break;
	default:
	    length = 0;
    }
    if (length && memcmp(a->sas_addr.bhli.hl_info,b->sas_addr.bhli.hl_info,
      length)) return 0;
    a_blli = a->sas_addr.blli;
    b_blli= b->sas_addr.blli;
    if (!a_blli && !b_blli) return 1; /* redundant */
    while (a_blli) {
	while (b_blli) {
	    if (match_blli(a_blli,b_blli)) return 1;
	    b_blli = b_blli->next;
	}
	a_blli = a_blli->next;
    }
    return 0;
}


SOCKET *lookup_sap(struct sockaddr_atmsvc *sap)
{
    SOCKET *walk;

    for (walk = sockets; walk; walk = walk->next)
	if (walk->state == ss_listening) {
	    if (!walk->local) break;
	    if (sap->sas_txtp.max_sdu && sap->sas_txtp.max_sdu <
	      walk->local->sas_txtp.max_sdu) continue;
	    if (walk->local->sas_rxtp.max_sdu && sap->sas_rxtp.max_sdu >
	      walk->local->sas_rxtp.max_sdu) continue;
	    /* check PCR compat for CBR @@@ */
	    if (!walk->local->sas_addr.bhli.hl_type &&
	      !walk->local->sas_addr.blli) break;
	    else if (match_sap(walk->local,sap)) break;
	}
    return walk;
}


const char *mid2name(unsigned char mid)
{
    switch (mid) {
	case QMSG_CALL_PROC:
	    return "CALL_PROCEEDING";
	case QMSG_CONNECT:
	    return "CONNECT";
	case QMSG_CONN_ACK:
	    return "CONNECT_ACK";
	case QMSG_SETUP:
	    return "SETUP";
	case QMSG_RELEASE:
	    return "RELEASE";
	case QMSG_REL_COMP:
	    return "REL_COMP";
	case QMSG_RESTART:
	    return "RESTART";
	case QMSG_REST_ACK:
	    return "REST_ACK";
	case QMSG_STATUS:
	    return "STATUS";
	case QMSG_STATUS_ENQ:
	    return "STATUS_ENQ";
	case QMSG_ADD_PARTY:
	    return "ADD_PARTY";
	case QMSG_ADD_PARTY_ACK:
	    return "ADD_PARTY_ACK";
	case QMSG_ADD_PARTY_REJ:
	    return "ADD_PARTY_REJECT";
	case QMSG_DROP_PARTY:
	    return "DROP_PARTY";
	case QMSG_DROP_PARTY_ACK:
	    return "DROP_PARTY_ACK";
	default:
	    return "???";
    }
}


void send_kernel(unsigned long vcc,unsigned long listen_vcc,
  enum atmsvc_msg_type type,int reply,SOCKET *sock)
{
    struct atmsvc_msg *msg;
    struct atm_blli *walk;
    int blli,i;

    blli = 0;
    for (walk = sock && sock->remote ? sock->remote->sas_addr.blli : NULL;
      walk; walk = walk->next) blli++;
    msg = alloc(bllis2msg(blli));
    msg->vcc = vcc;
    msg->listen_vcc = listen_vcc;
    msg->type = type;
    msg->reply = reply;
    memset(&msg->remote,0,sizeof(msg->remote));
    if (sock) msg->pvc = sock->pvc;
    if (sock && sock->remote) {
	msg->remote = *sock->remote;
	i = 0;
	for (walk = sock->remote->sas_addr.blli; walk; walk = walk->next)
	    msg->blli[i++] = *walk;
    }
    to_kernel(msg);
}


void send_release(SOCKET *sock,unsigned char reason,...)
{
    va_list ap;
    Q_DSC dsc;
    int size;
 
    q_create(&dsc,q_buffer,MAX_Q_MSG);
    q_assign(&dsc,QF_msg_type,QMSG_RELEASE);
    q_assign(&dsc,QF_call_ref,sock->call_ref);
    q_assign(&dsc,QF_cause,reason);
    va_start(ap,reason);
    switch (reason) {
	case ATM_CV_TIMER_EXP:
	    {
		char buf[4];

		sprintf(buf,"%d",va_arg(ap,int));
		q_write(&dsc,QF_timer,buf,3);
		break;
	    }
	default:
    }
    va_end(ap);
    size = q_close(&dsc);
    to_signaling(q_buffer,size);
}


void set_error(SOCKET *sock,int code)
{
    if (!sock->error) sock->error = code;
}


void send_close(SOCKET *sock)
{
    if (sock->error == 1234) diag(COMPONENT,DIAG_ERROR,"BUG! BUG! BUG!");
    send_kernel(sock->id,0,as_close,sock->error,NULL);
    sock->error = 1234;
}


void q_report(int severity,const char *msg,...)
{
    va_list ap;

    va_start(ap,msg);
    vdiag("QMSG",severity,msg,ap);
    va_end(ap);
}
