/*
 *  Copyright 2002 Tobias Ringstrom <tobias@ringstrom.mine.nu>
 *  Authentication Copyright 2002 Arcturus Networks Inc.
 *      by Norman Shulman <norm@arcturusnetworks.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <string.h>
#include <arpa/inet.h>

#include "ipsecadm.h"
#include "sa.h"
#include "tunnel.h"

int verbose;

void
usage(void)
{
	fprintf(stderr, "Usage:\n");
	sa_usage();
	tunnel_usage();
	fputs("    ipsecadm stats\n", stderr);
	exit(1);
}

const char*
ipv4_ntoa(uint32_t addr)
{
	struct in_addr a;

	if (addr == INADDR_ANY)
		return "any";

	a.s_addr = addr;

	return inet_ntoa(a);
}

uint32_t
ipv4_aton(const char *str)
{
	if (strcmp(str, "any") == 0)
		return INADDR_ANY;

	return inet_addr(str);
}

uint32_t
strtospi(const char *str)
{
	if (strcmp(str, "any") == 0)
		return IPSEC_SPI_ANY;

	return strtoul(str, NULL, 0);
}

int
ifname_to_ifindex(const char *name)
{
	struct ifreq ifr;
	int fd, st;

	if (strcmp(name, "any") == 0)
		return 0;

	fd = socket(PF_INET, SOCK_DGRAM, IPPROTO_IP);
	if (fd == -1)
	{
		fprintf(stderr, "Error: Cannot open socket\n");
		return -1;
	}

	memset(&ifr, 0, sizeof(struct ifreq));
	strcpy(ifr.ifr_name, name);
	st = ioctl(fd, SIOCGIFINDEX, &ifr);
	close(fd);

	return st == 0 ? ifr.ifr_ifindex : -1;
}

char*
ifindex_to_ifname(char *ifname, int ifindex)
{
	struct ifreq ifr;
	int fd, st;

	if (ifindex == 0)
	{
		strcpy(ifname, "any");
		return ifname;
	}

	fd = socket(PF_INET, SOCK_DGRAM, IPPROTO_IP);
	if (fd == -1)
	{
		fprintf(stderr, "Error: Cannot open socket\n");
		return NULL;
	}

	memset(&ifr, 0, sizeof(struct ifreq));
	ifr.ifr_ifindex = ifindex;
	st = ioctl(fd, SIOCGIFNAME, &ifr);
	close(fd);

	strcpy(ifname, st == 0 ? ifr.ifr_name : "unknown");

	return ifname;
}

int
ipsec_tunnel_open(const char *name, struct ifreq *ifr, int quiet)
{
	int fd, st;

	fd = socket(PF_INET, SOCK_DGRAM, IPPROTO_IP);
	if (fd == -1)
	{
		fprintf(stderr, "Error: Cannot open socket\n");
		return -1;
	}

	memset(ifr, 0, sizeof(struct ifreq));
	strcpy(ifr->ifr_name, name);
	st = ioctl(fd, SIOCGIFHWADDR, ifr);
	if (st != 0)
	{
		if (!quiet)
		{
			int err = errno;
			fprintf(stderr, "Error: Cannot open %s [%s]\n",
					name, strerror(err));
			if (err == ENODEV && strcmp(name, IPSECDEVNAME) == 0)
				fprintf(stderr, "Make sure the kernel module is loaded.\n");
		}
		close(fd);
		return -1;
	}

	if (ifr->ifr_hwaddr.sa_family != ARPHRD_IPSEC)
	{
		if (!quiet)
			fprintf(stderr, "Not an IPsec device!\n");
		close(fd);
		return -1;
	}

	return fd;
}

int
hex2dec(int c)
{
	if (c >= '0' && c <= '9')
		return c - '0';
	if (c >= 'a' && c <= 'f')
		return c - 'a' + 10;
	if (c >= 'A' && c <= 'F')
		return c - 'A' + 10;
	return -1;
}

int
parse_key(const char *str, void *key, int maxsize)
{
	int pos, n1, n0, keysize;
	unsigned char *ckey = key;

	keysize = strlen(str);
	if (keysize % 2 != 0)
	{
		fprintf(stderr, "Error: Key length is not an even number\n");
		return -1;
	}
	keysize /= 2;
	if (keysize > maxsize)
	{
		fprintf(stderr, "Error: Key is too long\n");
		return -1;
	}

	for (pos = 0; pos < keysize; ++pos)
	{
		n1 = hex2dec(str[2*pos]);
		n0 = hex2dec(str[2*pos+1]);
		if (n0 == -1 || n1 == -1)
		{
			fprintf(stderr, "Error: Key contains a non-hexadecimal character\n");
			return -1;
		}
		ckey[pos] = 16 * n1 + n0;
	}

	return keysize;
}

int
read_key_file(const char *filename, void *key, int maxsize)
{
	FILE *f = fopen(filename, "rb");
	int n;

	if (!f)
		return -1;

	n = fread(key, 1, maxsize, f);
	if (n == maxsize && fread(key, 1, maxsize, f) > 0)
	{
		fprintf(stderr, "Error: Key is too long\n");
		fclose(f);
		return -1;
	}

	fclose(f);

	return n;
}

void
opt_error(const char *fmt, ...)
{
	fputs(fmt, stderr);
	fputc('\n', stderr);
	exit(1);
}

int
stats_main(int argc, char *argv[])
{
	struct ipsec_stats stats;
	struct ifreq ifr;
	int fd, st;

	if (argc != 1)
		usage();

	fd = ipsec_tunnel_open(IPSECDEVNAME, &ifr, 0);
	if (fd == -1)
		return -1;

	memset(&stats, 0, sizeof(struct ipsec_stats));
	ifr.ifr_data = (char*)&stats;

	st = ioctl(fd, SIOCIPSEC_GET_STATS, &ifr);
	if (st != 0)
	{
		fprintf(stderr, "Cannot get statistics [%s]\n", strerror(errno));
		return -1;
	}

	printf("Accepted incoming packets %9lu\n", stats.rx_ok);
	printf("Dropped incoming packets  %9lu\n",
		   stats.rx_unknown_sa + stats.rx_auth_fail +
		   stats.rx_padding_fail + stats.rx_non_tunnel +
		   stats.rx_no_tunnel + stats.rx_mem +
		   stats.rx_other);
	printf("    Unknown SA            %9lu\n", stats.rx_unknown_sa);
	printf("    Failed authentication %9lu\n", stats.rx_auth_fail);
	printf("    Bad padding           %9lu\n", stats.rx_padding_fail);
	printf("    Non-tunnel mode       %9lu\n", stats.rx_non_tunnel);
	printf("    No matching tunnel    %9lu\n", stats.rx_no_tunnel);
	printf("    Memory shortage       %9lu\n", stats.rx_mem);
	printf("    Other errors          %9lu\n", stats.rx_other);

	printf("Accepted outgoing packets %9lu\n", stats.tx_ok);
	printf("Dropped outgoing packets  %9lu\n",
		   stats.tx_unknown_sa + stats.tx_recursion +
		   stats.tx_route + stats.tx_mtu +
		   stats.tx_mem + stats.tx_other);
	printf("    Unknown SA            %9lu\n", stats.tx_unknown_sa);
	printf("    Recursion             %9lu\n", stats.tx_recursion);
	printf("    No route              %9lu\n", stats.tx_route);
	printf("    MTU / packet too big  %9lu\n", stats.tx_mtu);
	printf("    Memory shortage       %9lu\n", stats.tx_mem);
	printf("    Other errors          %9lu\n", stats.tx_other);

	return 0;
}

int
main(int argc, char *argv[])
{
	if (argc < 2)
		usage();

	if (strcmp(argv[1], "sa") == 0)
		return sa_main(argc - 1, argv + 1);

	if (strncmp(argv[1], "tunnel", strlen(argv[1])) == 0)
		return tunnel_main(argc - 1, argv + 1);

	if (strncmp(argv[1], "stats", strlen(argv[1])) == 0)
		return stats_main(argc - 1, argv + 1);

	usage();

	return 1;
}
