/* ------------------------------------------------------------------------- */
/* 									     */
/* i2c.c - a device driver for the iic-bus interface			     */
/* 									     */
/* ------------------------------------------------------------------------- */
/*   Copyright (C) 1995 Simon G. Vogl

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		     */
/* ------------------------------------------------------------------------- */
/* TODO:								     */
/*		basic functionality established. To be implemented:	     */
/* 		o 10 bit adress support (another device driver?)	     */
/*		o combined write/read with repeated start condition	     */
/*		o adaptive transfer-rate (ioctl)			     */
/*		  -> sense when slow slave holds down scl-line		     */
/*		  including timeout sensing.				     */
/*		o slow start (start byte for uCs) ->ioctl		     */
/*		  + other limits in slo-mode(6 bytes max &c).		     */
/*		o multi-master support (only if needed)			     */
/*		  -> arbitration at the beginning of transmission: when a    */
/*		  two masters simoultanously write on the bus, the one, who  */
/*		  first produces a '1', when the other produces a '0' will   */
/*		  lose the bus, and must wait until transmission terminates. */
/*		  -> clock signals are synchronized due to wired-and con-    */
/*		  nection of the devices.				     */
/*		o split low-level routines into separate file to allow for   */
/*		  different hardware adapters (currently only parallel port  */
/*		  see adapter.txt; see 'Elektor', 3-92 for article on inter- */
/*		  face with a PCD8584)					     */
/*									     */
/*	Any comments, especially on the returned error codes & the ioctls    */
/*	heavily appreciated						     */
/* 									     */
/* 1995-03-08 Functioning version. The videotext-decoder finally works.	     */
/*		I do not know, if the busy-flag is necessary, since the      */
/*		arbitration procedure should work even between tasks.	     */
/*		Sometimes I experience the problem, that the port cannot be  */
/*		accessed - i.e. scl stays low (~0.8V) - and I do not know    */
/*		why - maybe hardware problems :-(			     */
/* 1995-01-26 Trying to adapt to very slow devices ( just got some EEPROMS)  */
/*		v0.12 rewritten to 1 minor / port			     */
/* 1995-01-14 v0.11 minor cleanups 					     */
/*		o implemented speed control				     */
/*		o arbitration 						     */
/*		o notice: ioctl is under construction and will NOT fonction  */
/*		    ( could not test it yet :-( )			     */
/* 1994-12-20 v0.1 first functioning version.				     */
/* 1994-11-30 created file. sigv					     */
/* 		Adapted code from linux/drivers/char/lp.c		     */
/*		Currently does not support 10byte adresses (read)	     */
/* ------------------------------------------------------------------------- */

#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/malloc.h>
#include <linux/module.h>
#include <linux/delay.h>
#include <asm/io.h>
#include <asm/system.h>

#include "i2c.h"
#include "kernel-version.h"

#define DEB(x)		/* should be reasonable open, close &c. */
#define DEB2(x) 	/* low level debugging - very slow */
#define DEB3(x) 
#define DEBE(x)	x	/* error messages */
/* debugging - slow down transfer to have a look at the data .. */
/*
#define SLO_IO      jif=jiffies;while(jiffies<=jif+i2c_table[minor].veryslow);
*/
#ifdef SLO_IO
	int jif;
#endif

/* ----- globals ----------------------------------------------------------- */

#define WHI	10
#define WLO	10

#define I2C_PORTS 1


/* ----- CHange here for other ports	 */
portinfo i2c_table[I2C_PORTS] = {
	/* base flag time udly mdly rtry vslo */
	{ 0x278,   0,  50,   5,   0,   5, 10,},
};

/* ------------------------------------------------------------------------- */

/* ----- level 1: controlling the i2c bus 	----- */

/* lo and hi correspond to the voltage levels on the i2c line ;) */

void scllo(int minor){
	outb_p(I2C_CMASK, CTRL); 
	udelay(WLO);
#ifdef SLO_IO
	SLO_IO
#endif 
	udelay(i2c_table[minor].udelay);
}

int sclhi(int minor){
	int start=jiffies;

	outb_p(I2C_SCL, CTRL); 
	udelay(WHI);
	udelay(i2c_table[minor].udelay);
	while (!inb_p(STAT)&0x10) {
		if (start+i2c_table[minor].timeout<=jiffies){
			DEBE(printk("sclhi failed\n"));
			return -ETIMEDOUT;
		}
	}
#ifdef SLO_IO
	SLO_IO
#endif 
	return 0;
} /* ( scl signal is inverted in philips adapter. ) */

void sdalo(int minor){
	outb_p(I2C_SDA , DATA);
        outb_p(I2C_SDA , DATA);
	udelay(WHI);
#ifdef SLO_IO
	SLO_IO
#endif 
}

void sdahi(int minor){
	outb_p(I2C_DMASK , DATA);
        outb_p(I2C_DMASK , DATA);
	udelay(WHI);
#ifdef SLO_IO
	SLO_IO
#endif 
}

#define sdain(minor) ( 0 != ( (inb_p(STAT)) & I2C_SDA ) )


/* --------------------------------------------	*/
/* for exact timings see: 			*/
/* 'The I2C-bus and how to use it', by Philips 	*/
/* Semiconductors, jan.'92			*/

void i2c_start(int minor) {
	DEB2(printk("i2c_start.\n"));
	sdalo(minor);
	udelay(WHI);		/* i.e. 4.7 usec */
	scllo(minor);
	udelay(WLO);		/* i.e. 4.7 usec */
}

void i2c_stop(int minor) {
	DEB2(printk("i2c_stop.\n"));
	/* assert: scl is low */
	sdalo(minor);
/*	sclhi(minor); gcc 2.5.8, kernel 1.1.79: got error with asm dir.	*/
	outb_p(I2C_SCL, CTRL); 	
	udelay(WHI);		
	sdahi(minor);
	udelay(WHI);		/* min. 4.7 u between stop to next start */
}

int i2c_outb(int minor, char c){ /*send byte via i2c port, without start/stop*/
	int i;
	char ack;
	
	/* assert: scl is low */
	DEB2(printk("i2c_outb:%x\n",c));
	for ( i=7 ; i>=0 ; i-- ) {
		if ( c&(1<<i) ){
			sdahi(minor);
		}else{		
			sdalo(minor);}
		if (sclhi(minor)<0){ /* timeout */
			sdahi(minor); /* we don't want to block the net */
			return 0;
		};
		/* arbitrate here: */
		udelay(WHI);
		if ( (c&(1<<i)) && !sdain(minor) ){
			/* we lost the arbitration process -> get lost */
			DEB(printk("i2c_write: arbitration bailout! \n"));
			return 0;
		}
		scllo(minor);	
	}
	sdahi(minor);
	if (sclhi(minor)<0){ /* timeout */
		return 0;
	};
	/* read ack */
	udelay(WHI);
	ack=!sdain(minor);	/* ack: sda is pulled low ->success. 	*/
	DEB3(printk("sdain: 0x%x\n",sdain(minor)));
	scllo(minor);
	return ack;		/* return 1 if ok 		*/
	/* assert: scl is low */
}

char i2c_inb(int minor) {
	/* read byte via i2c port, without start/stop sequence	*/
	/* acknowledge is sent in i2c_read.			*/
	/* assert: scl is low */
	int i;
	char data;

	DEB2(printk("i2c_inb.\n"));

	sdahi(minor); /* just to be sure ... */
	data=0;
	for (i=0;i<8;i++){
		if (sclhi(minor)<0){ /* timeout */
			sdahi(minor); /* we don't want to block the net */
			return 0;
		};
		data*=2;
		if ( sdain(minor) ) data++;
		scllo(minor);
	}
	/* assert: scl is low */
	return data;
}


/* ------------------------------------------------------------------------- */

/* ----- level 2: communicating with the kernel	----- */

static int i2c_lseek(struct inode * inode, struct file * file,
		    off_t offset, int origin) {
	return -ESPIPE;
}

/* ------------------------------------	*/
static int i2c_open(struct inode * inode, struct file * file) {
	unsigned int minor = MINOR(inode->i_rdev);

	if (minor >= 128) /* 7-bit adressing */
		return -ENODEV;	
	
/* More than one prg may use port - this may change 
	if (i2c_table[minor].flags & P_BUSY ) 
		return -EBUSY;
	i2c_table[minor].flags |= P_BUSY; 	*/

	if (I2C_PWON(minor)==0) /* bus switched off */
		return -ENODEV;
#ifdef MODULE
	MOD_INC_USE_COUNT;
#endif	
	DEB(printk("i2c_open: i2c%d\n",minor));
	return 0;
}
/* ------------------------------------	*/
static void i2c_release(struct inode * inode, struct file * file) {
/*	unsigned int minor = MINOR(inode->i_rdev);
*/
	DEB(printk("i2c_close: i2c%d\n",minor));

#ifdef MODULE
	MOD_DEC_USE_COUNT;
#endif		
}
/* ------------------------------------	*/
static int i2c_write(struct inode * inode, struct file * file,
			   char * buf, int count)
{
	unsigned int minor = MINOR(inode->i_rdev);
	char c, *temp = buf;
	int retval,i;

	DEB(printk("i2c_write: i2c%d writing %d bytes.\n",minor,count));

	if (minor>=I2C_PORTS){
		DEB2(printk("i2c_write: minor %d invalid\n",minor));
		return -EINVAL;
	}
	if (i2c_table[minor].flags&P_SLOW && count>7 )
		return -EFBIG;
	if (i2c_table[minor].flags&P_BUSY) return -EBUSY;
	i2c_table[minor].flags |= P_BUSY;
 
	i2c_start(minor);
	/* first byte always is adress! */
	i=0;
	while (!i2c_outb(minor,get_fs_byte(buf))) { 
		i2c_stop(minor);
		i++;
		i2c_start(minor);
		if (i>=i2c_table[minor].retries )  {
			DEBE(printk("i2c_read: i2c%d: address ack failed.\n",minor));
			i2c_stop(minor);
			i2c_table[minor].flags&=!P_BUSY;
			return -EREMOTEIO;
		};
	}
	count--;
	temp++;
	
	while (count > 0) {
		c = get_fs_byte(temp);

		DEB2(printk("i2c(%d)_write: writing %c\n",minor,c));

		retval = i2c_outb(minor,c);
		if (retval>0) { 
			count--; 
			temp++;
		} else { /* arbitration or no acknowledge */
			DEBE(printk("i2c(%d)_write: error - bailout.\n",minor));
			i2c_stop(minor);
			i2c_table[minor].flags&=!P_BUSY;
			return -EREMOTEIO; /* got a better one ?? */
		}
		__delay(i2c_table[minor].mdelay * (loops_per_sec / 1000) );
	}
	i2c_stop(minor);
	i2c_table[minor].flags&=!P_BUSY;
	DEB2(printk("i2c_write: wrote %d bytes.\n",temp-buf));
	return temp-buf ;
 }

/* ------------------------------------	*/
static int i2c_read(struct inode * inode, struct file * file,
			char * buf, int count) {	
	unsigned int minor = MINOR(inode->i_rdev);
	char *temp = buf;
	int i;

	DEB(printk("i2c_read: i2c%d reading %d bytes.\n",minor,count));

	if (minor>=I2C_PORTS){
		DEB2(printk("i2c_write: minor %d invalid\n",minor));
		return -EINVAL;
	}
	if (i2c_table[minor].flags&P_BUSY)
		return -EBUSY;
	i2c_table[minor].flags |= P_BUSY;

	i2c_start(minor);
	/* device adress must be in buf[0] - could change to a ioctl. !! */

	i=0;
        while (!i2c_outb(minor,get_fs_byte(buf)|0x01)) {
                i2c_stop(minor);
                i++;
                if (i>=i2c_table[minor].retries )  {
                        DEBE(printk("i2c_read: i2c%d: address ack failed.\n",minor));
                        i2c_stop(minor);
                        i2c_table[minor].flags&=!P_BUSY;
                        return -EREMOTEIO;
                };
                i2c_start(minor);
        }

	 
	while (count > 0) {
		put_user_byte(i2c_inb(minor),temp);	/* byte in	*/

		if (count==1){		/* send ack	*/
			sdahi(minor);
		}else{
			sdalo(minor);		/* reverse ack on last byte */
		}
		if (sclhi(minor)<0) { /* timeout */
			sdahi(minor);
			return 0;
		};
		scllo(minor);
		sdahi(minor);
		temp++;
		count--;
	}

	i2c_stop(minor);
	DEB(printk("i2c_read: %d bytes read.\n",temp-buf));
	i2c_table[minor].flags&=!P_BUSY;
	return temp-buf ;
}

/* ------------------------------------	*/
static int i2c_ioctl(struct inode *inode, struct file *file,
		    unsigned int cmd, unsigned long arg)
{
	unsigned int minor = MINOR(inode->i_rdev); 
	int retval = 0;

#ifdef I2C_DEBUG
	printk("i2c ioctl, cmd: 0x%x, arg: 0x%x\n", cmd, arg);
#endif
	switch ( cmd ) {
		case I2C_TIMEOUT:
			i2c_table[minor].timeout = arg;
			break;
#ifdef SLO_IO
		case I2C_V_SLOW: /* only for debugging! This will go away */
			i2c_table[minor].veryslow = arg;
#endif
		case I2C_RESET:
			outb_p(I2C_SCL, CTRL); 
        		outb_p(I2C_DMASK , DATA);
		case I2C_UDELAY:
			i2c_table[minor].udelay = arg;
		case I2C_MDELAY:
			i2c_table[minor].mdelay = arg;
		case I2C_RETRIES:
			i2c_table[minor].retries = arg;
		case I2C_CLEAR:
			i2c_table[minor].flags = 0 ;
/*		case I2C_WR_N:
			i2c_table[minor].wrbytes = arg ;
*/		default:
			retval = -EINVAL;
	}
	return retval;
}

/* ------------------------------------	*/
static struct file_operations i2c_fops = {
	i2c_lseek,
	i2c_read,
	i2c_write,
	NULL,		/* i2c_readdir */
	NULL,		/* i2c_select */
	i2c_ioctl,
	NULL,		/* i2c_mmap */
	i2c_open,
	i2c_release
};

void i2c_reset(int minor){
/*	outb_p(I2C_SCL, CTRL); 
	outb_p(I2C_SCL, CTRL); 
        outb_p(I2C_DMASK , DATA);
	outb_p(I2C_DMASK , DATA);
*/
	sdahi(minor);
	sclhi(minor);
	outb(I2C_SCL, CTRL); 
	outb(I2C_SCL, CTRL); 
}

/* ------------------------------------	*/
int init_module(void)
{
	int i;
	if (register_chrdev(I2C_MAJOR,"i2c",&i2c_fops)) {
		printk("i2c.o: unable to get major %d for i2c bus\n",I2C_MAJOR);
		printk("i2c.o:     Change the major number in i2c.h to a value\n");
		printk("i2c.o:     that is not used by other modules (See /usr/include/linux/major.h).\n");
		return -EIO;
	}

	/* no deteecting, only init. 	*/
	/* detect philips adapter:  	*/
	/* check PWON flag. If set check SDA, SCL (maybe several times) ? */

	for (i=0;i<I2C_PORTS;i++){
		i2c_reset(i);
		printk("i2c_init: i2c using port %#x.\n",i2c_table[i].base);
	}
	return 0;
}

void cleanup_module(void)
{
       if(MOD_IN_USE)
               printk("i2c: busy - remove delayed\n");
       else
               unregister_chrdev(I2C_MAJOR,"i2c");
}
