/* ------------------------------------------------------------------------- */
/* i2c.c - a device driver for the iic-bus interface			     */
/* ------------------------------------------------------------------------- */
/*   Copyright (C) 1995-96 Simon G. Vogl

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		     */
/* ------------------------------------------------------------------------- */
#define RCSID "$Id: i2c_main.c,v 1.3 1996/07/04 20:37:26 i2c Exp i2c $"
/* ------------------------------------------------------------------------- 
 * $Log: i2c_main.c,v $
 * Revision 1.3  1996/07/04 20:37:26  i2c
 * added init failure behaviour
 *
 * Revision 1.2  1996/03/28 20:35:22  i2c
 * implemented multi-adapter support
 *
 * Revision 1.4  1996/03/28 13:48:32  i2c
 * made private functions static
 *
 * Revision 1.3  1996/03/25 22:36:25  i2c
 * added checks for magic numbers, new ioctl
 *
 * Revision 1.2  1996/03/25 16:38:48  i2c
 * cleaned up code
 *
 * Revision 1.1  1996/03/15 09:06:07  root
 * Initial revision
 *
 -------------------------------------------------------------------------- */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/malloc.h>

#include "i2c.h"
#include "i2c_priv.h"
 
/* ----- global defines ---------------------------------------------------- */
#define DEB(x)		/* should be reasonable open, close &c. 	*/
#define DEB2(x) 	/* low level debugging - very slow 		*/
#define DEBE(x)	x	/* error messages 				*/
#define DEBI(x) 	/* ioctl and its arguments 			*/

#define CHK(data,MSG,RET)	if ((data->magic)!=I2C_MAGIC) {\
				printk("i2c magic failed:" MSG "\n");\
				return RET;}

#define I2C_ADAP(minor) 	i2c_adap[i2c_table[minor].flags & P_TMASK]
/* ----- global variables -------------------------------------------------- */

/* ----- Port list:  */

#define I2C_PORTS 1
struct i2c_struct i2c_table[I2C_PORTS] = {
	/* base 	flag time udly mdly rtry vslo */
	{ 0x278, ADAP_PHIL  ,  50,   5,   0,   5, 10,},
/*	{ 0x378, ADAP_LPPCF ,  50,   5,   0,   5, 10,},*/
};

struct i2c_opns *i2c_adap[]={		/* indexed by the types in i2c_priv  */
	&philops,	&lppcfops,	&illegops,	&illegops,
	&illegops,	&illegops,	&illegops,	&illegops,
};

/* ----- local functions --------------------------------------------------- */

static int i2c_lseek(struct inode * inode, struct file * file,
	off_t offset, int origin) 
{
	return -ESPIPE;
}


static int i2c_open(struct inode * inode, struct file * file) 
{
	unsigned int minor = MINOR(inode->i_rdev);
	struct i2c_data *data;

	if (minor >= I2C_PORTS || !(i2c_table[minor].flags & P_EXIST) ) {
		DEBE(printk("i2c: %s\n",(i2c_table[minor].flags & P_EXIST) ?
			"minor exceeded ports":"port not existant"));
		return -ENODEV;	
	}
#ifdef MODULE
	MOD_INC_USE_COUNT;
#endif
	data=kmalloc (sizeof(struct i2c_data),GFP_KERNEL);
	if (data == NULL) {
		DEBE(printk("i2c: failed to allocate local data"));
		return -ENOMEM;
	}
	data->magic	= I2C_MAGIC;
	data->address	= 0x00;
	data->flags	= 0x00;
	file->private_data = data;

	I2C_ADAP(minor)->open(inode,file);

	DEB(printk("i2c_open: i2c%d\n",minor));
	return 0;
}


static void i2c_release (struct inode * inode, struct file * file) 
{
	unsigned int minor = MINOR(inode->i_rdev);
	struct i2c_data *data;

	data=(struct i2c_data *)file->private_data;
	CHK(data,"i2c_close",/**/);
	kfree(data);
	file->private_data=NULL;

	i2c_table[minor].flags|=P_EXIST;/* we reset all values.*/

	DEB(printk("i2c_close: i2c%d\n",minor));
#ifdef MODULE
	MOD_DEC_USE_COUNT;
#endif    
}

static int i2c_write(struct inode * inode, struct file * file,
	const char * buf, int count)
{
	unsigned int minor = MINOR(inode->i_rdev);
	struct i2c_data *data;
	int ret;

	if (minor>=I2C_PORTS){
		DEBE(printk("i2c_write: minor %d invalid\n",minor));
		return -EINVAL;
	}
	data=(struct i2c_data *)file->private_data;
	CHK(data,"i2c_write",(-EINVAL));

	DEB(printk("i2c_write: i2c%d writing %d bytes.\n",minor,count));

	if (i2c_table[minor].flags & P_SLOW && count > 7 )
		return -EFBIG;
	if (i2c_table[minor].flags & P_BUSY) 
		return -EBUSY;
	i2c_table[minor].flags |= P_BUSY;

	ret = I2C_ADAP(minor)->write(inode,file,buf,count);

	i2c_table[minor].flags ^= P_BUSY;
	return ret;
}

static int i2c_read(struct inode * inode, struct file * file,
	char * buf, int count) 
{
	unsigned int minor = MINOR(inode->i_rdev);
	struct i2c_data *data;
	int ret;
	
	data=(struct i2c_data *)file->private_data;
	CHK(data,"i2c_read",(-EINVAL));
	DEB(printk("i2c_read: i2c%d reading %d bytes.\n",minor,count));

	if (minor>=I2C_PORTS){
		DEBE(printk("i2c_write: minor %d invalid\n",minor));
		return -EINVAL;
	}
	if (i2c_table[minor].flags&P_BUSY)
		return -EBUSY;
	i2c_table[minor].flags |= P_BUSY;

	ret = I2C_ADAP(minor)->read(inode,file,buf,count);

	i2c_table[minor].flags ^= P_BUSY;
	return ret;
}

static int i2c_ioctl(struct inode *inode, struct file *file,
	unsigned int cmd, unsigned long arg)
{
	unsigned int minor = MINOR(inode->i_rdev);
	struct i2c_data *data;
	int ret = 0;

	DEBI(printk("i2c ioctl, cmd: 0x%x, arg: %#lx\n", cmd, arg));
	data=(struct i2c_data*)file->private_data;
	CHK(data,"i2c_ioctl",(-EINVAL));	
	switch ( cmd ) {
		case I2C_RETRIES:
			i2c_table[minor].retries = arg;
			break;
		case I2C_SLAVE:
			data->address = arg%256;
			arg &= 0xfeff;
			if ( arg >= 0xf000 && arg <0xf700) {
				data->flags |= F_LONG ;
				data->flags &= (0xfffc) | ( (arg>>9) & 0x03 );
			}
			break;
		default:
			ret = I2C_ADAP(minor)->ioctl(inode,file,cmd,arg);
	}
	return ret;
}



/* ----- module functions ---------------------------------------------	*/
static struct file_operations i2c_fops = {
	i2c_lseek,
	i2c_read,
	i2c_write,
	NULL,  			/* i2c_readdir	*/
	NULL,			/* i2c_select 	*/
	i2c_ioctl,
	NULL,			/* i2c_mmap 	*/
	i2c_open,
	i2c_release
};

int init_module(void) {
	int i,success=0;

	if (register_chrdev(I2C_MAJOR,"i2c",&i2c_fops)) {
		printk("i2c: unable to get major %d for i2c bus\n",I2C_MAJOR);
		return -EIO;
	}

	for (i=0;i<I2C_PORTS;i++){
		if (!I2C_ADAP(i)->init(i)){
			i2c_table[i].flags |= P_EXIST;
			printk("i2c: i2c%d at port %#x\n",i,i2c_table[i].base);
			success++;
		} else {
			if (!success)	/* None worked */
				return -1;
		}
	}
	printk("i2c version: " RCSID "\n");
	return 0;
}

void cleanup_module(void) {
	int i;
	if(MOD_IN_USE) {
		printk("i2c: busy - remove delayed\n");
 		return;
	}
	for (i=0;i<I2C_PORTS;i++){
		if (i2c_table[i].flags & P_EXIST)
			I2C_ADAP(i)->exit(i);
	}
	unregister_chrdev(I2C_MAJOR,"i2c");
}
