/* ------------------------------------------------------------------------- */
/* i2c.c - a device driver for the iic-bus interface			     */
/* ------------------------------------------------------------------------- */
/*   Copyright (C) 1995-96 Simon G. Vogl

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.		     */
/* ------------------------------------------------------------------------- */
#define RCSID "$Id: i2c_main.c,v 2.2 1996/11/18 22:24:45 i2c Exp $"
/* ------------------------------------------------------------------------- 
 * $Log: i2c_main.c,v $
 * Revision 2.2  1996/11/18 22:24:45  i2c
 * removed warning message
 *
 * Revision 2.1  1996/11/17 14:02:36  i2c
 * Adapted i2c_main to new module layout.
 *
 * Revision 1.4  1996/07/23 20:46:46  i2c
 * bugfix: unregister major number when failing
 *
 * Revision 1.3  1996/07/04 20:37:26  i2c
 * added init failure behaviour
 *
 * Revision 1.2  1996/03/28 20:35:22  i2c
 * implemented multi-adapter support
 *
 * Revision 1.4  1996/03/28 13:48:32  i2c
 * made private functions static
 *
 * Revision 1.3  1996/03/25 22:36:25  i2c
 * added checks for magic numbers, new ioctl
 *
 * Revision 1.2  1996/03/25 16:38:48  i2c
 * cleaned up code
 *
 * Revision 1.1  1996/03/15 09:06:07  root
 * Initial revision
 *
 -------------------------------------------------------------------------- */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/malloc.h>

#include "i2c.h"
#include "i2c_priv.h"
 
/* ----- global defines ---------------------------------------------------- */
#define DEB(x)		/* should be reasonable open, close &c. 	*/
#define DEB2(x) 	/* low level debugging - very slow 		*/
#define DEBE(x)	x	/* error messages 				*/
#define DEBI(x) 	/* ioctl and its arguments 			*/

#define CHK(data,MSG,RET)	if ((data->magic)!=I2C_MAGIC) {\
				printk("i2c magic failed:" MSG "\n");\
				return RET;}

#define I2C_ALG(minor) 	(i2c_alg[i2c_table[minor].flags & ALG_MASK >> ALG_SHIFT])
/* ----- global variables -------------------------------------------------- */

/* Port list:  */
#define I2C_PORTS 1
struct i2c_struct i2c_table[] = {
	/* base 	flags      time  udly mdly rtry vslo */
	{ 0x378, ALG_BIT | IDX_B_LP ,  50,   5,   0,   5, 10,},
	{ 0x278, ALG_BIT | IDX_B_LP ,  50,   5,   0,   5, 10,},
};

/* */
struct i2c_opns *i2c_alg[]={	
	&illegops,
#if (BITADAPS)
	&alg_bit_opns,
#else
	&illegops,
#endif
#if (PCFADAPS)
	&alg_pcf_opns,
#else
	&illegops,
#endif
	&illegops,
};

/* ----- local functions --------------------------------------------------- */

static long long i2c_lseek(struct inode * inode, struct file * file,
	long long offset, int origin) 
{
	/*
	 * As a character device, we do not seek ...
	 * Although it might not be a bad idea to implement a
	 * device detect function without ioctl. But this would really
	 * be rather esoteric ;)
	 */
	return -ESPIPE;	
}


static int i2c_open(struct inode * inode, struct file * file) 
{
	unsigned int minor = MINOR(inode->i_rdev);
	struct i2c_data *data;

	if (minor >= I2C_PORTS || !(i2c_table[minor].flags & P_EXIST) ) {
		DEBE(printk("i2c: %s\n",(i2c_table[minor].flags & P_EXIST) ?
			"minor exceeded ports":"port does not exist"));
		return -ENODEV;	
	}
	data=kmalloc (sizeof(struct i2c_data),GFP_KERNEL);
	if (data == NULL) {
		DEBE(printk("i2c: failed to allocate local data"));
		return -ENOMEM;
	}
	data->magic	= I2C_MAGIC;
	data->address	= 0x00;
	data->flags	= 0x00;
	file->private_data = data;

	I2C_ALG(minor)->open(inode,file);

#ifdef MODULE
	MOD_INC_USE_COUNT;
#endif

	DEB(printk("i2c_open: i2c%d\n",minor));

	return 0;
}


static void i2c_release (struct inode * inode, struct file * file) 
{
	unsigned int minor = MINOR(inode->i_rdev);
	struct i2c_data *data;

	data=(struct i2c_data *)file->private_data;
	CHK(data,"i2c_close",/**/);
	kfree(data);
	file->private_data=NULL;

	i2c_table[minor].flags|=P_EXIST;/* we reset all values.*/

	DEB(printk("i2c_close: i2c%d\n",minor));
#ifdef MODULE
	MOD_DEC_USE_COUNT;
#endif    
}

static long i2c_write(struct inode * inode, struct file * file,
	const char * buf, unsigned long count)
{
	unsigned int minor = MINOR(inode->i_rdev);
	struct i2c_data *data;
	int ret;

	if (minor>=I2C_PORTS){
		DEBE(printk("i2c_write: minor %d invalid\n",minor));
		return -EINVAL;
	}
	data=(struct i2c_data *)file->private_data;
	CHK(data,"i2c_write",(-EINVAL));

	DEB(printk("i2c_write: i2c%d writing %d bytes.\n",minor,count));

	if (i2c_table[minor].flags & P_SLOW && count > 7 )
		return -EFBIG;
	if (i2c_table[minor].flags & P_BUSY) 
		return -EBUSY;
	i2c_table[minor].flags |= P_BUSY;

	ret = I2C_ALG(minor)->write(inode,file,buf,count);

	i2c_table[minor].flags ^= P_BUSY;
	return ret;
}

static long i2c_read(struct inode * inode, struct file * file,
	char * buf, unsigned long count) 
{
	unsigned int minor = MINOR(inode->i_rdev);
	struct i2c_data *data;
	int ret;
	
	data=(struct i2c_data *)file->private_data;
	CHK(data,"i2c_read",(-EINVAL));
	DEB(printk("i2c_read: i2c%d reading %d bytes.\n",minor,count));

	if (minor>=I2C_PORTS){
		DEBE(printk("i2c_write: minor %d invalid\n",minor));
		return -EINVAL;
	}
	if (i2c_table[minor].flags&P_BUSY)
		return -EBUSY;
	i2c_table[minor].flags |= P_BUSY;

	ret = I2C_ALG(minor)->read(inode,file,buf,count);

	i2c_table[minor].flags ^= P_BUSY;
	return ret;
}

static int i2c_ioctl(struct inode *inode, struct file *file,
	unsigned int cmd, unsigned long arg)
{
	unsigned int minor = MINOR(inode->i_rdev);
	struct i2c_data *data;
	int ret = 0;

	DEBI(printk("i2c ioctl, cmd: 0x%x, arg: %#lx\n", cmd, arg));
	data=(struct i2c_data*)file->private_data;
	CHK(data,"i2c_ioctl",(-EINVAL));	
	switch ( cmd ) {
		case I2C_RETRIES:
			i2c_table[minor].retries = arg;
			break;
		case I2C_SLAVE:
			data->address = arg%256;
			arg &= 0xfeff;
			if ( arg >= 0xf000 && arg <0xf700) {
				data->flags |= F_LONG ;
				data->flags &= (0xfffc) | ( (arg>>9) & 0x03 );
			}
			break;
		default:
			ret = I2C_ALG(minor)->ioctl(inode,file,cmd,arg);
	}
	return ret;
}



/* ----- module functions ---------------------------------------------	*/
static struct file_operations i2c_fops = {
	i2c_lseek,
	i2c_read,
	i2c_write,
	NULL,  			/* i2c_readdir	*/
	NULL,			/* i2c_select 	*/
	i2c_ioctl,
	NULL,			/* i2c_mmap 	*/
	i2c_open,
	i2c_release
};

int init_module(void) {
	int i;
	int success=0;

	if (register_chrdev(I2C_MAJOR,"i2c",&i2c_fops)) {
		printk("i2c: unable to get major %d for i2c bus\n",I2C_MAJOR);
		return -EIO;
	}
	/*  Now init the algorithm parts: */
#if (BITADAPS)
	alg_bit_init();
#endif
#if (PCFADAPS)
	alg_pcf_init();
#endif
	for (i=0;i<I2C_PORTS;i++) {
		if (!(I2C_ALG(i)->init(i)) ) {
			i2c_table[i].flags |= P_EXIST;
			printk("i2c: i2c%d at port %#x\n",i,i2c_table[i].base);
			success++;
		}
	}
	if (!success) {
		unregister_chrdev(I2C_MAJOR,"i2c");
		printk("Nope. Could not find any.\n");
		return -ENODEV;
	}
	printk("i2c version: " RCSID " initialized.\n");
	return 0;
}

void cleanup_module(void) {
	int i;

	if(MOD_IN_USE) {
		printk("i2c: busy - remove delayed\n");
 		return;
	}

	for (i=0;i<I2C_PORTS;i++){
		if (i2c_table[i].flags & P_EXIST)
			I2C_ALG(i)->exit(i);
	}
	unregister_chrdev(I2C_MAJOR,"i2c");

}
