#ifndef _ERRLOG_H_
#define _ERRLOG_H_

#define ERR_PANIC 0             /* Catastrophic failure */
#define ERR_INTERNAL 1          /* Internal inconsistency */
#define ERR_FAILURE 2           /* General failure */
#define ERR_PROTERR 3           /* Protocol error received */
#define ERR_STATUS 4            /* Status reports */
#define ERR_ALL 5               /* Print everything */

void 
ErrLogInit(void (*errlog)(const char *str), int level);

#ifdef __GNUC__
void _ErrLog(const char *filename, int line, int level, const char *s, ...);
#define ErrLog(level, format, args...) \
     _ErrLog(__FILE__, __LINE__, level, format , ## args)
#else 
void ErrLog(int level, const char *s, ...);
#endif

/*****************************************************************************/
/*                                                                           */
/*  Here the debugging functions begin                                       */
/*                                                                           */
/*****************************************************************************/
#if DEBUG > 0

#include <stdio.h>
void DebugPrintf(const char *s, ...);
void AssertPrintf(const char *file, int line, const char *s, ...);

/* All assertion functions jump here, you can put a breakpoint here. */
void AssertFailed(const char *s, int line);
#endif /* DEBUG > 0 */


/*****************************************************************************/
/*                                                                           */
/*  Insert a PathTest for every if/else and for every loop                   */
/*  Insert a Trap or TrapNot where you do not want the computer to go        */
/*                                                                           */
/*****************************************************************************/
#if DEBUG > 0
#define Trap(t)          ((t)?AssertFailed(__FILE__,__LINE__):)
#define TrapNot(t)       ((t)?:AssertFailed(__FILE__,__LINE__))
#define PathTest         AssertPrintf(__FILE__,__LINE__,"PathTest\n")
#else
#define Trap(t)
#define TrapNot(t)
#define PathTest
#endif

/*****************************************************************************/
/*                                                                           */
/*  Fill Memory with Garbage that we will notice                             */
/*                                                                           */
/*****************************************************************************/
#if DEBUG > 0
#include <string.h>
#define MemGarbage(p,len) (memset(p,0xfe,len))
#else
#define MemGarbage(p,len)
#endif /* DEBUG > 0 */


/*****************************************************************************/
/*                                                                           */
/*  General Assertion                                                        */
/*                                                                           */
/*****************************************************************************/
#if DEBUG > 0
#include <assert.h>
#define Assert(e) assert(e)     /* Just to be the same everywhere            */
#else
#define Assert(e)
#endif /* DEBUG > 0 */

/*****************************************************************************/
/*                                                                           */
/*  AssertStr(s) makes sure s is a String                                    */
/*                                                                           */
/*****************************************************************************/
#if DEBUG > 0
void _AssertStr(const char *f, int l, const char *s);
#define AssertStr(s) _AssertStr(__FILE__, __LINE__, s)
#else
#define AssertStr(s)
#endif /* DEBUG > 0 */

/*****************************************************************************/
/*                                                                           */
/*  AssertPrintStr(s) makes sure s is a String with len < 64k                */
/*                                                                           */
/*****************************************************************************/
#if DEBUG > 0
void _AssertPrintStr(const char *f, int l, const char *s);
#define AssertPrintStr(s) _AssertPrintStr(__FILE__, __LINE__, s)
#else
#define AssertPrintStr(s)
#endif /* DEBUG > 0 */

/*****************************************************************************/
/*                                                                           */
/*  AssertPtr(p) makes sure p is != NULL                                     */
/*                                                                           */
/*****************************************************************************/
#if DEBUG > 0
void _AssertPtr(const char *f, int l, const void *const p);
#define AssertPtr(p) _AssertPtr(__FILE__, __LINE__, p)
#else
#define AssertPtr(p)
#endif /* DEBUG > 0 */


#endif /* _ERRLOG_H_ */
