#include "inode.h"
#include "errlog.h"
#include "pstr.h"
#include "smbconst.h"
#include <sys/stat.h>
#include <errno.h>

/*****************************************************************************/
/*                                                                           */
/*  The list of all inodes                                                   */
/*                                                                           */
/*****************************************************************************/
static struct f_inode inodes;          /* one dummy element in list */

/*****************************************************************************/
/*                                                                           */
/*  Here is the directory cache, we cache only one directory (like NFS)      */
/*                                                                           */
/*****************************************************************************/
struct smb_dir_cache {
  struct f_inode *dir;                 /* The dir it comes from */
  int size;                     /* the potential size */
  int filled;                   /* how many entries are in here currently? */
  int last_found;               /* Where has lookup last found it? */
  struct smb_finfo *data;              /* Ok, the contents */
};

static struct smb_dir_cache dir_cache;

/*****************************************************************************/
/*                                                                           */
/*  inode_init                                                               */
/*                                                                           */
/*****************************************************************************/
void
inodes_init(void)
{
  /* initialize the inode list */
  inodes.prev = inodes.next = &inodes;

  /* initialize the directory cache */
  dir_cache.dir    = NULL;
  dir_cache.size   = 0;
  dir_cache.filled = 0;
  dir_cache.last_found = 0;
  dir_cache.data   = NULL;

  return;
}

/*****************************************************************************/
/*                                                                           */
/*  no_more_inodes                                                           */
/*                                                                           */
/*****************************************************************************/
Bool
no_more_inodes(void)
{
  return (   (inodes.next == &inodes)
          && (inodes.prev == &inodes));
}

/*****************************************************************************/
/*                                                                           */
/*  find_dir_cache                                                           */
/*                                                                           */
/*****************************************************************************/
static struct smb_dir_cache *
find_dir_cache(struct f_inode *dir)
{
  return (dir_cache.dir == dir ? &dir_cache : NULL);
}

/*****************************************************************************/
/*                                                                           */
/*  invalidate_dir_cache                                                     */
/*                                                                           */
/*****************************************************************************/
void
invalidate_dir_cache(struct f_inode *dir)
{
  struct smb_dir_cache *c = find_dir_cache(dir);
  if (c != NULL)
    c->dir = NULL;
  return;
}

/*****************************************************************************/
/*                                                                           */
/*  lookup_in_cache                                                          */
/*                                                                           */
/*****************************************************************************/
static struct smb_finfo *
lookup_in_cache(struct smb_dir_cache *c, const char *name)
{
  int i,n;
  AssertStr(name);

  i = (c->last_found < c->filled ? c->last_found : 0);
  
  for (n=c->filled; n>0; n--) {
    if (strcmp(name, (c->data[i]).name) == 0) {
      c->last_found = i;
      return &(c->data[i]);
    }
    i = (i + 1) % c->filled;
  }
  
  return NULL;
}

/*****************************************************************************/
/*                                                                           */
/*  rename_f_inode                                                           */
/*                                                                           */
/* newname is consumed !!                                                    */
/*                                                                           */
/*****************************************************************************/
void
rename_f_inode(struct f_inode *f, pstr newname)
{
  AssertInode(f);
  AssertStr(newname);
  
  pstr_delete(f->path);
  f->path = newname;
  return;
}  

/*****************************************************************************/
/*                                                                           */
/*  make_path                                                                */
/*  The caller has to delete it with pstr_delete                             */
/*                                                                           */
/*****************************************************************************/
pstr
make_path(pstr dir, pstr name)
{
  AssertStr(name);
  AssertStr(dir);
  
  if ((name[0] == '.') && (name[1] == 0))
    return pstr_1(dir);

  if ((name[0] == '.') && (name[1] == '.') && (name[2] == 0)) {
    char *result = (char *)pstr_2(dir, "\\");
    char *end;
    int len = strlen(dir);

    if (   (result == NULL)
        || (len == 1))          /* We're at the top */
      return result;

    end = strrchr(result, '\\');
    if (end == NULL) {          /* This should not happen */
      pstr_delete(result);
      return NULL;
    }
    *end = 0;                   /* Kill the last byte */
    end = strrchr(result, '\\');
    if (end == NULL) {          /* Again this should not happen */
      pstr_delete(result);
      return NULL;
    }
    *end = 0;                   /* Cut off the last path component */
    return (pstr)result;
  }

  return pstr_3(dir, "\\", name);
}

/*****************************************************************************/
/*                                                                           */
/*  new_f_inode                                                              */
/*                                                                           */
/* We do not check that it's not already there.                              */
/*                                                                           */
/*****************************************************************************/
struct f_inode *
new_f_inode(struct smb_conn *conn, struct f_inode *dir, const char *fname)
{
  pstr path;
  struct smb_finfo finfo;
  struct f_inode *result;
  struct smb_dir_cache *c;

  AssertConn(conn);
  AssertStr(fname);

  if (dir == NULL) {            /* We're making root */

    if ((path = pstr_1("\\")) == NULL) return NULL;

    if (smb_getatr(conn, path, &finfo))
      return NULL;

  } else {

    AssertInode(dir);
    if ((path = make_path(dir->path, fname)) == NULL) return NULL;

    if ((c = find_dir_cache(dir)) != NULL) {
      struct smb_finfo *f = lookup_in_cache(c, fname);
      if (f != NULL)
        finfo = *f;
      else
        if (smb_getatr(conn, path, &finfo))
          goto fail;
    } else {
      if (smb_getatr(conn, path, &finfo))
        goto fail;
    }
  }

  if ((result = malloc(sizeof(struct f_inode))) == NULL) {
fail:
    pstr_delete(path);
    return NULL;
  }

  MemGarbage(result, sizeof(*result));

  smb_finfo2inode(conn, &finfo, &(result->i));

  result->opened = False;
  result->fid    = -1;
  result->access = 0xfe;
  result->state  = INODE_INVALID;
  result->dir    = dir;
  result->nused  = 0;
  result->path   = path;

  if (dir == NULL) {
    *(char *)path = 0;          /* This is root */
  } else {
    (dir->nused)++;
  }

  result->prev = &inodes;
  result->next = inodes.next;
  inodes.next->prev = result;
  inodes.next = result;

  return result;
}

/*****************************************************************************/
/*                                                                           */
/*  make_root                                                                */
/*                                                                           */
/*****************************************************************************/
struct f_inode *
make_root(struct smb_conn *conn)
{
  return new_f_inode(conn, NULL, "");
}

/*****************************************************************************/
/*                                                                           */
/*  find_f_inode                                                             */
/*                                                                           */
/*****************************************************************************/
struct f_inode *
find_f_inode(struct smb_conn *conn, struct f_inode *dir, const char *fname)
{
  pstr path;
  struct f_inode *i = inodes.next;

  AssertConn(conn);
  AssertInode(dir);
  AssertStr(fname);

  if ((path = make_path(dir->path, fname)) == NULL) return NULL;

  i = inodes.next;
  while (i != &inodes) {
    if (strcmp(i->path, path) == 0) {
      pstr_delete(path);
      return i;
    }
    i = i->next;
  }
  pstr_delete(path);
  return NULL;
}

/*****************************************************************************/
/*                                                                           */
/*  free_f_inode                                                             */
/*                                                                           */
/*****************************************************************************/
void
free_f_inode(struct f_inode *i)
{
  AssertInode(i);

  invalidate_dir_cache(i);

  i->state = INODE_CACHED;

  while ((i->nused == 0) && (i->state == INODE_CACHED)) {
    struct f_inode *dir = i->dir;

    i->next->prev = i->prev;
    i->prev->next = i->next;

    pstr_delete(i->path);
    free(i);

    if (dir == NULL) return;

    (dir->nused)--;
    i = dir;
  }
}

/*****************************************************************************/
/*                                                                           */
/*  fill_one_entry                                                           */
/*                                                                           */
/*****************************************************************************/
#define CACHE_INCREASE 32
static void
fill_one_entry(struct smb_finfo *f, void *data)
{
  struct smb_dir_cache *c = data;

  if (   ((f->name[0] == '.') && (f->name[1] == 0))
      || ((f->name[0] == '.') && (f->name[1] == '.') && (f->name[2] == 0)))
    return; /* . and .. are not stored */

  if (c->size == c->filled) {
    struct smb_finfo *new = realloc(c->data,
                             sizeof(struct smb_finfo) * (c->size + CACHE_INCREASE));
    if (new == NULL) return;
    c->data = new;
    c->size += CACHE_INCREASE;
  }

  c->data[c->filled++] = *f;
  return;
}

/*****************************************************************************/
/*                                                                           */
/*  fill_dir_cache                                                           */
/*                                                                           */
/*****************************************************************************/
static void
fill_dir_cache(struct smb_conn *conn, struct f_inode *dir,
               struct smb_dir_cache *cache)
{
  pstr mask;
  
  AssertConn(conn);
  AssertInode(dir);

  cache->dir = dir;
  cache->filled = 0;
  cache->last_found = 0;

  if ((mask = make_path(dir->path, "????????.???")) == NULL) return;

  smb_do_dir(conn, mask, fill_one_entry, cache);
  pstr_delete(mask);
  return;
}

/*****************************************************************************/
/*                                                                           */
/*  dir_next_name                                                            */
/*                                                                           */
/* You must copy the string VERY SOON !!                                     */
/*                                                                           */
/*****************************************************************************/
pstr
dir_next_name(struct smb_conn *conn, struct f_inode *dir, off_t off)
{
  AssertInode(dir);
  Assert(S_ISDIR(dir->i.mode));

  switch(off) {
  case 0:
    return ".";                 /* No break necessary */
  case 1:
    return "..";                /* No break necessary */
  default:

    if (dir_cache.dir != dir)
      fill_dir_cache(conn, dir, &dir_cache);

    off -= 2;

    if ((off >= 0) && (off < dir_cache.filled))
      return (pstr)&( ( (dir_cache.data)[off] ).name);
  }
  return NULL;
}

/*****************************************************************************/
/*                                                                           */
/*  smb_finfo2inode                                                          */
/*                                                                           */
/*****************************************************************************/
void
smb_finfo2inode(struct smb_conn *conn, struct smb_finfo *finfo, Inode *i)
{
  AssertConn(conn);
  AssertPtr(finfo);
  AssertPtr(i);

  i->mode   = ((finfo->attrib & aDIR) != 0) ? conn->dir_mode : conn->file_mode;
  i->nlink  = 1;
  i->uid    = conn->uid;
  i->gid    = conn->gid;
  i->size   = finfo->size;
  i->mtime  = finfo->mtime;
  i->atime  = i->ctime = i->mtime;
  i->rdev   = 0;
  i->blksize= 512;
  i->blocks = (finfo->size != 0) ? (finfo->size - 1)/i->blksize + 1 : 0;
  return;
}

/*****************************************************************************/
/*                                                                           */
/*  AssertInode  (implies AssertConn)                                        */
/*                                                                           */
/*****************************************************************************/
#if DEBUG > 0
void
_AssertInode(const char *f, int l, struct f_inode *i)
{
  struct f_inode *test = inodes.next;

  _AssertPtr(f,l,i);

  if (test != NULL) {
    while (test != &inodes) {
      if (i == test) {
        _AssertStr(f,l,i->path);
        return;                 /* It's ok */
      }
      test = test->next;
    }
  }
  AssertPrintf(f,l, "No valid inode: %d\n", (ulong)i);
  AssertFailed(f,l);
}
#endif /* DEBUG > 0 */
