/*
 * portable.cpp
 *
 * $Id: portable.cpp,v 1.30.2.3 2002/12/06 15:13:05 lunakl Exp $
 *
 * Copyright (c) 1999 Paul Campbell <paul@taniwha.com>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

//
//	this file contains the machine specific laptop power management stuff
//	to add support for your own OS this is should be the only place you need
//	to change - to add your own stuff insert above the line marked 
//	'INSERT HERE' :
//
//		#ifdef MY_OS"
//			.. copy of linux code or whatever you want to use as a base
//		# else
//
//	then tag an extra '#endif' at the end
//
// There is support for the following OSsen right now:
//
//	Linux 		(#if __linux__)
//	FreeBSD		(#elif __FreeBSD__)
//	NetBSD		(#elif __NetBSD_APM__)
//	generic nothing	(#else)
//
//
//
//
//	If you have any problems, questions, whatever please get in touch
//
//		Paul Campbell
//		paul@taniwha.com
//		
//
//	Thanks to Cajus Pollmeier <C.Pollmeier@gmx.net>
//	and Robert Ellis Parrott <parrott@fas.harvard.edu>
//	who both provided ACPI support
//

#include <klocale.h>
#include <stdio.h>
#include "portable.h"

#ifdef __linux__

/*
** This is the Linux-specific laptop code.
*/
#include <sys/types.h>
#include <unistd.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <dirent.h>
#include <qpushbutton.h>
#include <qobject.h>
#include <kactivelabel.h>
#include "acpi_config.h"

static acpi_config *panel_start=0;


//
//	here's the Linux specific laptop control panel stuff
//

typedef struct apm_info {
   unsigned int apm_flags;
   unsigned int ac_line_status;
   int          battery_percentage;
   int          battery_time;
} apm_info;
 
static int
apm_read(apm_info *ap)
{
   	FILE	*f = 0;
   	char	tmp2[10];
	int	tmp, s;
	unsigned int utmp;
	char 	version[256];

	f = fopen("/proc/apm", "r");
	if (f == NULL)
		return(1);
	s = fscanf(f, "%255s %d.%d %x %x %x %x %d%% %d %s\n",
		version,
		&tmp,
		&tmp,
		&ap->apm_flags,
	   	&ap->ac_line_status,
		&utmp,
		&utmp,
	   	&ap->battery_percentage,
	   	&ap->battery_time,
	   	tmp2);
	if (s < 9)
		return(1);	
	if (version[0] == 'B') {
		fclose(f);
		return(2);
	}
   	if (ap->battery_percentage > 100) 
		ap->battery_percentage = -1;
	if (strcmp(tmp2, "sec") == 0)
		ap->battery_time /= 60;
   	fclose(f);
	return(0);
}

#define MAX_BATTERIES 4
static int acpi_num_batteries = 0;
static char acpi_batt_info0[NAME_MAX+50];
static char acpi_batt_info1[NAME_MAX+50];
static char acpi_batt_info2[NAME_MAX+50];
static char acpi_batt_info3[NAME_MAX+50];
static char acpi_batt_status0[NAME_MAX+50];
static char acpi_batt_status1[NAME_MAX+50];
static char acpi_batt_status2[NAME_MAX+50];
static char acpi_batt_status3[NAME_MAX+50];
static char *acpi_batt_info[MAX_BATTERIES] = {acpi_batt_info0, acpi_batt_info1, acpi_batt_info2, acpi_batt_info3};
static char *acpi_batt_status[MAX_BATTERIES] = {acpi_batt_status0, acpi_batt_status1, acpi_batt_status2, acpi_batt_status3};



static unsigned char acpi_ac_ok;

static int
getint(char *s)
{
	int v = 0, sgn=1;

	for (;;) {
		if (*s == 0)
			return(0);
		if (*s == ':')
			break;
		s++;
	}	
	s++;
	for (;;) {
		if (*s == 0)
			return(0);
		if (*s != ' ' && *s != '\t')
			break;
		s++;
	}
	if (*s == '-') {
		s++;
		sgn = -1;
	} else 
	if (*s == '+')
		s++;
	while (*s >= '0' && *s <= '9') {
		v = (v*10) + (*s-'0');
		s++;
	}
	return(sgn*v);
}

//
//	linux APCI doesn't return usefull stuff like how much TIME is left yet
//		the 'rate' is not smoothed over time so it's faked out here
//		it's not pretty and it's a quick estimate
//
//		for the moment we prefer APM
//

static int
acpi_ac_status()
{
	DIR *dfd;
	struct dirent *dp;
	FILE *f = NULL;
	static char buff[NAME_MAX+50];
	static bool inited=0;
	static bool bad=0;
	
	if (inited) {
		if (bad)
			return(-1);
		f = fopen(buff, "r");
		goto readit;
	}
	inited = 1;

	dfd = opendir("/proc/acpi/ac_adapter/");
	if (dfd) {
		for (;;) {
			dp = readdir(dfd);
			if (dp == 0)
				break;
			if (strcmp(dp->d_name, ".") == 0 ||
		            strcmp(dp->d_name, "..") == 0)
				continue;
			strcpy(buff, "/proc/acpi/ac_adapter/");
			strcat(buff, dp->d_name);
			strcat(buff, "/status");
			f = fopen(buff, "r");
			if (!f) {
				strcpy(buff, "/proc/acpi/ac_adapter/");
				strcat(buff, dp->d_name);
				strcat(buff, "/state");
				f = fopen(buff, "r");
			}
			if (f)
				break;
		}
		closedir(dfd);
readit:
		if (f) {
			for (;;) {
				char buff2[1024];
				if (fgets(buff2, sizeof(buff), f) == NULL)
					break;
				if (strstr(buff2, "Status:") != NULL ||
			    	    strstr(buff2, "state:") != NULL) {
					if (strstr(buff2, "on-line") != NULL) {
						fclose(f);
						return(1);
					}
				}
			}
			fclose(f);
			return(0);
		}
	}
	bad=1;
	return(-1);
}

static int
acpi_read(apm_info *ap)
{
	int	v, i;
	int rate, part, total; 
	int ret = 1;
	char buff[1024];
	FILE *f;

	part = 0;
	total = 0;
	rate = 0;
	for (i = 0; i < acpi_num_batteries; i++) {
		int remaining = 0;
		int caplow = 0;
		int cap = 0;
		int first=1;

		if (acpi_batt_info[i][0]) {
			f = fopen(acpi_batt_info[i], "r");
		} else {
			f = 0;
		}
again:
		if (f == NULL) {
			if (!first || !acpi_batt_status[i][0])
				continue;
			first = 0;
			f = fopen(acpi_batt_status[i], "r");
			goto again;
		}
		for (;;) {
			if (fgets(buff, sizeof(buff), f) == NULL)
				break;
			if (strstr(buff, "Design Capacity Low:") != NULL ||
			    strstr(buff, "design capacity low:") != NULL) {
				caplow = getint(buff);
			} else
			if (strstr(buff, "Present Rate:") != NULL ||
			    strstr(buff, "present rate:") != NULL) {
				v = getint(buff);
				if (v < 0)
					v = 0;
				rate += v;
			} else
			if (strstr(buff, "Last Full Capacity:") != NULL ||
			    strstr(buff, "last full capacity:") != NULL) {
				cap = getint(buff);
			} else
			if (strstr(buff, "Remaining Capacity:") != NULL ||
			    strstr(buff, "remaining capacity:") != NULL) {
				remaining = getint(buff);
			} 
		}
   		fclose(f);
		if (first && acpi_batt_status[i][0]) {
			first = 0;
			f = fopen(acpi_batt_status[i], "r");
			goto again;
		}
		ret = 0;
		if (caplow < 0)
			caplow = 0;
		cap -= caplow;
		if (cap < 0)
			cap = 0;
		remaining -= caplow;
		if (remaining < 0)
			remaining = 0;
		total += cap;
		part += remaining;
	}

	static int nrates = 0;
	static int saved_rate[8];
	static unsigned char ignore_next = 1;	// ignore the first
	
	ap->ac_line_status = 0;
	//
	//	ACPI (unlike nice systems like some APM implementations) doesn't
	//	tell us how much battery TIME we have left - here we
	//	do a weighted average of the discharge rate (in mW) and
	//	figure out how long we have left by dividing it into the 
	//	remaining capacity
	//
	//	because some ACPI implementations return bogus 
	//	rates when charging we can't estimate the battery life
	//	so we only collect discharge rate data when we're actually
	//	discharging
	//
	if (acpi_ac_status() == 1) {
		ap->ac_line_status |= 1;
		ignore_next = 1;
	} else {
		// after switching from power to unpowered we often get
		// a bad reading from ACPI resulting in bizarre
		// readings
		if (!ignore_next) {
			if (nrates < 8)				// smooth the power flow
				nrates++;			// simple box filter
			for (int i = 8-1; i > 0; i--) 
				saved_rate[i] = saved_rate[i-1];
			saved_rate[0] = rate;
		}
		ignore_next = 0;
	}
	//
	//	if we haven't got any discharge rate data yet don't return a
	//	battery time - probably happens when you start up with the
	//	ac adaptor plugged in
	//
	if (nrates == 0) {
		ap->battery_time = -1;
	} else {
		rate = 0;
		for (int i = 0; i < nrates; i++)
			rate += saved_rate[i];
		rate = (rate+2*saved_rate[0])/(nrates+2);		// weight it slighly
		ap->battery_time = (rate==0||part==0?0 : 60*part/rate);
	}
	ap->battery_percentage = (total==0?0:100*part/total);
	ap->apm_flags = 0;
	return(ret);
}

static int apm_no_time;
static apm_info apmx = {0,0,0,0};
static int
has_apm()
{
        static int init = 0;
        static int val;
        if (init)
                return(val);
        init = 1;
        val = 1;
        apm_no_time=0;
        if (apm_read(&apmx) || (apmx.apm_flags&0x20)) {
                val = 0;
                apm_no_time = 1;
        } else {
                apm_no_time = apmx.battery_time < 0;
        }
        return(val);
}

static int x_acpi_init = 0;
static int
has_acpi()
{
	int i;
	bool ok;
	DIR *dfd;
	struct dirent *dp;
	static int val;

	if (x_acpi_init)
		return(val);
	x_acpi_init = 1;
	val = 0;
	
	if (acpi_ac_status() >= 0)
		acpi_ac_ok = 1;

	dfd = opendir("/proc/acpi/battery");
	if (!dfd)
		return(val = 0);
	i = 0;
	acpi_num_batteries = 0;
	for (dp = readdir(dfd);dp && i < MAX_BATTERIES;dp = readdir(dfd)) {
		if (strcmp(dp->d_name, ".") == 0 ||
		    strcmp(dp->d_name, "..") == 0)
			continue;
		strcpy(acpi_batt_status[i], "/proc/acpi/battery/");
		strcat(acpi_batt_status[i], dp->d_name);
		strcpy(acpi_batt_info[i], acpi_batt_status[i]);
		ok = 0;
		strcat(acpi_batt_status[i], "/state");
		if (::access(acpi_batt_status[i], R_OK) != 0) {
			strcpy(acpi_batt_status[i], acpi_batt_info[i]);
			strcat(acpi_batt_status[i], "/status");
			if (::access(acpi_batt_status[i], R_OK) == 0) {
				ok = 1;
			} else {
				acpi_batt_status[i][0] = 0;
			}
		} else {
			ok = 1;
		}
		strcat(acpi_batt_info[i], "/info");
		if (::access(acpi_batt_info[i], R_OK) == 0) {
			ok = 1;
		} else {
			acpi_batt_info[i][0] = 0;
		}
		if (ok) {
			acpi_num_batteries++;
			i++;
			val = 1;
		}
	}
	closedir(dfd);
	if (!acpi_ac_ok) 
		val = 0;
	if (val && panel_start == 0) {
		KConfig *c = new KConfig("kcmlaptoprc");
		panel_start = new acpi_config(c);
		delete c;
	}
	return(val);
}
 
static int
apm_has_time()
{
        return(!apm_no_time);
}

//
//	something changed maybe we need to check out environment again
//
void
laptop_portable::power_management_restart()
{
	if (panel_start) {
		delete panel_start;
		panel_start = 0;
	}
	x_acpi_init = 0;
	acpi_num_batteries = 0;
}

//
//	returns 1 if we support power management
//
int laptop_portable::has_power_management()
{
	if (::has_apm())
		return(1);
	return (::has_acpi());
}
//
//	returns 1 if the BIOS returns the time left in the battery rather than a % of full
//
int laptop_portable::has_battery_time()
{
	if (::has_acpi())
		return(1);
	return (::apm_has_time());
}
//
//	returns 1 if we can perform a change-to-suspend-mode operation for the user
//	(has_power_management() has already returned 1)
//
int laptop_portable::has_suspend()
{
	struct stat s;
	if (::has_acpi()) {
		return(panel_start && panel_start->has_suspend());
	}
        if (stat("/usr/bin/apm", &s) || !(getuid() == 0 || s.st_mode&S_ISUID)) 
		return(0);
	return(1);
}
//
//	returns 1 if we can perform a change-to-standby-mode operation for the user
//	(has_power_management() has already returned 1)
//
int laptop_portable::has_standby()
{
	struct stat s;
	if (::has_acpi())
		return(panel_start && panel_start->has_sleep());
        if (stat("/usr/bin/apm", &s) || !(getuid() == 0 || s.st_mode&S_ISUID)) 
		return(0);
	return(1);
}
//
//	returns 1 if we can perform a change-to-hibernate-mode for a user
//      (has_power_management() has already returned 1)  [hibernate is the save-to-disk mode
//	not supported by linux APM]
//
static int hiber_type = 0;		// remember how we hibernate
int laptop_portable::has_hibernation()
{
	struct stat s;

	if (::has_acpi())
		return(panel_start && panel_start->has_hibernate());
	hiber_type = 0;
        if (stat("/usr/local/bin/tpctl", &s) == 0 && (getuid() == 0 || s.st_mode&S_ISUID))  {
		hiber_type = 1;
		return(1);
	}
	return(0);
}

//
//	explain to the user what they need to do if has_power_management() returned 0
//	to get any software they lack
//

KActiveLabel *laptop_portable::no_power_management_explanation(QWidget *parent)
{
	if (access("/proc/acpi", F_OK) == 0) {	// probably has default kernel ACPI installed
		KActiveLabel* explain = new KActiveLabel(i18n("Your computer seems to have a partial ACPI installation. ACPI was probably enabled, but some of the sub-options were not - you need to enable at least 'AC Adaptor' and 'Control Method Battery' and then rebuild your kernel."), parent);
      		explain->setMinimumSize(explain->sizeHint());       
		return(explain);
	}

	KActiveLabel* explain = new KActiveLabel(i18n("Your computer doesn't have the Linux APM (Advanced Power Management) or ACPI software installed, or doesn't have the APM kernel drivers installed - check out the <a href=\"http://www.linuxdoc.org/HOWTO/Laptop-HOWTO.html\">Linux Laptop-HOWTO</a> document for information on how to install APM."), parent);
 
     	explain->setMinimumSize(explain->sizeHint());       
	return(explain);
}

//
//	explain to the user what they need to do to get suspend/resume to work from user mode
//
QLabel *laptop_portable::how_to_do_suspend_resume(QWidget *parent)
{
	if (::has_apm()) {
 		QLabel* note = new QLabel(i18n("\nIf you make /usr/bin/apm setuid then you will also\nbe able to choose 'suspend' and 'standby' in the\nabove dialog - check out the help button below to\nfind out how to do this"), parent);
        	note->setMinimumSize(note->sizeHint()); 
		return(note);
	}
	if (::has_acpi()) {
 		QLabel* note = new QLabel(i18n("\nCurrently ACPI suspend/standby is not supported"), parent);
        	note->setMinimumSize(note->sizeHint()); 
		return(note);
	}
 	QLabel* note = new QLabel(i18n("\nYour system does not support suspend/standby"), parent);
        note->setMinimumSize(note->sizeHint()); 
	return(note);
}

static char tmp0[256], tmp1[256];
static int present=0;
static 
void get_pcmcia_info()
{
      FILE *f = fopen("/var/lib/pcmcia/stab", "r");
      if (!f) f = fopen("/var/run/stab", "r");
      if (f) {
	int c;
	char *cp;

	present = 1;
	cp = tmp0;
	for (;;) {
		c = fgetc(f);
		if (c == EOF || c == '\n')
			break;
		if (c == ':') {
			while ((c = fgetc(f)) == ' ')
				;
			if (c == EOF)
				break;
			for (;;) {
				*cp++ = c;
				c = fgetc(f);
				if (c == EOF || c == '\n')
					break;
			}
			break;
		}
	}
	*cp = 0;

	if (c != EOF) {
		cp = tmp1;
		for (;;) {
			c = fgetc(f);
			if (c == EOF)
				break;
			if (c == ':') {
				while ((c = fgetc(f)) == ' ')
					;
				if (c == EOF)
					break;
				for (;;) {
					*cp++ = c;
					c = fgetc(f);
					if (c == EOF || c == '\n')
						break;
				}
				break;
			}
		}
		*cp = 0;
	}
	fclose(f);
      } else {
	present = 0;
      }
}

//
//	pcmcia support - this will be replaced by better - pcmcia support being worked on by
//	others
//
QLabel *laptop_portable::pcmcia_info(int x, QWidget *parent)
{
	if (x == 0) 
		get_pcmcia_info();
	if (!present) {
		if (x == 1)
      			return(new QLabel(i18n("No PCMCIA controller detected"), parent));
      		return(new QLabel(i18n(""), parent));
	} else {
		switch (x) {
        	case 0: return(new QLabel(i18n("Card 0:"), parent));
        	case 1: return(new QLabel(tmp0, parent));
		case 2: return(new QLabel(i18n("Card 1:"), parent));
        	default:return(new QLabel(tmp1, parent));  
		}
	}
}

//
//	puts us into standby mode
//
void laptop_portable::invoke_standby()
{
	if (::has_acpi()) {
		if(panel_start)
			panel_start->do_sleep();
	} else {
	 	::system("/usr/bin/apm --standby");    
	}
}

//
//	puts us into suspend mode
//
void laptop_portable::invoke_suspend()
{
	if (::has_acpi()) {
		if(panel_start)
			panel_start->do_suspend();
	} else {
	 	::system("/usr/bin/apm --suspend");    
	}
}

//
//	puts us into hibernate mode
//
void laptop_portable::invoke_hibernation()
{
	if (::has_acpi()) {
		if(panel_start)
			panel_start->do_hibernate();
	} else 
	switch (hiber_type) {
	case 1:
	 	::system("/usr/local/bin/tpctl --hibernate");    
		break;
	// add other machine specific hibernates here
	}
}

void
laptop_portable::extra_config(QWidget *parent, KConfig *config, QVBoxLayout *layout)
{
	if (::has_acpi()) {
		if (panel_start)
			delete panel_start;
		panel_start = new acpi_config(parent, config, layout);
		return;
	}
	
}

//
//	return current battery state
//
struct power_result laptop_portable::poll_battery_state()
{
	struct power_result p;

	apm_info x = {0,0,0,0};
        if ((::has_acpi() ? ::acpi_read(&x) : ::apm_read(&x)) || (x.apm_flags&0x20)) {
                p.powered = 0;
                p.percentage=0;
                p.time = 0;
        } else {
		p.powered = x.ac_line_status&1;
                p.percentage = x.battery_percentage;
                p.time = x.battery_time;
        }                                 
	return(p);
}


//
//	returns true if any mouse or kdb activity has been detected
//	

int laptop_portable::poll_activity()
{
	static int mouse_count = 0, key_count = 0;

	int m=0, k = 0;
	int v;
	char	name[256];
	char *cp, *cp2;
	int *vp;
	static FILE *procint = 0;

	if (procint == 0) {
		procint = fopen("/proc/interrupts", "r");
		if (procint != 0)
			return(0);
		poll_activity();	// initialise statics
		return(1);
	}

	::rewind(procint);
	for (;;) {
		if (::fgets(name, sizeof(name), procint) == 0)
			break;
		vp = 0;
		if (strstr(name, "Mouse") || strstr(name, "mouse")) {
			vp = &m;
		} else
		if (strstr(name, "Keyboard") || strstr(name, "keyboard"))
			vp = &k;
		if (vp == 0)
			continue;
		v = 0;
		for (cp = name;*cp;cp++) {
			if (*cp != ':')
				continue;
			cp++;
			for (;;) {
				for(;;cp++) {
					if (*cp != ' ' && *cp != '\t')
						break;
				}
				if (*cp < '0' || *cp > '9')
					break;
				cp2 = cp;
				while (*cp >= '0' && *cp <= '9')
					cp++;
				
				*cp++ = 0;
				v += atoi(cp2);	
			}
			break;
		}
		if (v > *vp)
			*vp = v;
	}
	v = k != key_count || m != mouse_count;
	key_count = k;
	mouse_count = m;
	return(v);
}



#elif defined(__FreeBSD__)

/*
** This is the FreeBSD-specific code.
*/

#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <machine/apm_bios.h>
#include <sys/stat.h>
#include <qpushbutton.h>
#include <qobject.h>
#include <kactivelabel.h>

#define APMDEV "/dev/apm"

// FreeBSD support by yours truely.  Yay.
// Actually this code was "adapted" from apm(8) from
// FreeBSD's collection of tools.  The orignal apm program
// was pieced together by Tatsumi Hosokawa <hosokawa@jp.FreeBSD.org> in 1994

//
//	returns 1 if we support power management
//

#include <iostream.h>

//
//	something changed maybe we need to check out environment again
//
void
laptop_portable::power_management_restart()
{
}

int
laptop_portable::has_power_management()
{
	int ret, fd = ::open(APMDEV, O_RDWR);

	if (fd == -1) {
	  return 0;
	}

	struct apm_info info;
	ret=ioctl(fd, APMIO_GETINFO, &info);
	::close(fd);

	if (ret == -1) {
	  return 0;
	}

	return info.ai_status;
}
//
//	returns 1 if the BIOS returns the time left in the battery rather than a % of full
//
int laptop_portable::has_battery_time()
{
	int ret, fd = ::open(APMDEV, O_RDWR);

	if (fd == -1)
	  return 0;

	struct apm_info info;
	ret=ioctl(fd, APMIO_GETINFO, &info);
	::close(fd);

	if (ret == -1)
	  return 0;

	return (info.ai_batt_time != 0xffff);
}

//
//	returns 1 if we can perform a change-to-suspend-mode operation for the user
//	(has_power_management() has already returned 1)
//
int laptop_portable::has_suspend()
{
	int ret, fd = ::open(APMDEV, O_RDWR);

	if (fd == -1)
	  return 0;

	struct apm_info info;
	ret=ioctl(fd, APMIO_GETINFO, &info);
	::close(fd);

	if (ret == -1)
	  return 0;

	return (info.ai_capabilities & 0x02);
}
//
//	returns 1 if we can perform a change-to-standby-mode operation for the user
//	(has_power_management() has already returned 1)
//
int laptop_portable::has_standby()
{
	int ret, fd = ::open(APMDEV, O_RDWR);

	if (fd == -1)
	  return 0;

	struct apm_info info;
	ret=ioctl(fd, APMIO_GETINFO, &info);
	::close(fd);

	if (ret == -1)
	  return 0;

	return (info.ai_capabilities & 0x01);
}
//
//	returns 1 if we can perform a change-to-hibernate-mode for a user
//      (has_power_management() has already returned 1)  [hibernate is the save-to-disk mode
//	not supported by linux - different laptops have their own - the first here is for 
//	a ThinkPad]
//
static int hiber_type = 0;		// remember how we hibernate
int laptop_portable::has_hibernation()
{
	struct stat s;

	hiber_type = 0;
        if (stat("/usr/local/bin/tpctl", &s) == 0 && (getuid() == 0 || s.st_mode&S_ISUID))  {
		hiber_type = 1;	// ThinkPad
		return(1);
	}
	return(0);
}

//
//	explain to the user what they need to do if has_power_management() returned 0
//	to get any software they lack
//
KActiveLabel *laptop_portable::no_power_management_explanation(QWidget *parent)
{
  int fd;
  KActiveLabel *explain;

  fd = ::open(APMDEV, O_RDWR);
  if (fd == -1) {
    switch (errno) {
    case ENOENT:
      explain = new KActiveLabel("There is no /dev/apm file on this system.  Pleae review the FreeBSD handbook on how to create a device node for the apm device driver (man 4 apm)", parent);
      break;
    case EACCES:
      explain = new KActiveLabel("Your system has the proper device node for apm support, however you can't access it.  If you're root right now, you've got a problem, otherwise contact your local sysadmin and beg for read/write access to /dev/apm.", parent);
      break;
    case ENXIO:
      explain = new KActiveLabel("Your kernel lacks support for Advanced Power Managment.", parent);
      break;
      break;
    default:
      explain = new KActiveLabel("There was some generic error while opening /dev/apm.  Contact your local supermarket, there's a blue light special on FreeBSD, really.", parent);
      break;
    }
  } else {
    close(fd);
    explain = new KActiveLabel("APM has most likely been disabled.  Oops", parent);
  }
  
  explain->setMinimumSize(explain->sizeHint());       
  return(explain);
}

//
//	explain to the user what they need to do to get suspend/resume to work from user mode
//
QLabel *laptop_portable::how_to_do_suspend_resume(QWidget *parent)
{
 	QLabel* note = new QLabel(i18n(" "), parent);
        note->setMinimumSize(note->sizeHint()); 
	return(note);
}


//
//	pcmcia support - this will be replaced by better - pcmcia support being worked on by
//	others
//
QLabel *laptop_portable::pcmcia_info(int x, QWidget *parent)
{
      	if (x == 0)
		return(new QLabel(i18n("No PCMCIA controller detected"), parent));
      	return(new QLabel(i18n(""), parent));
}
//
//	puts us into standby mode
//
void laptop_portable::invoke_standby()
{
  	int fd = ::open(APMDEV, O_RDWR);

	if (fd == -1)
	  return;

	ioctl(fd, APMIO_STANDBY, NULL);
	::close(fd);

	return;
}

//
//	puts us into suspend mode
//
void laptop_portable::invoke_suspend()
{
  	int fd = ::open(APMDEV, O_RDWR);

	if (fd == -1)
	  return;

	ioctl(fd, APMIO_SUSPEND, NULL);
	::close(fd);

	return;
}
//
//	puts us into hibernate mode
//
void laptop_portable::invoke_hibernation()
{
	switch (hiber_type) {
	case 1:
	 	::system("/usr/local/bin/tpctl --hibernate");    
		break;
	// add other macheine specific hibernates here
	}
}

//
//	adds extra widgets to the battery panel
//
void
laptop_portable::extra_config(QWidget *parent, KConfig *config, QVBoxLayout *layout)
{
}



//
//	return current battery state
//
struct power_result laptop_portable::poll_battery_state()
{
	struct power_result p;
       	int ret;

       	int fd = ::open(APMDEV, O_RDWR);

       	if (fd == -1)
         	goto bad;

       	struct apm_info info;
       	ret=ioctl(fd, APMIO_GETINFO, &info);
       	::close(fd);
	
       	if (ret == -1)
         	goto bad;

       	p.powered = info.ai_acline;
       	p.percentage = (info.ai_batt_life==255 ? 100 : info.ai_batt_life);
       	p.time = (info.ai_batt_time != 0xffff ? info.ai_batt_time/60 : -1);
       	return(p);

bad:
       	p.powered = 1; 
	p.percentage = 100;
	p.time = 0;
	return(p);
}

//
//
//	returns true if any mouse or kdb activity has been detected
//	

int laptop_portable::poll_activity()
{
	return(1);
}

#elif  defined(__NetBSD_APM__)

#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <machine/apmvar.h>
#include <iostream.h>

//
// klaptopdeamon interface to NetBSD 1.5 apm.
// Scott Presnell, srp@zgi.com, srp@tworoads.net
// Fri Jun 29 17:21:25 PDT 2001
// Tested on Dell I4K running NetBSD 1.5R
//
#define APMDEV "/dev/apm"

//
//     Check for apm in kernel by talking to /dev/apm
//     (opening read only is allowed by any process).
//     returns 1 if we support power management
//
int
laptop_portable::has_power_management()
{
       int ret, fd = ::open(APMDEV, O_RDONLY);

       if (fd == -1) {
         return 0;
       }

       struct apm_power_info info;
       ret=ioctl(fd, APM_IOC_GETPOWER, &info);
       ::close(fd);

       if (ret == -1) {
         return 0;
       }

       return 1;
}

//
//     returns 1 if the BIOS returns the time left in the battery rather than a % of full
//
int laptop_portable::has_battery_time()
{
       int ret, fd = ::open(APMDEV, O_RDONLY);

       if (fd == -1)
         return 0;

       struct apm_power_info info;
       ret=ioctl(fd, APM_IOC_GETPOWER, &info);
       ::close(fd);

       if (ret == -1)
         return 0;

       return (info.minutes_left != 0xffff);
}

//
//     returns 1 if we can perform a change-to-suspend-mode operation for the user
//     (ust check to see if we have the binary)
//     (has_power_management() has already returned 1)
//
int laptop_portable::has_suspend()
{

       struct stat s;
        if (stat("/usr/sbin/apm", &s))
               return(0);
       return(1);
}

//
//     returns 1 if we can perform a change-to-standby-mode operation for the user
//     (just check to see if we have the binary)
//     (has_power_management() has already returned 1)
//
int laptop_portable::has_standby()
{

       struct stat s;
        if (stat("/usr/sbin/apm", &s))
               return(0);
       return(1);
}

//
//     returns 1 if we can perform a change-to-hibernate-mode for a user
//      (has_power_management() has already returned 1)  [hibernate is the save-to-disk mode
//     not supported by linux - different laptops have their own - the first here is for 
//     a ThinkPad]
//     No support in NetBSD at this time.
//
int laptop_portable::has_hibernation()
{
       return(0);
}

//
//     explain to the user what they need to do if has_power_management() returned 0
//     to get any software they lack
//
KActiveLabel *laptop_portable::no_power_management_explanation(QWidget *parent)
{
  int fd;
  KActiveLabel *explain;

  fd = ::open(APMDEV, O_RDONLY);
  if (fd == -1) {
    switch (errno) {
    case ENOENT:
      explain = new KActiveLabel("There is no /dev/apm file on this system.  Pleae review the NetBSD documentation on how to create a device node for the apm device driver (man 4 apm)", parent);
      break;
    case EACCES:
      explain = new KActiveLabel("Your system has the proper device node for apm support, however you can't access it. If you have apm in the kernel this should not happen", parent);
      break;
    case ENXIO:
      explain = new KActiveLabel("Your kernel lacks support for Advanced Power Managment.", parent);
      break;
      break;
    default:
      explain = new KActiveLabel("There was some generic error while opening /dev/apm.", parent);
      break;
    }
  } else {
    close(fd);
    explain = new KActiveLabel("APM has most likely been disabled.  Oops", parent);
  }
  
  explain->setMinimumSize(explain->sizeHint());       
  return(explain);
}

//
//     explain to the user what they need to do to get suspend/resume to work from user mode
//
QLabel *laptop_portable::how_to_do_suspend_resume(QWidget *parent)
{
       QLabel* note = new QLabel(i18n(" "), parent);
        note->setMinimumSize(note->sizeHint()); 
       return(note);
}

//
//     pcmcia support - this will be replaced by better - pcmcia support being worked on by
//     others
//
QLabel *laptop_portable::pcmcia_info(int x, QWidget *parent)
{
       if (x == 0)
               return(new QLabel(i18n("No PCMCIA controller detected"), parent));
       return(new QLabel(i18n(""), parent));
}

//
//     puts us into standby mode
//     Use apm rather than ioctls in case they are running apmd
//     (as they should be).
//
void laptop_portable::invoke_standby()
{
       ::system("/usr/sbin/apm -S");    
}

//
//     puts us into suspend mode
//     Use apm rather than ioctls in case they are running apmd
//     (as they should be).
//
void laptop_portable::invoke_suspend()
{

       ::system("/usr/sbin/apm -z");    
}

//
//     puts us into hibernate mode
//     No hibernate mode for NetBSD.
//
void laptop_portable::invoke_hibernation()
{
       return;
}


//
//     return current battery state
//
struct power_result laptop_portable::poll_battery_state()
{
       struct power_result p;
               int ret;

               int fd = ::open(APMDEV, O_RDONLY);

               if (fd == -1)
               goto bad;

               struct apm_power_info info;
               ret=ioctl(fd, APM_IOC_GETPOWER, &info);
               ::close(fd);
       
               if (ret == -1)
               goto bad;

               p.powered = (info.ac_state == APM_AC_ON);
               p.percentage = (info.battery_life==255 ? 100 : info.battery_life);
               p.time = (info.minutes_left != 0xffff ? info.minutes_left : -1);
               return(p);

bad:
               p.powered = 1; 
       p.percentage = 100;
       p.time = 0;
       return(p);
}

//
//
//     returns true if any mouse or kdb activity has been detected
//     
int laptop_portable::poll_activity()
{
       return(1);
}
#else

/*
** This is utterly generic code.
*/

//
//	something changed maybe we need to check out environment again
//
void
laptop_portable::power_management_restart()
{
}

//
//	returns 1 if we support power management
//
int
laptop_portable::has_power_management()
{
	return(0);
}
//
//	returns 1 if the BIOS returns the time left in the battery rather than a % of full
//
int laptop_portable::has_battery_time()
{
	return (0);
}

//
//	returns 1 if we can perform a change-to-suspend-mode operation for the user
//	(has_power_management() has already returned 1)
//
int laptop_portable::has_suspend()
{
	return(0);
}
//
//	returns 1 if we can perform a change-to-standby-mode operation for the user
//	(has_power_management() has already returned 1)
//
int laptop_portable::has_standby()
{
	return(0);
}
//
//	returns 1 if we can perform a change-to-hibernate-mode for a user
//      (has_power_management() has already returned 1)  [hibernate is the save-to-disk mode
//	not supported by linux]
//
int laptop_portable::has_hibernation()
{
	return(0);
}

//
//	explain to the user what they need to do if has_power_management() returned 0
//	to get any software they lack
//
KActiveLabel *laptop_portable::no_power_management_explanation(QWidget *parent)
{
	KActiveLabel* explain = new KActiveLabel(i18n("Your computer or operating system is not supported by the current version of the\nKDE laptop control panels. If you want help porting these panels to work with it\nplease contact paul@taniwha.com."), parent);
      	explain->setMinimumSize(explain->sizeHint());       
	return(explain);
}

//
//	explain to the user what they need to do to get suspend/resume to work from user mode
//
QLabel *laptop_portable::how_to_do_suspend_resume(QWidget *parent)
{
 	QLabel* note = new QLabel(i18n(" "), parent);
        note->setMinimumSize(note->sizeHint()); 
	return(note);
}


//
//	pcmcia support - this will be replaced by better - pcmcia support being worked on by
//	others
//
QLabel *laptop_portable::pcmcia_info(int x, QWidget *parent)
{
      	if (x == 0)
		return(new QLabel(i18n("No PCMCIA controller detected"), parent));
      	return(new QLabel(i18n(""), parent));
}
//
//	puts us into standby mode
//
void laptop_portable::invoke_standby()
{
}

//
//	puts us into suspend mode
//
void laptop_portable::invoke_suspend()
{
}
//
//	puts us into hibernate mode
//
void laptop_portable::invoke_hibernation()
{
}


//
//	adds extra widgets to the battery panel
//
void
laptop_portable::extra_config(QWidget *parent, KConfig *config, QVBoxLayout *layout)
{
}

//
//	return current battery state
//
struct power_result laptop_portable::poll_battery_state()
{
	struct power_result p;
	p.powered = 0;
	p.percentage = 0;
	p.time = 0;
	return(p);
}

//
//	returns true if any mouse or kdb activity has been detected
//	

int laptop_portable::poll_activity()
{
	return(1);
}
#endif
