/*  ksim - a system monitor for kde
 *
 *  Copyright (C) 2001  Robbie Ward <linuxphreak@gmx.co.uk>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "fsystemconfig.h"
#include "fsystemconfig.moc"
#include <config.h>

#include <klocale.h>
#include <kiconloader.h>
#include <klistview.h>
#include <kapplication.h>
#include <knuminput.h>
#include <kdebug.h>

#include <qcheckbox.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <qlabel.h>
#include <qwhatsthis.h>

#include <stdlib.h>
#ifdef HAVE_MNTENT_H
#include <mntent.h>
#endif

#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif
#ifdef HAVE_SYS_UCRED_H
#include <sys/ucred.h>
#endif
#ifdef HAVE_SYS_MOUNT_H
#include <sys/mount.h>
#endif

#define MAXMOUNT 32

class FSysViewItem : public QCheckListItem
{
  public:
    FSysViewItem(QListView *parent, const QString &text1,
       const QString &text2, const QString &text3)
        : QCheckListItem(parent, text1, CheckBox)
    {
      setText(1, text2);
      setText(2, text3);
    }
};

FsystemConfig::FsystemConfig(KSim::PluginObject *parent, const char *name)
   : KSim::PluginPage(parent, name)
{
  m_mainLayout = new QGridLayout(this);
  m_mainLayout->setSpacing(6);

  m_availableMounts = new KListView(this);
  m_availableMounts->addColumn(i18n("Mounted Partitions"));
  m_availableMounts->addColumn(i18n("Device"));
  m_availableMounts->addColumn(i18n("Type"));
  m_mainLayout->addMultiCellWidget(m_availableMounts, 0, 0, 0, 3);

  m_stackItems = new QCheckBox(this);
  m_stackItems->setText(i18n("Align items vertically"));
  m_mainLayout->addMultiCellWidget(m_stackItems, 1, 1, 0, 1);

  m_showPercentage = new QCheckBox(this);
  m_showPercentage->setText(i18n("Show percentage"));
  m_mainLayout->addMultiCellWidget(m_showPercentage, 1, 1, 2, 3);

  m_splitNames = new QCheckBox(this);
  m_splitNames->setText(i18n("Display short mount point names"));
  QWhatsThis::add(m_splitNames, i18n("This option shortens the text"
     " to shrink down a mount point. E.G: a mount point /home/myuser"
     " would become myuser."));
  m_mainLayout->addMultiCellWidget(m_splitNames, 2, 2, 0, 3);

  m_scrollLabel = new QLabel(this);
  m_scrollLabel->setText(i18n("Scroll speed:"));
  m_scrollLabel->setAlignment(AlignVCenter | AlignRight);
  m_mainLayout->addMultiCellWidget(m_scrollLabel, 3, 3, 0, 1);

  m_scrollUpdate = new KIntSpinBox(this);
  m_scrollUpdate->setMinValue(15);
  m_scrollUpdate->setMaxValue(200);
  m_scrollUpdate->setValue(65);
  m_scrollUpdate->setEnabled(false);
  QToolTip::add(m_scrollUpdate, i18n("0 means no scroll"));
  m_mainLayout->addMultiCellWidget(m_scrollUpdate, 3, 3, 2, 2);
  connect(m_stackItems, SIGNAL(toggled(bool)),
     m_scrollUpdate, SLOT(setDisabled(bool)));

  QLabel *scrollLabel = new QLabel(this);
  scrollLabel->setText(i18n("Seconds"));
  scrollLabel->setSizePolicy(QSizePolicy(QSizePolicy::Fixed,
     QSizePolicy::Fixed));
  m_mainLayout->addMultiCellWidget(scrollLabel, 3, 3, 3, 3);

  m_intervalLabel = new QLabel(this);
  m_intervalLabel->setText( i18n("Update interval:"));
  m_intervalLabel->setAlignment(AlignVCenter | AlignRight);
  m_mainLayout->addMultiCellWidget(m_intervalLabel, 4, 4, 0, 1);

  m_updateTimer = new KIntSpinBox(this);
  m_updateTimer->setValue(60);
  QToolTip::add(m_updateTimer, i18n("0 means no update"));
  m_mainLayout->addMultiCellWidget(m_updateTimer, 4, 4, 2, 2);

  QLabel *intervalLabel = new QLabel(this);
  intervalLabel->setText(i18n("Seconds"));
  intervalLabel->setSizePolicy(QSizePolicy(QSizePolicy::Fixed,
     QSizePolicy::Fixed));
  m_mainLayout->addMultiCellWidget(intervalLabel, 4, 4, 3, 3);

  getStats();
}

FsystemConfig::~FsystemConfig()
{
}

void FsystemConfig::readConfig()
{
  config()->setGroup("Fsystem");
  m_stackItems->setChecked(config()->readBoolEntry("stackItems", false));
  m_showPercentage->setChecked(config()->readBoolEntry("ShowPercentage", true));
  m_scrollUpdate->setValue(config()->readNumEntry("scrollerUpdate", 65));
  m_updateTimer->setValue(config()->readNumEntry("updateValue", 60));
  m_splitNames->setChecked(config()->readBoolEntry("ShortenEntries", false));

  if (!m_availableMounts->childCount())
    return;

  QStringList list = config()->readListEntry("mountEntries");
  for (QListViewItemIterator it(m_availableMounts); it.current(); ++it) {
    QString string = it.current()->text(0) + ":" + splitString(it.current()->text(0));
    static_cast<QCheckListItem *>(it.current())->setOn(list.contains(string) > 0);
  }
}

void FsystemConfig::saveConfig()
{
  config()->setGroup("Fsystem");
  config()->writeEntry("stackItems", m_stackItems->isChecked());
  config()->writeEntry("ShowPercentage", m_showPercentage->isChecked());
  config()->writeEntry("scrollerUpdate", m_scrollUpdate->value());
  config()->writeEntry("updateValue", m_updateTimer->value());
  config()->writeEntry("ShortenEntries", m_splitNames->isChecked());

  QStringList list;
  for (QListViewItemIterator it(m_availableMounts); it.current(); ++it) {
    if (static_cast<QCheckListItem *>(it.current())->isOn())
      list.append(it.current()->text(0) + ":" + splitString(it.current()->text(0)));
  }

  config()->writeEntry("mountEntries", list);
}

void FsystemConfig::showEvent(QShowEvent *)
{
  // Update the entries to take into account
  // any mounted/unmounted filesystems
  m_availableMounts->clear();
  getStats();
}

void FsystemConfig::getStats()
{
#ifdef HAVE_MNTENT_H
  FILE *fp = setmntent("/etc/mtab", "r");
  struct mntent *mntEntries;

  while ((mntEntries = getmntent(fp)) != 0) {
    if (!m_availableMounts->findItem(mntEntries->mnt_dir, 0))
      (void) new FSysViewItem(m_availableMounts, mntEntries->mnt_dir,
         mntEntries->mnt_fsname, mntEntries->mnt_type);
  }

  endmntent(fp);
#elif defined(HAVE_SYS_PARAM_H) && \
  defined(HAVE_SYS_UCRED_H) && defined(HAVE_SYS_MOUNT_H)
  struct statfs sfs[MAXMOUNT];
  int fs_count;
  if ((fs_count = getfsstat(sfs, sizeof(sfs), 0)) != -1) {
    for (int i = 0; i < fs_count; i++) {
      if (!m_availableMounts->findItem(sfs[i].f_mntonname, 0))
        (void) new FSysViewItem(m_availableMounts, sfs[i].f_mntonname,
#ifdef __osf__
           sfs[i].f_mntfromname, "");
#else
           sfs[i].f_mntfromname, sfs[i].f_fstypename);
#endif
    }
  }
#else
  kdError() << "unable to get the mounted partitions" << endl;
#endif

  if (!m_availableMounts->childCount())
    return;

  config()->setGroup("Fsystem");
  QStringList list = config()->readListEntry("mountEntries");
  for (QListViewItemIterator it(m_availableMounts); it.current(); ++it) {
    QString string = it.current()->text(0) + ":" + splitString(it.current()->text(0));
    static_cast<QCheckListItem *>(it.current())->setOn(list.contains(string) > 0);
  }
}

QString FsystemConfig::splitString(const QString &string) const
{
  if (string == "/" || !m_splitNames->isChecked())
    return string;

  int location = string.findRev("/");
  QString newString(string);
  return newString.remove(0, location + 1);
}
