/****************************************************************************
 * This module is based on Twm, but has been siginificantly modified 
 * by Rob Nation (nation@rocket.sanders.lockheed.com 
 ****************************************************************************/
/*****************************************************************************/
/**       Copyright 1988 by Evans & Sutherland Computer Corporation,        **/
/**                          Salt Lake City, Utah                           **/
/**  Portions Copyright 1989 by the Massachusetts Institute of Technology   **/
/**                        Cambridge, Massachusetts                         **/
/**                                                                         **/
/**                           All Rights Reserved                           **/
/**                                                                         **/
/**    Permission to use, copy, modify, and distribute this software and    **/
/**    its documentation  for  any  purpose  and  without  fee is hereby    **/
/**    granted, provided that the above copyright notice appear  in  all    **/
/**    copies and that both  that  copyright  notice  and  this  permis-    **/
/**    sion  notice appear in supporting  documentation,  and  that  the    **/
/**    names of Evans & Sutherland and M.I.T. not be used in advertising    **/
/**    in publicity pertaining to distribution of the  software  without    **/
/**    specific, written prior permission.                                  **/
/**                                                                         **/
/**    EVANS & SUTHERLAND AND M.I.T. DISCLAIM ALL WARRANTIES WITH REGARD    **/
/**    TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES  OF  MERCHANT-    **/
/**    ABILITY  AND  FITNESS,  IN  NO  EVENT SHALL EVANS & SUTHERLAND OR    **/
/**    M.I.T. BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL  DAM-    **/
/**    AGES OR  ANY DAMAGES WHATSOEVER  RESULTING FROM LOSS OF USE, DATA    **/
/**    OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER    **/
/**    TORTIOUS ACTION, ARISING OUT OF OR IN  CONNECTION  WITH  THE  USE    **/
/**    OR PERFORMANCE OF THIS SOFTWARE.                                     **/
/*****************************************************************************/
/***********************************************************************
 *
 * fvwm menu code
 *
 ***********************************************************************/

#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include "fvwm.h"
#include "menus.h"
#include "misc.h"
#include "parse.h"
#include "screen.h"

extern XEvent Event;
extern int menuFromFrameOrWindowOrTitlebar;

extern FvwmWindow *Tmp_win;


void FocusOn(FvwmWindow *t);
extern char **g_argv;

/***********************************************************************
 *
 *  Procedure:
 *	ExecuteFunction - execute a fvwm built in function
 *
 *  Inputs:
 *	func	- the function to execute
 *	action	- the menu action to execute 
 *	w	- the window to execute this function on
 *	tmp_win	- the fvwm window structure
 *	event	- the event that caused the function
 *	context - the context in which the button was pressed
 *      val1,val2 - the distances to move in a scroll operation 
 *
 ***********************************************************************/
void ExecuteFunction(int func,char *action, Window w, FvwmWindow *tmp_win, 
		     XEvent *eventp, int context,int val1, int val2,
		     MenuRoot *menu)
{
  FvwmWindow *t;
  char *junk;
  int x,y,delta_x,delta_y,warp_x,warp_y;
  Pixel TextColor,BackColor;
  Pixmap BackPixmap;

  switch (func)
    {
    case F_NOP:
    case F_TITLE:
      break;
      
    case F_BEEP:
      XBell(dpy, Scr.screen);
      break;
      
    case F_RESIZE:
      if (DeferExecution(eventp,&w,&tmp_win,&context, MOVE, ButtonPress))
	break;

      tmp_win->flags &= ~MAXIMIZED;
      resize_window(eventp,w,tmp_win);
      break;
      
    case F_MOVE:
      if (DeferExecution(eventp,&w,&tmp_win,&context, MOVE,ButtonPress))
	break;
      move_window(eventp,w,tmp_win,context);
      break;

#ifndef NON_VIRTUAL      
    case F_SCROLL:
      if((val1 > -100000)&&(val1 < 100000))
	x=Scr.Vx + val1*Scr.MyDisplayWidth/100;
      else
	x = Scr.Vx + (val1/1000)*Scr.MyDisplayWidth/100;
	  
      if((val2 > -100000)&&(val2 < 100000))
	y=Scr.Vy + val2*Scr.MyDisplayHeight/100;
      else
	y = Scr.Vy + (val2/1000)*Scr.MyDisplayHeight/100;
      
      if(((val1 <= -100000)||(val1 >= 100000))&&(x>Scr.VxMax))
	{
	  x = 0;
	  y += Scr.MyDisplayHeight;
	  if(y > Scr.VxMax)
	    y=0;
	}
      if(((val1 <= -100000)||(val1 >= 100000))&&(x<0))
	{
	  x = Scr.VxMax;
	  y -= Scr.MyDisplayHeight;
	  if(y < 0)
	    y=Scr.VyMax;
	}
      if(((val2 <= -100000)||(val2>= 100000))&&(x>Scr.VyMax))
	{
	  y = 0;
	  x += Scr.MyDisplayWidth;
	  if(x > Scr.VxMax)
	    x=0;
	}
      if(((val2 <= -100000)||(val2>= 100000))&&(y<0))
	{
	  y = Scr.VyMax;
	  x -= Scr.MyDisplayWidth;
	  if(x < 0)
	    x=Scr.VxMax;
	}
      MoveViewport(x,y,True);
      CheckAndSetFocus();
      break;
#endif
    case F_MOVECURSOR:
      XQueryPointer( dpy, Scr.Root, &JunkRoot, &JunkChild,
		    &x,&y,&JunkX, &JunkY, &JunkMask);
#ifndef NON_VIRTUAL
      delta_x = 0;
      delta_y = 0;
      warp_x = 0;
      warp_y = 0;
      if(x >= Scr.MyDisplayWidth -2)
	{
	  delta_x = Scr.EdgeScrollX;
	  warp_x = Scr.EdgeScrollX - 4;
	}
      if(y>= Scr.MyDisplayHeight -2)
	{
	  delta_y = Scr.EdgeScrollY;
	  warp_y = Scr.EdgeScrollY - 4;      
	}
      if(x < 2)
	{
	  delta_x = -Scr.EdgeScrollX;
	  warp_x =  -Scr.EdgeScrollX + 4;
	}
      if(y < 2)
	{
	  delta_y = -Scr.EdgeScrollY;
	  warp_y =  -Scr.EdgeScrollY + 4;
	}
      if(Scr.Vx + delta_x < 0)
	delta_x = -Scr.Vx;
      if(Scr.Vy + delta_y < 0)
	delta_y = -Scr.Vy;
      if(Scr.Vx + delta_x > Scr.VxMax)
	delta_x = Scr.VxMax - Scr.Vx;
      if(Scr.Vy + delta_y > Scr.VyMax)
	delta_y = Scr.VyMax - Scr.Vy;
      if((delta_x!=0)||(delta_y!=0))
	{
	  MoveViewport(Scr.Vx + delta_x,Scr.Vy+delta_y,True);
	  XWarpPointer(dpy, Scr.Root, Scr.Root, 0, 0, Scr.MyDisplayWidth, 
		       Scr.MyDisplayHeight, 
		       x - warp_x,
		       y - warp_y);
	}
#endif
      XWarpPointer(dpy, Scr.Root, Scr.Root, 0, 0, Scr.MyDisplayWidth, 
		   Scr.MyDisplayHeight, x + val1*Scr.MyDisplayWidth/100-warp_x,
		   y+val2*Scr.MyDisplayHeight/100 - warp_y);
      
      break;
    case F_ICONIFY:
      if (DeferExecution(eventp,&w,&tmp_win,&context, SELECT, ButtonRelease))
	break;
      if (tmp_win->flags & ICON)
	DeIconify(tmp_win);
      else
	Iconify(tmp_win, eventp->xbutton.x_root-5,eventp->xbutton.y_root-5);
      break;

    case F_RAISE:
      if (DeferExecution(eventp,&w,&tmp_win,&context, SELECT,ButtonRelease))
	break;
      
      RaiseWindow(tmp_win);

      if (LookInList(Scr.TheList,tmp_win->name, &tmp_win->class, &junk)
	  & STAYSONTOP_FLAG)
	tmp_win->flags |= ONTOP;
      KeepOnTop();
      break;
      
    case F_LOWER:
      if (DeferExecution(eventp,&w,&tmp_win,&context, SELECT, ButtonRelease))
	break;
      
      LowerWindow(tmp_win);

      tmp_win->flags &= ~ONTOP;
      break;
      
    case F_DESTROY:
      if (DeferExecution(eventp,&w,&tmp_win,&context, DESTROY, ButtonRelease))
	break;

#ifndef NO_PAGER
      /* Dont delete the pager - it crashes the program! */
      if((tmp_win->w == Scr.Pager_w)||(tmp_win == Scr.FvwmPager))
	break;
#endif

      if (XGetGeometry(dpy, tmp_win->w, &JunkRoot, &JunkX, &JunkY,
		       &JunkWidth, &JunkHeight, &JunkBW, &JunkDepth) == 0)
	Destroy(tmp_win);
      else
	XKillClient(dpy, tmp_win->w);
      XSync(dpy,Scr.screen);
      break;
      
    case F_DELETE:
      if (DeferExecution(eventp,&w,&tmp_win,&context, DESTROY,ButtonRelease))
	break;

#ifndef NO_PAGER
      /* Dont delete the pager - it crashes the program! */
      if((tmp_win->w == Scr.Pager_w)||(tmp_win == Scr.FvwmPager))
	break;
#endif
      if (tmp_win->flags & DoesWmDeleteWindow)
	send_clientmessage (tmp_win->w, _XA_WM_DELETE_WINDOW, CurrentTime);
      else
	XBell (dpy, Scr.screen);
      XSync(dpy,Scr.screen);
      break;
      
    case F_RESTART:
      Done(1, action);
      fvwm_err("Restart of %s failed",action,NULL,NULL);
      XBell(dpy,Scr.screen);
      XSync(dpy,Scr.screen);
      sleep_a_little(1000000);
      /* Backup plan - do a conventional restart */
      execvp(g_argv[0],g_argv);      
      XBell(dpy,Scr.screen);
      break;

    case F_EXEC:
      XGrabPointer(dpy, Scr.Root, True,
		   ButtonPressMask | ButtonReleaseMask,
		   GrabModeAsync, GrabModeAsync,
		   Scr.Root, Scr.FvwmCursors[WAIT], CurrentTime);
      XSync (dpy, Scr.screen);
      system(action);
      XUngrabPointer(dpy,CurrentTime);
      XSync (dpy, Scr.screen);
      break;
      
    case F_REFRESH:
      {
	XSetWindowAttributes attributes;
	unsigned long valuemask;
	
	valuemask = (CWBackPixel);
	attributes.background_pixel = Scr.StdColors.fore;
	attributes.backing_store = NotUseful;
	w = XCreateWindow (dpy, Scr.Root, 0, 0,
			   (unsigned int) Scr.MyDisplayWidth,
			   (unsigned int) Scr.MyDisplayHeight,
			   (unsigned int) 0,
			   CopyFromParent, (unsigned int) CopyFromParent,
			   (Visual *) CopyFromParent, valuemask,
			   &attributes);
	XMapWindow (dpy, w);
	XDestroyWindow (dpy, w);
	XFlush (dpy);
      }
      break;

    case F_STICK:
      /* stick/unstick a window */
      if (DeferExecution(eventp,&w,&tmp_win,&context,SELECT,ButtonRelease))
	break;
      if(tmp_win->flags & STICKY)
	{
	  tmp_win->flags &= ~STICKY;
	  SetStickyProp(tmp_win, False,tmp_win->frame_x,tmp_win->frame_y);
	}
      else
	{
	  tmp_win->flags |=STICKY;
	  SetStickyProp(tmp_win, True,tmp_win->frame_x,tmp_win->frame_y);
	}
#ifdef MORE_COLORS
      if(Scr.Hilite != tmp_win)
	{
	  FvwmWindow *temp;
	  /* Need to make SetBorder change the window back color */
	  temp=Scr.Hilite;
	  SetBorder(tmp_win,True,True,True,None);
	  SetBorder(tmp_win,False,True,True,None);
	  SetBorder(temp,True,True,True,None);
	}
#endif
#ifndef NO_PAGER
      MoveResizePagerView(tmp_win);

      /* Need to re-draw pager_view in case the window
       * is unsticking */
      if(Scr.Hilite == tmp_win)
	{
	  TextColor = Scr.HiColors.fore;
	  BackPixmap= Scr.gray_pixmap;
	  BackColor = Scr.HiColors.back;
	}
      else
	{
	  TextColor = Scr.StdColors.fore;
	  BackPixmap = Scr.light_gray_pixmap;
	  BackColor = Scr.StdColors.back;
	}
      if(Scr.d_depth < 2)
	XSetWindowBackgroundPixmap(dpy,tmp_win->pager_view,BackPixmap);
      else
	XSetWindowBackground(dpy,tmp_win->pager_view,BackColor);
      XClearWindow(dpy,tmp_win->pager_view);
      if((tmp_win->icon_name != NULL)&&(Scr.PagerFont.height > 0))
	{
	  NewFontAndColor(Scr.PagerFont.font->fid,TextColor,BackColor);
	  XDrawString (dpy, tmp_win->pager_view,Scr.FontGC,2,Scr.PagerFont.y+2,
		       tmp_win->icon_name, strlen(tmp_win->icon_name));
	}
#endif
      break;

#ifndef NON_VIRTUAL
    case F_GOTO_PAGE:
      /* back up 1 virtual desktop page */
      x=val1*Scr.MyDisplayWidth;
      y=val2*Scr.MyDisplayHeight;
      MoveViewport(x,y,True);
      break;
#endif

    case F_CIRCULATE_UP:
      t=Circulate(tmp_win,action,UP);
      FocusOn(t);
      break;

    case F_CIRCULATE_DOWN:
      t=Circulate(tmp_win,action,DOWN);
      FocusOn(t);
      break;

    case F_WARP:
      t=Circulate(tmp_win,action,DOWN);
      if(t->flags & ICON)
	{
	  FocusOn(t);
	  DeIconify(t);
	}
      FocusOn(t);
      break;

    case F_RAISELOWER:
      if (DeferExecution(eventp,&w,&tmp_win,&context, SELECT,ButtonRelease))
	break;

      if((tmp_win == Scr.LastWindowRaised)||
	 (tmp_win->flags & VISIBLE))
	{
	  LowerWindow(tmp_win);
	  tmp_win->flags &= ~ONTOP;
	}
      else
	{
	  RaiseWindow(tmp_win);
	  if (LookInList(Scr.TheList,tmp_win->name, &tmp_win->class,&junk)&
	      STAYSONTOP_FLAG)
	    tmp_win->flags |= ONTOP;	    
	  KeepOnTop();
	}
      break;

    case F_POPUP:
      ActiveItem = NULL;
      ActiveMenu = NULL;
      menuFromFrameOrWindowOrTitlebar = FALSE;
      do_menu(menu);
      break;

     case F_MAXIMIZE:
      if (DeferExecution(eventp,&w,&tmp_win,&context, SELECT,ButtonRelease))
	break;
      Maximize(tmp_win,val1,val2);
      break; 

    case F_QUIT:
      Done(0, NULL);
      break;

    case F_WINDOWLIST:
      do_windowList();
      break;

    case F_RAISE_IT:
      RaiseThisWindow(val1);
      break;

    case F_FUNCTION:
      ComplexFunction(w, tmp_win, eventp, context,menu);
      break;
    }

  WaitForButtonsUp();

  return;
}

FvwmWindow *Circulate(FvwmWindow *tmp_win, char *action,Bool Direction)
{
  FvwmWindow *t;
  Bool found;
  int count = 0;
  char *junk;

  /* move focus to the next window */
  found = FALSE;
  t = tmp_win;
  while((!found)&&(count < 2))
    {
      if(Direction == DOWN)
	{
	  if((t == (FvwmWindow *)0)||(t->next == NULL))
	    {
	      t = Scr.FvwmRoot.next;
	      count++;
	    }
	  else
	    t =t->next;
	}
      else /* Direction Up */
	{
	  if ((t == (FvwmWindow *)0)||(t == &Scr.FvwmRoot)||
	      (t->prev == &Scr.FvwmRoot)||(t->prev == (FvwmWindow *)NULL))
	    {
	      for(t=Scr.FvwmRoot.next;t->next != (FvwmWindow *)NULL;t=t->next);
	      count++;
	    }
	  else
	    t=t->prev;
	}
      found = TRUE;
      if (LookInList(Scr.TheList,t->name, &t->class,&junk) & 
	  CIRCULATESKIP_FLAG)
	found = FALSE;
      if (LookInList(Scr.TheList,t->icon_name, &t->class,&junk)& 
	  CIRCULATESKIP_FLAG)
	found = FALSE;
      if((t->flags & ICON)&&(Scr.flags & SuppressIcons))
	found = FALSE;
      
      /* optional skip over icons */
#ifdef CIRCULATESKIPICONS
      if(t->flags & ICON)
	found = FALSE;
#endif
      
      /* Make CirculateUp and CirculateDown take args. by Y.NOMURA */
      if (action &&
	  strncmp(action, t->name, strlen(action)) &&
	  strncmp(action, t->icon_name, strlen(action)) &&
	  t->class.res_name &&
	  strcmp(action, t->class.res_name))
	found = FALSE;
    }
  return t;
}
	  

/***********************************************************************
 *
 *  Procedure:
 *	DeferExecution - defer the execution of a function to the
 *	    next button press if the context is C_ROOT
 *
 *  Inputs:
 *      eventp  - pointer to XEvent to patch up
 *      w       - pointer to Window to patch up
 *      tmp_win - pointer to FvwmWindow Structure to patch up
 *	context	- the context in which the mouse button was pressed
 *	func	- the function to defer
 *	cursor	- the cursor to display while waiting
 *      finishEvent - ButtonRelease or ButtonPress; tells what kind of event to
 *                    terminate on.
 *
 ***********************************************************************/
int DeferExecution(XEvent *eventp, Window *w,FvwmWindow **tmp_win,
		   int *context, int cursor, int FinishEvent)

{
  int done;
  int finished = 0;
  Window dummy;

  if((*context != C_ROOT)&&(*context != C_NO_CONTEXT))
    {
      if((FinishEvent == ButtonPress)||((FinishEvent == ButtonRelease) &&
					(eventp->type != ButtonPress)))
	return FALSE;
    }
  if(!GrabEm(cursor))
    {
      XBell(dpy,Scr.screen);
      return False;
    }
  
  while (!finished)
    {
      done = 0;
      /* block until there is an event */
      XMaskEvent(dpy, ButtonPressMask | ButtonReleaseMask |
		 ExposureMask |KeyPressMask | VisibilityChangeMask |
		 ButtonMotionMask| PointerMotionMask, eventp);

      if(eventp->type == KeyPress)
	Keyboard_shortcuts(eventp,FinishEvent);	
      if(eventp->type == FinishEvent)
	finished = 1;
      if(eventp->type == ButtonPress)
	{
	  XAllowEvents(dpy,ReplayPointer,CurrentTime);
	  done = 1;
	}
      if(eventp->type == ButtonRelease)
	done = 1;
      if(eventp->type == EnterNotify)
	done = 1;
      if(eventp->type == LeaveNotify)
	done = 1;
      if(!done)DispatchEvent();
    }

  *w = eventp->xany.window;
  if(((*w == Scr.Root)||(*w == Scr.NoFocusWin))
      && (eventp->xbutton.subwindow != (Window)0))
    {
      *w = eventp->xbutton.subwindow;
      eventp->xany.window = *w;
    }
  if (*w == Scr.Root)
    {
      *context = C_ROOT;
      XBell(dpy,Scr.screen);
      UngrabEm();
      return TRUE;
    }
  if (XFindContext (dpy, *w, FvwmContext, (caddr_t *)tmp_win) == XCNOENT)
    {
      *tmp_win = NULL;
      XBell(dpy,Scr.screen);
      UngrabEm();
      return (TRUE);
    }
  *context = GetContext(*tmp_win,eventp,&dummy);

  UngrabEm();
  return FALSE;
}


/***********************************************************************
 *
 *  Procedure:
 *	DeIconify a window
 *
 ***********************************************************************/
void DeIconify(FvwmWindow *tmp_win)
{
  FvwmWindow *t;
  int new_x,new_y,w2,h2;

  /* now de-iconify transients */
  for (t = Scr.FvwmRoot.next; t != NULL; t = t->next)
    {
      if ((t == tmp_win)|| 
	  ((t->flags & TRANSIENT) &&(t->transientfor == tmp_win->w)))
	{
	  t->flags |= MAPPED;
	  if(Scr.Hilite == t)
	    SetBorder (t, False,True,True,None);
	  /* make sure that the window is on this screen */
	  if((t->frame_x < 0)||(t->frame_y<0)||
	     (t->frame_x >= Scr.MyDisplayWidth)||
	     (t->frame_y >= Scr.MyDisplayHeight))
	    
	    {
	      /* try to put at least half the window
	       * in the current screen */
	      new_x = t->frame_x;
	      new_y = t->frame_y;
	      w2 = (t->frame_width>>1);
	      h2 = (t->frame_height>>1);
	      if (( new_x < -w2) || (new_x > (Scr.MyDisplayWidth-w2 )))
		{
		  new_x = new_x % Scr.MyDisplayWidth;
		  if ( new_x < -w2 )
		    new_x += Scr.MyDisplayWidth;
		}
	      if ((new_y < -h2) || (new_y > (Scr.MyDisplayHeight-h2 )))
		{
		  new_y = new_y % Scr.MyDisplayHeight;
		  if ( new_y < -h2 )
		    new_y += Scr.MyDisplayHeight;
		}
	      SetupFrame(t,new_x,new_y,
			 t->frame_width,t->frame_height,False);
	    }
	  XMapWindow(dpy, t->w);
	  RaiseWindow(t);
	  XMapWindow(dpy, t->frame);
	  XMapWindow(dpy, t->Parent);
	  SetMapStateProp(t, NormalState);
	  XRaiseWindow(dpy,t->w);
	  if (t->icon_w) 
	    XUnmapWindow(dpy, t->icon_w);
	  if (t->icon_pixmap_w) 
	    XUnmapWindow(dpy, t->icon_pixmap_w);
	  t->flags &= ~ICON;
	}
    }
  if (Scr.flags & ClickToFocus) 
    FocusOn(tmp_win);
  else if(Scr.Focus == tmp_win)
    {
      SetFocus(Scr.NoFocusWin,NULL);
    }

  KeepOnTop();

  MoveResizePagerView(tmp_win);

  return;
}


/****************************************************************************
 *
 * Iconifies the selected window
 *
 ****************************************************************************/
void Iconify(FvwmWindow *tmp_win, int def_x, int def_y)
{
  FvwmWindow *t;
  XWindowAttributes winattrs;
  unsigned long eventMask;
  
  XGetWindowAttributes(dpy, tmp_win->w, &winattrs);
  eventMask = winattrs.your_event_mask;

  if((tmp_win)&&(tmp_win == Scr.Hilite)&&
     (Scr.flags & ClickToFocus)&&(tmp_win->next))
    SetFocus(tmp_win->next->w,tmp_win->next);

  /* iconify transients first */
  for (t = Scr.FvwmRoot.next; t != NULL; t = t->next)
    {
      if ((t==tmp_win)||
	  ((t->flags & TRANSIENT) && (t->transientfor == tmp_win->w)))
	{
	  /*
	   * Prevent the receipt of an UnmapNotify, since that would
	   * cause a transition to the Withdrawn state.
	   */
	  t->flags &= ~MAPPED;
	  XSelectInput(dpy, t->w, eventMask & ~StructureNotifyMask);
	  XUnmapWindow(dpy, t->w);
	  XSelectInput(dpy, t->w, eventMask);
	  XUnmapWindow(dpy, t->frame);
	  if (t->icon_w)
	    XUnmapWindow(dpy, t->icon_w);
	  if (t->icon_pixmap_w)
	    XUnmapWindow(dpy, t->icon_pixmap_w);
	  SetMapStateProp(t, IconicState);
	  SetBorder (t, False,False,False,None);
	  t->flags |= ICON;
	}
    } 

  if (tmp_win->icon_w == (int)NULL)
    CreateIconWindow(tmp_win, def_x, def_y);
  AutoPlace(tmp_win);

  LowerWindow(tmp_win);
  XMapWindow(dpy, tmp_win->icon_w);
  if(tmp_win->icon_pixmap_w != None)
    XMapWindow(dpy, tmp_win->icon_pixmap_w);

  KeepOnTop();

  MoveResizePagerView(tmp_win);

  if (Scr.flags & ClickToFocus) 
    {
      if (tmp_win == Scr.Focus)
	{
	  if(Scr.PreviousFocus == Scr.Focus)
	    Scr.PreviousFocus = NULL;
	  if((Scr.flags & ClickToFocus)&&(tmp_win->next))
	    SetFocus(tmp_win->next->w, Tmp_win->next);
	  else
	    {
	      SetFocus(Scr.NoFocusWin, NULL);
	    }
	}
    }
  else
    {
      SetFocus(Scr.NoFocusWin,NULL);
    }
  
  return;
}


/****************************************************************************
 *
 * This is used to tell applications which windows on the screen are
 * top level appication windows, and which windows are the icon windows
 * that go with them.
 *
 ****************************************************************************/
void SetMapStateProp(FvwmWindow *tmp_win, int state)
{
  unsigned long data[3];		/* "suggested" by ICCCM version 1 */
  
  data[0] = (unsigned long) state;
  data[1] = (unsigned long) tmp_win->icon_w;
  
  XChangeProperty (dpy, tmp_win->w, _XA_WM_STATE, _XA_WM_STATE, 32, 
		   PropModeReplace, (unsigned char *) data, 3);
  return;
}

/****************************************************************************
 *
 * This is used to tell applications which windows on the screen are
 * top level appication windows, and which windows are the icon windows
 * that go with them.
 *
 ****************************************************************************/
void SetStickyProp(FvwmWindow *tmp_win, int state,int x, int y)
{
  unsigned long data[5];
  
  data[0] = (unsigned long) state;
  data[1] = (unsigned long) x;
  data[2] = (unsigned long) y;
  data[3] = (unsigned long) Scr.Vx;
  data[4] = (unsigned long) Scr.Vy;

  XChangeProperty (dpy, tmp_win->w, _XA_FVWM_STICKS_TO_GLASS, 
		   _XA_FVWM_STICKS_TO_GLASS, 32, 
		   PropModeReplace, (unsigned char *) data, 5);
  return;
}

/****************************************************************************
 *
 * This is used to tell applications which windows on the screen are
 * top level appication windows, and which windows are the icon windows
 * that go with them.
 *
 ****************************************************************************/
void SetClientProp(FvwmWindow *tmp_win)
{
  unsigned long data[3];		/* "suggested" by ICCCM version 1 */
  
  data[0] = (unsigned long) tmp_win->w;
  data[1] = (unsigned long) tmp_win->icon_w;
  
  XChangeProperty (dpy, tmp_win->frame, _XA_FVWM_CLIENT, 
		   _XA_FVWM_CLIENT, 32, 
		   PropModeReplace, (unsigned char *) data, 3);
  return;
}


/****************************************************************************
 *
 * Keeps the "StaysOnTop" windows on the top of the pile.
 * This is achieved by clearing a flag for OnTop windows here, and waiting
 * for a visibility notify on the windows. Eception: OnTop windows which are
 * obscured by other OnTop windows, which need to be raised here.
 *
 ****************************************************************************/
void KeepOnTop()
{
  FvwmWindow *t;

  /* flag that on-top windows should be re-raised */
  for (t = Scr.FvwmRoot.next; t != NULL; t = t->next)
    {
      if((t->flags & ONTOP)&&!(t->flags & VISIBLE))
	{
	  RaiseWindow(t);
	  t->flags &= ~RAISED;
	}
      else
	t->flags |= RAISED;
    }
}


/***************************************************************************
 *
 *  Moves the viewport within thwe virtual desktop
 *
 ***************************************************************************/
#ifndef NON_VIRTUAL
void MoveViewport(int newx, int newy, Bool grab)
{
  FvwmWindow *t;
  int deltax,deltay;

  if(grab)
    XGrabServer(dpy);


  if(newx > Scr.VxMax)
    newx = Scr.VxMax;
  if(newy > Scr.VyMax)
    newy = Scr.VyMax;
  if(newx <0)
    newx = 0;
  if(newy <0)
    newy = 0;

  deltay = Scr.Vy - newy;
  deltax = Scr.Vx - newx;

  Scr.Vx = newx;
  Scr.Vy = newy;
  if((deltax!=0)||(deltay!=0))
    {
      for (t = Scr.FvwmRoot.next; t != NULL; t = t->next)
	{
	  /* If the window is iconified, and sticky Icons is set,
	   * then the window should essentially be sticky */
	  if(!((t->flags & ICON)&&(Scr.flags & StickyIcons)) &&
	     (!(t->flags & STICKY)))
	    {
	      if((t->icon_w)&&(!(Scr.flags & StickyIcons)))
		{
		  t->icon_x_loc += deltax;
		  t->icon_xl_loc += deltax;
		  t->icon_y_loc += deltay;
		  if(t->icon_pixmap_w != None)
		    XMoveWindow(dpy,t->icon_pixmap_w,t->icon_x_loc,
				t->icon_y_loc);
		  XMoveWindow(dpy,t->icon_w,t->icon_x_loc,
			      t->icon_y_loc+t->icon_p_height);
		}
	      SetupFrame (t, t->frame_x+ deltax, t->frame_y + deltay,
			  t->frame_width, t->frame_height,FALSE);
	    }
	  else
	    SetStickyProp(t,True,t->frame_x,t->frame_y);
	}
      for (t = Scr.FvwmRoot.next; t != NULL; t = t->next)
	{
	  /* If its an icon, and its sticking, autoplace it so
	   * that it doesn't wind up on top a a stationary
	   * icon */
	  if(((t->flags & STICKY)||(Scr.flags & StickyIcons))&&
	     (t->flags & ICON)&&(!(t->flags & ICON_MOVED)))
	    AutoPlace(t);
	}

    }
  /* fix up the viewport indicator */
  MoveResizeViewPortIndicator();
  while(XCheckTypedEvent(dpy,MotionNotify,&Event));
  if(grab)
    XUngrabServer(dpy);
}
#endif


/**************************************************************************
 *
 * Moves focus to specified window 
 *
 *************************************************************************/
void FocusOn(FvwmWindow *t)
{
  int dx,dy;
  int cx,cy,x,y;
  
  if(t == (FvwmWindow *)0)
    return;

#ifndef NON_VIRTUAL
  if(t->flags & ICON)
    {
      cx = t->icon_xl_loc + t->icon_w_width/2;
      cy = t->icon_y_loc + t->icon_p_height + ICON_HEIGHT/2;
    }
  else
    {
      cx = t->frame_x + t->frame_width/2;
      cy = t->frame_y + t->frame_height/2;
    }
  /* Put center of window on the visible screen */
  if(Scr.flags & CenterOnCirculate)
    {
      dx = cx - Scr.MyDisplayWidth/2 + Scr.Vx;
      dy = cy - Scr.MyDisplayHeight/2 + Scr.Vy;
    }
  else
    {
      dx = (cx + Scr.Vx)/Scr.MyDisplayWidth*Scr.MyDisplayWidth;
      dy = (cy +Scr.Vy)/Scr.MyDisplayHeight*Scr.MyDisplayHeight;
    }
  MoveViewport(dx,dy,True);
#endif

  if(t->flags & ICON)
    {
      x = t->icon_xl_loc + t->icon_w_width/2;
      y = t->icon_y_loc + t->icon_p_height + ICON_HEIGHT/2;
    }
  else
    {
      x = t->frame_x;
      y = t->frame_y;
    }
  if(!(Scr.flags & ClickToFocus))
    XWarpPointer(dpy, None, Scr.Root, 0, 0, 0, 0, x+2,y+2);
  RaiseWindow(t);
  /* To make really sure that we focus on the window as expected,
   * move focus to NoFocusWin for a sec. */
  SetFocus(Scr.NoFocusWin,NULL);
  SetFocus(t->w,t);
}


   
/***********************************************************************
 *
 *  Procedure:
 *	(Un)Maximize a window.
 *
 ***********************************************************************/
void Maximize(FvwmWindow *tmp_win,int val1,int val2)
{
  int new_width, new_height,new_x,new_y;

  if (tmp_win->flags & MAXIMIZED)
    {
      tmp_win->flags &= ~MAXIMIZED;
      SetupFrame(tmp_win, tmp_win->orig_x, tmp_win->orig_y, tmp_win->orig_wd,
		 tmp_win->orig_ht,TRUE);
    }
  else
    {
      new_width = tmp_win->frame_width;      
      new_height = tmp_win->frame_height;
      new_x = tmp_win->frame_x;
      new_y = tmp_win->frame_y;
      if(val1 >0)
	{
	  new_width = val1*Scr.MyDisplayWidth/100;
	  new_x = 0;
	}
      if(val2 >0)
	{
	  new_height = val2*Scr.MyDisplayHeight/100;
	  new_y = 0;
	}
      if((val1==0)&&(val2==0))
	{
	  new_x = 0;
	  new_y = 0;
	  new_height = Scr.MyDisplayHeight;
	  new_width = Scr.MyDisplayWidth;
	}
      tmp_win->flags |= MAXIMIZED;
      ConstrainSize (tmp_win, &new_width, &new_height);
      SetupFrame(tmp_win,new_x,new_y,new_width,new_height,TRUE);
    }
#ifndef NO_PAGER  
  RedrawPager();
#endif
}

/*****************************************************************************
 *
 * Grab the pointer and keyboard
 *
 ****************************************************************************/
Bool GrabEm(int cursor)
{
  int i=0,val;

  XSync(dpy,Scr.screen);  
  /* move the keyboard focus prior to grabbing the pointer to
   * eliminate the enterNotify and exitNotify events that go
   * to the windows */
  if(Scr.PreviousFocus == NULL)
    Scr.PreviousFocus = Scr.Focus;
  SetFocus(Scr.NoFocusWin,NULL);
  while((i<1000)&&(val=XGrabPointer(dpy, Scr.Root, True, ButtonPressMask | 
			       ButtonReleaseMask | ButtonMotionMask |
			       PointerMotionMask|PointerMotionHintMask,
			       GrabModeAsync, GrabModeAsync, Scr.Root,
			       Scr.FvwmCursors[cursor], CurrentTime)!=
		  GrabSuccess))
    {
      i++;
      /* If you go too fast, other windows may not get a change to release
       * any grab that they have. */
      sleep_a_little(1000);
    }
  /* If we fall out of the loop without grabbing the pointer, its
     time to give up */
  XSync(dpy,Scr.screen);
  if(val!=GrabSuccess)
    {
      return False;
    }
  return True;
}


/*****************************************************************************
 *
 * UnGrab the pointer and keyboard
 *
 ****************************************************************************/
void UngrabEm()
{
  Window w;

  XSync(dpy,Scr.screen);
  XUngrabPointer(dpy,CurrentTime);

  if(Scr.PreviousFocus != NULL)
    {
      w = Scr.PreviousFocus->w;

      /* if the window still exists, focus on it */
      if (w)
	{
	  SetFocus(w,Scr.PreviousFocus);
	}
      Scr.PreviousFocus = NULL;
    }

  XSync(dpy,Scr.screen);
}

/*****************************************************************************
 *
 * Builtin which determines if the button press was a click or double click...
 *
 ****************************************************************************/
void ComplexFunction(Window w, FvwmWindow *tmp_win, XEvent *eventp,int context,
		     MenuRoot *mr)
{
  char type = MOTION;
  char c;
  XEvent *ev;
  MenuItem *mi;
  XEvent d;
  Bool Persist = False;

  if(mr == NULL)
    return;

  mi = mr->first;
  while(mi != NULL)
    {
      /* make lower case */
      c = *(mi->item);
      if(isupper(c))
	c=tolower(c);
      if(c == 'i')
	{
	  if(tmp_win)
	    w = tmp_win->frame;
	  else
	    w = None;
	  ExecuteFunction(mi->func, mi->action,w,
			  tmp_win, eventp, context,mi->val1,mi->val2,mi->menu);
	}
      else
	Persist = True;
      mi = mi->next;
    }

  if(!Persist)
    return;

  if (DeferExecution(eventp,&w,&tmp_win,&context, SELECT,ButtonPress))
    return;

  if(!GrabEm(SELECT))
    {
      XBell(dpy,Scr.screen);
      return;
    }

  /* A window has already been selected */
  ev = eventp;

  /* Wait and see if we have a click, or a move */
  /* wait 100 msec, see if the used releases the button */
  sleep_a_little(Scr.ClickTime*1000);
  if(XCheckMaskEvent (dpy,ButtonReleaseMask, &d))
    {
      type = CLICK;
      ev = &d;
    }

  /* If it was a click, wait to see if its a double click */
  if(type == CLICK)
    {
      sleep_a_little(Scr.ClickTime*1000);
      if(XCheckMaskEvent (dpy,ButtonPressMask, &d))
	{
	  type = ONE_AND_A_HALF_CLICKS;
	  ev = &d;
	}
      if(type == ONE_AND_A_HALF_CLICKS)
	{
	  sleep_a_little(Scr.ClickTime*1000);
	  if(XCheckMaskEvent (dpy,ButtonReleaseMask, &d))
	    {
	      type = DOUBLE_CLICK;      
	      ev = &d;
	    }
	}
    }

  /* some functions operate on button release instead of 
   * presses. These gets really weird for complex functions ... */
  if(eventp->type == ButtonPress)
    eventp->type = ButtonRelease;

  mi = mr->first;
  while(mi != NULL)
    {
      /* make lower case */
      c = *(mi->item);
      if(isupper(c))
	c=tolower(c);
      if(c == type)
	{
	  if(tmp_win)
	    w = tmp_win->frame;
	  else
	    w = None;
	  ExecuteFunction(mi->func, mi->action,w,
			  tmp_win, eventp, context,mi->val1,mi->val2,mi->menu);
	}
      mi = mi->next;
    }
  UngrabEm();
}


/* For Ultrix 4.2 */
#include <sys/types.h>

#include <sys/time.h>


/**************************************************************************
 * 
 * Sleep for n microseconds
 *
 *************************************************************************/
void sleep_a_little(int n)
{
  struct timeval value;
  
  if (n <= 0)
    return;
  
  value.tv_usec = n % 1000000;
  value.tv_sec = n / 1000000;
  
  (void) select(1, 0, 0, 0, &value);
}
