/****************************************************************************
 * This module is based on Twm, but has been siginificantly modified 
 * by Rob Nation (nation@rocket.sanders.lockheed.com)
 ****************************************************************************/
/*****************************************************************************/
/**       Copyright 1988 by Evans & Sutherland Computer Corporation,        **/
/**                          Salt Lake City, Utah                           **/
/**  Portions Copyright 1989 by the Massachusetts Institute of Technology   **/
/**                        Cambridge, Massachusetts                         **/
/**                                                                         **/
/**                           All Rights Reserved                           **/
/**                                                                         **/
/**    Permission to use, copy, modify, and distribute this software and    **/
/**    its documentation  for  any  purpose  and  without  fee is hereby    **/
/**    granted, provided that the above copyright notice appear  in  all    **/
/**    copies and that both  that  copyright  notice  and  this  permis-    **/
/**    sion  notice appear in supporting  documentation,  and  that  the    **/
/**    names of Evans & Sutherland and M.I.T. not be used in advertising    **/
/**    in publicity pertaining to distribution of the  software  without    **/
/**    specific, written prior permission.                                  **/
/**                                                                         **/
/**    EVANS & SUTHERLAND AND M.I.T. DISCLAIM ALL WARRANTIES WITH REGARD    **/
/**    TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES  OF  MERCHANT-    **/
/**    ABILITY  AND  FITNESS,  IN  NO  EVENT SHALL EVANS & SUTHERLAND OR    **/
/**    M.I.T. BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL  DAM-    **/
/**    AGES OR  ANY DAMAGES WHATSOEVER  RESULTING FROM LOSS OF USE, DATA    **/
/**    OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER    **/
/**    TORTIOUS ACTION, ARISING OUT OF OR IN  CONNECTION  WITH  THE  USE    **/
/**    OR PERFORMANCE OF THIS SOFTWARE.                                     **/
/*****************************************************************************/


/***********************************************************************
 * fvwm - "F? Virtual Window Manager"
 ***********************************************************************/

#include <stdio.h>
#include <signal.h>
#include <fcntl.h>
#include <string.h>
#include <sys/wait.h>
#include <unistd.h>
#include "fvwm.h"
#include "menus.h"
#include "misc.h"
#include "screen.h"
#include <X11/Xproto.h>
#include <X11/Xatom.h>
/* need to get prototype for XrmUniqueQuark for XUniqueContext call */
#include <X11/Xresource.h>
#ifdef SHAPE
#include <X11/extensions/shape.h>
#endif /* SHAPE */

ScreenInfo Scr;		        /* structures for the screen */
Display *dpy;			/* which display are we talking to */
extern char *config_file;

XErrorHandler CatchRedirectError(Display *, XErrorEvent *);
XErrorHandler FvwmErrorHandler(Display *, XErrorEvent *);
void newhandler(int sig);
void CreateCursors(void);
void NoisyExit(int);

XContext FvwmContext;		/* context for fvwm windows */

XClassHint NoClass;		/* for applications with no class */

int JunkX = 0, JunkY = 0;
Window JunkRoot, JunkChild;		/* junk window */
unsigned int JunkWidth, JunkHeight, JunkBW, JunkDepth, JunkMask;

/* assorted gray bitmaps for decorative borders */
#define g_width 2
#define g_height 2
static char g_bits[] = {0x02, 0x01};

#define l_g_width 4
#define l_g_height 2
static char l_g_bits[] = {0x08, 0x02};
Bool debugging = False,PPosOverride;

char **g_argv;

#ifndef NO_PAGER
extern Pixel PagerForeColor;
#endif

#ifdef SHAPE
int ShapeEventBase, ShapeErrorBase;
#endif

#define USAGE "Fvwm Ver 0.99\n\nusage:  fvwm [-d dpy] [-debug] [-f config_file]\n"

/***********************************************************************
 *
 *  Procedure:
 *	main - start of fvwm
 *
 ***********************************************************************
 */
void main(int argc, char **argv)
{
    char *display_name = NULL;
    unsigned long valuemask;	/* mask for create windows */
    XSetWindowAttributes attributes;	/* attributes for create windows */
    void InternUsefulAtoms (void);
    void InitVariables(void);
    void enterAlarm(int);
    int i,child;
    extern int x_fd;
    int len;
    char *display_string;
    char message[255];
    char num[10];
    Bool single = False;

    for (i = 1; i < argc; i++) 
      {
	if (strncasecmp(argv[i],"-debug",6)==0)
	  debugging = True;
	if (strncasecmp(argv[i],"-s",2)==0)
	  single = True;
	else if (strncasecmp(argv[i],"-d",2)==0)
	  {
	    if (++i >= argc)
	      fprintf (stderr,USAGE);	      
	    display_name = argv[i];
	  }
	else if (strncasecmp(argv[i],"-f",2)==0)
	  {
	    if (++i >= argc)
	      fprintf (stderr,USAGE);	      
	    config_file = argv[i];
	  }
	else 
	  fprintf (stderr,USAGE);
      }

    g_argv = argv;

    newhandler (SIGINT);
    newhandler (SIGHUP);
    newhandler (SIGQUIT);
    newhandler (SIGTERM);

    if (signal (SIGSEGV, SIG_IGN) != SIG_IGN)
      signal (SIGSEGV, NoisyExit);

    signal(SIGALRM,enterAlarm);
    /* catch dying processes so they don't become zombies */
    /* Massive reduction in zombie zapping complexity for fvwm-0.97 */
    signal (SIGCHLD,SIG_IGN);


    if (!(dpy = XOpenDisplay(display_name))) 
      {
	fvwm_err("can't open display %s", XDisplayName(display_name),
		 NULL,NULL);
	exit (1);
      }
    x_fd = XConnectionNumber(dpy);
    
    if (fcntl(x_fd, F_SETFD, 1) == -1) 
      {
	fvwm_err("close-on-exec failed",NULL,NULL,NULL);
	exit (1);
      }
    Scr.screen= DefaultScreen(dpy);
    Scr.NumberOfScreens = ScreenCount(dpy);

    if(!single)
      {
	for(i=0;i<Scr.NumberOfScreens;i++)
	  {
	    if(i!= Scr.screen)
	      {
		sprintf(message,"%s -d %s",argv[0],XDisplayString(dpy));
		len = strlen(message);
		message[len-1] = 0;
		sprintf(num,"%d",i);
		strcat(message,num);
		strcat(message," -s ");
		if(debugging)
		  strcat(message," -debug");
		strcat(message," -f ");
		strcat(message,config_file);
		strcat(message," &\n");
		system(message);
	      }
	  }
      }
	    
	    

    /*  Add a DISPLAY entry to the environment, incase we were started
     * with rxvt -display term:0.0
     */
    len = strlen(XDisplayString(dpy));
    display_string = malloc(len+10);
    sprintf(display_string,"DISPLAY=%s",XDisplayString(dpy));
    putenv(display_string);
    
    Scr.Root = RootWindow(dpy, Scr.screen);
    if(Scr.Root == None) 
      {
	fvwm_err("Screen %d is not a valid screen",(char *)Scr.screen,
		 NULL,NULL);
	exit(1);
      }

#ifdef SHAPE
    XShapeQueryExtension(dpy, &ShapeEventBase, &ShapeErrorBase);
#endif /* SHAPE */

    InternUsefulAtoms ();

    /* Make sure property priority colors is empty */
    XChangeProperty (dpy, Scr.Root, _XA_MIT_PRIORITY_COLORS,
		     XA_CARDINAL, 32, PropModeReplace, NULL, 0);

    XSetErrorHandler((XErrorHandler)CatchRedirectError);
    XSelectInput(dpy, Scr.Root,
		 ColormapChangeMask | EnterWindowMask | PropertyChangeMask | 
		 SubstructureRedirectMask | KeyPressMask |
		 ButtonPressMask | ButtonReleaseMask | PointerMotionMask);
    XSync(dpy, Scr.screen);

    XSetErrorHandler((XErrorHandler)FvwmErrorHandler);
    CreateCursors();
    InitVariables();

    /* read config file, set up menus, colors, fonts */
    MakeMenus();

    if(Scr.d_depth<2)
      {
	Scr.gray_pixmap = 
	XCreatePixmapFromBitmapData(dpy,Scr.Root,g_bits, g_width,g_height,
				    Scr.StdColors.fore,Scr.StdColors.back,
				    Scr.d_depth);	
	Scr.light_gray_pixmap = 
	XCreatePixmapFromBitmapData(dpy,Scr.Root,l_g_bits,l_g_width,l_g_height,
				    Scr.StdColors.fore,Scr.StdColors.back,
				    Scr.d_depth);
      }

    /* create a window which will accept the keyboard focus when no other 
       windows have it */
    attributes.event_mask = KeyPressMask;
    attributes.override_redirect = True;
    Scr.NoFocusWin=XCreateWindow(dpy,Scr.Root,-10, -10, 10, 10, 0, 0,
				 InputOnly,CopyFromParent,
				 CWEventMask|CWOverrideRedirect,
				 &attributes);
    XMapWindow(dpy, Scr.NoFocusWin);


    XSetInputFocus (dpy, Scr.NoFocusWin, RevertToParent, CurrentTime);
    Scr.TitleHeight=Scr.WindowFont.font->ascent+Scr.WindowFont.font->descent+3;

    XGrabServer(dpy);
    XSync(dpy, Scr.screen);
    if(debugging)
      XSynchronize(dpy,1);

    CaptureAllWindows();

    Scr.SizeStringWidth = XTextWidth (Scr.StdFont.font,
				      " 8888 x 8888 ", 13);
    attributes.border_pixel = Scr.StdColors.fore;
    attributes.background_pixel = Scr.StdColors.back;
    attributes.bit_gravity = NorthWestGravity;
    valuemask = (CWBorderPixel | CWBackPixel | CWBitGravity);
    Scr.SizeWindow = XCreateWindow (dpy, Scr.Root, 0, 0, 
				     (unsigned int)(Scr.SizeStringWidth +
						    SIZE_HINDENT*2),
				     (unsigned int) (Scr.StdFont.height +
						     SIZE_VINDENT*2),
				     (unsigned int) 0, 0,
				     (unsigned int) CopyFromParent,
				     (Visual *) CopyFromParent,
				     valuemask, &attributes);
    XUngrabServer(dpy);
    MoveResizeViewPortIndicator();
    HandleEvents();
    return;
}

/***********************************************************************
 *
 *  Procedure:
 *      CaptureAllWindows
 *
 *   Decorates all windows at start-up
 *
 ***********************************************************************/

void CaptureAllWindows(void)
{
  int i,j;
  unsigned int nchildren;
  Window root, parent, *children;

  PPosOverride = TRUE;
  XQueryTree(dpy, Scr.Root, &root, &parent, &children, &nchildren);
  /*
   * weed out icon windows
   */
  for (i = 0; i < nchildren; i++) 
    {
      if (children[i]) 
	{
	  XWMHints *wmhintsp = XGetWMHints (dpy, children[i]);
	  if (wmhintsp) 
	    {
	      if (wmhintsp->flags & IconWindowHint) 
		{
		  for (j = 0; j < nchildren; j++) 
		    {
		      if (children[j] == wmhintsp->icon_window) 
			{
			  children[j] = None;
			  break;
			}
		    }
		}
	      XFree ((char *) wmhintsp);
	    }
	}
    }
  /*
   * map all of the non-override windows
   */
  
  for (i = 0; i < nchildren; i++)
    {
      if (children[i] && MappedNotOverride(children[i]))
	{
	  XUnmapWindow(dpy, children[i]);
	  Event.xmaprequest.window = children[i];
	  HandleMapRequest ();
	}
    }
  
  XFree((char *)children);
  /* after the windows already on the screen are in place,
   * don't use PPosition */
  PPosOverride = FALSE;
}

/***********************************************************************
 *
 *  Procedure:
 *	MappedNotOverride - checks to see if we should really
 *		put a fvwm frame on the window
 *
 *  Returned Value:
 *	TRUE	- go ahead and frame the window
 *	FALSE	- don't frame the window
 *
 *  Inputs:
 *	w	- the window to check
 *
 ***********************************************************************/
int MappedNotOverride(Window w)
{
  XWindowAttributes wa;
  
  XGetWindowAttributes(dpy, w, &wa);
  return ((wa.map_state != IsUnmapped) && (wa.override_redirect != True));
}


/***********************************************************************
 *
 *  Procedure:
 *	InternUsefulAtoms:
 *            Dont really know what it does
 *
 ***********************************************************************
 */
Atom _XA_MIT_PRIORITY_COLORS;
Atom _XA_WM_CHANGE_STATE;
Atom _XA_WM_STATE;
Atom _XA_WM_COLORMAP_WINDOWS;
Atom _XA_WM_PROTOCOLS;
Atom _XA_WM_TAKE_FOCUS;
Atom _XA_WM_DELETE_WINDOW;

void InternUsefulAtoms (void)
{
  /* 
   * Create priority colors if necessary.
   */
  _XA_MIT_PRIORITY_COLORS = XInternAtom(dpy, "_MIT_PRIORITY_COLORS", False);   
  _XA_WM_CHANGE_STATE = XInternAtom (dpy, "WM_CHANGE_STATE", False);
  _XA_WM_STATE = XInternAtom (dpy, "WM_STATE", False);
  _XA_WM_COLORMAP_WINDOWS = XInternAtom (dpy, "WM_COLORMAP_WINDOWS", False);
  _XA_WM_PROTOCOLS = XInternAtom (dpy, "WM_PROTOCOLS", False);
  _XA_WM_TAKE_FOCUS = XInternAtom (dpy, "WM_TAKE_FOCUS", False);
  _XA_WM_DELETE_WINDOW = XInternAtom (dpy, "WM_DELETE_WINDOW", False);
  return;
}

/***********************************************************************
 *
 *  Procedure:
 *	newhandler: Installs new signal handler
 *
 ***********************************************************************
 */
void newhandler(int sig)
{
  if (signal (sig, SIG_IGN) != SIG_IGN)
    signal (sig, SigDone);
}

/***********************************************************************
 *
 *  Procedure:
 *	CreateCursors - Loads fvwm cursors
 *
 ***********************************************************************
 */
void CreateCursors(void)
{
  /* define cursors */
  Scr.FvwmCursors[POSITION] = XCreateFontCursor(dpy,XC_top_left_corner);
  Scr.FvwmCursors[FRAME] = XCreateFontCursor(dpy, XC_top_left_arrow);
  Scr.FvwmCursors[SYS] = XCreateFontCursor(dpy, XC_hand2);
  Scr.FvwmCursors[TITLE_CURSOR] = XCreateFontCursor(dpy, XC_top_left_arrow);
  Scr.FvwmCursors[MOVE] = XCreateFontCursor(dpy, XC_fleur);
  Scr.FvwmCursors[MENU] = XCreateFontCursor(dpy, XC_sb_left_arrow);
  Scr.FvwmCursors[WAIT] = XCreateFontCursor(dpy, XC_watch);
  Scr.FvwmCursors[SELECT] = XCreateFontCursor(dpy, XC_dot);
  Scr.FvwmCursors[DESTROY] = XCreateFontCursor(dpy, XC_pirate);
  Scr.FvwmCursors[LEFT] = XCreateFontCursor(dpy, XC_left_side);
  Scr.FvwmCursors[RIGHT] = XCreateFontCursor(dpy, XC_right_side);
  Scr.FvwmCursors[TOP] = XCreateFontCursor(dpy, XC_top_side);
  Scr.FvwmCursors[BOTTOM] = XCreateFontCursor(dpy, XC_bottom_side);
}

/***********************************************************************
 *
 *  Procedure:
 *	InitVariables - initialize fvwm variables
 *
 ***********************************************************************
 */

void InitVariables(void)
{
  FvwmContext = XUniqueContext();
  NoClass.res_name = NoName;
  NoClass.res_class = NoName;

  Scr.d_depth = DefaultDepth(dpy, Scr.screen);
  Scr.d_visual = DefaultVisual(dpy, Scr.screen);
  Scr.FvwmRoot.w = Scr.Root;
  Scr.FvwmRoot.next = 0;
  Scr.root_pushes = 0;
  Scr.pushed_window = &Scr.FvwmRoot;

  Scr.MyDisplayWidth = DisplayWidth(dpy, Scr.screen);
  Scr.MyDisplayHeight = DisplayHeight(dpy, Scr.screen);
    
  Scr.BorderWidth = BW;
  Scr.NoBorderWidth = NOFRAME_BW;
  Scr.BoundaryWidth = BOUNDARY_WIDTH;
  Scr.CornerWidth = CORNER_WIDTH;
  Scr.Focus = NULL;
  
  Scr.StdFont.font = NULL;
  Scr.StdFont.name = "fixed";
  Scr.WindowFont.name = "fixed";

  Scr.VScale = 32;
#ifndef NON_VIRTUAL  
  Scr.VxMax = 3;
  Scr.VyMax = 3;
#else
  Scr.VxMax = 1;
  Scr.VyMax = 1;
#endif
  Scr.Vx = Scr.Vy = 0;

  Scr.EdgeScrollX = Scr.EdgeScrollY = -100000;
  Scr.ScrollResistance = Scr.MoveResistance = 0;
  Scr.OpaqueSize = 5;
  Scr.ClickTime = 150;
  Scr.AutoRaiseDelay = -750;

  /* set major operating modes */
  Scr.flags = 0;
#ifdef NO_ICONS
  Scr.flags |= SuppressIcons;
#endif
  Scr.NumBoxes = 0;

  Scr.randomx = Scr.randomy = 0;
  Scr.buttons2grab = 7;

#ifndef NO_PAGER
  Scr.PagerFont.name = NULL;
  Scr.PagerFont.height = 0;
  Scr.PagerFont.y = 0;
  Scr.FvwmPager = (FvwmWindow *)0;
  Scr.Pager_w = None;
#endif


  Scr.left_button_styles[0][0] = -55;
  Scr.left_button_styles[1][0] = -25;
  Scr.left_button_styles[0][1] = -35;
  Scr.left_button_styles[1][1] = -10;
  Scr.left_button_styles[0][2] = 1;
  Scr.left_button_styles[1][2] = 1;
  Scr.left_button_styles[0][3] = 35;
  Scr.left_button_styles[1][3] = 10;
  Scr.left_button_styles[0][4] = 56;
  Scr.left_button_styles[1][4] = 25;

  Scr.right_button_styles[0][0] = -28;
  Scr.right_button_styles[1][0] = -28;
  Scr.right_button_styles[0][1] = -50;
  Scr.right_button_styles[1][1] = -50;
  Scr.right_button_styles[0][2] = 1;
  Scr.right_button_styles[1][2] = 1;
  Scr.right_button_styles[0][3] = 50;
  Scr.right_button_styles[1][3] = 50;
  Scr.right_button_styles[0][4] = 28;
  Scr.right_button_styles[1][4] = 28;
  return;
}



/***********************************************************************
 *
 *  Procedure:
 *	Reborder - Removes fvwm border windows
 *
 ***********************************************************************
 */

void Reborder(void)
{
  FvwmWindow *tmp;			/* temp fvwm window structure */

  /* put a border back around all windows */

  XGrabServer (dpy);
  
  InstallWindowColormaps (0, &Scr.FvwmRoot);	/* force reinstall */
  for (tmp = Scr.FvwmRoot.next; tmp != NULL; tmp = tmp->next)
    {
      RestoreWithdrawnLocation (tmp); 
      XMapWindow (dpy, tmp->w);
    }
  XUngrabServer (dpy);
  XSetInputFocus (dpy, PointerRoot, RevertToPointerRoot,CurrentTime);
}

/***********************************************************************
 *
 *  Procedure: NoisyExit
 *	Print error messages and die. (segmentation violation)
 *
 ***********************************************************************
 */
void NoisyExit(int nonsense)
{
  XErrorEvent event;
  
  fvwm_err("Seg Fault",NULL,NULL,NULL);
  event.error_code = 0;
  event.request_code = 0;
  FvwmErrorHandler(dpy, &event);

  /* Attempt to do a re-start of fvwm */
  Done(0,NULL);
}




/***********************************************************************
 *
 *  Procedure:
 *	Done - cleanup and exit fvwm
 *
 ***********************************************************************
 */
void SigDone(int nonsense)
{
  Done(0, NULL);
  SIGNAL_RETURN;
}

void Done(int restart, char *command)
{

#ifndef NON_VIRTUAL
  MoveViewport(0,0,True);
#endif

  Reborder ();
  if(restart)
    execvp(command,g_argv);
  else
    {
      XCloseDisplay(dpy);
      exit(0);
    }
}

/***********************************************************************
 *
 *  Procedure:
 *	CatchRedirectError - Figures out if there's another WM running
 *
 ***********************************************************************
 */
XErrorHandler CatchRedirectError(Display *dpy, XErrorEvent *event)
{
  fvwm_err("another WM is running",NULL,NULL,NULL);
  exit(1);
}


/***********************************************************************
 *
 *  Procedure:
 *	FvwmErrorHandler - displays info on internal errors
 *
 ************************************************************************/
XErrorHandler FvwmErrorHandler(Display *dpy, XErrorEvent *event)
{
  extern int last_event_type;

  /* some errors are acceptable, mostly they're caused by 
   * trying to update a lost  window */
  if((event->error_code == BadWindow)||(event->request_code == X_GetGeometry)||
     (event->error_code==BadDrawable)||(event->request_code==X_SetInputFocus)||
     (event->request_code == X_InstallColormap))
    return 0 ;

  fvwm_err("internal error",NULL,NULL,NULL);
  fprintf(stderr,"      Request %d, Error %d\n", event->request_code,
	  event->error_code);
  fprintf(stderr,"      EventType: %d",last_event_type);
  fprintf(stderr,"\n");
  return 0;
}

void fvwm_err(char *message, char *arg1, char *arg2, char *arg3)
{
  fprintf(stderr,"fvwm: ");
  fprintf(stderr,message,arg1,arg2,arg3);
  fprintf(stderr,"\n");
}




