/*  Copyright 1992, 1993 Robert Nation (nation@rocket.sanders.lockheed.com)
 *
 *  You can do what you like with this source code  as long as you include an
 *  unaltered copy of this message (including the copyright).
 *
 * As usual, the author accepts no responsibility for anything, nor does
 * he guarantee anything whatsoever.
 */
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#define CLASS	        "Clock"
#define MESS_CLASS	"Appointment"

#define MW_EVENTS   (ExposureMask | StructureNotifyMask)
#define MESS_EVENTS (ExposureMask | StructureNotifyMask | ButtonPressMask)

/* screen info */
Display		*display;
int             screen;		   /* the X screen number */
int             MyDisplayWidth, MyDisplayHeight;
int             x_fd;              /* file descriptor of X server connection */

/* windows and their sizes */
Window		main_win;               /* parent window */
int             main_height=100, main_width = 100;

#ifdef ICONWIN
Window		icon_win;               /* icon window */
int             icon_height=65, icon_width = 65;
#endif

#ifdef REMINDERS
Window		mess_win;                /* message window */
int             mess_w,mess_h;
#endif

/* assorted graphics info */
GC 		gc;	                 /* GC for drawing text */

#ifdef REMINDERS
XFontStruct     *font;
int             font_height;
char            *font_string = "fixed";
#endif

XColor          background_color;
XColor          foreground_color;
#define BW 1                            /* Window border width */

char            *app_name = CLASS;      /* the resource name */
#ifdef REMINDERS
char            *mess_name = MESS_CLASS; 
#endif

#ifdef ICONWIN
int             iconic = 0;             /* iconic startup? */
#endif

#ifdef REMINDERS
int             Mapped = 0;             /* is message win mapped? */
int             alarmTime= -1;
char            message[256]="";
#endif

static XSizeHints sizehints = {
	PMinSize | PResizeInc | PBaseSize,
	0, 0, 80, 80,	/* x, y, width and height */
	1, 1,		/* Min width and height */
	0, 0,		/* Max width and height */
	1, 1,		/* Width and height increments */
	{1, 1},           /* x,y increments */
	{1, 1},	        /* Aspect ratio */
	0, 0 ,		/* base size */
	0
};

/* save from computing sine(x), use pre-computed values
 * There are *100, to avoid using floats */
short sine[]={0,105,208,309,407,500,588,669,743,809,866,914,951,
	      978,995,999,995,978,951,914,866,809,743,669,588,500,
	      407,309,208,105,0,-105,-208,-309,-407,-500,-588,-669,
	      -743,-809,-866,-914,-951,-978,-995,-999,-994,-978,-951,
	      -914,-866,-809,-743,-669,-588,-500,-407,-309,-208,-105 };

/* subroutine declarations */
void extractResources(char *, char *, char *);
void createWindows(int,char **);
void getEvent(void);
void error(char *);
void DrawWindow(Window, int, int);
void SetOffAlarm(void);
void GetNewAlarm(int);
int GetOneNum(char **,int);
void CantAlloc(char *);
void NoColor(char *);
void NoProp(void);

/* Arguments for GetNewAlarm() */
#define REPLACE 0
#define UPDATE 1

/* name of the .rclock file */
#ifdef REMINDERS
char *home_file;
#endif

#ifdef MAIL
char *mail_file;
#endif

/****************************************************************************
 *
 * rclock - Robs clock - simple X windows clock with appointment reminder
 *
 ****************************************************************************/
void main(int argc,char **argv)
{
#ifdef REMINDERS
  char *config_file = ".rclock";
  int HomeLen;
  char *Home;
#endif
  int i;
  char *display_name = NULL;
  char *bg_color = "white";
  char *fg_color = "black";
  char *geom_string = "100x100";
  XGCValues gcv;

  display_name = getenv("DISPLAY");

  /* parse the command line */
  for(i=1;i<argc;i+=2)
    {
      if(strcmp(argv[i],"-display")==0)
	display_name = argv[i+1];
      else if(strcmp(argv[i],"-geometry")==0)
	geom_string = argv[i+1];
      else if(strcmp(argv[i],"-fg")==0)  
	fg_color = argv[i+1];
      else if(strcmp(argv[i],"-bg")==0)
	bg_color = argv[i+1];
#ifdef REMINDERS
      else if(strcmp(argv[i],"-font")==0)
	font_string = argv[i+1];
#endif
#ifdef ICONWIN
      else if(strcmp(argv[i],"-ic")==0)
	{
	  iconic = 1;
	  i--;
	}
#endif
      else 
	{
	  fprintf(stderr,"rclock arguments:
-display <name>	        specify the display (server)
-geometry <spec>	the initial window geometry\n");
	  fprintf(stderr,"-bg <colour>		background color
-fg <colour>		foreground color\n");
#ifdef ICONWIN
	  fprintf(stderr,"-ic                     start iconic\n");
#endif
	  error("exiting");
	}
    }

  /* open display */
  if (!(display = XOpenDisplay(display_name)))
    error("Can't open display");

  /* get display info */
  x_fd = XConnectionNumber(display);
  screen = DefaultScreen(display);
  MyDisplayWidth = DisplayWidth(display, screen);
  MyDisplayHeight = DisplayHeight(display, screen);

  extractResources(geom_string, fg_color, bg_color);
  createWindows(argc,argv);
#ifdef REMINDERS
  /* load the font for messages */
  if ((font = XLoadQueryFont(display,font_string)) == NULL) 
    {
      fprintf(stderr,"can't access font %s\n",font_string);
      error("Exiting");
    }
  font_height = font->ascent + font->descent;
#endif
  /*  Create the graphics contexts. */
  gcv.foreground = foreground_color.pixel;
  gcv.background = background_color.pixel;
#ifdef REMINDERS
  gcv.font = font->fid;
  gc = XCreateGC(display,main_win,GCFont|GCForeground|GCBackground,&gcv);
#else
  gc = XCreateGC(display,main_win,GCForeground|GCBackground,&gcv);
#endif

  /* get the mail file name */
#ifdef MAIL
  mail_file = getenv("MAIL");
#endif

  /* find the .rclock file */
#ifdef REMINDERS
  Home = getenv("HOME");
  if (Home == NULL)
    Home = "./";
  HomeLen = strlen(Home);
  
  home_file = malloc(HomeLen+strlen(config_file)+3);
  if(home_file == (char *)0)
    {
      fprintf(stderr,"rclock: couldn't malloc file name\n");
      exit(-1);
    }
  strcpy(home_file,Home);
  strcat(home_file,"/");
  strcat(home_file,config_file);
#endif
  getEvent();

}


/*****************************************************************************
 *
 *  Extract the resource fields that are needed to open the window.
 *  (geometry and colors)
 *
 *****************************************************************************/
void extractResources(char *geom_string, char *fg_color, char *bg_color)
{
  int x, y, width, height;
  int flags;
  Colormap	colormap;

  colormap = DefaultColormap(display,screen);

  flags = XParseGeometry(geom_string,&x,&y,&width,&height);
  
  if (flags & WidthValue) 
    {
      main_width = width + sizehints.base_width;
      sizehints.flags |= USSize;
    }
  if (flags & HeightValue) 
    {
      main_height = height + sizehints.base_height;
      sizehints.flags |= USSize;
    }
  if (flags & XValue) 
    {
      if (flags & XNegative)
	x = MyDisplayWidth + x - main_width - 2*BW;
      sizehints.x = x;
      sizehints.flags |= USPosition;
    }
  if (flags & YValue) 
    {
      if (flags & YNegative)
	y = DisplayHeight(display,screen) + y - main_height - 2*BW;
      sizehints.y = y;
      sizehints.flags |= USPosition;
    }
  
  /*  Do the foreground, and background colors.*/
  if (XParseColor(display,colormap,fg_color,&foreground_color) == 0)
    NoColor(fg_color);
  else 
    XAllocColor(display,colormap,&foreground_color);

  if (XParseColor(display,colormap,bg_color,&background_color) == 0)
    NoColor(bg_color);
  else
    XAllocColor(display,colormap,&background_color);

}


/**************************************************************************
 *
 *  Open and map the windows.
 *
 **************************************************************************/
static Atom wm_del_win;
void createWindows(int argc,char **argv)
{
  XTextProperty wname, iname,mname;
  XClassHint class1, class2;
  XWMHints wmhints;
  
  sizehints.width = main_width;
  sizehints.height = main_height;
  main_win = XCreateSimpleWindow(display,DefaultRootWindow(display),
				 sizehints.x,sizehints.y,
				 sizehints.width,sizehints.height,
				 BW,foreground_color.pixel,
				 background_color.pixel);
  XStringListToTextProperty(&app_name,1,&wname);

#ifdef ICONWIN
  sizehints.width = icon_width;
  sizehints.height = icon_height;
  icon_win = XCreateSimpleWindow(display,DefaultRootWindow(display),
				 sizehints.x,sizehints.y,
				 sizehints.width,sizehints.height,
				 BW,foreground_color.pixel,
				 background_color.pixel);
#endif  
  XStringListToTextProperty(&app_name,1,&iname);

  class1.res_name = app_name;
  class1.res_class = CLASS;
  wmhints.input = True;
#ifdef ICONWIN
  if(!iconic)
    wmhints.initial_state = NormalState;
  else
    wmhints.initial_state = IconicState;
  wmhints.icon_window = icon_win;
  wmhints.flags = InputHint | StateHint | IconWindowHint;
#else
  wmhints.flags = InputHint;
#endif
  XSetWMProperties(display,main_win,&wname,&iname,argv,argc,
		   &sizehints,&wmhints,&class1);

  XSelectInput(display,main_win,MW_EVENTS);
#ifdef ICONWIN
  XSelectInput(display,icon_win,MW_EVENTS);
#endif
  XMapWindow(display,main_win);

  /* create, but don't map a window for appointment reminders */
#ifdef REMINDERS
  mess_win = XCreateSimpleWindow(display,DefaultRootWindow(display),
				 sizehints.x,sizehints.y,
				 sizehints.width,sizehints.height,
				 BW,foreground_color.pixel,
				 background_color.pixel);
  if (XStringListToTextProperty(&mess_name,1,&mname) == 0) 
    error("cannot allocate window name");
  class2.res_name = mess_name;
  class2.res_class = MESS_CLASS;
  wmhints.input = True;
  wmhints.initial_state = NormalState;
  wmhints.flags = InputHint | StateHint;
  sizehints.flags |= USPosition;
  XSetWMProperties(display,mess_win,&mname,&mname,argv,argc,
		   &sizehints,&wmhints,&class2);

  XSelectInput(display,mess_win,MESS_EVENTS);
#endif

  /* want to accept the delete window protocol */
  wm_del_win = XInternAtom(display,"WM_DELETE_WINDOW",False);
  XSetWMProtocols(display,main_win,&wm_del_win,1);
#ifdef ICONWIN
  XSetWMProtocols(display,icon_win,&wm_del_win,1);
#endif
#ifdef REMINDERS
  XSetWMProtocols(display,mess_win,&wm_del_win,1);
#endif

}



/****************************************************************************
 *
 *  Redraw the whole window after an exposure or size change.
 *  Sound an alarm if needed 
 *
 ****************************************************************************/
void DrawWindow(Window win, int width, int height)
{
  time_t t;
  struct tm *tmval;
  int i,j;
  int angle1,angle2;
  int h_x,h_y,m_x,m_y,center_x,center_y;
#ifdef ICONWIN
  static int saved_day = -100;
  char time_string[100];
  char *str_ptr;
  XTextProperty name;
#endif
#ifdef REMINDERS
  int currentTime;
  static int lastUpdateTime = -10;
#endif
#ifdef MAIL
  static int mail_up = 0;
  XSetWindowAttributes attributes;
  XGCValues gcv;
  struct stat stval;

  if((mail_file != NULL)&&(!stat(mail_file, &stval) &&
			   (stval.st_mtime >= stval.st_atime) &&
			   (stval.st_blocks >0)))
     {
      if(mail_up == 0)
	{
	  /* enter reverse video */
	  gcv.background = foreground_color.pixel;
	  gcv.foreground = background_color.pixel;
	  XChangeGC(display,gc,GCForeground|GCBackground,&gcv);
	  mail_up = 1;
	}
      attributes.background_pixel = foreground_color.pixel;
      XChangeWindowAttributes(display,win,CWBackPixel, &attributes);
    }
  else
    {
      if (mail_up == 1)
	{
	  /* enter reverse video */
	  gcv.foreground = foreground_color.pixel;
	  gcv.background = background_color.pixel;
	  XChangeGC(display,gc,GCForeground|GCBackground,&gcv);
	}
      attributes.background_pixel = background_color.pixel;
      XChangeWindowAttributes(display,win,CWBackPixel, &attributes);
      mail_up = 0;
    }
#endif

  XClearWindow(display,win);

#ifdef REMINDERS
  /* for a message window, just re-draw the message */
  if(win == mess_win)
    {
      XDrawString (display, mess_win,gc, 20, 20+font->ascent,
		   message, strlen(message));  
      return;
    }
#endif

  center_x = width>>1;
  center_y = height>>1;

  /* get the current time */
  t=time(0);
  tmval = localtime(&t);

  /* draw the hands */
  angle1 = (tmval->tm_hour%12)*5 + tmval->tm_min/12;
  h_x =   sine[angle1] * width  *60/200000 + center_x;
  h_y =   -(sine[(angle1+15)%60])* height *60/200000 + center_y;

  angle2 = tmval->tm_min;
  m_x =   sine[angle2] * width  *85/200000 + center_x;
  m_y = -(sine[(angle2+15)%60])* height *85/200000 + center_y;
  for(i=-1;i<2;i++)
    for(j=-1;j<2;j++)
      {
	XDrawLine(display,win,gc,center_x+i,center_y+j,h_x,h_y);
	XDrawLine(display,win,gc,center_x+i,center_y+j,m_x,m_y);
      }

  /* draw the clock face */
  for(i=0;i<60;i+=5)
    {
      angle1  = sine[i]*width;
      angle2  = -sine[(i+15)%60]*height;
      h_x = angle1 * 9 / 20000 + center_x;
      h_y = angle2 * 9 / 20000 + center_y;
      m_x = angle1 *10 / 20000 + center_x;
      m_y = angle2 *10 / 20000 + center_y;
      XDrawLine(display,win,gc,m_x,m_y,h_x,h_y);
    }

  /* See if we need to sound an alarm 
   * convert multi-field time info to a single int */
#ifdef REMINDERS
  currentTime = tmval->tm_min + tmval->tm_hour*60 +
    tmval->tm_mday*24*60 + (tmval->tm_mon+1)*31*24*60+
      tmval->tm_year*12*31*24*60;
    
  /* see if we have an alarm to set off */

  if((alarmTime >= 0)&&
     (alarmTime <= currentTime))
    SetOffAlarm();

  /* every 10 minutes, check for revised alarm entries in .rclock */
  if((!Mapped)&&(currentTime - lastUpdateTime > 10))
    {
      GetNewAlarm(UPDATE);
      lastUpdateTime = currentTime;
    }
#endif

  /* once every day, update the window and icon name */
#ifdef ICONWIN
  if(tmval->tm_yday != saved_day)
    {
      saved_day = tmval->tm_yday;
      strftime(time_string,100,"%a %m/%d",tmval);  
      str_ptr = time_string;
      if (XStringListToTextProperty(&str_ptr,1,&name) == 0) 
	{
	  fprintf(stderr,"cannot allocate icon name");
	  return;
	}
      XSetWMName(display,main_win,&name);
      XFree(name.value);
      strftime(time_string,100,"%a %h %d",tmval);  
      str_ptr = time_string;
      if (XStringListToTextProperty(&str_ptr,1,&name) == 0) 
	{
	  fprintf(stderr,"cannot allocate window name");
	  return;
	}
      XSetWMIconName(display,main_win,&name);
      XFree(name.value);
    }
#endif
}

/**************************************************************************
 *
 * Sounds an alarm by mapping the message window
 *
 **************************************************************************/
#ifdef REMINDERS
void SetOffAlarm(void)
{
  if(Mapped)
    return;

  XBell(display,screen);

  /* compute the window size */
  mess_w=XTextWidth(font,message,strlen(message));
  mess_w += 40;

  mess_h = font_height + 40;

  /* resize and center the window */
  XMoveResizeWindow(display, mess_win, (MyDisplayWidth - mess_w)>>1,
		    (MyDisplayHeight - mess_h)>>1, mess_w, mess_h);

  /* map the sucker */
  XMapRaised(display,mess_win);

  /* Make it noisy */
  XBell(display,screen);

  Mapped = 1;
}
#endif
/****************************************************************************
 *
 * Read the .rclock file and find the next alarm to sound 
 * if update_only = 1, we look for a reminder whose time is
 *                     greater than the current time, but
 *                     less than the currently set alarm time
 * if update_only = 0, we just look for a reminder whose time
 *                     is greater than the alarm that just went
 *                     off.
 *
 ***************************************************************************/
#ifdef REMINDERS
void GetNewAlarm(int update_only)
{
  time_t t;
  struct tm *tmval;
  char line[256];
  int yy,mo,dy,mm,hh,testTime,savedTime = 1e8;
  int currentTime;
  FILE *fd;
  char *tline;
  
  fd = fopen(home_file,"r");
  if(fd == (FILE *)0)
    {
      /* no config file, don't make alarms */
      alarmTime = -1;
      return;
    }
  
  /* get the current time */
  t=time(0);
  tmval = localtime(&t);
  
  currentTime = tmval->tm_min + tmval->tm_hour*60 +
	tmval->tm_mday*24*60 + (tmval->tm_mon+1)*31*24*60+
	  tmval->tm_year*12*31*24*60;

  /* on initial startup, want to discard reminders that should
   * already have occured. */
  if(alarmTime <0)
    alarmTime = currentTime;
  
  /* now scan for next alarm */
  tline = fgets(line,(sizeof(line))-1, fd);
  while(tline != (char *)0)
    {
      /* parse the line */
      /* line format is hh:mm mo/dy/yr message */
      /* any of hh, mm, mo, dy, yr could be a *, indicating a wildcard. */
      while(isspace(*tline))tline++;
      hh = GetOneNum(&tline, tmval->tm_hour);
      tline++;
      mm = GetOneNum(&tline, tmval->tm_min);
      tline++;
      mo = GetOneNum(&tline,tmval->tm_mon+1);
      tline++;
      dy = GetOneNum(&tline,tmval->tm_mday);
      tline++;
      yy = GetOneNum(&tline,tmval->tm_year);
      testTime =  mm + hh*60 + dy*24*60 + mo*31*24*60+ yy*12*31*24*60;
      if(((testTime > alarmTime)&&(!update_only))||
	 ((testTime > currentTime)&&(update_only)))
	{
	  /* have an alarm whose time is greater than the 
	   * last alarm, now make sure its the smallest avail */
	  if(testTime < savedTime)
	    {
	      savedTime = testTime;
	      strcpy(message,(tline+1));
	      message[strlen(message)-1] = 0;
	    }
	}
      tline = fgets(line,(sizeof(line))-1, fd);
    }
  if(savedTime < 1e8)
    {
      alarmTime = savedTime;
    }
  else
    alarmTime = -1;
  fclose(fd);
  return;
}
#endif

/****************************************************************************
 *
 * Reads a single integer from *tline, returns default value if it finds "*"
 *
 ***************************************************************************/
#ifdef REMINDERS
int GetOneNum(char **tline,int def)
{
  int hh = 0;
  int hit = 0;

  while(isdigit(**tline))
    {
      hit = 1;
      hh = hh*10+(**tline-'0');
      (*tline) = *tline +1;
    }
  if(!hit)
    {
      while(**tline == '*')(*tline)++;
      hh = def;
    }
  return hh;
}
#endif

/***************************************************************************
 *
 * Loops forever, looking for stuff to do. Sleeps 1 minute if nothing to do
 *
 **************************************************************************/
void getEvent(void)
{
  fd_set in_fdset;
  XEvent event;
  int fd_width = 10;
  struct timeval tm;
  Window root;
  int x,y,border_width,depth;

  for (;;) 
    {
      /* take care of all pending X events */
      while(XPending(display))
	{
	  XNextEvent(display,&event); 
	  switch(event.type)
	    {
	    case ClientMessage:
	      /* check for delete window requests */
	      if((event.xclient.format == 32)&&
		 (event.xclient.data.l[0] == wm_del_win))
		{
		  if(event.xany.window == main_win)
		    exit(0);
#ifdef ICONWIN
		  if (event.xany.window == icon_win)
		    exit(0);
#endif
#ifdef REMINDERS
		  else
		    {
		      XUnmapWindow(display,mess_win);
		      Mapped = 0;
		      GetNewAlarm(REPLACE);
		    }
#endif
		}
	      break;
	    case Expose:
	    case GraphicsExpose:
	      /* need to re-draw a window */
	      if (event.xany.window == main_win)
		DrawWindow(main_win,main_width,main_height);
#ifdef ICONWIN
	      else if(event.xany.window == icon_win)
		DrawWindow(icon_win,icon_width,icon_height);
#endif
#ifdef REMINDERS
		else
		  DrawWindow(mess_win,mess_w,mess_h);
#endif
	      break;
	    case ConfigureNotify:
	      /* window has been re-sized */
	      if (event.xany.window == main_win)
		XGetGeometry(display,main_win,&root,&x,&y,
			     &main_width,&main_height,&border_width,&depth);
#ifdef ICONWIN
	      else if(event.xany.window == icon_win)
		XGetGeometry(display,icon_win,&root,&x,&y,
			     &icon_width,&icon_height,&border_width,&depth);
#endif
#ifdef REMINDERS
	      else
		XGetGeometry(display,mess_win,&root,&x,&y,
			     &mess_w,&mess_h,&border_width,&depth);
#endif
	      break;
	    case ButtonPress:
#ifdef REMINDERS
	      /* button press to dismiss message window */
	      if(event.xany.window == mess_win)
		{
		  XUnmapWindow(display,mess_win);
		  Mapped = 0;
		  GetNewAlarm(REPLACE);
		}
#endif
	      break;
	    }
	}

      /* Now wait for time out or new X event */
      FD_ZERO(&in_fdset);
      FD_SET(x_fd,&in_fdset);
      tm.tv_sec = 60;
      tm.tv_usec = 0;
      select(fd_width,&in_fdset,NULL,NULL,&tm);

      DrawWindow(main_win,main_width,main_height);
#ifdef ICONWIN
      DrawWindow(icon_win,icon_width,icon_height);
#endif
    }
}


/***************************************************************************
 *
 * Assorted error message routines 
 * 
 **************************************************************************/
void error(char *message)
{
  fprintf(stderr,"rclock: %s\n",message);
  exit(1);
}

void NoColor(char *s)
{
  fprintf(stderr,"rclock: invalid color %s",s);
}


