/*****************************************************************************
 *                                                                           *
 *  Copyright (c) 1993, 1994, Elan Feingold (feingold@zko.dec.com)           *
 *                                                                           *
 *     PERMISSION TO USE, COPY, MODIFY, AND TO DISTRIBUTE THIS SOFTWARE      *
 *     AND ITS DOCUMENTATION FOR ANY PURPOSE IS HEREBY GRANTED WITHOUT       *
 *     FEE, PROVIDED THAT THE ABOVE COPYRIGHT NOTICE APPEAR IN ALL           *
 *     COPIES AND MODIFIED COPIES AND THAT BOTH THAT COPYRIGHT NOTICE AND    *
 *     THIS PERMISSION NOTICE APPEAR IN SUPPORTING DOCUMENTATION.  THERE     *
 *     IS NO REPRESENTATIONS ABOUT THE SUITABILITY OF THIS SOFTWARE FOR      *
 *     ANY PURPOSE.  THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS       *
 *     OR IMPLIED WARRANTY.                                                  *
 *                                                                           *
 *****************************************************************************/

#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>

#include "types.h"
#include "network.h"
#include "riskgame.h"
#include "server.h"
#include "deck.h"
#include "client.h"
#include "debug.h"


#define MAX_DESCRIPTORS (MAX_CLIENTS+5)*2

Client   pClients[MAX_CLIENTS];
Int      iNumClients;
Int      iServerSock;
Int      iState;
fd_set   fdSet, fdBackup;
Boolean  pfDescInUse[MAX_DESCRIPTORS];
Deck    *pFreePlayers;
Int      iReply, iAllClients;

/* Private functions */
void SRV_DistributeCountries(void);
void SRV_CollectPlayers(Int iClientsStarted);
Int  SRV_AllocClient(void);
void SRV_FreeClient(Int i);
Int  SRV_GetPlayerIDOfPlayer(Int iIndex);
void UTIL_ExitProgram(Int iExitValue);

/* Server message handlers */
void SRV_HandleEXIT(void);
void SRV_HandleALLOCPLAYER(Int iClient);
void SRV_HandleFREEPLAYER(void *pvMessage);
void SRV_HandleREPLYPACKET(void *pvMessage);
void SRV_HandleMESSAGEPACKET(Int iClient, void *pvMessage);
void SRV_HandleENTERSTATE(void *pvMessage);
void SRV_HandleDEREGISTERCLIENT(Int iClient);


/************************************************************************ 
 *  FUNCTION: main
 *  HISTORY: 
 *     01.23.94  ESF  Created.
 *     02.22.94  ESF  Cleaned up a bit, removing warnings.
 *     05.08.94  ESF  Fixed MSG_MESSAGEPACKET handling.
 *     05.10.94  ESF  Fixed, not registering needed callback with DistObj.
 *     05.12.94  ESF  Added fd usage map.
 *     05.19.94  ESF  Fixed bug, passing &pvMessage instead of pvMessage.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void main(void)
{
  struct sockaddr_in   server;
  Int                  i;

  /* Startup the memory debugging system */
  MEM_BootStrap("server-memory.log");
  
  /* Init. clients and players */
  iNumClients = 0;

  /* Init the fields of the clients */
  for (i=0; i!=MAX_CLIENTS; i++)
    {
      pClients[i].strAddress   =  NULL;
      pClients[i].iReadSocket  =  -1;
      pClients[i].iWriteSocket =  -1;
      pClients[i].fActive      =  FALSE;
    }
  
  /* Init the fd usage map */
  for (i=0; i!=MAX_DESCRIPTORS; i++)
    pfDescInUse[i] = FALSE;

  /* Initialize the pool of free players */
  pFreePlayers = DECK_Create(MAX_PLAYERS);

  /* Initialize the distributed object */
  RISK_Initialize(MODE_SERVER | MODE_PERSISTANT, 0, 0);
  RISK_SetBroadcastCallback(SRV_BroadcastMessage);
  RISK_SetFinishBroadcastCallback(SRV_BroadcastToOthers);
  RISK_SetSendMsgCallback(NET_SendMessage);

  /* Create server socket */
  if ((iServerSock=socket(AF_INET, SOCK_STREAM, 0)) < 0)
    {
      perror("Creating socket");
      UTIL_ExitProgram(1);
    }
  
  /* Name sockets using wildcards */
  server.sin_family = AF_INET;
  server.sin_addr.s_addr = INADDR_ANY;
  server.sin_port = htons(RISK_PORT);
  
  /* Bind the socket to the port */
  if (bind(iServerSock, (struct sockaddr *)&server, sizeof(server)))
    {
      perror("Binding stream socket");
      UTIL_ExitProgram(1);
    }

  SRV_CollectPlayers(0);

  /* Don't accept any more connections -- Change this!!! */
  /* close(iServerSock); */

  SRV_PlayGame();
}


/************************************************************************ 
 *  FUNCTION: SRV_BroadcastTextMessage
 *  HISTORY: 
 *     01.26.94  ESF  Created 
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_BroadcastTextMessage(String strMessage)
{
  MsgMessagePacket   msgMess;

  msgMess.strMessage  = strMessage;
  msgMess.strFrom     = "Server";
  
  /* Send the message out to all clients */
  SRV_BroadcastMessage(MSG_MESSAGEPACKET, &msgMess);
}


/************************************************************************ 
 *  FUNCTION: SRV_SocketToClientName
 *  HISTORY: 
 *     01.29.94  ESF  Created 
 *     05.15.94  ESF  Fixed bug, didn't work for general client #s.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
char *SRV_SocketToClientName(Int iSocket)
{
  Int i;

  for (i=0; i!=MAX_CLIENTS; i++)
    if ((pClients[i].iWriteSocket == iSocket ||
	 pClients[i].iReadSocket == iSocket) &&
	pClients[i].fActive == TRUE)
      return(pClients[i].strAddress);
  
  return ("Unknown Client");
}


/************************************************************************ 
 *  FUNCTION: SRV_SocketToClientName
 *  HISTORY: 
 *     01.29.94  ESF  Created 
 *     05.15.94  ESF  Fixed bug, didn't work for general client #s.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Int SRV_SocketToClient(Int iSocket)
{
  Int i;

  for (i=0; i!=MAX_CLIENTS; i++)
    if ((pClients[i].iWriteSocket == iSocket ||
	 pClients[i].iReadSocket == iSocket) &&
	pClients[i].fActive == TRUE)
      return(i);
  
  return (-1);
}


/************************************************************************ 
 *  FUNCTION: SRV_PlayGame
 *  HISTORY: 
 *     02.04.94  ESF  Created.
 *     02.05.94  ESF  Fixed broadcast loop bug. 
 *     02.05.94  ESF  Fixed message receive bug.
 *     03.03.94  ESF  Changed to send _UPDATE to all clients but sender.
 *     03.28.94  ESF  Added _DEADPLAYER & _ENDOFGAME.
 *     03.29.94  ESF  Added _REQUESTCARD.
 *     04.01.94  ESF  Fixed card exchange to work right with jokers.
 *     04.11.94  ESF  Fixed CARDPACKET to broadcast the card.
 *     05.05.94  ESF  Added MSG_OBJ* msgs.
 *     05.06.94  ESF  Factored out dealing cards code.
 *     05.15.94  ESF  Added MSG_[ALLOC|FREE]PLAYER.
 *     05.15.94  ESF  Added MSG_REPLYPACKET.
 *     05.17.94  ESF  Added MSG_NETMESSAGE.
 *     06.24.94  ESF  Fixed memory leak bug.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_PlayGame(void)
{
  Int               i, n, iMessageType, iTurn;
  void             *pvMessage;
  Deck             *pCardDeck;
  MsgTurnNotify     msgTurn;
  Int               iNumVoted=0, iNeedToRegister=0, iPlayer;
  Boolean           fGotMessage;

  /* Create the decks */
  pCardDeck = DECK_Create(NUM_COUNTRIES + 2);

  /* Deal out the countries */
  SRV_DistributeCountries();

  /* Select the turn */
  srand((Int)clock());
  iTurn = rand() % RISK_GetNumLivePlayers();
  iPlayer = SRV_GetPlayerIDOfPlayer(iTurn);

  /* Let the first player know that it is his/her turn */
  msgTurn.iPlayer = iPlayer;
  msgTurn.iClient = RISK_GetClientOfPlayer(iPlayer);
  SRV_BroadcastMessage(MSG_TURNNOTIFY, &msgTurn);

  /* Add all of the client's file descriptors to the set */
  FD_ZERO(&fdBackup);
  for (i=0; i!=iNumClients; i++)
    FD_SET(pClients[i].iReadSocket, &fdBackup);
  
  /* Loop for the entirety of the game */
  for(;;)
    {
      fdSet = fdBackup;
      
      if (select(22, &fdSet, (fd_set *)0, (fd_set *)0, NULL) < 0)
	perror("Select");
      
      /* Something happened, find out what */
      for (i=0, fGotMessage=FALSE; i!=MAX_CLIENTS && !fGotMessage; i++)
	if (FD_ISSET(pClients[i].iReadSocket, &fdSet) &&
	    pClients[i].fActive == TRUE)
	  {
	    /* We got the message we were looking */
	    fGotMessage = TRUE;

	    /* Actually pick it up */
	    NET_RecvMessage(pClients[i].iReadSocket, &iMessageType, 
			    &pvMessage);
	    
	    switch (iMessageType)
	      {
	      /********************/
	      case MSG_ALLOCPLAYER:
		SRV_HandleALLOCPLAYER(i);
		break;
		
	      /********************/
	      case MSG_FREEPLAYER:
		SRV_HandleFREEPLAYER(pvMessage);
		break;

	      /********************/
	      case MSG_NETMESSAGE:
		SRV_BroadcastToOthers(i, iMessageType, pvMessage);
		break;
		
	      /********************/
	      case MSG_REPLYPACKET:
		SRV_HandleREPLYPACKET(pvMessage);
		break;
		
	      /********************/
	      case MSG_MESSAGEPACKET:
		SRV_HandleMESSAGEPACKET(i, pvMessage);
		break;

	      /********************/
	      case MSG_DELETEMSGDST:
		SRV_BroadcastToOthers(i, iMessageType, pvMessage);
		  break;

	      /*********************/
	      case MSG_GAMENEWPLAYERS:
	      case MSG_GAMEMOREPLAYERS:
	      case MSG_GAMEANOTHER:
	      case MSG_DEREGISTERCLIENT:
	      /*********************/
		
		/* This message is coming from a client who may want to
		 * be involved in another game.  Based on the message,
		 * we need to fiddle with the players of the client.
		 * Only start the new game when all of the clients have
		 * "voted."
		 */

		/* If this is a departing client, destroy it */
		if (iMessageType == MSG_DEREGISTERCLIENT)
		  {
		    MsgFreePlayer  m;
		    Int            j;
		    
		    SRV_HandleDEREGISTERCLIENT(i);

		    /* Get rid of the players that lived at this client */
		    for (j=0; j!=MAX_PLAYERS; j++)
		      if (RISK_GetClientOfPlayer(j) == i)
			{
			  m.iPlayer = j;
			  SRV_HandleFREEPLAYER(&m);
			}
		  }

		/* Do this on the first pass */
		if (iNumVoted == 0)
		  RISK_ResetGame();
		
		/* Someone just voted */
		iNumVoted++;

		/* Are they going to need to register players? */
		if(iMessageType == MSG_GAMENEWPLAYERS ||
		   iMessageType == MSG_GAMEMOREPLAYERS)
		  iNeedToRegister++;

		/* If the number of clients that have voted is the number
		 * of clients that existed in this last game, then go
		 * ahead and start the next game.
		 */

		if (iNumVoted == iAllClients)
		  {
		    Int iPlayer;

		    /* Set the number of AllClients to be the real number */
		    iAllClients = iNumClients;

		    /* Switch to a new deck of cards */
		    DECK_Destroy(pCardDeck);
		    pCardDeck = DECK_Create(NUM_COUNTRIES + 2);
		    
		    /* Init. DistObj based on type of new game */
		    if (iNeedToRegister)
		      SRV_CollectPlayers(iAllClients-iNeedToRegister);

		    SRV_DistributeCountries();
		    
		    /* Let the first player know that it is his/her turn */
		    iTurn = rand() % RISK_GetNumLivePlayers();
		    iPlayer = SRV_GetPlayerIDOfPlayer(iTurn);
		    msgTurn.iPlayer = iPlayer;
		    msgTurn.iClient = RISK_GetClientOfPlayer(iPlayer);
		    SRV_BroadcastMessage(MSG_TURNNOTIFY, &msgTurn);

		    /* Reset these for the next time */
		    iNumVoted = iNeedToRegister = 0;
		  }
		break;


	      /*****************/
	      case MSG_STARTGAME:
		/* The client finished registering already */
		iNeedToRegister--;
		break;

	      /********************/
	      case MSG_OBJSTRUPDATE:
	      case MSG_OBJINTUPDATE:
	      /********************/
		RISK_ProcessMessage(iMessageType, pvMessage, i);
		break;

	      /***************/
	      case MSG_ENDTURN:
	      /***************/
		{
		  Int iPlayer;

		  /* Assert: There is a player still alive. */
		  iTurn = (iTurn+1) % RISK_GetNumLivePlayers();
		  
		  iPlayer = SRV_GetPlayerIDOfPlayer(iTurn);

		  msgTurn.iPlayer = iPlayer;
		  msgTurn.iClient = RISK_GetClientOfPlayer(iPlayer);
		  SRV_BroadcastMessage(MSG_TURNNOTIFY, &msgTurn);
		}
		break;

	      /******************/
	      case MSG_ENTERSTATE:
		SRV_HandleENTERSTATE(pvMessage);
		break;
		
	      /*****************/
	      case MSG_ENDOFGAME:
	      /*****************/		
		{
		  /* Let everyone else know */
		  SRV_BroadcastMessage(MSG_ENDOFGAME, NULL);
		}
		break;
		
	      /*********************/
	      case MSG_EXCHANGECARDS:
	      /*********************/
		{
		  MsgExchangeCards *pMess = (MsgExchangeCards *)pvMessage;
		  MsgReplyPacket    msgMess;
		  Int               iNumJokers;

		  /* Put the cards back on the deck and change them to type */
		  for (n=iNumJokers=0; n!=3; n++)
		    {
		      DECK_PutCard(pCardDeck, pMess->piCards[n]);

		      if (pMess->piCards[n] < NUM_COUNTRIES)
			pMess->piCards[n] %= 3;
		      else
			pMess->piCards[n] = -1, iNumJokers++;
		    }

		  /* Find out how many armies the player gets in 
		   * exchange for the cards and send them to him or
		   * her, in an _UPDATEARMIES message.  Right now
		   * the only option is fixed return values for card
		   * exchanges.
		   */

		  /* Do we have one of each (possibly with jokers)? */
		  if ((pMess->piCards[0] != pMess->piCards[1] &&
		       pMess->piCards[1] != pMess->piCards[2] &&
		       pMess->piCards[0] != pMess->piCards[2]) ||
		      iNumJokers == 2)
		    {
		      msgMess.iReply = 10;
		    }
		  else if (pMess->piCards[0]==0 ||
			   pMess->piCards[1]==0 ||
			   pMess->piCards[2]==0)
		    {
		      msgMess.iReply = 8;  
		    }
		  else if (pMess->piCards[0]==1 ||
			   pMess->piCards[1]==1 ||
			   pMess->piCards[2]==1)
		    {
		      msgMess.iReply = 6;  
		    }
		  else 
		    {
		      msgMess.iReply = 4;  
		    }
			
		  NET_SendMessage(pClients[i].iWriteSocket, MSG_REPLYPACKET,
				  &msgMess);
		}
		break;

	      /*******************/
	      case MSG_REQUESTCARD:
	      /*******************/
		{
		  MsgRequestCard   *pMess = (MsgRequestCard *)pvMessage;

		  RISK_SetCardOfPlayer(pMess->iPlayer, 
				      RISK_GetNumCardsOfPlayer(pMess->iPlayer),
				      DECK_GetCard(pCardDeck));
		  RISK_SetNumCardsOfPlayer(pMess->iPlayer,
				   RISK_GetNumCardsOfPlayer(pMess->iPlayer)+1);
		}
		break;

	      /************/
	      case MSG_EXIT:
	      /************/
		SRV_HandleEXIT();
		break;
		  
	      default:
		printf("Exiting, nasty message [%d].\n", iMessageType);
		
		/* Let other clients know */
		SRV_BroadcastMessage(MSG_EXIT, NULL);

		UTIL_ExitProgram(-1);
	      }

	    /* Free up the memory the message was taking */
	    NET_DeleteMessage(iMessageType, pvMessage);
	  }
    }
}


/************************************************************************ 
 *  FUNCTION: SRV_BroadcastMessage
 *  HISTORY: 
 *     02.05.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_BroadcastMessage(Int iMessageType, void *pvMess)
{
  Int i;

  for (i=0; i!=MAX_CLIENTS; i++)
    if (pClients[i].fActive)
      NET_SendMessage(pClients[i].iWriteSocket, iMessageType, pvMess);
}


/************************************************************************ 
 *  FUNCTION: SRV_BroadcastToOthers
 *  HISTORY: 
 *     03.28.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_BroadcastToOthers(Int iClientExclude, Int iMessageType, void *pvMess)
{
  Int i;

  for (i=0; i!=MAX_CLIENTS; i++)
    if (i!=iClientExclude && pClients[i].fActive)
      NET_SendMessage(pClients[i].iWriteSocket, iMessageType, pvMess);
}


/************************************************************************ 
 *  FUNCTION: 
 *  HISTORY: 
 *     05.12.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_DistributeCountries(void)
{
  Deck  *pCountryDeck = DECK_Create(NUM_COUNTRIES);
  Int    iCountry, iPlayer, i, iTurn;

  /* Dole out the countries */
  for(i=0, iPlayer=0; i!=NUM_COUNTRIES; i++)
    {
      iTurn = SRV_GetPlayerIDOfPlayer(iPlayer);
      iCountry = DECK_GetCard(pCountryDeck);

      RISK_SetOwnerOfCountry(iCountry, iTurn);
      RISK_SetNumArmiesOfCountry(iCountry, 1);
      RISK_SetNumCountriesOfPlayer(iTurn, 
				   RISK_GetNumCountriesOfPlayer(iTurn)+1);
      RISK_SetNumArmiesOfPlayer(iTurn, RISK_GetNumArmiesOfPlayer(iTurn)-1);
      iPlayer = (++iPlayer) % RISK_GetNumPlayers();
    }
  
  DECK_Destroy(pCountryDeck);  
}


/************************************************************************ 
 *  FUNCTION: SRV_GetPlayerIDOfPlayer
 *  HISTORY: 
 *     05.15.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Int SRV_GetPlayerIDOfPlayer(Int iIndex)
{
  Int i, iCount;

  for (i=0, iCount=-1; i!=MAX_PLAYERS && iIndex!=iCount; i++)
    if (RISK_GetClientOfPlayer(i) != -1 &&
	RISK_GetStateOfPlayer(i) == TRUE)
      iCount++;

  if (iIndex == iCount)
    return (i-1);
  else 
    return -1;
}


/************************************************************************ 
 *  FUNCTION: SRV_CollectPlayers
 *  HISTORY: 
 *     05.12.94  ESF  Created from existing code in main().
 *     05.15.94  ESF  Added MSG_REPLYPACKET.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_CollectPlayers(Int iClientsStarted)
{
  void                *pvMessage;
  Int                  iMessageType, iNewSocket;
  Int                  i, j, n;
  fd_set               fdRead;
  char                 strScratch[256];
  Boolean              fGotMessage;

  /* Start accepting connections */
  if (iServerSock>=0)
    listen(iServerSock, 5);
  
  /* This is only useful when we are starting new games */
  SRV_BroadcastMessage(MSG_STARTREGISTRATION, NULL);

  while (RISK_GetNumPlayers()<2 || iNumClients!=iClientsStarted)
    {
      /* Set up a select mask for connections */
      FD_ZERO(&fdRead);

      /* Add client's sockets */
      for (i=0; i<=MAX_DESCRIPTORS; i++)
	if (pfDescInUse[i]==TRUE)
	  FD_SET(i, &fdRead);

      /* Plus listen to where the conections are going to come in off of */
      if (iServerSock>=0)
	FD_SET(iServerSock, &fdRead);

      /* Wait for something to happen */
      if (select(22, &fdRead, (fd_set *)0, (fd_set *)0, NULL) < 0)
	perror("Select");
      
      /* If there is a new client and we haven't too many already, accept */
      if (FD_ISSET(iServerSock, &fdRead) && iNumClients <= MAX_CLIENTS)
	{
	  if((iNewSocket = accept(iServerSock, 0, 0)) == -1)
	    perror("Accept");
	  else 
	    {
	      if (iNewSocket < MAX_DESCRIPTORS)
		pfDescInUse[iNewSocket] = TRUE;
	      else
		{
		  printf("Fatal Error! SERVER: fd out of range for map.\n");
		  UTIL_ExitProgram(0);
		}
	    }
	}
      else
	{
	  for (i=0, fGotMessage=FALSE; 
	       i!=MAX_DESCRIPTORS && !fGotMessage; 
	       i++)
	    if (FD_ISSET(i, &fdRead))
	      {
		/* We just found the source of the message */
		fGotMessage = TRUE;

		/* Pick the message up */
		NET_RecvMessage(i, &iMessageType, &pvMessage);
		
		switch (iMessageType)
		  {
		  /********************/
		  case MSG_ALLOCPLAYER:
		    SRV_HandleALLOCPLAYER(SRV_SocketToClient(i));
		    break;
		    
		  /********************/
		  case MSG_FREEPLAYER:
		    SRV_HandleFREEPLAYER(pvMessage);
		    break;

		  /********************/
		  case MSG_OBJSTRUPDATE:
		  case MSG_OBJINTUPDATE:
		    RISK_ProcessMessage(iMessageType, pvMessage, 
					SRV_SocketToClient(i));
		    break;

		  /**********************/		    
		  case MSG_MESSAGEPACKET:
		    SRV_HandleMESSAGEPACKET(SRV_SocketToClient(i), pvMessage);
		    break;
		   
		  /**********************/
		  case MSG_REGISTERCLIENT:
		    {
		      Boolean              fFound=FALSE;
		      MsgRegisterClient   *pmsgRegisterClient;
		      
		      pmsgRegisterClient = (MsgRegisterClient *)pvMessage;
		      
		      /* See if this is first or second socket */
		      for (j=0; j!=iNumClients && fFound==FALSE; j++)
			if (!strcmp(pmsgRegisterClient->strClientAddress, 
				    pClients[j].strAddress))
			  fFound = TRUE, j--;
		      
		      if (fFound)
			{
			  MsgClientIdent msgID;

			  printf("Receiving second CommLink from %s\n", 
				 pClients[j].strAddress);
			  pClients[j].iWriteSocket = i;

			  /* Let the client know its identification # */
			  msgID.iClientID = j;
			  NET_SendMessage(i, MSG_CLIENTIDENT, &msgID);

			  /* Let the new client know about old clients */
			  if (iNumClients>1)
			    {
			      MsgMessagePacket  msgMess;

			      /* Let the new client know about old clients */
			      for (n=0; n!=iNumClients-1; n++)
				{
				  sprintf(strScratch, 
					  "Client, \"%s\", has registered.",
					  pClients[n].strAddress);

				  msgMess.strMessage = strScratch;
				  msgMess.strFrom = "Server";
				  NET_SendMessage(i, MSG_MESSAGEPACKET,
						  &msgMess);
				}
			    }
  
			  /* If there are any players, let the new client 
			   * know about them, so that it can set up the 
			   * colors.
			   */

			  if (RISK_GetNumPlayers())
			    {
			      /* Send it the needed fields */
			      RISK_SelectiveReplicate(i, GEN_NUMPLAYERS,
						      0, 0, 0, 0);
			      RISK_SelectiveReplicate(i, PLR_COLORINDEX,
						      0, 
						      RISK_GetNumPlayers()-1,
						      0, 0);
			      RISK_SelectiveReplicate(i, PLR_COLORSTRING,
						      0, 
						      RISK_GetNumPlayers()-1,
						      0, 0); 
			      RISK_SelectiveReplicate(i, PLR_NAME,
						      0, 
						      RISK_GetNumPlayers()-1,
						      0, 0);
			    }

			  /* Let all clients know about it */
			  sprintf(strScratch, 
				  "A new client, %s, just registered.",
				  pClients[j].strAddress);
			  SRV_BroadcastTextMessage(strScratch);
			}
		      else
			{
			  Int iNewClient;

			  /* Get the handle to a new client */
			  iNewClient = SRV_AllocClient();

			  /* Set up its fields */
			  printf("A new client, %s, just showed up.\n",
				 pmsgRegisterClient->strClientAddress);
			  pClients[iNewClient].fActive = TRUE;
			  pClients[iNewClient].iReadSocket = i;
			  pClients[iNewClient].strAddress = (String)malloc
			    (strlen(pmsgRegisterClient->strClientAddress)+1);
			  strcpy(pClients[iNewClient].strAddress, 
				 pmsgRegisterClient->strClientAddress);
			}
		    }
		    break;
		    
		  /*****************/
		  case MSG_STARTGAME:
		    {
		      sprintf(strScratch, 
			      "%s has finished registering players...", 
			      SRV_SocketToClientName(i));
		      printf("%s\n", strScratch);
		      SRV_BroadcastTextMessage(strScratch); 
		      
		      /* We have a client ready to go */
		      iClientsStarted++;
		    }

		    break;

                  /************************/
		  case MSG_DEREGISTERCLIENT:
		    SRV_HandleDEREGISTERCLIENT(SRV_SocketToClient(i));
		    break;

                  /******************/
		  case MSG_ENTERSTATE:
		    SRV_HandleENTERSTATE(pvMessage);
		    break;
    
		  /********************/
		  case MSG_REPLYPACKET:
		    SRV_HandleREPLYPACKET(pvMessage);
		    break;

		  /**********************/
		  case MSG_EXIT:
		    SRV_HandleEXIT();
		    break;
		    
		  default:
		    printf("Received garbage [%d], I'm outta here.\n", 
			   iMessageType);

		    /* Let other clients know */
		    SRV_BroadcastMessage(MSG_EXIT, NULL);

		    UTIL_ExitProgram(0);
		  }

		/* Free message memory */
		NET_DeleteMessage(iMessageType, pvMessage);
	      }
	}
    }

  /* Set the initial number of armies */
  for (i=0; i!=RISK_GetNumPlayers(); i++)
    RISK_SetNumArmiesOfPlayer(SRV_GetPlayerIDOfPlayer(i), 
			      40-RISK_GetNumPlayers());

  /* Don't let any more clients connect */
  close(iServerSock); iServerSock = -1;
}



/************************************************************************ 
 *  FUNCTION: SRV_AllocClient
 *  HISTORY: 
 *     05.12.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Int SRV_AllocClient(void)
{
  Int i;

  /* Find a free client slot and return it.
   * Since we wouldn't have accepted the connection if there weren't
   * a slot available, this call cannot fail.
   */

  for(i=0; i!=MAX_CLIENTS; i++)
    if (pClients[i].fActive == FALSE)
      {
	iNumClients++;
	iAllClients++;
	return(i);
      }

  printf("Fatal Error!  CLIENT:  No free slots available.\n");
  SRV_BroadcastMessage(MSG_EXIT, NULL);
  UTIL_ExitProgram(-1);

  /* For the compiler :) */
  return(-1);
}


/************************************************************************ 
 *  FUNCTION: SRV_FreeClient
 *  HISTORY: 
 *     05.12.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void SRV_FreeClient(Int i)
{
  /* Get rid of the client */
  iNumClients--;
  
  /* Reset the fields, close the sockets */
  pClients[i].fActive = FALSE;
  close(pClients[i].iWriteSocket);
  close(pClients[i].iReadSocket);

  /* Remove them from the map */
  pfDescInUse[pClients[i].iWriteSocket] = FALSE;
  pfDescInUse[pClients[i].iReadSocket] = FALSE;

  /* Remove the sockets from the fd_set */
  FD_CLR(pClients[i].iReadSocket, &fdBackup); 
}


/***********************/
void SRV_HandleEXIT(void)
{
  Int n;

  /* Let other clients know */
  SRV_BroadcastMessage(MSG_EXIT, NULL);
  
  /* close the listening socket */
  close(iServerSock);

  /* close all of the player sockets */
  for(n=0; n!=iNumClients; n++)
    {
      close(pClients[n].iReadSocket);
      close(pClients[n].iWriteSocket);
    }
  UTIL_ExitProgram(0);
}


/*************************************/
void SRV_HandleALLOCPLAYER(Int iClient)
{
  MsgReplyPacket mess;
  
  /* Get a player and return it, or -1 if none available */
  mess.iReply = DECK_GetCard(pFreePlayers);
  
  /* If there was a valid player, make a note */
  if (mess.iReply != -1)
    {
      RISK_SetNumPlayers(RISK_GetNumPlayers()+1);
      RISK_SetNumLivePlayers(RISK_GetNumLivePlayers()+1);
    }
  
  NET_SendMessage(pClients[iClient].iWriteSocket, 
		  MSG_REPLYPACKET, &mess);
}


/*****************************************/
void SRV_HandleREPLYPACKET(void *pvMessage)
{
  iReply = ((MsgReplyPacket *)pvMessage)->iReply; 
}


/*****************************************************/
void SRV_HandleFREEPLAYER(void *pvMessage)
{
  Int iPlayer;

  /* Put the player ID back onto pool of free players */
  iPlayer = ((MsgFreePlayer *)(pvMessage))->iPlayer;
  DECK_PutCard(pFreePlayers, iPlayer);
  
  /* Make a note of this */
  RISK_SetClientOfPlayer(iPlayer, -1);
  RISK_SetNumPlayers(RISK_GetNumPlayers()-1);
  RISK_SetNumLivePlayers(RISK_GetNumLivePlayers()-1);
}


/********************************************************/
void SRV_HandleMESSAGEPACKET(Int iClient, void *pvMessage)
{
  MsgMessagePacket *pMess = (MsgMessagePacket *)pvMessage;
  
  if (pMess->iTo == DST_ALLPLAYERS)
    SRV_BroadcastMessage(MSG_MESSAGEPACKET, pvMessage);
  else if (pMess->iTo == DST_ALLBUTME)
    SRV_BroadcastToOthers(iClient, MSG_MESSAGEPACKET, pvMessage);
  else
    NET_SendMessage(pClients[pMess->iTo].iWriteSocket, 
		    MSG_MESSAGEPACKET, pvMessage);
}


/****************************************/
void SRV_HandleENTERSTATE(void *pvMessage)
{
  iState = ((MsgEnterState *)pvMessage)->iState; 
}


/******************************************/
void SRV_HandleDEREGISTERCLIENT(Int iClient)
{
  MsgDeleteMsgDst mess;
  
  mess.iClient = iClient;
  SRV_FreeClient(iClient); 
  SRV_BroadcastToOthers(iClient, MSG_DELETEMSGDST, &mess);
}


/************************************************************************ 
 *  FUNCTION: 
 *  HISTORY: 
 *     06.16.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void UTIL_ExitProgram(Int iExitValue)
{
  MEM_TheEnd();
  exit(iExitValue);
}
