/*
**                      Softwarepraktikum iMaze
**                              1993/94
**                Joerg Czeranski    Hans-Ulrich Kiel
**
** Datei: xv_ereignisse.c
**
** Kommentar:
**  Gibt bei Auftreten von Ereignissen audio-visuelle Signale aus
*/


#include <xview/xview.h>
#include <xview/canvas.h>
#include <xview/panel.h>
#include <xview/defaults.h>

#ifdef SOUND
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <time.h>
#include <sys/ioctl.h>

#ifdef sun

#ifdef SUN_AUDIOIO_H
#include <sun/audioio.h>
#else
#include <sys/audioio.h>
#endif

#include <stropts.h>

#define AUDIO_CTL "/dev/audioctl"
#define DEV_AUDIO "/dev/audio"
#define ULAW
#undef SOUND_NICHT_MISCHEN

#else
#ifdef linux

#include <linux/soundcard.h>

#undef AUDIO_CTL
#define DEV_AUDIO "/dev/dsp"
#undef ULAW

#else
#undef SOUND
#endif
#endif

#endif

#include "system.h"
#include "signale.h"
#include "ereignisse.h"
#include "einaus.h"
#include "xv_einaus.h"

static char sccsid[] = "@(#)xv_ereignisse.c	1.41 17 Dec 1994";


#ifdef SOUND

/* Daten pro Sekunde */
#define SAMPLING_RATE 8000

#define	PUFFER_GROESSE (SAMPLING_RATE / 10)
#define	SCHLUSS_STILLE (SAMPLING_RATE / 200)

/* maximale Anzahl von Sounds, die gemischt werden */
#define MAX_MISCH 10

/* Verzeichnis, in dem die Sounds gesucht werden */
#ifndef DEFAULT_SOUND_DIR
#define DEFAULT_SOUND_DIR "."
#endif

#ifdef ULAW
/* Parameter fuer die linear -> u-law Wandlung */
static int ulaw_parameter[16][3] =
{
	1152, 258, 128 * 1157,
	960, 900, 64 * 964,
	480, 0, 32 * 482,
	224, 17, 16 * 225,
	1, 0, 8 * 1,
	1, 0, 4 * 1,
	18, 3, 2 * 19,
	1, 0, 1 * 1,
	-1, 0, 1 * 1,
	-1, 0, 2 * 1,
	-1, 0, 4 * 1,
	-104, 9, 8 * 105,
	-224, 17, 16 * 225,
	-224, 33, 32 * 225,
	-256, 129, 64 * 257,
	-256, 0, 128 * 257
};

/* linear -> u-law Tabelle */
static unsigned char ulaw[8192];

#endif

/* u-law -> linear Tabelle */
static short linear[256] =
{
	-32256, -31228, -30200, -29172, -28143, -27115, -26087, -25059,
	-24031, -23002, -21974, -20946, -19918, -18889, -17861, -16833,
	-16062, -15548, -15033, -14519, -14005, -13491, -12977, -12463,
	-11949, -11435, -10920, -10406, -9892, -9378, -8864, -8350,
	-7964, -7707, -7450, -7193, -6936, -6679, -6422, -6165,
	-5908, -5651, -5394, -5137, -4880, -4623, -4365, -4108,
	-3916, -3787, -3659, -3530, -3402, -3273, -3144, -3016,
	-2887, -2759, -2630, -2502, -2373, -2245, -2116, -1988,
	-1891, -1827, -1763, -1698, -1634, -1570, -1506, -1441,
	-1377, -1313, -1249, -1184, -1120, -1056, -992, -927,
	-879, -847, -815, -783, -751, -718, -686, -654,
	-622, -590, -558, -526, -494, -461, -429, -397,
	-373, -357, -341, -325, -309, -293, -277, -261,
	-245, -228, -212, -196, -180, -164, -148, -132,
	-120, -112, -104, -96, -88, -80, -72, -64,
	-56, -48, -40, -32, -24, -16, -8, 0,
	32256, 31228, 30200, 29172, 28143, 27115, 26087, 25059,
	24031, 23002, 21974, 20946, 19918, 18889, 17861, 16833,
	16062, 15548, 15033, 14519, 14005, 13491, 12977, 12463,
	11949, 11435, 10920, 10406, 9892, 9378, 8864, 8350,
	7964, 7707, 7450, 7193, 6936, 6679, 6422, 6165,
	5908, 5651, 5394, 5137, 4880, 4623, 4365, 4108,
	3916, 3787, 3659, 3530, 3402, 3273, 3144, 3016,
	2887, 2759, 2630, 2502, 2373, 2245, 2116, 1988,
	1891, 1827, 1763, 1698, 1634, 1570, 1506, 1441,
	1377, 1313, 1249, 1184, 1120, 1056, 992, 927,
	879, 847, 815, 783, 751, 718, 686, 654,
	622, 590, 558, 526, 494, 461, 429, 397,
	373, 357, 341, 325, 309, 293, 277, 261,
	245, 228, 212, 196, 180, 164, 148, 132,
	120, 112, 104, 96, 88, 80, 72, 64,
	56, 48, 40, 32, 24, 16, 8, 0
};

/* Namen der Ereignisse fuer Dateinamen */
static char *ereignis_namen[ERG_ANZ + 1] =
{
	"failure",
	"action",
	"shoot",
	"miss",
	"injured",
	"dead",
	"wound",
	"kill",
	"pause",
	"suspended",
	"awake",
	"title"
};

/* Puffer fuer die verschiedenen Sounds */
struct
{
	int laenge;    /* Laenge des Sounds */
	short *inhalt; /* Inhalt des Sounds (linear) */
} sounds[ERG_ANZ + 1];

/* Puffer fuer die zur Zeit abzuspielenden Sounds */
struct
{
	int laenge;      /* restliche Laenge */
	short *position; /* restlicher Inhalt (linear) */
} abspiel[MAX_MISCH];

static int abspiel_anz;      /* Anzahl der zur Zeit abzuspielenden Sounds */
static int audio_laeuft;     /* Audio aktiviert? */
static int audio_deskriptor; /* Deskriptor fuer das Audio-Device */

static Frame audio_frame; /* Fenster, zu dem audio_handler gehoert */

/* Puffer fuer zu spielende Audio-Daten */
static unsigned char audio_puffer[PUFFER_GROESSE];

#ifdef AUDIO_CTL
static int audio_ctl_deskriptor; /* Deskriptor fuer das
                                    Audio-Kontroll-Device */
static int eof_zaehler;          /* Anzahl bisher abgespielter
                                    Audio-Pakete */
#endif
#endif

static Window flash_window, beep_window; /* Fenster, in dem Signale
                                            ausgegeben werden sollen */


#ifdef SOUND
/*
** fehler_text
**  ermittelt den zur Systemvariable errno passenden Fehlertext
**
** Rueckgabewert:
**  Zeiger in das Feld sys_errlist oder in eine lokale Variable, die
**  den Fehler als Nummer enthaelt (falls kein Standardtext vorhanden ist)
**
** Seiteneffekte:
**  der zuletzt zurueckgegebene Zeiger wird evtl. ungueltig
*/
static char *fehler_text()
{
	extern int errno;           /* Systemvariable: letzter aufgetretener
	                               Fehlercode oder 0 */
	extern char *sys_errlist[]; /* Liste der Standard-Fehlertexte */
	extern int sys_nerr;        /* Anzahl der Eintraege in sys_errlist */

	/* falls errno kein gueltiger Index fuer sys_errlist ist, nur die
	   Nummer als String zurueckgeben */
	if (errno < 0 || errno >= sys_nerr)
	{
		static char text[20]; /* Puffer fuer Fehlernummer */

		sprintf(text, "Error #%d", errno);

		/* Zeiger auf Puffer zurueckgeben; Achtung: dieser Zeiger
		   ist immer derselbe */
		return text;
	}

	/* Standard-Fehlertext zurueckgeben */
	return sys_errlist[errno];
}
#endif


#ifdef SOUND
#ifdef ULAW
/*
** init_ulaw
**  initialisiert die linear -> u-law Wandlungstabelle
**
** Seiteneffekte:
**  ulaw wird initialisiert
*/
static void init_ulaw()
{
	int i, j, k, offset, von, bis;

	for (j = k = 0; k < 16; k++)
		for (i = 0;; i++)
		{
			int wert;

			wert = ulaw_parameter[k][0] * (i + ulaw_parameter[k][1]) /
				ulaw_parameter[k][2];
			if (!i)
			{
				if (k < 8)
				{
					von = 16 * k;
					bis = von + 16 - (k == 7);
				}
				else
				{
					von = 383 - 16 * k;
					bis = von - 16;
				}
				offset = von - wert;
			}
			wert += offset;
			if (wert == bis)
				break;
			ulaw[j++] = wert;
		}
}
#endif


/*
** xv_audio_handler
**  wird aufgerufen, wenn wieder Audio-Daten geschrieben werden koennen
**
** Parameter:
**  client: Dummy-Parameter
**  signal: Dummy-Parameter
**  mode: Dummy-Parameter
**
** Rueckgabewert:
**  NOTIFY_DONE oder NOTIFY_ERROR
*/
static Notify_value xv_audio_handler(client, signal, mode)
Notify_client client;
int signal;
Notify_signal_mode mode;
{
	static void audio_handler();

	audio_handler(0);

	return NOTIFY_DONE;
}


/*
** audio_handler
**  berechnet und spielt Audio-Daten
**
** Parameter:
**  neu: Flag, ob neue Ereignisse ausgeloest wurden
**
** Seiteneffekte:
**  abspiel und abspiel_anz werden veraendert
*/
static void audio_handler(neu)
int neu;
{
	static int audio_puffer_laenge = 0; /* Laenge der noch
	                                       abzuspielenden Audio-Daten */
#ifdef AUDIO_CTL
	int freigeben; /* muss SIGPOLL wieder freigegeben werden? */

	if (freigeben = !signal_blockiert(SIGPOLL))
		signal_blockieren(SIGPOLL); /* Signal blockieren */
#endif

	for (;;)
	{
		/* keine Daten mehr im Puffer? */
		if (!audio_puffer_laenge)
		{
			int i;                             /* Index fuer die
			                                      abzuspielenden Sounds */
			int linear_puffer[PUFFER_GROESSE]; /* Puffer fuer die linearen
			                                      Audio-Daten */

			/* alle aktuell abzuspielenden Sounds bearbeiten */
			for (i = 0; i < abspiel_anz; i++)
			{
				int laenge; /* restliche Laenge des Sounds; hoechstens
				               soviel wie ein Puffer-Inhalt */
				int j;      /* Index fuer den Audio-Puffer */

				/* Laenge berechnen */
				laenge = abspiel[i].laenge > PUFFER_GROESSE ?
					PUFFER_GROESSE : abspiel[i].laenge;

				/* war Audio-Puffer kuerzer als jetzt
				   abzuspielende Daten? */
				if (laenge > audio_puffer_laenge)
				{
					/* Puffer initialisieren */
					for (j = audio_puffer_laenge; j < laenge; j++)
						linear_puffer[j] = 0;

					/* Laenge anpassen */
					audio_puffer_laenge = laenge;
				}

				/* lineare Daten aufaddieren */
				for (j = 0; j < laenge; j++)
					linear_puffer[j] += abspiel[i].position[j];

				/* Sound zuende? */
				if (!(abspiel[i].laenge -= laenge))
				{
					/* Sound muss nicht mehr weiter abgespielt werden */
					abspiel_anz--;
					abspiel[i].laenge = abspiel[abspiel_anz].laenge;
					abspiel[i].position = abspiel[abspiel_anz].position;

					/* Schleifenindex fuer die Sounds korrigieren */
					i--;

					/* naechster Sound */
					continue;
				}

				/* Position der als naechstes abzuspielenden Daten */
				abspiel[i].position += laenge;
			}

			/* Audio-Puffer nicht leer? */
			if (audio_puffer_laenge)
				for (i = 0; i < audio_puffer_laenge; i++)
				{
					int wert; /* linearer Audio-Wert */

					wert = linear_puffer[i];

					/* Wert begrenzen */
					if (wert < -32768)
						wert = -32768;
					else if (wert > 32767)
						wert = 32767;

					/* Wert konvertieren (u-law oder ein Byte linear) */
#ifdef ULAW
					audio_puffer[i] = ulaw[(wert + 32768) >> 3];
#else
					audio_puffer[i] = (wert + 32768) >> 8;
#endif
				}
		}

		/* wenn Audio-Puffer leer, Schleife beenden */
		if (!audio_puffer_laenge)
			break;

#ifdef AUDIO_CTL
		{
			/* Audio-Paket-Zaehler abfragen */

			struct audio_info info; /* Status des Audio-Kontroll-Devices */

			/* Status abfragen */
			if (ioctl(audio_ctl_deskriptor, AUDIO_GETINFO, &info))
			{
				static char *meldung[] = { "iMaze - Sound Error", "",
					"Can't check audio play counter:", NULL, NULL };

				meldung[3] = fehler_text();
				uebler_fehler(meldung, NULL);
			}

			/* Puffer des Audio-Devices noch nicht abgespielt? */
			if (eof_zaehler != info.play.eof)
			{
				/* dann noch keine neuen Daten spielen */
				if (freigeben)
					signal_freigeben(SIGPOLL); /* Signal wieder freigeben */

				return;
			}
		}
#endif

		{
			int laenge; /* Lange der geschriebenen Daten */

#ifdef AUDIO_CTL
			/* Audio-Paket-Zaehler erhoehen */
			eof_zaehler++;

			/* Signal ausloesen, wenn fertig */
			write(audio_deskriptor, audio_puffer, 0);
#endif

#ifdef SOUND_NICHT_MISCHEN
			/* neue Sounds? */
			if (neu)
			{
				/* alte Sounds abbrechen */
				ioctl(audio_deskriptor, SNDCTL_DSP_RESET, 0);
				neu = 0;
			}
#endif

			/* Daten in das Audio-Device schreiben */
			laenge = write(audio_deskriptor, audio_puffer,
				audio_puffer_laenge);

			/* Fehler? */
			if (laenge < 0)
			{
#ifdef AUDIO_CTL
				/* Signal freigeben */
				if (freigeben)
					signal_freigeben(SIGPOLL);
#else
				/* xv_audio_handler aufrufen, wenn weitere Daten
				   geschrieben werden koennen */
				notify_set_output_func(audio_frame,
					xv_audio_handler, audio_deskriptor);
#endif

				/* fertig */
				return;
			}

			/* Puffer verkuerzen */
			audio_puffer_laenge -= laenge;
			if (audio_puffer_laenge)
			{
				int i; /* Index */

				/* Daten nach vorne verschieben */
				for (i = audio_puffer_laenge; i--;)
					audio_puffer[i] = audio_puffer[i + laenge];

#ifdef AUDIO_CTL
				/* Signal freigeben */
				if (freigeben)
					signal_freigeben(SIGPOLL);
#endif

				/* fertig */
				return;
			}

			/* Puffer noch nicht leer, neu versuchen */
		}
	}

#ifdef AUDIO_CTL
	/* Signal freigeben */
	if (freigeben)
		signal_freigeben(SIGPOLL);
#else
	/* xv_audio_handler nicht mehr aufrufen */
	notify_set_output_func(audio_frame, NULL, audio_deskriptor);
#endif
}
#endif


/* bis hier lokaler Teil                       */
/***********************************************/
/* ab hier globaler Teil                       */


/*
** ereignisse_darstellen
**  fuehrt die Ausgabe der Signale aus
**
** Parameter:
**  ereignisse: Feld aller Ereignisse, in dem die auszugebenen gesetzt sind
**
** Seiteneffekte:
**  abspiel und abspiel_anz werden veraendert
*/
void ereignisse_darstellen(ereignisse)
char ereignisse[ERG_ANZ];
{
#ifdef SOUND
	/* Audio spielen? */
	if (audio_laeuft)
	{
		char erg_nr;          /* Index fuer Ereignisse */
		int neu;              /* neues Ereignis aufgetreten? */
		void audio_handler(); /* Routine zum Abspielen */
#ifdef AUDIO_CTL
		int freigeben;        /* muss SIGPOLL wieder freigegeben werden? */

		if (freigeben = !signal_blockiert(SIGPOLL))
			signal_blockieren(SIGPOLL); /* Signal blockieren */
#endif

		for (erg_nr = neu = 0; erg_nr < ERG_ANZ; erg_nr++)

			/* Ereignis aufgetreten und passender Sound vorhanden? */
			if (ereignisse[erg_nr] && sounds[erg_nr].laenge)
			{
#ifdef SOUND_NICHT_MISCHEN
				/* erstes neu hinzugekommenes Ereignis? */
				if (!neu)

					/* vorige Sounds beenden */
					abspiel_anz = 0;
#endif

				/* noch Platz fuer neue Sounds? */
				if (abspiel_anz < MAX_MISCH)
				{
					/* Startposition des Sounds setzen */
					abspiel[abspiel_anz].laenge = sounds[erg_nr].laenge;
					abspiel[abspiel_anz].position = sounds[erg_nr].inhalt;

					/* ein Sound mehr zum Abspielen */
					abspiel_anz++;

					/* neue Ereignisse aufgetreten, neue Sounds */
					neu = 1;
				}
			}

#ifdef AUDIO_CTL
		if (freigeben)
			signal_freigeben(SIGPOLL); /* Signal wieder freigeben */
#endif

		audio_handler(neu); /* Sound starten */

		/* nicht Blinken/Piepen ausloesen */
		return;
	}
#endif

	/* Falls Spieler selbst abgeschossen wurde, wird das Fenster kurz
	   invertiert und ein Piep ausgegeben*/
	if (ereignisse[ABGESCHOSSEN_ERG])
		xv_set(flash_window,
			WIN_ALARM,
			0);

	/* Falls Spieler selbst getroffen wurde, wird ein Piep ausgegeben */
	else if (ereignisse[GETROFFEN_ERG])
		xv_set(beep_window,
			WIN_ALARM,
			0);

	/* Falls eine Aktion nicht moeglich war, wird ein Piep ausgegeben */
	else if (ereignisse[AKTION_FEHLER_ERG])
		xv_set(beep_window,
			WIN_ALARM,
			0);

	/* Falls Spieler vom Server aus dem Spiel genommen wurde, wird
	   ein Piep ausgegeben */
	else if (ereignisse[SUSPENDIERT_ERG])
		xv_set(beep_window,
			WIN_ALARM,
			0);
}


/*
** xv_ereignisse_init
**  sucht nach XView-Objekten, in denen die Signale ausgegeben werden
**
** Parameter:
**  hauptframe: Fenster, von dem alle anderen Fenster abstammen
**  panel: Panel, in dem Buttons ergaenzt werden duerfen
**
** Rueckgabewert:
**  0 fuer Erfolg, 1 fuer Fehler
**
** Seiteneffekte:
**  beep_window und flash_window werden gesetzt
*/
int xv_ereignisse_init(hauptframe, panel, argc, argv)
Frame hauptframe;
Panel panel;
int *argc;
char **argv;
{
	int i;            /* Index fuer Unterfenster */
	Window subwindow; /* Unterfenster */
#ifdef SOUND
    static char audio_verzeichnis[64]; /* Verzeichnis, in dem die
	                                      Sound-Dateien liegen */
#endif

	/* Pieps werden im Hauptfenster ausgeloest */
	beep_window = hauptframe;

	/* Suche nach einem Unterfenster vom Typ CANVAS */
	for (i = 1; (subwindow = xv_get(hauptframe,
		FRAME_NTH_SUBWINDOW, i)) != 0; i++)
		if ((Xv_pkg *)xv_get(subwindow, XV_TYPE) == CANVAS)
		{
			/* Das canvas_paint_window wird zum Invertieren
			   genommen */
			flash_window = canvas_paint_window(subwindow);
			goto gefunden;
		}

	/* falls kein CANVAS gefunden */
	flash_window = beep_window;

gefunden:
	;

#ifdef SOUND
	/* Fenster fuer Audio-Signal-Handler merken */
	audio_frame = hauptframe;

	{
		int ueberspringen; /* Anzahl der ausgewerteten
		                      Kommandozeilenparameter */

		/* Parameter initialisieren */
		audio_laeuft = 0;

		/* Name des Audio-Verzeichnisses abfragen und kopieren */
		strncpy(audio_verzeichnis,
			defaults_get_string("iMaze.sound.directory",
			"IMaze.Sound.Directory", DEFAULT_SOUND_DIR),
			sizeof audio_verzeichnis);
		audio_verzeichnis[sizeof audio_verzeichnis - 1] = 0;

		/* Kommandozeilenparameter auswerten */
		for (ueberspringen = 1; ueberspringen < *argc &&
			argv[ueberspringen][0] == '-'; ueberspringen++)
		{
			for (i = 1; argv[ueberspringen][i] != 0 &&
				strchr("s", argv[ueberspringen][i]) != NULL; i++);

			if (strchr("S", argv[ueberspringen][i]) != NULL)
			{
				if (argv[ueberspringen][i + 1] == 0 &&
					ueberspringen == *argc - 1)
					break;
			}
			else if (argv[ueberspringen][i] != 0 || i < 2)
				break;

			/* argv[ueberspingen] faengt mit "-" an und enthaelt
			   nur bekannte Optionen; Optionen auswerten */
			for (i = 1; i > 0 && argv[ueberspringen][i] != 0;)
				switch (argv[ueberspringen][i++])
				{
					case 'S': /* Sounds in alternativem Verzeichnis
					             suchen und spielen */
						if (argv[ueberspringen][i] == 0)
							ueberspringen++, i = 0;
						strncpy(audio_verzeichnis, argv[ueberspringen] + i,
							sizeof audio_verzeichnis);
						audio_verzeichnis[sizeof audio_verzeichnis - 1] = 0;
						i = 0;

					case 's': /* Sounds spielen */
						audio_laeuft = 1;
				}
		}

		/* "--" ignorieren */
		if (ueberspringen < *argc && !strcmp(argv[ueberspringen], "--"))
			ueberspringen++;

		/* mehr als argv[0] ausgewertet? */
		if (ueberspringen > 1)
		{
			/* Aufrufparameter verschieben, ausgewertete ueberschreiben */
			ueberspringen--;
			*argc -= ueberspringen;
			for (i = 1; i < *argc; i++)
				argv[i] = argv[i + ueberspringen];
		}
	}

	/* keine Sounds spielen? */
	if (!audio_laeuft)
		return 0;

	/* Audio-Device oeffnen bis Erfolg oder Abbruch */
	for (;;)
	{
		static char *meldung[] = { "iMaze - Sound Error", "",
			NULL, NULL, NULL };
		char text[80];

		/* Audio-Device oeffnen */
		if ((audio_deskriptor = open(DEV_AUDIO,
			O_WRONLY | O_NDELAY, 0)) >= 0)
			break;

		/* Fehlertext erzeugen */
		sprintf(text, "Can't open %s:", DEV_AUDIO);
		meldung[2] = text;
		meldung[3] = fehler_text();

		/* sonstiger Fehler? */
		if (errno != EBUSY)
			if (rueckfrage(meldung, "No Sound", NULL))
			{
				audio_laeuft = 0;
				return 0;
			}
			else
				return 1;
		/* Device gerade in Benutzung? */
		else if (!rueckfrage(meldung, "Try again", "No Sound"))
		{
			audio_laeuft = 0;
			return 0;
		}
	}

#ifdef AUDIO_CTL
	{
		/* Audio-Kontroll-Device oeffnen */

		struct audio_info info; /* Status des Audio-Devices */

		/* Audio-Kontroll-Device oeffnen */
		if ((audio_ctl_deskriptor = open(AUDIO_CTL, O_RDWR, 0)) < 0)
		{
			static char *meldung[] = { "iMaze - Sound Error", "",
				NULL, NULL, NULL };
			char text[80];

			sprintf(text, "Can't open %s:", AUDIO_CTL);
			meldung[2] = text;
			meldung[3] = fehler_text();
			uebler_fehler(meldung, NULL);
		}

		/* Audio-Status abfragen */
		if (ioctl(audio_ctl_deskriptor, AUDIO_GETINFO, &info))
		{
			static char *meldung[] = { "iMaze - Sound Error", "",
				"Can't check audio play counter:", NULL, NULL };

			meldung[3] = fehler_text();
			uebler_fehler(meldung, NULL);
		}

		/* Startwert des Audio-Paket-Zaehlers merken */
		eof_zaehler = info.play.eof;
	}
#else
#ifndef ULAW
	{
		int faktor;  /* Verkleinerungs-Faktor fuer System-Audio-Puffer */
		int groesse; /* Puffer-Groesse */

		/* Faktoren 16, 8, 4, 2, 1 probieren */
		for (faktor = 16;; faktor >>= 1)
		{
			int status; /* Fehler-Status */

			/* schon alle Faktoren probiert, oder anderer Fehler? */
			if (!faktor || (status = ioctl(audio_deskriptor,
				SNDCTL_DSP_SUBDIVIDE, &faktor)) && errno != EINVAL)
			{
				static char *meldung[] = { "iMaze - Sound Error", "",
					"Can't enable small audio buffer:", NULL, NULL };

				meldung[3] = fehler_text();
				uebler_fehler(meldung, NULL);
			}

			/* gefunden? */
			if (!status)
				break;
		}

		/* Puffer-Groesse abfragen */
		if (ioctl(audio_deskriptor, SNDCTL_DSP_GETBLKSIZE, &groesse))
		{
			static char *meldung[] = { "iMaze - Sound Error", "",
				"Can't check audio buffer size:", NULL, NULL };

			meldung[3] = fehler_text();
			uebler_fehler(meldung, NULL);
		}

		/* Puffer viel zu gross? */
		if (groesse > 4096)
		{
			static char *meldung[] = { "iMaze - Sound Error", "",
				"Can't enable small audio buffer:", NULL, NULL };
			char text[80];

			sprintf(text, "buffer size is %d bytes", groesse);
			meldung[3] = text;

			/* trotzdem spielen? */
			if (!rueckfrage(meldung, NULL, "No Sound"))
			{
				close(audio_deskriptor);
				audio_laeuft = 0;

				return 0;
			}
		}
	}
#endif
#endif

#ifdef AUDIO_CTL
	{
		struct audio_info info; /* Audio-Status */

		/* Status initialisieren */
		AUDIO_INITINFO(&info);

		/* Abtastrate setzen */
		info.play.sample_rate = SAMPLING_RATE;
		if (ioctl(audio_deskriptor, AUDIO_SETINFO, &info))
		{
			static char *meldung[] = { "iMaze - Sound Error", "",
				"Can't set audio sample rate:", NULL, NULL };

			meldung[3] = fehler_text();
			uebler_fehler(meldung, NULL);
		}
	}
#else
	{
		int speed = SAMPLING_RATE;

		/* Abtastrate setzen */
		if (ioctl(audio_deskriptor, SNDCTL_DSP_SPEED, &speed))
		{
			static char *meldung[] = { "iMaze - Sound Error", "",
				"Can't set audio sample rate:", NULL, NULL };

			meldung[3] = fehler_text();
			uebler_fehler(meldung, NULL);
		}
	}
#endif

	{
		int i;            /* Index fuer Ereignisse */
		int geladen;      /* mindestens eine Sound-Datei gefunden? */
		char *datei_name; /* Dateiname fuer Sound-Dateien */

#ifdef ULAW
		init_ulaw(); /* linear -> u-law Tabelle initialisieren */
#endif

		for (i = 0; i <= ERG_ANZ; i++)
			sounds[i].laenge = 0;

		/* Speicherplatz fuer Dateinamen belegen */
		speicher_belegen((void **)&datei_name,
			strlen(audio_verzeichnis) + 40);

		for (i = geladen = 0; i <= ERG_ANZ; i++)
		{
			int laenge;     /* Laenge der Sound-Datei */
			int deskriptor; /* Deskriptor fuer die Sound-Datei */

			/* Ereignis-Name definiert oder Nummer verwenden? */
			if (ereignis_namen[i] != NULL)
				sprintf(datei_name, "%s/%s.au", audio_verzeichnis,
					ereignis_namen[i]);
			else
				sprintf(datei_name, "%s/%d.au", audio_verzeichnis, i);

			/* Datei oeffnen */
			if ((deskriptor = open(datei_name, O_RDONLY)) >= 0)
			{
				/* Laenge abfragen */
				laenge = lseek(deskriptor, 0, SEEK_END);
				lseek(deskriptor, 0, SEEK_SET);

				/* Datei lang genug? */
				if (laenge >= 7)
				{
					unsigned char *puffer; /* Puffer fuer Sound */
					int offset;            /* Position der Sounddaten */
					int j;                 /* Index fuer Sounddaten */

					/* temporaeren Puffer belegen */
					speicher_belegen((void **)&puffer, laenge);

					/* Daten lesen */
					read(deskriptor, puffer, laenge);

					/* au-Header vorhanden? */
					if (!strncmp((char *)puffer, ".snd", 4))
					{
						offset = (int)puffer[4] << 24 |
							(int)puffer[5] << 16 |
							(int)puffer[6] << 8 |
							(int)puffer[7];

						/* offset ungueltig? */
						if (offset < 0 || offset >= laenge)
							continue;
					}
					else
						offset = 0;

					/* offset ueberspringen, SCHLUSS_STILLE anhaengen */
					laenge -= offset - SCHLUSS_STILLE;

					/* Speicher fuer Sounddaten belegen */
					speicher_belegen((void **)&sounds[i].inhalt,
						sizeof *sounds[i].inhalt * laenge);

					/* Daten wandeln */
					for (j = 0; j < laenge - SCHLUSS_STILLE; j++)
						sounds[i].inhalt[j] = linear[puffer[offset + j]];

					/* Schlussstille anhaengen */
					for (; j < laenge; j++)
						sounds[i].inhalt[j] = 0;

					/* Laenge speichern */
					sounds[i].laenge = laenge;

					/* temporaeren Puffer anlegen */
					speicher_freigeben((void **)&puffer);

					/* es wurden Sound-Daten gefunden */
					geladen++;
				}

				/* Datei schliessen */
				close(deskriptor);
			}
		}

		/* Speicher fuer Dateiname freigeben */
		speicher_freigeben((void **)&datei_name);

		/* keine Datei gefunden? */
		if (!geladen)
		{
			static char *meldung[] = { "iMaze - Sound Error", "",
				"No audio file found in:", NULL, NULL };

			meldung[3] = audio_verzeichnis;

			if (!rueckfrage(meldung, "No Sound", NULL))
				return 1;

			audio_laeuft = 0;
			close(audio_deskriptor);

#ifdef AUDIO_CTL
			close(audio_ctl_deskriptor);
#endif

			return 0;
		}

		deskriptor_nicht_blockieren(audio_deskriptor);

		/* Titelmusik abspielen */
		if (sounds[ERG_ANZ].laenge)
		{
			abspiel_anz = 1;
			abspiel[0].laenge = sounds[ERG_ANZ].laenge;
			abspiel[0].position = sounds[ERG_ANZ].inhalt;
		}

#ifdef AUDIO_CTL
		/* Handler fuer Audio-Signal installieren */
		notify_set_signal_func(audio_frame, xv_audio_handler,
			SIGPOLL, NOTIFY_ASYNC);

		/* Signale durch das Audio-Kontroll-Device zulassen */
		if (ioctl(audio_ctl_deskriptor, I_SETSIG, S_MSG))
		{
			static char *meldung[] = { "iMaze - Sound Error", "",
				"Can't enable signals on audio control device:",
				NULL, NULL };

			meldung[3] = fehler_text();
			uebler_fehler(meldung, NULL);
		}
#endif
	}
#endif

	return 0;
}


/*
** xv_ereignisse_ende
**  Routine fuer das Schliessen von Ausgabekanaelen etc.
*/
void xv_ereignisse_ende()
{
#ifdef SOUND
	int i; /* Index fuer Ereignisse */

	/* laeuft Audio? */
	if (!audio_laeuft)
		return;

	/* Speicher freigeben */
	for (i = 0; i <= ERG_ANZ; i++)
		if (sounds[i].laenge)
			speicher_freigeben((void **)&sounds[i].inhalt);

	/* Audio-Device schliessen */
	close(audio_deskriptor);

#ifdef AUDIO_CTL
	/* Audio-Kontroll-Device schliessen */
	close(audio_ctl_deskriptor);
#endif

	/* Audio laeuft nicht mehr */
	audio_laeuft = 0;
#endif
}
