# angband.tcl --
#
#	Commands called by Angband from C.
#

# angband_notice --
#
#	Called by Angband when something has changed.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc angband_notice condition {

	global Notice
	global Windows

	switch -- $condition {

		close_game {

			### The game is just about to exit. Perform housekeeping here.
			if {[catch {
				NSValueManager::CloseModule
				WriteAlternateFile
				WriteAssignFile
				WriteSoundFile
				WriteSpriteFile
			} result]} {
				set message "An error occured during shutdown:\n    $result"
				tk_messageBox -title "$Angband(name) Error" -message $message \
					-icon error]
			}
		}

		inventory -
		koff {
			set Notice($condition) 1
		}
	}
}

# angband_display --
#
#	Called by Angband to hide/show a window.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

proc angband_display {window action args} {

	global Angband
	global Display
	global NSOptions
	global PYPX
	global Windows

	if {$action == "hide"} {

		switch -- $window {

			floor -
			inven -
			equip {
				set window inventory
			}

			map {
				set mainId $Windows(main,oop)
				set mapId [Global bigmap,mapId]
				place forget [NSMap::Info $mapId frame]
				place forget [Global mapdetail,widget]
				place [Global micromap,widget] \
					-relx 0.5 -rely 0.5 -anchor center
					
				# Focus for KeyPress bindings
				focus $Windows(main,win)
				return
			}

			message {
				set window messages
			}
		}

		# Undisplay the window
		if {[info exists NSWindowManager::Priv($window,win)]} {
			eval NSWindowManager::Undisplay $window $args
		} else {
			wm withdraw $Windows($window,win)
		}

		# Restore focus
		catch {focus $Display($window,oldFocus)}

		set Display(window) none

	} else {

		switch -- $window {

			kingly {
				if {[catch {
					KinglyWindow
				} result]} {
					set message "An error occured:\n    $result\nQuit now?"
					set answer [tk_messageBox -title "$Angband(name) Error" \
						-message $message -type yesno -icon question]
					if {$answer == "yes"} [angband game abort]
				}
				return
			}
	
			tomb {
				if {[catch {
					TombWindow
				} result]} {
					set message "An error occured:\n    $result\nQuit now?"
					set answer [tk_messageBox -title "$Angband(name) Error" \
						-message $message -type yesno -icon question]
					if {$answer == "yes"} [angband game abort]
				}
				return
			}

			book {
			}

			building {
				NSModule::LoadIfNeeded NSBuilding
			}
			
			choosemonster {
				NSModule::LoadIfNeeded NSChooseMonster
			}

			equip {
				NSInventory::SetList $Windows(inventory,oop) equipment {}
				set window inventory
			}

			floor {
				NSInventory::SetList $Windows(inventory,oop) floor {}
				set window inventory
			}	
	
			help {
				NSModule::LoadIfNeeded NSHelp
			}
	
			highscore {

				set loaded [NSModule::LoadIfNeeded NSHighScore]

				# Hack -- If the Tomb Window is showing, then it means
				# the game is over, and we are about to display the
				# High Scores Window. To ensure there is always an
				# active frontmost window, we don't hide the Tomb Window
				# until after the High Scores Window is displayed.
				if {[winfo exists .tomb]} {

					if {[catch {
	
						# Tk Bug? When a transient window with a menubar
						# is made no longer transient, the menubar is no
						# longer displayed. As a workaround, I recreate the
						# High Scores Window.
	
						if !$loaded {
							destroy $Windows(highscore,win)
							NSObject::Delete NSHighScore $Windows(highscore,oop)
							NSHighScore::InitModule
						}
	
						# Important for positioning and yview'ing
						NSToplevel::NaturalSize $Windows(highscore,win) ""
	
						NSHighScore::SetList $Windows(highscore,oop)
	
						# Position and select the window
						WindowPosition $Windows(highscore,win) 2 3
						focus $Windows(highscore,win)
	
						# Now destroy the tomb window
						destroy .tomb
	
						# Show the window, and wait
						tkwait window $Windows(highscore,win)
	
					} result]} {
						set message "An error occured while trying to display the High Scores Window:\n    $result\nQuit now?"
						set answer [tk_messageBox -title "$Angband(name) Error" \
							-message $message -type yesno -icon question]
						if {$answer == "yes"} [angband game abort]
					}

					# Done
					return
				}

				wm title $Windows(highscore,win) [lindex $args 0]
				set args {}
			}

			info {
				NSInfoWindow::SetList $Windows(info,oop) [lindex $args 0] \
					[lindex $args 1]
			}

			inven {
				NSInventory::SetList $Windows(inventory,oop) inventory {}
				set window inventory
			}

			knowledge {
				NSModule::LoadIfNeeded NSKnowledge
			}
	
			map {
				set mainId $Windows(main,oop)
				set mapId [Global bigmap,mapId]
				place [NSMap::Info $mapId frame] -x 0 -y 0 \
					-relwidth 1.0 -relheight 1.0
				place forget [Global micromap,widget]
				place [Global mapdetail,widget] \
					-relx 0.5 -rely 0.5 -anchor center
				update idletasks

				# We must map the widget so it may resize to fit its parent
				# before setting the view. Also, the scrollbars must be
				# mapped before they will update properly.
				eval NSMap::SetView $mapId $PYPX
				eval [Global mapdetail,widget] center $PYPX

				# Focus for KeyPress bindings
				focus [Global bigmap,widget]
				return
			}
	
			message {
				NSMessage::SetList $Windows(messages,oop)
				set window messages
			}

			macros {
				NSModule::LoadIfNeeded NSMacros
			}

			mindcraft {
#				NSMindcraft::SetList $Windows(mindcraft,oop)
			}

			options {
				NSModule::LoadIfNeeded NSOptions
			}

			player {
				NSModule::LoadIfNeeded NSPlayer
			}

			playerflags {
				NSModule::LoadIfNeeded NSPlayerFlags
			}

			power {
				NSPower::SetList $Windows(power,oop)
			}
	
			store {
				NSModule::LoadIfNeeded NSStore
			}

			default {
				error "unknown window \"$window\""
			}
		}

		# The window isn't already displayed
		if {$Display(window) != $window} {

			# Save current focus
			set Display($window,oldFocus) [focus]

			set Display(window) $window
		}

		# Display the window
		if {[info exists NSWindowManager::Priv($window,win)]} {
			eval NSWindowManager::Display $window $args
		} else {
			WindowBringToFront $Windows($window,win)
		}
	}

	# Update the windows now
	update
}

# angband_prompt --
#
#	Called by Angband to prompt the user. Rather complex interface to
#	handle multiple messages per line, and getting a string from the
#	user. This routine builds the string to be displayed in the
#	Message Window. It is not actually displayed until a Term-fresh
#	quasi-event occurs, at which point Fresh_Prompt() is called.
#	This behaviour prevents unwanted redraws during macro invocations.
#
# Arguments:
#	arg1					about arg1
#
# Results:
#	What happened.

set AngbandPriv(prompt,old) ""
set AngbandPriv(prompt,more) 0
set AngbandPriv(prompt,fresh) 0
qebind PROMPT <Term-fresh> "Fresh_Prompt"

proc Fresh_Prompt {} {

	global Windows
	global AngbandPriv

	if !$AngbandPriv(prompt,fresh) return
	if {[string compare $AngbandPriv(prompt,text) $AngbandPriv(prompt,old)]} {
		set text $Windows(message,win).message
		$text delete 1.0 end
		$text insert end $AngbandPriv(prompt,text)
		if {$AngbandPriv(prompt,more)} {
			$text tag add tagMore "end - 7 chars" end
		}
		set AngbandPriv(prompt,old) $AngbandPriv(prompt,text)
	}
	set AngbandPriv(prompt,fresh) 0
	set AngbandPriv(prompt,more) 0
}

proc angband_prompt {action {arg1 ""} {arg2 ""}} {

	global AngbandPriv

	switch -- $action {

		open {
			set AngbandPriv(prompt,length) [string length $arg1]
			set AngbandPriv(prompt,prefix) $arg1
			set AngbandPriv(prompt,text) $arg1$arg2
		}

		update {
			set AngbandPriv(prompt,text) $AngbandPriv(prompt,prefix)$arg1
		}

		set {
			set AngbandPriv(prompt,text) $arg1
		}

		append {
			if {$arg1 == " -more-"} {
				set AngbandPriv(prompt,more) 1
			}
			append AngbandPriv(prompt,text) $arg1
		}

		wipe {
			set AngbandPriv(prompt,text) ""
			set AngbandPriv(prompt,more) 0
		}
	}

	set AngbandPriv(prompt,fresh) 1
}
