/*
**                      Softwarepraktikum iMaze
**                              1993/94
**                Joerg Czeranski    Hans-Ulrich Kiel
**
** Datei: ip_server.c
**
** Kommentar:
**  Netzwerkkommunikation ueber TCP/IP und
**  Interprozesskommunikation ueber Sockets
*/


#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include "speicher.h"
#include "server_netz.h"
#include "system.h"

static char sccsid[] = "@(#)ip_server.c	1.26 5/26/94";


/* Default-TCP-Port des Servers */
#ifndef DEFAULT_PORT
#define DEFAULT_PORT /* phone +49-*/5323/*-72-3896 */
#endif


/* Makro, das aus einem Verbindungsdeskriptor den zugehoerigen
   Prolog-Socket ausliest */
#define SOCKET(verbindungs_deskriptor) \
	(((struct verbindung *)(verbindungs_deskriptor))->socket)


/* Struktur, in der alle relevanten Daten ueber eine Verbindung
   zu einem Client gespeichert werden */
struct verbindung
{
	struct sockaddr_in adresse; /* IP-Adresse des Clients */
	int socket;    /* Deskriptor des Sockets, ueber den die
	                  Kommunikation laeuft (nur fuer den Prolog) */
	int auf_spiel; /* Flag, ob die Verbindung vom Prolog auf das Spiel
	                  umgeschaltet wurde */
};

/* Struktur, in der eine Nachricht fuer die
   Interprozesskommunikation gespeichert werden kann */
union nachricht
{
	/* inklusive Beschreibung eines Clients */
	struct
	{
		struct sockaddr_in client_adresse; /* IP-Adresse des Clients */
		int auf_spiel; /* Client ist schon in der Spiel-Phase */

		char typ; /* Typ der Nachricht */
	} lang;

	/* ohne Beschreibung des Clients */
	char kurz_typ; /* Typ der Nachricht */
};


/* Systemvariable: letzter aufgetretener Fehlercode oder 0 */
extern int errno;

/* im Server-Prozess: */
static int server_prolog_socket; /* Socket, auf dem die Prolog-Verbindungen
                                    der Clients entgegengenommen werden */

/* im Session-Prozess: */
static int socket_zum_server;    /* Socket fuer die Kommunikation mit
                                    dem Server-Prozess */
static int session_spiel_socket; /* Socket, ueber den mit den Clients die
                                    Spiel-Pakete ausgetauscht werden */

/* im Initialisierungs-Prozess: */
static void *init_verbindung = NULL; /* Verbindung, auf der der Prozess
                                        den Client initialisiert */


/*
** fehler_text
**  ermittelt den zur Systemvariable errno passenden Fehlertext
**
** Rueckgabewert:
**  Zeiger in das Feld sys_errlist oder in eine lokale Variable, die
**  den Fehler als Nummer enthaelt (falls kein Standardtext vorhanden ist)
**
** Seiteneffekte:
**  der zuletzt zurueckgegebene Zeiger wird evtl. ungueltig
*/
static char *fehler_text()
{
	extern char *sys_errlist[]; /* Liste der Standard-Fehlertexte */
	extern int sys_nerr;        /* Anzahl der Eintraege in sys_errlist */

	/* falls errno kein gueltiger Index fuer sys_errlist ist, nur die
	   Nummer als String zurueckgeben */
	if (errno < 0 || errno >= sys_nerr)
	{
		static char text[20]; /* Puffer fuer Fehlernummer */

		sprintf(text, "Error #%d", errno);

		/* Zeiger auf Puffer zurueckgeben; Achtung: dieser Zeiger
		   ist immer derselbe */
		return text;
	}

	/* Standard-Fehlertext zurueckgeben */
	return sys_errlist[errno];
}


/*
** netzwerk_fehler
**  ruft uebler_fehler mit einem Standard-Fehlertext auf
**
** Parameter:
**  ursache: Textzeile, die die Fehlerursache beschreibt
*/
static void netzwerk_fehler(ursache)
char *ursache;
{
	static char *meldung[] = { "iMaze - Network Error", "",
		NULL, NULL, NULL }; /* Fehlermeldung */

	meldung[2] = ursache;       /* Fehlerursache */
	meldung[3] = fehler_text(); /* Standard-Fehlertext */

	/* Fehlermeldung ausgeben */
	uebler_fehler(meldung, NULL);
}


/*
** verbindung_init
**  belegt Speicher und initialisiert eine Struktur zur Beschreibung
**  der Verbindung zu einem Client
**
** Parameter:
**  adresse: Zeiger auf die IP-Adresse des Clients
**
** Rueckgabewert:
**  Zeiger auf die Beschreibungsstruktur
*/
static struct verbindung *verbindung_init(adresse)
struct sockaddr_in *adresse;
{
	struct verbindung *verbindung; /* Zeiger auf die Struktur */

	/* Speicher belegen */
	speicher_belegen((void **)&verbindung, sizeof(struct verbindung));

	/* Adresse kopieren */
	memcpy(&verbindung->adresse, adresse, sizeof(struct sockaddr_in));

	/* Zeiger zurueckgeben */
	return verbindung;
}


/* bis hier lokaler Teil                       */
/***********************************************/
/* ab hier globaler Teil                       */


/*
** prolog_paket_senden
**  sendet ein Prolog-Paket zu einem Client
**
** Parameter:
**  verbindung: Deskriptor fuer die Verbindung zum Client
**  paket_laenge: Laenge des Pakets
**  paket: das Paket
**
** Rueckgabewert:
**  1, falls ein Fehler aufgetreten ist, sonst 0
*/
int prolog_paket_senden(verbindung, paket_laenge, paket)
void *verbindung;
int paket_laenge;
void *paket;
{
	unsigned char laenge[2]; /* Paket, in dem die Laenge des Datenpaketes
	                            gesendet wird */

	/* Laengen ueber 65535 sind nicht moeglich */
	if (paket_laenge > 65535)
		return 1;

	/* Laenge der folgenden Daten senden */
	laenge[0] = paket_laenge / 256;
	laenge[1] = paket_laenge % 256;
	if (puffer_schreiben(SOCKET(verbindung), 2, laenge) != 2)
		return 1;

	/* falls Laenge der Daten nicht 0 ist, die Daten senden */
	if (paket_laenge)
		if (puffer_schreiben(SOCKET(verbindung),
			paket_laenge, paket) != paket_laenge)
			return 1;

	/* kein Fehler aufgetreten */
	return 0;
}


/*
** prolog_paket_empfangen
**  empfaengt ein Prolog-Paket von einem Client;
**  belegt Speicher dafuer, der spaeter freigegeben werden muss
**
** Parameter:
**  verbindung: Deskriptor fuer die Verbindung zum Client
**  paket_laenge: Zeiger auf die Laenge des Pakets
**  paket: Zeiger auf das Paket
**
** Rueckgabewert:
**  1, falls ein Fehler aufgetreten ist, sonst 0
*/
int prolog_paket_empfangen(verbindung, paket_laenge, paket)
void *verbindung;
int *paket_laenge;
void **paket;
{
	unsigned char laenge[2]; /* Puffer fuer die Laenge der folgenden Daten */

	/* erst die Laenge der darauffolgenden Daten empfangen */
	if (puffer_lesen(SOCKET(verbindung), 2, laenge) != 2)
		return 1;
	*paket_laenge = (unsigned short)laenge[0] * 256 |
		(unsigned short)laenge[1];

	/* Speicher fuer das Paket belegen */
	speicher_belegen(paket, *paket_laenge);

	/* die Daten empfangen */
	if (puffer_lesen(SOCKET(verbindung),
		*paket_laenge, *paket) != *paket_laenge)
		return 1;

	/* kein Fehler aufgetreten */
	return 0;
}


/*
** spiel_paket_senden
**  sendet in der Spiel-Phase ein Paket an einen Client
**
** Parameter:
**  verbindung: Deskriptor fuer die Verbindung zum Client
**  paket_laenge: Laenge des Pakets
**  paket: das Paket
*/
void spiel_paket_senden(verbindung, paket_laenge, paket)
void *verbindung, *paket;
int paket_laenge;
{
	/* Paket senden */
	if (sendto(session_spiel_socket, paket, paket_laenge, 0,
		(struct sockaddr *)&((struct verbindung *)verbindung)->adresse,
		sizeof(struct sockaddr_in)) != paket_laenge)
		/* Fehler melden */
		netzwerk_fehler("Can't send game packet:");
}


/*
** spiel_paket_empfangen
**  empfaengt in der Spiel-Phase ein Paket von einem der Clients;
**  belegt Speicher dafuer, der spaeter freigegeben werden muss
**
** Parameter:
**  verbindungen_anz: Anzahl der Clients
**  verbindungen: Feld mit Deskriptoren fuer die Verbindungen
**  paket_laenge: Zeiger auf die Laenge des Pakets
**  paket: Zeiger auf das Paket
**
** Rueckgabewert:
**  -1 bei Fehler, sonst einen Index in das Feld verbindungen
*/
int spiel_paket_empfangen(verbindungen_anz, verbindungen,
	paket_laenge, paket)
int verbindungen_anz, *paket_laenge;
void **verbindungen, **paket;
{
	int i;                            /* Index des Clients */
	int paket_adresse_laenge;         /* Groesse der IP-Adresse */
	struct sockaddr_in paket_adresse; /* IP-Adresse */
	char paket_tmp[1024];             /* Puffer fuer das Paket */

	/* Schleife, bis ein Paket zugeordnet werden konnte oder
	   keins mehr bereitsteht */
	for (;;)
	{
		/* Paket empfangen */
		paket_adresse_laenge = sizeof paket_adresse;
		*paket_laenge = recvfrom(session_spiel_socket, paket_tmp,
			sizeof paket_tmp, 0, (struct sockaddr *)&paket_adresse,
			&paket_adresse_laenge);

		/* Fehler? */
		if (*paket_laenge < 0)
		{
			/* kein Paket mehr bereit? */
			if (errno == EWOULDBLOCK)
				return -1;

			/* bei anderem Fehler Programm mit Fehlermeldung beenden */
			netzwerk_fehler("Can't receive game packet:");
		}

		/* Client suchen, dem die IP-Adresse gehoert */
		for (i = 0; i < verbindungen_anz &&
			(((struct verbindung *)verbindungen[i])->
			adresse.sin_port != paket_adresse.sin_port ||
			((struct verbindung *)verbindungen[i])->
			adresse.sin_addr.s_addr != paket_adresse.sin_addr.s_addr);
			i++);

		/* gefunden? dann Schleife verlassen */
		if (i < verbindungen_anz)
			break;
	}

	/* Speicher fuer Paket belegen und kopieren */
	speicher_belegen(paket, *paket_laenge);
	memcpy(*paket, paket_tmp, *paket_laenge);

	/* Index zurueckgeben */
	return i;
}


/*
** verbindung_freigeben
**  gibt einen Verbindungsdeskriptor wieder frei;
**  beendet evtl. den Initialisierungs-Prozess
**
** Parameter:
**  verbindung: Deskriptor fuer die Verbindung zum Client
**
** Seiteneffekte:
**  der Speicher fuer *verbindung wird freigegeben;
**  evtl. wird dieser Prozess beendet
*/
void verbindung_freigeben(verbindung)
void *verbindung;
{
	/* ist dies der zugehoerige Initialisierungs-Prozess? */
	if (verbindung == init_verbindung)
	{
		/* den Speicher freigeben */
		speicher_freigeben(&verbindung);

		/* und den Prozess beenden */
		exit(0);
	}

	/* sonst nur den Speicher freigeben */
	speicher_freigeben(&verbindung);
}


/*
** verbindung_annehmen
**  nimmt eine Prolog-Verbindung zu einem Client entgegen
**
** Rueckgabewert:
**  Deskriptor fuer die Verbindung zum Client; NULL, falls
**  keine Verbindung bereit war oder ein Initialisierungs-Prozess
**  abgespalten wurde und dies der Server-Prozess ist
*/
void *verbindung_annehmen()
{
	int deskriptor;                    /* IP-Deskriptor der Verbindung */
	int client_adresse_laenge;         /* Groesse der IP-Adresse */
	struct sockaddr_in client_adresse; /* IP-Adresse */
	struct verbindung *verbindung;     /* Deskriptor der Verbindung */

	/* Verbindung entgegennehmen */
	client_adresse_laenge = sizeof client_adresse;
	if ((deskriptor = accept(server_prolog_socket,
		(struct sockaddr *)&client_adresse, &client_adresse_laenge)) < 0)
	{
		/* keine Verbindung bereit? */
		if (errno == EINTR || errno == EWOULDBLOCK)
			return NULL;

		/* bei anderem Fehler Programm mit Fehlermeldung beenden */
		netzwerk_fehler("Can't accept connection from client:");
	}

	/* Initialisierungs-Prozess abspalten */
	switch (fork())
	{
		case -1: /* Fehler */
			netzwerk_fehler("Can't fork client initialization process:");

		case 0: /* Initialisierungs-Prozess */
			/* Deskriptor initialisieren */
			verbindung = verbindung_init(&client_adresse);
			verbindung->socket = deskriptor;
			verbindung->auf_spiel = 0;

			/* merken, dass dies der zu verbindung gehoerige
			   Initialisierungs-Prozess ist */
			init_verbindung = verbindung;

			/* Verbindungs-Deskriptor zurueckgeben */
			return verbindung;

		default: /* Server-Prozess */
			/* Verbindung wieder schliessen */
			close(deskriptor);

			/* keine Verbindung zurueckgeben */
			return NULL;
	}
}


/*
** verbindung_abbrechen
**  bricht waehrend der Prolog-Phase die Verbindung ab und
**  beendet den zugehoerigen Initialisierungs-Prozess
**
** Parameter:
**  verbindung: Deskriptor fuer die Verbindung zum Client
**
** Seiteneffekte:
**  gibt den Speicher fuer *verbindung frei und
**  beendet den Initialisierungs-Prozess
*/
void verbindung_abbrechen(verbindung)
void *verbindung;
{
	/* Verbindung schliessen */
	close(SOCKET(verbindung));

	/* Speicher freigeben */
	speicher_freigeben(&verbindung);

	/* Initialisierungs-Prozess beenden */
	exit(0);
}


/*
** verbindung_auf_spiel
**  schaltet eine Verbindung aus der Prolog-Phase in die
**  Spiel-Phase
**
** Parameter:
**  session: Session, zu der der Client verbunden werden soll
**  verbindung: Deskriptor fuer die Verbindung zum Client
**
** Rueckgabewert:
**  1, falls ein Fehler aufgetreten ist, sonst 0
*/
int verbindung_auf_spiel(session, verbindung)
void *session, *verbindung;
{
	unsigned char port[6];                  /* Puffer zum Uebertragen der
	                                           Adresse ueber das Netz */
	int spiel_adresse_laenge;               /* Grosse der IP-Adresse */
	struct sockaddr_in spiel_adresse;       /* IP-Adresse */
	struct sockaddr_in *verbindung_adresse; /* temporaerer Zeiger */

	/* Adresse dieser Seite der Verbindung feststellen */
	spiel_adresse_laenge = sizeof spiel_adresse;
	if (getsockname(SOCKET(verbindung), (struct sockaddr *)&spiel_adresse,
		&spiel_adresse_laenge))
		return 1;

	/* die Adresse ist die eben abgefragte */
	port[0] = ntohl(spiel_adresse.sin_addr.s_addr) >> 24;
	port[1] = ntohl(spiel_adresse.sin_addr.s_addr) >> 16;
	port[2] = ntohl(spiel_adresse.sin_addr.s_addr) >> 8;
	port[3] = ntohl(spiel_adresse.sin_addr.s_addr);

	/* der Port haengt von der Session ab */
	port[4] = ((int *)session)[1] >> 8;
	port[5] = ((int *)session)[1];

	/* IP-Adresse des Session-Prozesses an den Client senden */
	if (puffer_schreiben(SOCKET(verbindung), 6, port) != 6)
		return 1;

	/* IP-Adresse des Clients fuer die Spiel-Phase empfangen */
	if (puffer_lesen(SOCKET(verbindung), 6, port) != 6)
		return 1;

	/* Prolog-Verbindung schliessen */
	if (close(SOCKET(verbindung)))
		return 1;

	/* Zeiger auf die IP-Adresse im Verbindungs-Deskriptor */
	verbindung_adresse = &((struct verbindung *)verbindung)->adresse;

	/* IP-Adresse des Clients speichern */
	verbindung_adresse->sin_family = AF_INET;
	verbindung_adresse->sin_port =
		htons((unsigned short)port[4] << 8 | (unsigned short)port[5]);
	verbindung_adresse->sin_addr.s_addr =
		htonl((unsigned long)port[0] << 24 | (unsigned long)port[1] << 16 |
		(unsigned long)port[2] << 8 | (unsigned long)port[3]);

	/* Verbindung ist jetzt auf Spiel geschaltet */
	((struct verbindung *)verbindung)->auf_spiel = 1;

	/* kein Fehler aufgetreten */
	return 0;
}


/*
** netzwerk_init
**  initialisiert den Netzwerkteil des Server-Prozesses
**
** Parameter:
**  argc: Zeiger auf Anzahl der an das Programm uebergebenen Parameter
**  argv: Zeiger auf Liste mit Strings, die die Aufrufparameter enthalten
**
** Seiteneffekte:
**  *argc und *argv werden veraendert
*/
void netzwerk_init(argc, argv)
int *argc;
char **argv;
{
	struct sockaddr_in server_adresse; /* IP-Adresse des TCP-Sockets */
	int port_wiederverwenden;          /* Flag, ob das Wiederverwenden
	                                      desselben TCP-Ports erzwungen
	                                      werden soll */
	char *portname;                    /* Name/Nummer des TCP-Ports */
	int ueberspringen;                 /* Anzahl bereits ausgewerteter
	                                      Aufrufparameter */

	/* argv[0] ueberspringen */
	ueberspringen = 1;

	/* normalerweise das Wiederverwenden nicht erzwingen */
	port_wiederverwenden = 0;

	/* Name/Nummer des Ports noch unbekannt */
	portname = NULL;

	/* bei Aufrufparameter "-f" Wiederverwenden erzwingen */
	if (*argc > ueberspringen && !strcmp(argv[ueberspringen], "-f"))
		port_wiederverwenden = 1, ueberspringen++;

	/* bei Aufrufparameter "-p" oder "-fp" Portname/nummer merken */
	if (*argc > ueberspringen + 1 && (!strcmp(argv[ueberspringen], "-p") ||
		!strcmp(argv[ueberspringen], "-fp")))
	{
		/* bei Aufrufparameter "-fp" Wiederverwenden erzwingen */
		if (!strcmp(argv[ueberspringen], "-fp"))
			port_wiederverwenden = 1;

		/* Portname/nummer merken */
		portname = argv[ueberspringen + 1];
		ueberspringen += 2;
	}

	/* bei Aufrufparameter "-f" (auch nach "-p")
	   Wiederverwenden erzwingen */
	if (*argc > ueberspringen && !strcmp(argv[ueberspringen], "-f"))
		port_wiederverwenden = 1, ueberspringen++;

	/* mehr als argv[0] ausgewertet? */
	if (ueberspringen > 1)
	{
		int i;

		/* Aufrufparameter verschieben, ausgewertete ueberschreiben */
		ueberspringen--;
		*argc -= ueberspringen;
		for (i = 1; i < *argc; i++)
			argv[i] = argv[i + ueberspringen];
	}

	/* IP-Adresse des TCP-Sockets initialisieren */
	server_adresse.sin_family = AF_INET;
	server_adresse.sin_addr.s_addr = INADDR_ANY;

	/* kein Port angegeben? */
	if (portname == NULL)
	{
#ifndef GETSERVBYNAME_BUG
		struct servent *service_eintrag; /* Portname/-nummer-Zuordnung */

		/* Portname/-nummer-Zuordnung des Ports "imaze" im Protokoll "tcp"
		   abfragen */
		if ((service_eintrag = getservbyname("imaze", "tcp")) != NULL)
			/* Nummer ist bekannt */
			server_adresse.sin_port = service_eintrag->s_port;
		else
#endif
			/* Nummer ist unbekannt, Default-Portnummer verwenden */
			server_adresse.sin_port = htons(DEFAULT_PORT);
	}
	else
	{
		/* Portname ist angegeben, als Nummer interpretieren und merken */
		if ((server_adresse.sin_port = htons(atoi(portname))) == 0)
#ifndef GETSERVBYNAME_BUG
		{
			struct servent *service_eintrag; /* Portname/-nummer-
			                                    Zuordnung */

			/* Portname war keine Nummer; Portname/-nummer-Zuordnung des
			   angegebenen Ports mittels des Portname abfragen; falls
			   keine Zuordnung gefunden, Fehler */
			if ((service_eintrag = getservbyname(portname, "tcp")) == NULL)
			{
				/* Fehlermeldung: */
				static char *meldung[] = { "iMaze - Network Error", "",
					"Unknown TCP service:", NULL, NULL };

				/* Portnamen mit anzeigen */
				meldung[3] = portname;

				uebler_fehler(meldung, NULL);
			}

			/* gefundene Nummer uebernehmen */
			server_adresse.sin_port = service_eintrag->s_port;
		}
#else
		{
			/* Fehlermeldung: */
			static char *meldung[] = { "iMaze - Network Error", "",
					"Ports by name not supported", NULL };

			uebler_fehler(meldung, NULL);
		}
#endif
	}

	/* TCP-Socket erzeugen */
	if ((server_prolog_socket = socket(PF_INET, SOCK_STREAM,
		IPPROTO_TCP)) < 0)
		netzwerk_fehler("Can't create prolog socket:");

	/* Wiederverwenden erzwingen? */
	if (port_wiederverwenden)
		/* entsprechende Option setzen */
		if (setsockopt(server_prolog_socket, SOL_SOCKET,
			SO_REUSEADDR, NULL, 0))
			netzwerk_fehler("Can't enable address reuse on prolog socket:");

	/* TCP-Socket an gewuenschte Adresse binden */
	if (bind(server_prolog_socket, (struct sockaddr *)&server_adresse,
		sizeof server_adresse))
	{
		char fehler_text[80]; /* Fehlermeldung */

		/* Fehlermeldung erzeugen */
		sprintf(fehler_text, "Can't bind prolog socket to port %d:",
			server_adresse.sin_port);

		/* Fehlermeldung anzeigen */
		netzwerk_fehler(fehler_text);
	}

	/* auf dem TCP-Socket ankommende Verbindungen zulassen */
	if (listen(server_prolog_socket, 5))
		netzwerk_fehler("Cant listen on prolog socket:");
}


/*
** session_starten
**  spaltet einen Session-Prozess ab
**
** Rueckgabewert:
**  ein Deskriptor fuer die Kommunikation mit dem Session-Prozess
**  bzw. NULL, wenn dies der Session-Prozess ist
*/
void *session_starten()
{
	int sockpair[2];     /* Sockets fuer die Interprozess-Kommunikation */
	int session_prozess; /* Prozess-ID des Session-Prozesses */
	struct sockaddr_in session_adresse; /* IP-Adresse (unspezifiziert) */

	/* UDP-Socket fuer die Verbindung zu den Clients in
	   der Spiel-Phase erzeugen */
	if ((session_spiel_socket = socket(PF_INET, SOCK_DGRAM,
		IPPROTO_UDP)) < 0)
		netzwerk_fehler("Can't create session socket:");

	/* UDP-Socket an einen beliebigen Port binden */
	session_adresse.sin_family = AF_INET;
	session_adresse.sin_port = 0;
	session_adresse.sin_addr.s_addr = INADDR_ANY;
	if (bind(session_spiel_socket, (struct sockaddr *)&session_adresse,
		sizeof session_adresse))
		netzwerk_fehler("Can't bind session socket to any address:");

	/* UDP-Socket in einen Modus versetzen, in dem das Empfangen eines
	   Pakets den Prozess nicht blockiert, wenn noch kein Paket bereit
	   ist, sondern einen Fehlercode erzeugt */
	if (fcntl(session_spiel_socket, F_SETFL, O_NDELAY))
		netzwerk_fehler("Can't switch session socket to non-blocking mode:");

	/* Sockets fuer die Interprozess-Kommunikation erzeugen */
	if (socketpair(PF_UNIX, SOCK_DGRAM, 0, sockpair))
		netzwerk_fehler("Can't create interprocess sockets:");

	/* Session-Prozess abspalten */
	if ((session_prozess = fork()) < 0)
		netzwerk_fehler("Can't fork session process:");

	/* ist dies der Session-Prozess? */
	if (session_prozess == 0)
	{
		/* die Server-Seite des Kommunikations-Sockets schliessen */
		if (close(sockpair[0]))
			netzwerk_fehler("Can't close other interprocess socket:");

		/* den TCP-Socket des Servers schliessen */
		if (close(server_prolog_socket))
			netzwerk_fehler("Can't close prolog socket:");

		/* den Kommunikations-Socket in den nicht blockierenden
		   Modus schalten */
		if (fcntl(sockpair[1], F_SETFL, O_NDELAY))
			netzwerk_fehler(
				"Can't switch interprocess socket to non-blocking mode:");

		/* den Kommunikations-Socket speichern */
		socket_zum_server = sockpair[1];

		/* dies ist der Session-Prozess */
		return NULL;
	}
	else
	{
		int *session;                     /* Deskriptor fuer die
		                                     Kommunikation mit dem
		                                     Session-Prozess */
		int spiel_adresse_laenge;         /* Groesse der IP-Adresse */
		struct sockaddr_in spiel_adresse; /* IP-Adresse des UDP-Sockets
		                                     der Session */

		/* IP-Adresse des UDP-Sockets der Session abfragen */
		spiel_adresse_laenge = sizeof spiel_adresse;
		if (getsockname(session_spiel_socket,
			(struct sockaddr *)&spiel_adresse, &spiel_adresse_laenge))
			netzwerk_fehler("Can't figure out session socket address:");

		/* die Session-Seite des Kommunikations-Sockets schliessen */
		if (close(sockpair[1]))
			netzwerk_fehler("Can't close other interprocess socket:");

		/* den UDP-Socket der Session schliessen */
		if (close(session_spiel_socket))
			netzwerk_fehler("Can't close session socket:");

		/* den Kommunikations-Socket in den nicht blockierenden
		   Modus schalten */
		if (fcntl(sockpair[0], F_SETFL, O_NDELAY))
			netzwerk_fehler(
				"Can't switch interprocess socket to non-blocking mode:");

		/* Speicher fuer den Session-Deskriptor belegen */
		speicher_belegen((void **)&session, 2 * sizeof(int));

		/* den Kommunikations-Socket und die Portnummer speichern */
		session[0] = sockpair[0];
		session[1] = htons(spiel_adresse.sin_port);

		/* den Session-Deskriptor an den Server-Prozess zurueckgeben */
		return session;
	}
}


/*
** nachricht_senden
**  sendet eine Nachricht an einen bestimmten Prozess
**
** Parameter:
**  session: Deskriptor fuer den Session-Prozess bzw. NULL
**           fuer den Server-Prozess
**  verbindung: Deskriptor, der in der Nachricht enthalten sein
**              soll oder NULL
**  typ: Typ der Nachricht
*/
void nachricht_senden(session, verbindung, typ)
void *session, *verbindung;
int typ;
{
	int socket;                /* Socket, ueber den die Nachricht
	                              gesendet werden soll */
	union nachricht nachricht; /* die Nachricht */

	if (session == NULL)
		/* Nachricht zum Server senden */
		socket = socket_zum_server;
	else
		/* Nachricht an die angegebene Session senden */
		socket = *(int *)session;

	/* kurze Nachricht ohne Deskriptor senden? */
	if (verbindung == NULL)
	{
		/* nur Typ kopieren */
		nachricht.kurz_typ = typ;

		/* Nachricht senden */
		if (send(socket, &nachricht.kurz_typ,
			sizeof nachricht.kurz_typ, 0) != sizeof nachricht.kurz_typ)
			/* bei Fehler Prozess mit oder ohne Meldung beenden */
			if (init_verbindung != NULL)
				exit(1);
			else
				netzwerk_fehler("Can't send interprocess message:");

		/* Nachricht abgesandt */
		return;
	}

	/* Typ kopieren */
	nachricht.lang.typ = typ;

	/* Verbindungs-Deskriptor kopieren */
	memcpy(&nachricht.lang.client_adresse,
		&((struct verbindung *)verbindung)->adresse,
		sizeof(struct sockaddr_in));
	nachricht.lang.auf_spiel = ((struct verbindung *)verbindung)->auf_spiel;

	/* Nachricht senden */
	if (send(socket, (void *)&nachricht.lang,
		sizeof nachricht.lang, 0) != sizeof nachricht.lang)
		/* bei Fehler Prozess mit oder ohne Meldung beenden */
		if (init_verbindung != NULL)
			exit(1);
		else
			netzwerk_fehler("Can't send interprocess message:");
}


/*
** nachricht_empfangen
**  empfaengt eine Nachricht von einem bestimmten Prozess
**
** Parameter:
**  session: Deskriptor fuer den Session-Prozess bzw. NULL
**           fuer den Server-Prozess
**  verbindung: Zeiger auf einen Deskriptor, der in der Nachricht
**              enthalten ist oder NULL
**  typ: Zeiger auf den Typ der Nachricht
**
** Rueckgabewert:
**  1, falls keine Nachricht empfangen werden konnte, sonst 0
**
** Seiteneffekte:
**  *verbindung und *typ werden gesetzt
*/
int nachricht_empfangen(session, verbindung, typ)
void *session, **verbindung;
char *typ;
{
	int socket;                /* Socket, ueber den die Nachricht
	                              empfangen werden soll */
	int laenge;                /* Laenge der Nachricht */
	union nachricht nachricht; /* die Nachricht */

	if (session == NULL)
		/* Nachricht vom Server empfangen */
		socket = socket_zum_server;
	else
		/* Nachricht von der angegebenen Session empfangen */
		socket = *(int *)session;

	/* Nachricht empfangen */
	if ((laenge = recv(socket, (void *)&nachricht, sizeof nachricht, 0)) < 0)
	{
		/* Unterbrechung durch ein Signal oder keine Nachricht vorhanden? */
		if (errno == EINTR || errno == EWOULDBLOCK)
			/* keine Nachricht */
			return 1;

		/* bei anderem Fehler Programm mit Fehlermeldung beenden */
		netzwerk_fehler("Can't receive interprocess message:");
	}

	/* kurze Nachricht ohne Verbindungs-Deskriptor? */
	if (laenge == sizeof nachricht.kurz_typ)
	{
		/* Daten zurueckgeben */
		*verbindung = NULL;
		*typ = nachricht.kurz_typ;

		/* Nachricht empfangen */
		return 0;
	}

	/* stimmt die Laenge auch nicht fuer eine lange Nachricht
	   inklusive Verbindungs-Deskriptor? */
	if (laenge != sizeof nachricht.lang)
		/* keine Nachricht */
		return 1;

	/* Daten zurueckgeben */
	*verbindung = verbindung_init(&nachricht.lang.client_adresse);
	((struct verbindung *)*verbindung)->auf_spiel =
		nachricht.lang.auf_spiel;
	*typ = nachricht.lang.typ;

	/* Nachricht empfangen */
	return 0;
}


/*
** nachricht_erwarten
**  wartet auf eine Nachricht von einem bestimmten Prozess
**
** Parameter:
**  session: Deskriptor fuer den Session-Prozess bzw. NULL
**           fuer den Server-Prozess
*/
void nachricht_erwarten(session)
void *session;
{
	fd_set readfds; /* Deskriptoren, auf die gewartet wird */

	/* initialisieren */
	FD_ZERO(&readfds);

	if (session == NULL)
		/* auf Nachricht vom Server warten */
		FD_SET(socket_zum_server, &readfds);
	else
		/* auf Nachricht von der angegebenen Session warten */
		FD_SET(*(int *)session, &readfds);

	/* warten, bei Fehler ausser Unterbrechung durch ein Signal
	   Programm mit Fehlermeldung beenden */
	if (select(max_deskriptor_anzahl(), &readfds, NULL, NULL, NULL) < 0 &&
		errno != EINTR)
		netzwerk_fehler("Can't wait for interprocess message:");
}


/*
** nachricht_oder_verbindung_erwarten
**  wartet, bis fuer den Server-Prozess eine Nachricht von einem
**  der anderen Prozesse oder eine neue Verbindung zu einem
**  Client bereitsteht
**
** Parameter:
**  anzsessions: Anzahl der Sessions
**  sessions: Feld mit den Deskriptoren der Sessions
**
** Rueckgabewert:
**  NULL, bei einer Verbindung;
**  ein Deskriptor fuer die Session bei einer Nachricht
*/
void *nachricht_oder_verbindung_erwarten(anzsessions, sessions)
int anzsessions;
void **sessions;
{
	/* Schleife, bis etwas eintrifft */
	for (;;)
	{
		fd_set readfds; /* Deskriptoren, auf die gewartet wird */
		int i;          /* Index in die Sessionliste */

		/* auf Daten auf dem Prolog-Socket des Servers oder
		   den Sockets zu den Sessions warten */
		FD_ZERO(&readfds);

		FD_SET(server_prolog_socket, &readfds);
		for (i = 0; i < anzsessions; i++)
			FD_SET(*(int *)sessions[i], &readfds);

		if (select(max_deskriptor_anzahl(), &readfds, NULL, NULL, NULL) < 0)
		{
			/* Unterbrechung durch ein Signal? */
			if (errno == EINTR)
				/* weiter warten */
				continue;

			/* bei anderem Fehler Programm mit Fehlermeldung beenden */
			netzwerk_fehler(
			"Can't wait for interprocess message or connection from client:");
		}

		/* Daten auf dem Prolog-Socket? */
		if (FD_ISSET(server_prolog_socket, &readfds))
			/* dann NULL zurueckgeben */
			return NULL;

		/* Daten auf einem der Session-Sockets? */
		for (i = 0; i < anzsessions; i++)
			if (FD_ISSET(*(int *)sessions[i], &readfds))
				/* Deskriptor der Session zurueckgeben */
				return sessions[i];
	}
}
