/*
**                      Softwarepraktikum iMaze
**                              1993/94
**                Joerg Czeranski    Hans-Ulrich Kiel
**
** Datei: system.c
**
** Kommentar:
**  Globale systemnahe Kompatibiltaetsroutinen;
**  Signal-Handling, Timing etc.
*/


#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <sys/time.h>

#include "speicher.h"

static char sccsid[] = "@(#)system.c	1.11 5/17/94";


/* Systemvariable: letzter aufgetretener Fehlercode oder 0 */
extern int errno;


/* Struktur, in der der Timerzustand gerettet werden kann */
struct timer_zustand
{
	void (*handler)();
	struct itimerval wert;
};


/* bis hier lokaler Teil                       */
/***********************************************/
/* ab hier globaler Teil                       */


#ifdef SYSV
/*
** signal
**  simuliert auf System V das Verhalten des BSD-signal mit sigaction
**
** Parameter:
**  signum: zu behandelndes Signal
**  handler: Signal-Handler
**
** Rueckgabewert:
**  vorheriger Signal-Handler
*/
void (*signal(signum, handler))()
int signum;
void (*handler)();
{
	struct sigaction neue_sa; /* neues Verhalten bei diesem Signal */
	struct sigaction alte_sa; /* vorheriges Verhalten bei diesem Signal */

	neue_sa.sa_handler = handler; /* Signal-Handler setzen */
	neue_sa.sa_mask = 0;          /* keine zusaetzlichen Signale
	                                im Handler sperren */
	neue_sa.sa_flags = SA_NOMASK; /* sa_mask ignorieren */

	/* neues Verhalten aktivieren, vorheriges auslesen */
	if (sigaction(signum, &neue_sa, &alte_sa) == -1)
		/* bei Fehler -1 zurueckgeben */
		return (void *)-1;

	/* vorherigen Signal-Handler zurueckgeben */
	return alte_sa.sa_handler;
}
#endif


/*
** max_deskriptor_anzahl
**  stellt die maximale Anzahl der Deskriptoren dieses Prozesses fest
**
** Rueckgabewert:
**  maximale Anzahl der Deskriptoren
*/
int max_deskriptor_anzahl()
{
	/* bei SunOS identisch mit getdtablesize() */
	return ulimit(4);
}


/*
** puffer_lesen
**  liest durch einen Deskriptor einen Puffer, falls moeglich,
**  komplett voll
**
** Parameter:
**  deskriptor: Deskriptor, durch den gelesen werden soll
**  puffer_laenge: Groesse des Puffers
**  puffer: der Puffer
**
** Rueckgabewert:
**  -1 fuer Fehler, sonst die Anzahl der gelesenen Bytes; puffer_laenge,
**  falls kein EOT gelesen wurde
*/
int puffer_lesen(deskriptor, puffer_laenge, puffer)
int deskriptor, puffer_laenge;
void *puffer;
{
	int laenge_gesamt; /* Anzahl der bisher gelesenen Bytes */

	/* Schleife, solange noch Platz im Puffer ist */
	for (laenge_gesamt = 0; puffer_laenge;)
	{
		int laenge_neu; /* Anzahl der aktuell gelesenen Bytes oder -1 */

		/* maximal puffer_laenge Bytes lesen */
		if ((laenge_neu = read(deskriptor, puffer, puffer_laenge)) < 0)
		{
			/* Fehler EINTR (Interrupted System Call) ignorieren */
			if (errno == EINTR)
				continue;

			/* bei anderem Fehler Wert < 0 zurueckgeben */
			return laenge_neu;
		}

		/* EOT, d.h. keine Bytes gelesen? */
		if (!laenge_neu)
			break;

		puffer = (char *)puffer + laenge_neu; /* Pufferzeiger erhoehen */
		puffer_laenge -= laenge_neu; /* verbleibende Puffergroesse
		                                vermindern */
		laenge_gesamt += laenge_neu; /* Anzahl gelesener Bytes erhoehen */
	}

	/* Anzahl insgesamt gelesener Bytes zurueckgeben */
	return laenge_gesamt;
}


/*
** puffer_schreiben
**  schreibt einen Puffer, falls moeglich, komplett durch einen Deskriptor
**
** Parameter:
**  deskriptor: Deskriptor, durch den geschrieben werden soll
**  puffer_laenge: Groesse des Puffers
**  puffer: der Puffer
**
** Rueckgabewert:
**  -1 fuer Fehler, sonst die Anzahl der geschriebenen Bytes; puffer_laenge,
**  falls write nicht 0 zurueckgegeben hat (sollte nicht passieren)
*/
int puffer_schreiben(deskriptor, puffer_laenge, puffer)
int deskriptor, puffer_laenge;
void *puffer;
{
	int laenge_gesamt; /* Anzahl der bisher geschriebenen Bytes */

	/* Schleife, solange noch Bytes im Puffer sind */
	for (laenge_gesamt = 0; puffer_laenge;)
	{
		int laenge_neu; /* Anzahl der aktuell geschriebenen Bytes oder -1 */

		/* maximal puffer_laenge Bytes schreiben */
		if ((laenge_neu = write(deskriptor, puffer, puffer_laenge)) < 0)
		{
			/* Fehler EINTR (Interrupted System Call) ignorieren */
			if (errno == EINTR)
				continue;

			/* bei anderem Fehler Wert < 0 zurueckgeben */
			return laenge_neu;
		}

		/* keine Bytes geschrieben? d.h. kein Schreiben moeglich,
		   aber kein Fehler */
		if (!laenge_neu)
			break;

		puffer = (char *)puffer + laenge_neu; /* Pufferzeiger erhoehen */
		puffer_laenge -= laenge_neu; /* Anzahl verbleibender Bytes
		                                vermindern */
		laenge_gesamt += laenge_neu; /* Anzahl geschriebener Bytes
		                                erhoehen */
	}

	/* Anzahl insgesamt geschriebener Bytes zurueckgeben */
	return laenge_gesamt;
}


/*
** timer_starten
**  startet den Timer; nach der gewaehlten Zeit wird ein SIGALRM ausgeloest
**
** Parameter:
**  ms: Zeit in Millisekunden
*/
void timer_starten(ms)
int ms;
{
	struct itimerval neuer_wert; /* neuer Timerzustand */

	/* Timer nach Ablauf nicht automatisch neu starten */
	neuer_wert.it_interval.tv_sec = 0;
	neuer_wert.it_interval.tv_usec = 0;

	/* Timer auf ms Millisekunden setzen */
	neuer_wert.it_value.tv_sec = ms / 1000;
	neuer_wert.it_value.tv_usec = (ms % 1000) * 1000;

	/* Timer setzen */
	setitimer(ITIMER_REAL, &neuer_wert, NULL);
}


/*
** timer_starten
**  stoppt den Timer; danach wird kein SIGALRM mehr ausgeloest
*/
void timer_stoppen()
{
	struct itimerval neuer_wert; /* neuer Timerzustand */

	/* Timer nach Ablauf nicht automatisch neu starten */
	neuer_wert.it_interval.tv_sec = 0;
	neuer_wert.it_interval.tv_usec = 0;

	/* Timer auf "abgelaufen" setzen */
	neuer_wert.it_value.tv_sec = 0;
	neuer_wert.it_value.tv_usec = 0;

	/* Timer setzen */
	setitimer(ITIMER_REAL, &neuer_wert, NULL);
}


/*
** timer_restzeit
**  fragt die verbleibende Zeit bis zum Ablauf des Timers ab
**
** Rueckgabewert:
**  verbleibende Zeit in Millisekunden
*/
int timer_restzeit()
{
	struct itimerval wert; /* Timerzustand */

	/* Timerzustand abfragen */
	getitimer(ITIMER_REAL, &wert);

	/* verbleibende Zeit umrechnen und zurueckgeben */
	return wert.it_value.tv_sec * 1000 +
		(wert.it_value.tv_usec + 999) / 1000;
}


/*
** timer_abwarten
**  wartet, bis der Timer abgelaufen ist
*/
void timer_abwarten()
{
	struct itimerval wert; /* Timerzustand */

	/* Timerzustand abfragen */
	getitimer(ITIMER_REAL, &wert);

	/* Sekunden warten */
	if (wert.it_value.tv_sec)
		sleep(wert.it_value.tv_sec);

	/* Mikrosekunden warten */
	if (wert.it_value.tv_usec)
		usleep(wert.it_value.tv_usec);
}


/*
** timer_retten
**  fragt den aktuellen Timerzustand ab und haelt den Timer an
**
** Rueckgabewert:
**  Zeiger auf eine Struktur, die den Timerzustand enthaelt
**
** Seiteneffekte:
**  fuer die Struktur wird Speicher belegt
*/
void *timer_retten()
{
	struct timer_zustand *zustand; /* vorheriger Timerzustand und
	                                  Signal-Handler */
	struct itimerval neuer_wert;   /* neuer Timerzustand */

	/* Speicher fuer Timerzustand und Signal-Handler belegen */
	speicher_belegen((void **)&zustand, sizeof(struct timer_zustand));

	/* Timer nach Ablauf nicht automatisch neu starten */
	neuer_wert.it_interval.tv_sec = 0;
	neuer_wert.it_interval.tv_usec = 0;

	/* Timer auf "abgelaufen" setzen */
	neuer_wert.it_value.tv_sec = 0;
	neuer_wert.it_value.tv_usec = 0;

	/* Timer abfragen und setzen */
	setitimer(ITIMER_REAL, &neuer_wert, &zustand->wert);

	/* Signal-Handler merken und loeschen */
	zustand->handler = signal(SIGALRM, SIG_IGN);

	/* alten Zustand als typenlosen Zeiger zurueckgeben */
	return (void *)zustand;
}


/*
** timer_restaurieren
**  laesst den Timer ab dem geretteten Zustand weiterlaufen
**
** Parameter:
**  zustand: Zeiger auf Struktur mit dem geretteten Zustand
**
** Seiteneffekte:
**  Speicher, auf den zustand zeigt, wird freigegeben
*/
void timer_restaurieren(zustand)
void *zustand;
{
	/* Timer stoppen, damit nicht der Signal-Handler unbeabsichtigt
	   aufgerufen wird */
	timer_stoppen();

	/* den geretteten Signal-Handler restaurieren, bevor der Timer neu
	   gestartet wird */
	signal(SIGALRM, ((struct timer_zustand *)zustand)->handler);

	/* Timer mit den geretteten Parametern neu starten */
	setitimer(ITIMER_REAL, &((struct timer_zustand *)zustand)->wert, NULL);

	/* Speicher wieder freigeben */
	speicher_freigeben((void **)&zustand);
}
