/*
**                      Softwarepraktikum iMaze
**                              1993/94
**                Joerg Czeranski    Hans-Ulrich Kiel
**
** Datei: xv_fenster.c
**
** Kommentar:
**  Fensterverwaltung auf Basis von XView
*/


#include <stdio.h>
#include <xview/xview.h>
#include <xview/icon.h>
#include <xview/panel.h>
#include <xview/notice.h>
#include <xview/canvas.h>
#include <xview/xv_xrect.h>
#include <xview/notify.h>
#include <xview/attr.h>
#include <xview/scrollbar.h>

#include "speicher.h"
#include "grafik.h"
#include "fehler.h"
#include "client.h"
#include "system.h"
#include "X_grafik.h"
#include "xv_einaus.h"

static char sccsid[] = "@(#)xv_fenster.c	1.58 5/18/94";


/* die schwarz-weiss-Daten des Icons: */
static short icon_daten[] = {
#include "xv_icon.h"
};

/* die Maske des Icons: */
static short icon_maske_daten[] = {
#include "xv_icon_maske.h"
};


/* Struktur enthaelt alle benoetigten Informationen ueber ein Fenster
   und Zeiger auf Funktionen, die beim Neuaufbau und Veraendern der
   Groesse auszufuehren sind. Diese Struktur wird dann an ein Canvas
   angehaengt */
struct fenster_beschreibung
{
	struct fenster fenster;  /* Fensterinformation, die von xview an X11
	                            uebergeben werden muessen */
	void *daten;             /* Zeiger auf eine Kopien der im Fenster zu
	                            zu zeichnenden Grafikobjekte */
	void *X_daten;           /* private Daten des X11-Teil */
	/* Zeiger auf Funktionen, die neu zeichnen,
	   X_daten setzen bzw. X_daten freigeben */
#ifdef __STDC__
	void (*X_zeichne)(void *X_daten, struct fenster *fenster, void *daten);
	void (*X_init)(void **X_daten, struct fenster *fenster);
	void (*X_freigeben)(void **X_daten, struct fenster *fenster);
#else
	void (*X_zeichne)();
	void (*X_init)();
	void (*X_freigeben)();
#endif
};

/* Strukturen mit den Informationen zu den einzelnen Fenstern */
static struct fenster_beschreibung blickfeld_fenster;
static struct fenster_beschreibung rueckspiegel_fenster;
static struct fenster_beschreibung karte_fenster;
static struct fenster_beschreibung kompass_fenster;

/* die Frames zu den Fenstern muessen global gespeichert werden */
static Frame hauptframe, rueckspiegel_frame, karte_frame, kompass_frame;

/* der Canvas der Karte wird zum Neuerzeugen der Scrollbars benoetigt */
static Canvas karte_canvas;

/* Scrollbars fuer karte_canvas */
static Scrollbar vertikaler_scrollbar;
static Scrollbar horizontaler_scrollbar;

/* Panel-Items werden von properties_apply/reset_knopf benoetigt, um
   die Werte abzufragen oder zurueckzusetzen */
static Panel_choice_item hintergrund_aktiv_checkbox;
static Panel_slider_item hintergrund_timeout_slider;
static Panel_choice_item karte_scrollbar_checkbox;
static Panel_choice_item karte_zentriert_checkbox;
static Panel_slider_item zentrier_schritt_slider;

/* Werte der Panel-Items bei letztem Klicken auf Apply */
static int hintergrund_aktiv = 1;
static int hintergrund_timeout = 250;
static int karte_hat_scrollbars = 0;
static int karte_ist_zentriert = 0;
static int zentrier_schritt = 25;

/* aktuelle Position der Scrolllbars von karte_canvas */
static struct punkt kartenzentrum =
	{ FENSTER_BREITE / 2, FENSTER_HOEHE / 2 };

/* der Hauptframe ist am Anfang nicht vorhanden */
static int hauptframe_gueltig = 0;

/* zu Anfang besteht noch keine Verbindung zum Server */
static int verbindung_aktiv = 0;

/* diese Strukturen werden an fenster_beschreibung.daten zugewiesen und
   enthalten die Kopien der zu zeichnenden Grafikobjekte */
static struct blickfelddaten objekte, objekte_rueck;
static struct grundrissdaten grundriss;
static int blickrichtung;


/*
** fenster_init
**  setzt die Fensterinformationen, die an Canvas angehaengt sind
**
** Paramter:
**  canvas: Canvas des Fensters
**
** Seiteneffekte:
**  die an Canvas angehaengten Daten werden veraendert
*/
static void fenster_init(canvas)
Canvas canvas;
{
	Xv_Window window;                     /* Window des Canvas */
	struct fenster_beschreibung *fenster; /* Daten und Funktionen
	                                         zum Fenster */

	/* Zeiger auf Informationen zum Fenster, die an Canvas angehaengt
	   sind */
	fenster = (struct fenster_beschreibung *)xv_get(canvas, XV_KEY_DATA, 1);

	/* Window ermitteln und ablegen */
	window = canvas_paint_window(canvas);
	fenster->fenster.window = xv_get(window, XV_XID);
	fenster->fenster.display = XV_DISPLAY_FROM_WINDOW(window);

	/* Ausmasse und Farbtiefe des Fensters ablegen */
	fenster->fenster.breite = (int)xv_get(window, XV_WIDTH);
	fenster->fenster.hoehe = (int)xv_get(window, XV_HEIGHT);
	fenster->fenster.farbtiefe = xv_get(window, WIN_DEPTH);

	/* der X11-Teil legt ebenfalls seine Informationen zum Fenster ab */
	fenster->X_init(&fenster->X_daten, &fenster->fenster);
}


/*
** repaint_fenster
**  wird aufgerufen, wenn ein Repaint-Signal ausgeloest wurde;
**  der komplette Fensterinhalt wird neu gezeichnet
**
** Parameter:
**  canvas: Canvas des Fensters
**  window: Dummy-Parameter
**  display: Dummy-Parameter
**  xid: Dummy-Parameter
**  area: Dummy-Parameter
*/
static void repaint_fenster(canvas, window, display, xid, area)
Canvas canvas;
Xv_Window window;
Display *display;
Window xid;
Xv_xrectlist *area;
{
	struct fenster_beschreibung *fenster; /* Daten und Funktionen
	                                         zum Fenster */

	/* Zeiger auf Informationen zum Fenster, die an Canvas angehaengt
	   sind */
	fenster = (struct fenster_beschreibung *)xv_get(canvas, XV_KEY_DATA, 1);

	/* Anfang des Neuzeichnens synchronisieren */
	X_zeichne_sync_anfang(fenster->fenster.display);

	/* X11-Routine, die das Neuzeichnen uebernimmt */
	fenster->X_zeichne(fenster->X_daten, &fenster->fenster,
		fenster->daten);

	/* Ende des Neuzeichnens synchronisieren */
	X_zeichne_sync_ende(fenster->fenster.display);
}


/*
** resize_fenster
**  wird aufgerufen, wenn die Groesse eines Fensters veraendert wurde;
**  die privaten Daten aus dem X11-Grafikteil werden neu gesetzt;
**  Neuzeichnen erfolgt jedoch erst beim repaint
**
** Parameter:
**  canvas: Canvas des Fensters
**  breite: Dummy-Parameter
**  hoehe: Dummy-Parameter
*/
static void resize_fenster(canvas, breite, hoehe)
Canvas canvas;
int breite, hoehe;
{
	struct fenster_beschreibung *fenster; /* Daten und Funktionen
	                                         zum Fenster */

	/* Zeiger auf Informationen zum Fenster, die an Canvas
	   angehaengt sind */
	fenster = (struct fenster_beschreibung *)xv_get(canvas, XV_KEY_DATA, 1);

	/* Anfang des Neuzeichnens synchronisieren */
	X_zeichne_sync_anfang(fenster->fenster.display);

	/* Freigeben der privaten Daten des X11-Grafikteil */
	fenster->X_freigeben(&fenster->X_daten, &fenster->fenster);

	/* private Daten des X11-Grafikteil neu ermitteln */
	fenster_init(canvas);

	/* Ende des Neuzeichnens synchronisieren */
	X_zeichne_sync_ende(fenster->fenster.display);
}


/*
** objekte_kopieren
**  kopiert Liste, in der die Grafikobbjekte fuer die 3D-Darstellung
**  zwischengespeichert werden
**
** Parameter:
**  objekte: Liste, Ziel
**  objekte_neu: Liste, Quelle
**
** Seiteneffekte:
**  objekte_neu wird gesetzt
*/
static void objekte_kopieren(objekte, objekte_neu)
struct objektdaten *objekte, *objekte_neu;
{
	int i; /* Index fuer die Kugeln */

	objekte->hintergrund_zeichnen = objekte_neu->hintergrund_zeichnen;

	/* Liste der Waende freigeben und dann neue Kopie erstellen */
	liste_freigeben(&objekte->anzwaende, (void **)&objekte->waende);
	liste_kopieren(&objekte->anzwaende, (void **)&objekte->waende,
		&objekte_neu->anzwaende, (void **)&objekte_neu->waende,
		sizeof(struct flaeche));

	/* in Liste mit Kugeln jeweils Unterliste mit Sichtbarkeitsbereichen
	   freigeben */
	for (i = 0; i < objekte->anzkugeln; i++)
		liste_freigeben(&objekte->kugeln[i].sichtanz,
			(void **)&objekte->kugeln[i].sichtbar);

	/* Liste mit Kugeln freigeben und dann neu kopieren */
	liste_freigeben(&objekte->anzkugeln, (void **)&objekte->kugeln);
	liste_kopieren(&objekte->anzkugeln, (void **)&objekte->kugeln,
		&objekte_neu->anzkugeln, (void **)&objekte_neu->kugeln,
		sizeof(struct kugel));

	/* in Liste mit Kugeln jeweils Unterliste mit Sichtbarkeitsbereichen
	   neu kopieren */
	for (i = 0; i < objekte->anzkugeln; i++)
		liste_kopieren(&objekte->kugeln[i].sichtanz,
			(void **)&objekte->kugeln[i].sichtbar,
			&objekte_neu->kugeln[i].sichtanz,
			(void **)&objekte_neu->kugeln[i].sichtbar,
			sizeof(struct ausschnitt));
}


/*
** objekt_listen_init
**  initialisiert Liste, in der die Grafikobbjekte fuer die 3D-Darstellung
**  zwischengespeichert werden
**
** Parameter:
**  objekte: Liste, die initialisiert werden soll
**
** Seiteneffekte:
**  *objekte wird gesetzt
*/
static void objekt_listen_init(objekte)
struct objektdaten *objekte;
{
	/* zu Anfang ist das Bild leer */
	objekte->hintergrund_zeichnen = 0;

	/* objekte enthaelt ihrerseits eine Liste von Waenden und Kugeln */
	liste_initialisieren(&objekte->anzwaende, (void **)&objekte->waende);
	liste_initialisieren(&objekte->anzkugeln, (void **)&objekte->kugeln);
}


/*
** disconnect_knopf
**  wird aufgerufen, falls der Disconnect-Knopf gedrueckt wird, und beendet
**  nach einer Sicherheitsabfrage die Verbindung
**
** Parameter:
**  knopf: betroffener Knopf
**  event: Event der den Knopf ausgeloest hat
**
** Rueckgabewert:
**  XV_OK oder XV_ERROR
*/
static int disconnect_knopf(knopf, event)
Panel_item knopf;
Event *event;
{
	disconnect_abfrage();

	return XV_OK;
}


/*
** fenster_knopf
**  behandelt alle Knoepfe, mit denen ein Unterfenster geoeffnet wird,
**  in dem eine X11-Grafik aufgebaut ist
**
** Parameter:
**  menu: Menue, in dem sich der betroffene Knopf befindet
**  knopf: betroffener Knopf
**
** Rueckgabewert:
**  XV_OK oder XV_ERROR
*/
static int fenster_knopf(menu, knopf)
Menu menu;
Menu_item knopf;
{
	/* enthaelt Daten und Funktionen zum Fenster */
	struct fenster_beschreibung *fenster;

	/* Zeiger auf Informationen zum Fenster, die an Panel_item angehaengt
	   sind */
	fenster = (struct fenster_beschreibung *)xv_get(knopf, XV_KEY_DATA, 1);

	/* Betroffenes Fenster wird auf sichtbar gesetzt */
	xv_set(xv_get(knopf, XV_KEY_DATA, 0),
		XV_SHOW, TRUE,
		0);

	/* Anfang des Neuzeichnens synchronisieren */
	X_zeichne_sync_anfang(fenster->fenster.display);

	/* Funktion zum Neuzeichnen des Fensters aufrufen */
	fenster->X_zeichne(fenster->X_daten,
		&fenster->fenster, fenster->daten);

	/* Ende des Neuzeichnens synchronisieren */
	X_zeichne_sync_ende(fenster->fenster.display);

	return XV_OK;
}


/*
** fenster_oeffnen_knopf
**  behandelt alle Knoepfe, mit denen ein Unterfenster geoeffnet wird
**
** Parameter:
**  knopf: betroffener Knopf
**  event: Event, durch den der betroffene Knopf ausgeloest wurde
**
** Rueckgabewert:
**  XV_OK oder XV_ERROR
*/
static int fenster_oeffnen_knopf(knopf, event)
Panel_item knopf;
Event *event;
{
	/* Betroffenes Fenster wird auf sichtbar gesetzt */
	xv_set(xv_get(knopf, XV_KEY_DATA, 0),
		XV_SHOW, TRUE,
		0);

	return XV_OK;
}


/*
** properties_kategorie_knopf
**  schaltet des Properties-Panel auf eine andere Kategorie um
**
** Parameter:
**  knopf: Knopf, durch den die Funktion ausgeloest wurde
**  wert: neuer Zustand des Knopfes
**  event: Event, durch den der betroffene Knopf ausgeloest wurde
**
** Rueckgabewert:
**  XV_OK oder XV_ERROR
*/
static int properties_kategorie_knopf(knopf, wert, event)
Panel_item knopf;
int wert;
Event *event;
{
	Panel properties_panel; /* gesamter Properties-Panel */
	Panel subpanel;         /* ein Teilpanel */
	Frame properties_frame; /* Frame, den der Panel ausfuellt */
	int i;                  /* Index fuer die Teilpanels */

	/* Panel und Frame aus Knopf ermitteln */
	properties_panel = xv_get(knopf, XV_OWNER);
	properties_frame = xv_get(properties_panel, XV_OWNER);

	/* alle Teilpanels pruefen */
	for (i = 0;; i++)
	{
		/* kein weiterer Teilpanel? */
		if (!(subpanel = xv_get(knopf, XV_KEY_DATA, i)))
			break;

		/* Teilpanel auf sichtbar/unsichtbar setzen */
		xv_set(subpanel,
			XV_SHOW, i == wert ? TRUE : FALSE,
			0);
	}

	/* gewaehlten Teilpanel feststellen */
	subpanel = xv_get(knopf, XV_KEY_DATA, wert);

	/* Framegroesse an Groesse des Teilpanels anpassen */
	xv_set(properties_frame,
		XV_WIDTH, xv_get(subpanel, XV_KEY_DATA, 0),
		XV_HEIGHT, xv_get(subpanel, XV_KEY_DATA, 1),
		0);

	return XV_OK;
}


/*
** slider_aktivieren_knopf
**  passt die Sichtbarkeit eines Sliders an den Wert
**  der zugehoerigen Checkbox an
**
** Parameter:
**  knopf: die Checkbox
**  wert: neuer Zustand der Checkbox
**  event: Event, durch den die Checkbox veraendert wurde
**
** Rueckgabewert:
**  XV_OK oder XV_ERROR
*/
static int slider_aktivieren_knopf(knopf, wert, event)
Panel_item knopf;
int wert;
Event *event;
{
	/* Slider auf sichtbar/faded out setzen */
	xv_set(xv_get(knopf, XV_KEY_DATA, 0),
		PANEL_INACTIVE, wert ? FALSE : TRUE,
		0);

	return XV_OK;
}


/*
** properties_apply_knopf
**  der Apply-Knopf uebernimmt die gewaehlten Werte
**
** Parameter:
**  knopf: der Apply-Knopf
**  event: Event, durch den der Knopf ausgeloest wurde
**
** Rueckgabewert:
**  XV_OK oder XV_ERROR
*/
static int properties_apply_knopf(knopf, event)
Panel_item knopf;
Event *event;
{
	/* welche Kategorie ist sichtbar? */
	switch (xv_get(knopf, XV_KEY_DATA, 0))
	{
		case 0: /* Kategorie: Game */
			/* Timer evtl. stoppen */
			if (hintergrund_aktiv)
				timer_stoppen();

			/* Timerparameter uebernehmen */
			hintergrund_aktiv = xv_get(hintergrund_aktiv_checkbox,
				PANEL_VALUE);
			hintergrund_timeout = xv_get(hintergrund_timeout_slider,
				PANEL_VALUE);

			/* Timer evtl. neu starten */
			if (hintergrund_aktiv)
				timer_starten(hintergrund_timeout);

			break;

		case 1: /* Kategorie: Map */
			{
				int scrollbars_existieren; /* Scrollbars existieren; nicht
				                              notwendig sichtbar */

				/* Scrollbars werden auch fuer das Zentrieren benoetigt */
				scrollbars_existieren = karte_hat_scrollbars ||
					karte_ist_zentriert;

				/* Scrolling-Parameter uebernehmen */
				karte_hat_scrollbars = xv_get(karte_scrollbar_checkbox,
					PANEL_VALUE);
				karte_ist_zentriert = xv_get(karte_zentriert_checkbox,
					PANEL_VALUE);
				zentrier_schritt = xv_get(zentrier_schritt_slider,
					PANEL_VALUE);

				/* Scrollbars erzeugen/zerstoeren? */
				if (scrollbars_existieren !=
					(karte_hat_scrollbars || karte_ist_zentriert))
					if (scrollbars_existieren)
					{
						/* Scrollbars zerstoeren */
						xv_destroy(vertikaler_scrollbar);
						xv_destroy(horizontaler_scrollbar);
					}
					else
					{
						/* Scrollbars neu erzeugen */
						vertikaler_scrollbar = xv_create(karte_canvas,
								SCROLLBAR,
							SCROLLBAR_DIRECTION, SCROLLBAR_VERTICAL,
							0);
						horizontaler_scrollbar = xv_create(karte_canvas,
								SCROLLBAR,
							SCROLLBAR_DIRECTION, SCROLLBAR_HORIZONTAL,
							0);
					}

				/* existieren jetzt Scrollbars? */
				scrollbars_existieren = karte_hat_scrollbars ||
					karte_ist_zentriert;

				if (scrollbars_existieren)
				{
					/* Scrollbars auf sichtbar/unsichtbar/faded out
					   setzen */
					xv_set(vertikaler_scrollbar,
						XV_SHOW, karte_hat_scrollbars ? TRUE : FALSE,
						SCROLLBAR_INACTIVE, karte_ist_zentriert ?
							TRUE : FALSE,
						0);
					xv_set(horizontaler_scrollbar,
						XV_SHOW, karte_hat_scrollbars ? TRUE : FALSE,
						SCROLLBAR_INACTIVE, karte_ist_zentriert ?
							TRUE : FALSE,
						0);
				}

				/* automatisches Anpassen der Canvasgroesse an die
				   Fenstergroesse nur, falls keine Scrollbars existieren */
				xv_set(karte_canvas,
					CANVAS_AUTO_SHRINK, scrollbars_existieren ?
						FALSE : TRUE,
					0);
			}
			break;
	}

	return XV_OK;
}


/*
** properties_reset_knopf
**  der Reset-Knopf setzt die Werte auf die vom letzten Apply zurueck
**
** Parameter:
**  knopf: der Reset-Knopf
**  event: Event, durch den der Knopf ausgeloest wurde
**
** Rueckgabewert:
**  XV_OK oder XV_ERROR
*/
static int properties_reset_knopf(knopf, event)
Panel_item knopf;
Event *event;
{
	/* welche Kategorie ist sichtbar? */
	switch (xv_get(knopf, XV_KEY_DATA, 0))
	{
		case 0: /* Kategorie: Game */
			xv_set(hintergrund_aktiv_checkbox,
				PANEL_VALUE, hintergrund_aktiv,
				0);
			xv_set(hintergrund_timeout_slider,
				PANEL_VALUE, hintergrund_timeout,
				PANEL_INACTIVE, hintergrund_aktiv ? FALSE : TRUE,
				0);
			break;

		case 1: /* Kategorie: Map */
			xv_set(karte_scrollbar_checkbox,
				PANEL_VALUE, karte_hat_scrollbars,
				0);
			xv_set(karte_zentriert_checkbox,
				PANEL_VALUE, karte_ist_zentriert,
				0);
			xv_set(zentrier_schritt_slider,
				PANEL_INACTIVE, karte_ist_zentriert ? FALSE : TRUE,
				PANEL_VALUE, zentrier_schritt,
				0);
			break;
	}

	return XV_OK;
}


/*
** spiel_deskriptor
**  fragt den Spiel-Deskriptor des Netzwerkteils ab
**
** Rueckgabewert:
**  der Deskriptor oder -1 falls der Deskriptor nicht
**  abgefragt werden konnte
*/
static int spiel_deskriptor()
{
	int *deskriptor; /* Zeiger auf den Deskriptor */

	/* Deskriptor abfragen */
	if ((deskriptor = bewegung_deskriptor()) == NULL)
		/* es wurde kein Deskriptor zurueckgegeben */
		return -1;

	/* Deskriptor weiterreichen */
	return *deskriptor;
}


/*
** ende_ausloesen
**  leitet das Programmende ohne Sicherheitsabfrage ein
*/
static void ende_ausloesen()
{
	/* muss die Verbindung noch beendet werden? */
	if (verbindung_aktiv)
	{
		int deskriptor; /* Spiel-Deskriptor vom Netzwerkteil */

		verbindung_aktiv = 0;

		/* falls es einen Spiel-Deskriptor gibt, Handler loeschen */
		if ((deskriptor = spiel_deskriptor()) >= 0)
			notify_set_input_func(hauptframe, NULL, deskriptor);

		/* Verbindung zum Server beenden (mit Timeout) */
		client_spiel_beenden();
	}

	/* Standard-xview-Funktion fuer Programmende */
	xv_destroy_safe(hauptframe);
}


/*
** abbruch_signal_handler
**  wird aufgerufen, falls Conrol-C gedrueckt wird (o.ae.) und veranlasst
**  nach einer Sicherheitsabfrage das Programmende
**
** Parameter:
**  client: Dummy-Parameter
**  signal: Dummy-Parameter
**  mode: Dummy-Parameter
**
** Rueckgabewert:
**  NOTIFY_DONE oder NOTIFY_ERROR
*/
static Notify_value abbruch_signal_handler(client, signal, mode)
Notify_client client;
int signal;
Notify_signal_mode mode;
{
	ende_ausloesen();

	return NOTIFY_DONE;
}


/*
** paket_abfragen_handler
**  wird durch den Hintergrund-Timer oder aus dem
**  deskriptor_bereit_handler aufgerufen; laesst den Netzwerkteil
**  Pakete empfangen
**
** Parameter:
**  signum: Dummy-Parameter
*/
static void paket_abfragen_handler(signum)
int signum;
{
	/* falls die Verbindung bereits beendet wurde, Handler loeschen */
	if (!verbindung_aktiv)
	{
		/* falls Timer noch laeuft, Timer stoppen */
		if (hintergrund_aktiv)
			timer_stoppen();

		signal(SIGALRM, SIG_DFL);
		return;
	}

	/* Timer evtl. stoppen */
	if (hintergrund_aktiv)
		timer_stoppen();

	/* der Netzwerkteil kann hier aufgelaufene Pakete abarbeiten,
	   kein Aufruf der Zeichenroutinen */
	bewegung(0, 0);

	/* Timer evtl. neu starten */
	if (hintergrund_aktiv)
		timer_starten(hintergrund_timeout);
}


/*
** deskriptor_bereit_handler
**  wird aufgerufen, wenn der Spiel-Deskriptor des Netzwerkteils
**  Daten bereit haelt
**
** Parameter:
**  client: Dummy-Parameter
**
** Rueckgabewert:
**  NOTIFY_DONE oder NOTIFY_ERROR
*/
static Notify_value deskriptor_bereit_handler(client)
Notify_client client;
{
	/* falls die Verbindung bereits beendet wurde, nichts tun */
	if (!verbindung_aktiv)
		return;

	/* Timer evtl. neu starten */
	if (hintergrund_aktiv)
		timer_starten(hintergrund_timeout);

	/* der Netzwerkteil kann hier aufgelaufene Pakete abarbeiten,
	   Ergebnis grafisch darstellen durch Aufruf der Zeichenroutinen */
	bewegung(0, 1);

	/* Timer evtl. neu starten */
	if (hintergrund_aktiv)
		timer_starten(hintergrund_timeout);

	return NOTIFY_DONE;
}


/*
** hauptframe_ende_handler
**  wird vor dem Zerstoeren des Hauptframes aufgerufen und setzt
**  hauptframe_gueltig zurueck, falls notwendig
**
** Parameter:
**  client: Dummy-Parameter
**  status: Grund des Aufrufs
**
** Rueckgabewert:
**  NOTIFY_DONE oder NOTIFY_ERROR
**
** Seiteneffekte:
**  hauptframe_gueltig wird veraendert
*/
static Notify_value hauptframe_ende_handler(client, status)
Notify_client client;
Destroy_status status;
{
	switch (status)
	{
		case DESTROY_CHECKING: /* Endeabfrage moeglich */
			/* muss die Verbindung noch beendet werden? */
			if (verbindung_aktiv)
			{
				int deskriptor; /* Spiel-Deskriptor vom Netzwerkteil */

				verbindung_aktiv = 0;

				/* falls es einen Spiel-Deskriptor gibt, Handler loeschen */
				if ((deskriptor = spiel_deskriptor()) >= 0)
					notify_set_input_func(hauptframe, NULL, deskriptor);

				/* Verbindung zum Server beenden (mit Timeout) */
				client_spiel_beenden();
			}
			break;

		case DESTROY_CLEANUP: /* Hauptframe zerstoert */
		case DESTROY_PROCESS_DEATH: /* Prozessende */
			hauptframe_gueltig = 0;
			break;
	}

	/* naechsten Handler aufrufen */
	return notify_next_destroy_func(client, status);
}


/*
** notice_oeffnen
**  oeffnet ein Notice-Fenster (eine Abfragebox mit Knoepfen), wartet bis
**  ein Knopf gedrueckt wurde und gibt dessen Nummer zurueck
**
** Parameter:
**  attribute: mit attr_create_list erzeugt Liste von Attributen
**             fuer das Fenster
**
** Rueckgabewert:
**  Nummer des Knopfes oder NOTICE_FAILED fuer Fehler
*/
static int notice_oeffnen(attribute)
Attr_avlist attribute;
{
	Frame frame;      /* Frame, zu dem das Notice-Fenster gehoeren soll */
	Xv_Notice notice; /* Notice-Fenster */
	int wahl;         /* Nummer des gewaehlten Knopfes */

	/* falls keine Attributliste in der uebergeordneten Routine
	   erzeugt werden konnte, Fehler zurueckgeben */
	if (attribute == NULL)
		return NOTICE_FAILED;

	/* falls hauptframe noch nicht gueltig, extra einen Frame erzeugen */
	if (hauptframe_gueltig)
		frame = hauptframe;
	else
		frame = xv_create(XV_NULL, FRAME,
			0);

	/* Notice-Fenster erzeugen, anzeigen, auf Knopf warten und
	   wieder schliessen */
	notice = xv_create(frame, NOTICE,
		ATTR_LIST, attribute,
		XV_SHOW, TRUE,
		0);

	/* welcher Knopf wurde gedrueckt? */
	wahl = xv_get(notice, NOTICE_STATUS);

	/* Notice-Fenster zerstoeren */
	xv_destroy(notice);

	/* falls extra ein Frame erzeugt wurde, wieder freigeben */
	if (!hauptframe_gueltig)
		xv_destroy(frame);

	/* Attributliste wieder freigeben */
	free(attribute);

	/* Nummer des gewaehlten Knopfes zurueckgeben */
	return wahl;
}


/*
** scrollbar_zentrieren
**  zentriert einen Scrollbar neu
**
** Parameter:
**  scrollbar: der Scrollbar
**  zentrum: das neue Zentrum in virtuellen Bildpunkten
**  skalierung: die Gesamtgroesse in virtuellen Bildpunkten
*/
static void scrollbar_zentrieren(scrollbar, zentrum, skalierung)
Scrollbar scrollbar;
int zentrum, skalierung;
{
	int gesamt;        /* Gesamtgroesse des zu scrollenden Objekts */
	int ausschnitt;    /* Groesse des sichtbaren Ausschnitts */
	int neue_position; /* neue Position des Ausschnitts */
	int alte_position; /* alte Position des Ausschnitts */

	/* Gesamt-, Ausschnittgroesse und alte Position abfragen */
	gesamt = xv_get(scrollbar, SCROLLBAR_OBJECT_LENGTH);
	ausschnitt = xv_get(scrollbar, SCROLLBAR_VIEW_LENGTH);
	alte_position = xv_get(scrollbar, SCROLLBAR_VIEW_START);

	/* neue Position */
	neue_position = zentrum * gesamt / skalierung - ausschnitt / 2;

	/* weicht die neue Position schon um die Zentrier-Schrittweite von
	   der alten ab? */
	if (neue_position < alte_position -
		ausschnitt * zentrier_schritt / 100 ||
		neue_position > alte_position +
		ausschnitt * zentrier_schritt / 100)
	{
		/* neue Position mit 0 und gesamt - ausschnitt beschraenken */
		if (neue_position < 0)
			neue_position = 0;
		if (neue_position > gesamt - ausschnitt)
			neue_position = gesamt - ausschnitt;

		if (neue_position != alte_position)
			/* neue Position setzen */
			xv_set(scrollbar,
				SCROLLBAR_VIEW_START, neue_position,
				0);
	}
}


/* bis hier lokaler Teil                       */
/***********************************************/
/* ab hier globaler Teil                       */


/*
** zeichne_blickfeld
**  zeichnet 3D-Darstellung im Hauptfenster
**
** Parameter:
**  objekte_neu: Liste mit allen zu zeichnenden Grafikobjekten
**
** Seiteneffekte:
**  objekte wird neu gesetzt
*/
void zeichne_blickfeld(objekte_neu)
struct objektdaten *objekte_neu;
{
	/* lokale Kopie der Daten anlegen */
	objekte_kopieren(&objekte.objekte, objekte_neu);

	/* falls Fenster geoeffnet, X11-Routine aufrufen */
	if (xv_get(hauptframe, FRAME_CLOSED) == FALSE)
		blickfeld_fenster.X_zeichne(blickfeld_fenster.X_daten,
			&blickfeld_fenster.fenster, blickfeld_fenster.daten);
}


/*
** zeichne_rueckspiegel
**  zeichnet 3D-Darstellung im Rueckspiegel
**
** Parameter:
**  objekte_rueck_neu: Liste mit allen zu zeichnenden Grafikobjekten
**
** Seiteneffekte:
**  objekte_rueck wird neu gesetzt
*/
void zeichne_rueckspiegel(objekte_rueck_neu)
struct objektdaten *objekte_rueck_neu;
{
	/* lokale Kopie der Daten anlegen */
	objekte_kopieren(&objekte_rueck.objekte, objekte_rueck_neu);

	/* falls Fenster geoeffnet und sichtbar, X11-Routine aufrufen */
	if (xv_get(rueckspiegel_frame, FRAME_CLOSED) == FALSE &&
		xv_get(rueckspiegel_frame, XV_SHOW) == TRUE)
		rueckspiegel_fenster.X_zeichne(rueckspiegel_fenster.X_daten,
			&rueckspiegel_fenster.fenster, rueckspiegel_fenster.daten);
}


/*
** zeichne_karte
**  bewegt Spieler auf der Karte
**
** Parameter
**  spieler_index: Index des Spielers selbst in der Liste
**                 oder -1, falls Spieler tot
**  anzkreise_neu: Anzahl der Spieler
**  kreise_neu: Liste mit Positionen der Spieler
**
** Seiteneffekte:
**  grundriss wird neu gesetzt
*/
void zeichne_karte(spieler_index, anzkreise_neu, kreise_neu)
int spieler_index, anzkreise_neu;
struct kartenkreis *kreise_neu;
{
	if (spieler_index >= 0)
	{
		kartenzentrum.x = kreise_neu[spieler_index].mittelpunkt.x;
		kartenzentrum.y = kreise_neu[spieler_index].mittelpunkt.y;

		if (karte_ist_zentriert)
		{
			scrollbar_zentrieren(horizontaler_scrollbar,
				kartenzentrum.x, FENSTER_BREITE);
			scrollbar_zentrieren(vertikaler_scrollbar,
				kartenzentrum.y, FENSTER_HOEHE);
		}
	}

	/* lokale Kopie der Daten fuer X_zeichne_karte retten und neu anlegen */
	grundriss.anzkreise_alt = grundriss.anzkreise;
	grundriss.kreise_alt = grundriss.kreise;
	liste_kopieren(&grundriss.anzkreise, (void **)&grundriss.kreise,
		&anzkreise_neu, (void **)&kreise_neu, sizeof(struct kartenkreis));

	/* falls Fenster sichtbar, X11-Routine aufrufen */
	if (xv_get(karte_frame, XV_SHOW) == TRUE)
		X_zeichne_karte(karte_fenster.X_daten, &karte_fenster.fenster,
			karte_fenster.daten);

	/* lokale Kopie der veralteten Daten freigeben */
	liste_freigeben(&grundriss.anzkreise_alt,
		(void **)&grundriss.kreise_alt);
	grundriss.anzkreise_alt = 0;
	grundriss.kreise_alt = NULL;
}


/*
** zeichne_grundriss
**  zeichnet den Grundriss der Karte und die Spieler neu
**
** Parameter:
**  anzlinien: Anzahl der Linienstuecke
**  linien_neu: Liste mit Daten ueber Linienstuecke
**
** Seiteneffekte:
**  grundriss wird neu gesetzt
*/
void zeichne_grundriss(anzlinien_neu, linien_neu)
int anzlinien_neu;
struct kartenlinie *linien_neu;
{
	/* lokale Kopie der Daten freigeben und neu anlegen */
	liste_freigeben(&grundriss.anzlinien, (void **)&grundriss.linien);
	liste_kopieren(&grundriss.anzlinien, (void **)&grundriss.linien,
		&anzlinien_neu, (void **)&linien_neu, sizeof(struct kartenlinie));

	/* falls Fenster sichtbar, X11-Routine aufrufen */
	if (xv_get(karte_frame, XV_SHOW) == TRUE)
		karte_fenster.X_zeichne(karte_fenster.X_daten,
			&karte_fenster.fenster, karte_fenster.daten);
}


/*
** zeichne_kompass
**  zeichnet Kompass
**
** Parameter:
**  blickrichtung_neu: Blickrichtung des Spieler (0..TRIGANZ)
**                     oder -1, falls Spieler tot
**
** Seiteneffekte:
**  setzt blickrichtung
*/
void zeichne_kompass(blickrichtung_neu)
int blickrichtung_neu;
{
	/* lokale Kopie der Blickrichtung */
	blickrichtung = blickrichtung_neu;

	/* falls Fenster geoeffnet und sichtbar, X11-Routine aufrufen */
	if (xv_get(kompass_frame, FRAME_CLOSED) == FALSE &&
		xv_get(kompass_frame, XV_SHOW) == TRUE)
		kompass_fenster.X_zeichne(kompass_fenster.X_daten,
			&kompass_fenster.fenster, kompass_fenster.daten);
}


/*
** zeichne_sync_anfang
**  synchronisiert den Anfang des Zeichnens
*/
void zeichne_sync_anfang()
{
	/* weiterreichen an X11-Teil */
	X_zeichne_sync_anfang(XV_DISPLAY_FROM_WINDOW(hauptframe));
}


/*
** zeichne_sync_ende
**  synchronisiert das Ende des Zeichnens
*/
void zeichne_sync_ende()
{
	/* weiterreichen an X11-Teil */
	X_zeichne_sync_ende(XV_DISPLAY_FROM_WINDOW(hauptframe));
}


/*
** grafik_schleife
**  startet xview-grafik-schleife,
**  wird erst nach Abbau der Grafik wieder verlassen
*/
void grafik_schleife()
{
	/* zur Zeit ist hier die Verbindung bereits aufgebaut */
	verbindung_aktiv = 1;

	/* Hintergrund-Handler aktivieren */
	signal(SIGALRM, paket_abfragen_handler);
	if (hintergrund_aktiv)
		/* evtl. Timer starten */
		timer_starten(hintergrund_timeout);

	{
		int deskriptor; /* Spiel-Deskriptor vom Netzwerkteil */

		/* falls es einen Spiel-Deskriptor gibt, Bereit-Handler
		   aktivieren */
		if ((deskriptor = spiel_deskriptor()) >= 0)
			notify_set_input_func(hauptframe, deskriptor_bereit_handler,
				deskriptor);
	}

	/* Haupt-Schleife von xview */
	xv_main_loop(hauptframe);

	/* Timer stoppen und Hintergrund-Handler loeschen */
	timer_stoppen();
	signal(SIGALRM, SIG_DFL);

	/* der Hauptframe ist nach dem Ende nicht mehr vorhanden */
	hauptframe_gueltig = 0;
}


/*
** grafik_init
**  erzeugt die Fensterstruktur mit Knoepfen etc.
**
** Parameter:
**  argc: Zeiger auf Anzahl der an das Programm uebergebenen Parameter
**  argv: Zeiger auf Liste mit Strings, die die Aufrufparameter enthalten
**
** Rueckgabewert:
**  0 bei ordentlicher Initialisierung, 1 bei Fehler
**
** Seiteneffekte:
**  *argc und *argv werden veraendert
**  alle globalen Variablen werden gesetzt bzw. initialisiert
*/
int grafik_init(argc, argv)
int *argc;
char **argv;
{
	Panel hauptpanel; /* Panel mit allen linksbuendigen Knoepfen */

	/* Listen fuer Zwischenspeicherung der Grafikobjekte initialisieren */
	objekt_listen_init(&objekte.objekte);
	objekt_listen_init(&objekte_rueck.objekte);
	liste_initialisieren(&grundriss.anzlinien, (void **)&grundriss.linien);
	liste_initialisieren(&grundriss.anzkreise, (void **)&grundriss.kreise);
	grundriss.anzkreise_alt = 0;
	grundriss.kreise_alt = NULL;

	/* Rueckspiegel hat im Gegensatz zur Frontansicht kein Zielkreuz */
	objekte.fadenkreuz = 1;
	objekte_rueck.fadenkreuz = 0;

	/* Initialisierung der blickrichtung (Kompassnadel nicht zeichnen) */
	blickrichtung = -1;

	/* xview wertet seine Kommandozeilen-Parameter aus und fuehrt
	   Initialisierung aus */
	xv_init(XV_INIT_ARGC_PTR_ARGV, argc, argv, 0);

	/* Erzeugen des Haupfensters */
	hauptframe = xv_create(XV_NULL, FRAME,
		FRAME_LABEL, "iMaze - Version 1.0",
		FRAME_ICON, xv_create(XV_NULL, ICON, /* das Icon */
			ICON_IMAGE, xv_create(XV_NULL, SERVER_IMAGE,
				XV_WIDTH, 64,
				XV_HEIGHT, 64,
				SERVER_IMAGE_BITS, icon_daten,
				0),
			ICON_MASK_IMAGE, xv_create(XV_NULL, SERVER_IMAGE,
				XV_WIDTH, 64,
				XV_HEIGHT, 64,
				SERVER_IMAGE_BITS, icon_maske_daten,
				0),
			XV_LABEL, "iMaze", /* der Text unter dem Icon */
			0),
		0);

	/* Handler fuer Zerstoeren des Hauptframes */
	notify_interpose_destroy_func(hauptframe, hauptframe_ende_handler);

	/* das Hauptfenster ist vorhanden */
	hauptframe_gueltig = 1;

	/* X11-Funktion, die die Farben initialisiert */
	if (X_farben_init(XV_DISPLAY_FROM_WINDOW(hauptframe),
		(int)xv_get(hauptframe, SCREEN_NUMBER)))
	{
		/* Standard-xview-Funktion zum Zerstoeren des Hauptframes */
		xv_destroy_safe(hauptframe);

		/* das Hauptfenster ist schon wieder weg */
		hauptframe_gueltig = 0;

		return 1;
	}

	/* Handler fuer die Verwaltung von Signalen vom TTY
	   (Control-C und Hangup) und normalem Kill */
	notify_set_signal_func(hauptframe, abbruch_signal_handler,
		SIGINT, NOTIFY_SYNC);
	notify_set_signal_func(hauptframe, abbruch_signal_handler,
		SIGHUP, NOTIFY_SYNC);
	notify_set_signal_func(hauptframe, abbruch_signal_handler,
		SIGTERM, NOTIFY_SYNC);


	{
		/* initialisiert das Kartenfenster als Unterfenster */
		karte_frame = xv_create(hauptframe, FRAME_CMD,
			XV_X, 0,
			XV_Y, 0,
			FRAME_LABEL, "iMaze: Map",
			FRAME_CMD_PUSHPIN_IN, TRUE,
			FRAME_SHOW_RESIZE_CORNER, TRUE, /* Groesse veraendern erlaubt */
			0);

		/* kein Panel-Bereich erwuenscht */
		xv_destroy(xv_get(karte_frame, FRAME_CMD_PANEL));

		/* Zeichenfeld initialisieren */
		karte_canvas = xv_create(karte_frame, CANVAS,
			CANVAS_X_PAINT_WINDOW, TRUE,
			CANVAS_FIXED_IMAGE, FALSE,
			CANVAS_RETAINED, FALSE,
			/* Anhaengen von Struktur mit Daten und Funktion, die beim
			   Neuzeichnen erforderlich sind */
			XV_KEY_DATA, 1, &karte_fenster,
			0);

		if (karte_hat_scrollbars || karte_ist_zentriert)
		{
			/* Scrollbars erzeugen und auf sichtbar/unsichtbar/faded out
			   setzen */
			vertikaler_scrollbar = xv_create(karte_canvas, SCROLLBAR,
				SCROLLBAR_DIRECTION, SCROLLBAR_VERTICAL,
				XV_SHOW, karte_hat_scrollbars ? TRUE : FALSE,
				SCROLLBAR_INACTIVE, karte_ist_zentriert ? TRUE : FALSE,
				0);
			horizontaler_scrollbar = xv_create(karte_canvas, SCROLLBAR,
				SCROLLBAR_DIRECTION, SCROLLBAR_HORIZONTAL,
				XV_SHOW, karte_hat_scrollbars ? TRUE : FALSE,
				SCROLLBAR_INACTIVE, karte_ist_zentriert ? TRUE : FALSE,
				0);
		}

		/* Standard-Fenstergroesse setzen */
		xv_set(karte_canvas,
			XV_WIDTH, xv_get(karte_canvas, XV_HEIGHT) * 3 / 4,
			XV_HEIGHT, xv_get(karte_canvas, XV_HEIGHT) * 3 / 4,
			0);
		window_fit(karte_frame);

		/* Fenstergroesse anpassen */
		xv_set(xv_get(karte_frame, FRAME_CMD_PANEL),
			XV_WIDTH, WIN_EXTEND_TO_EDGE,
			0);
		window_fit_height(xv_get(karte_frame, FRAME_CMD_PANEL));

		/* Initialisierung der oben genannten Struktur */
		karte_fenster.daten = &grundriss;
		karte_fenster.X_zeichne = X_zeichne_grundriss;
		karte_fenster.X_init = X_karte_init;
		karte_fenster.X_freigeben = X_karte_freigeben;

		/* initialisiert Daten fuer X11-Grafik in karte_fenster */
		fenster_init(karte_canvas);

		/* Funktionen fuer Neuzeichen setzen */
		xv_set(karte_canvas,
			CANVAS_REPAINT_PROC, repaint_fenster,
			CANVAS_RESIZE_PROC, resize_fenster,
			0);

		/* Kartengroesse automatisch an Fenstergroesse anpassen? */
		xv_set(karte_canvas,
			CANVAS_AUTO_SHRINK, karte_hat_scrollbars ||
				karte_ist_zentriert ? FALSE : TRUE,
			0);
	}


	{
		Canvas rueckspiegel_canvas;

		/* initialisiert das Rueckspiegelfenster als Unterfenster */
		rueckspiegel_frame = xv_create(hauptframe, FRAME_CMD,
			XV_X, xv_get(karte_frame, XV_WIDTH) + 12,
			XV_Y, 0,
			FRAME_LABEL, "iMaze: Look Back",
			FRAME_CMD_PUSHPIN_IN, TRUE,
			FRAME_SHOW_RESIZE_CORNER, TRUE, /* Groesse veraendern erlaubt */
			0);

		/* kein Panel-Bereich erwuenscht */
		xv_destroy(xv_get(rueckspiegel_frame, FRAME_CMD_PANEL));

		/* Zeichenfeld initialisieren */
		rueckspiegel_canvas = xv_create(rueckspiegel_frame, CANVAS,
			CANVAS_X_PAINT_WINDOW, TRUE,
			CANVAS_FIXED_IMAGE, FALSE,
			CANVAS_RETAINED, FALSE,
			/* Anhaengen von Struktur mit Daten und Funktion, die beim
			   Neuzeichnen erforderlich sind */
			XV_KEY_DATA, 1, &rueckspiegel_fenster,
			0);

		/* Standard-Fenstergroesse setzen */
		xv_set(rueckspiegel_canvas,
			XV_WIDTH, xv_get(rueckspiegel_canvas, XV_WIDTH) / 3,
			XV_HEIGHT, xv_get(rueckspiegel_canvas, XV_HEIGHT) / 3,
			0);
		window_fit(rueckspiegel_frame);

		/* Initialisierung der oben genannten Struktur */
		rueckspiegel_fenster.daten = &objekte_rueck;
		rueckspiegel_fenster.X_zeichne = X_zeichne_blickfeld;
		rueckspiegel_fenster.X_init = X_fenster_init;
		rueckspiegel_fenster.X_freigeben = X_fenster_freigeben;

		/* initialisiert Daten fuer X11-Grafik in rueckspiegel_fenster */
		fenster_init(rueckspiegel_canvas);

		/* Funktionen fuer Neuzeichen setzen */
		xv_set(rueckspiegel_canvas,
			CANVAS_REPAINT_PROC, repaint_fenster,
			CANVAS_RESIZE_PROC, resize_fenster,
			0);
	}


	{
		Canvas kompass_canvas;

		/* initialisiert das Kompass-Fenster als Unterfenster */
		kompass_frame = xv_create(hauptframe, FRAME_CMD,
			XV_X, xv_get(karte_frame, XV_WIDTH) + 12 +
				xv_get(rueckspiegel_frame, XV_WIDTH) + 12,
			XV_Y, 0,
			FRAME_LABEL, "iMaze: Compass",
			FRAME_CMD_PUSHPIN_IN, TRUE,
			FRAME_SHOW_RESIZE_CORNER, TRUE, /* Groesse veraendern erlaubt */
			0);

		/* kein Panel-Bereich erwuenscht */
		xv_destroy(xv_get(kompass_frame, FRAME_CMD_PANEL));

		/* Zeichenfeld initialisieren */
		kompass_canvas = xv_create(kompass_frame, CANVAS,
			CANVAS_X_PAINT_WINDOW, TRUE,
			CANVAS_FIXED_IMAGE, FALSE,
			CANVAS_RETAINED, FALSE,
			/* Anhaengen von Struktur mit Daten und Funktion, die beim
			   Neuzeichnen erforderlich sind */
			XV_KEY_DATA, 1, &kompass_fenster,
			0);

		/* Standard-Fenstergroesse setzen */
		xv_set(kompass_canvas,
			XV_WIDTH, xv_get(kompass_canvas, XV_HEIGHT) / 4,
			XV_HEIGHT, xv_get(kompass_canvas, XV_HEIGHT) / 4,
			0);
		window_fit(kompass_frame);

		/* Initialisierung der oben genannten Struktur */
		kompass_fenster.daten = &blickrichtung;
		kompass_fenster.X_zeichne = X_zeichne_kompass;
		kompass_fenster.X_init = X_fenster_init;
		kompass_fenster.X_freigeben = X_fenster_freigeben;

		/* initialisiert Daten fuer X11-Grafik in kompass_fenster */
		fenster_init(kompass_canvas);

		/* Funktionen fuer Neuzeichen setzen */
		xv_set(kompass_canvas,
			CANVAS_REPAINT_PROC, repaint_fenster,
			CANVAS_RESIZE_PROC, resize_fenster,
			0);
	}


	/* erzeugt die Panel-Leiste */
	hauptpanel = xv_create(hauptframe, PANEL,
		PANEL_LAYOUT, PANEL_HORIZONTAL, /* Ausrichtung horizontal */
		0);

	/* erzeugt das "Open"-Menue */
	xv_create(hauptpanel, PANEL_BUTTON,
		PANEL_LABEL_STRING, "Open",
		PANEL_ITEM_MENU, xv_create(XV_NULL, MENU,
			MENU_ITEM,
				MENU_STRING, "Map",
				MENU_ACTION_PROC, fenster_knopf,
				/* zu oeffnender Frame */
				XV_KEY_DATA, 0, karte_frame,
				/* Anhaengen von Struktur mit Daten und Funktion, die beim
				   Neuzeichnen erforderlich sind */
				XV_KEY_DATA, 1, &karte_fenster,
				0,
			MENU_ITEM,
				MENU_STRING, "Look Back",
				MENU_ACTION_PROC, fenster_knopf,
				/* zu oeffnender Frame */
				XV_KEY_DATA, 0, rueckspiegel_frame,
				/* Anhaengen von Struktur mit Daten und Funktion, die beim
				   Neuzeichnen erforderlich sind */
				XV_KEY_DATA, 1, &rueckspiegel_fenster,
				0,
			MENU_ITEM,
				MENU_STRING, "Compass",
				MENU_ACTION_PROC, fenster_knopf,
				/* zu oeffnender Frame */
				XV_KEY_DATA, 0, kompass_frame,
				/* Anhaengen von Struktur mit Daten und Funktion, die beim
				   Neuzeichnen erforderlich sind */
				XV_KEY_DATA, 1, &kompass_fenster,
				0,
			0),
		0);

	{
		Frame properties_frame;            /* Properties-Fenster */
		Panel properties_panel;            /* Properties-Panel */
		Panel_button_item category_button; /* Kategorie-Button */
		int i;

		/* Properties-Fenster mit Panel erzeugen */
		properties_frame = xv_create(karte_frame, FRAME_PROPS,
			XV_X, xv_get(karte_frame, XV_WIDTH) + 12,
			XV_Y, xv_get(rueckspiegel_frame, XV_HEIGHT) + 32,
			FRAME_LABEL, "iMaze: Properties",
			0);

		/* Properties-Button erzeugen */
		xv_create(hauptpanel, PANEL_BUTTON,
			PANEL_LABEL_STRING, "Properties",
			/* auszufuehrende Funktion: */
			PANEL_NOTIFY_PROC, fenster_oeffnen_knopf,
			XV_KEY_DATA, 0, properties_frame, /* zu oeffnender Frame */
			0);

		/* Properties-Panel von Frame abfragen */
		properties_panel = xv_get(properties_frame, FRAME_PROPS_PANEL);

		/* Kategorie-Button erzeugen */
		category_button = xv_create(properties_panel, PANEL_CHOICE_STACK,
			PANEL_LABEL_STRING, "Category:",
			PANEL_CHOICE_STRINGS,
				"Game",
				"Map",
				NULL,
			PANEL_NOTIFY_PROC, properties_kategorie_knopf,
			0);

		/* Panelhoehe anpassen */
		window_fit_height(properties_panel);

		{
			Panel spiel_properties;

			/* Kategorie Game erzeugen */
			spiel_properties = xv_create(properties_frame, PANEL,
				WIN_BORDER, TRUE,
				PANEL_LAYOUT, PANEL_VERTICAL,
				0);
			xv_set(category_button,
				XV_KEY_DATA, 0, spiel_properties,
				0);

			/* Checkbox erzeugen */
			hintergrund_aktiv_checkbox = xv_create(spiel_properties,
					PANEL_CHECK_BOX,
				PANEL_LABEL_STRING, "  Keep game alive: ",
				PANEL_VALUE, hintergrund_aktiv,
				0);

			/* Slider erzeugen */
			hintergrund_timeout_slider = xv_create(spiel_properties,
					PANEL_SLIDER,
				PANEL_LABEL_STRING, "  Retransmit time (in ms): ",
				PANEL_INACTIVE, hintergrund_aktiv ? FALSE : TRUE,
				PANEL_VALUE, hintergrund_timeout,
				PANEL_MIN_VALUE, 20,
				PANEL_MAX_VALUE, 1000,
				PANEL_SLIDER_WIDTH, 100 - 2,
				0);

			/* Slider-aktivieren-Knopf einbinden */
			xv_set(hintergrund_aktiv_checkbox,
				PANEL_NOTIFY_PROC, slider_aktivieren_knopf,
				XV_KEY_DATA, 0, hintergrund_timeout_slider,
				0);
		}

		{
			Panel karte_properties;

			/* Kategorie Map erzeugen */
			karte_properties = xv_create(properties_frame, PANEL,
				WIN_BORDER, TRUE,
				PANEL_LAYOUT, PANEL_VERTICAL,
				0);
			xv_set(category_button,
				XV_KEY_DATA, 1, karte_properties,
				0);

			/* Checkbox erzeugen */
			karte_scrollbar_checkbox = xv_create(karte_properties,
					PANEL_CHECK_BOX,
				PANEL_LABEL_STRING, "  Map scrolling: ",
				PANEL_CHOICE_STRING, 0, "with scrollbars",
				PANEL_VALUE, karte_hat_scrollbars,
				0);

			/* Checkbox erzeugen */
			karte_zentriert_checkbox = xv_create(karte_properties,
					PANEL_CHECK_BOX,
				PANEL_NEXT_ROW, 0,
				PANEL_LABEL_STRING, " ",
				PANEL_CHOICE_STRING, 0, "player is centered",
				PANEL_VALUE, karte_ist_zentriert,
				0);

			/* Slider erzeugen */
			zentrier_schritt_slider = xv_create(karte_properties,
					PANEL_SLIDER,
				PANEL_LABEL_STRING, "  Step size (% of window size): ",
				PANEL_INACTIVE, karte_ist_zentriert ? FALSE : TRUE,
				PANEL_VALUE, zentrier_schritt,
				PANEL_MAX_VALUE, 50,
				0);

			/* Slider-aktivieren-Knopf einbinden */
			xv_set(karte_zentriert_checkbox,
				PANEL_NOTIFY_PROC, slider_aktivieren_knopf,
				XV_KEY_DATA, 0, zentrier_schritt_slider,
				0);
		}

		/* alle Teilpanels durchlaufen */
		for (i = 0;; i++)
		{
			Panel subpanel;                 /* Teilpanel fuer eine
			                                   Kategorie */
			Panel_item item;                /* ein Item des Teilpanels */
			Panel_button_item apply_button; /* Objekt Apply-Knopf */
			Panel_button_item reset_button; /* Objekt Reset-Knopf */
			int max_label_breite;           /* Breite der laengsten
			                                   Beschriftung */

			/* kein weiterer Teilpanel? */
			if (!(subpanel = xv_get(category_button, XV_KEY_DATA, i)))
				break;

			/* Teilpanel direkt unter dem Properties-Panel angeordnet */
			xv_set(subpanel,
				XV_X, 0,
				XV_Y, xv_get(properties_panel, XV_HEIGHT),
				XV_SHOW, i == 0 ? TRUE : FALSE,
				PANEL_LAYOUT, PANEL_VERTICAL,
				0);

			/* Apply-Knopf erzeugen */
			apply_button = xv_create(subpanel, PANEL_BUTTON,
				PANEL_NEXT_ROW, 3 * xv_get(subpanel, PANEL_ITEM_X_GAP),
				PANEL_LABEL_STRING, "Apply",
				PANEL_NOTIFY_PROC, properties_apply_knopf,
				XV_KEY_DATA, 0, i, /* Index des Teilpanels */
				0);

			/* Reset-Knopf neben dem Apply-Knopf erzeugen */
			xv_set(subpanel,
				PANEL_DEFAULT_ITEM, apply_button,
				PANEL_LAYOUT, PANEL_HORIZONTAL,
				0);
			reset_button = xv_create(subpanel, PANEL_BUTTON,
				PANEL_LABEL_STRING, "Reset",
				PANEL_NOTIFY_PROC, properties_reset_knopf,
				XV_KEY_DATA, 0, i, /* Index des Teilpanels */
				0);

			/* Breite der laengsten Beschriftung feststellen */
			max_label_breite = 0;
			PANEL_EACH_ITEM(subpanel, item)
				if (item != apply_button && item != reset_button &&
					xv_get(item, PANEL_LABEL_WIDTH) > max_label_breite)
					max_label_breite = xv_get(item, PANEL_LABEL_WIDTH);
			PANEL_END_EACH

			/* Beschriftungen ausser fuer Apply-/Reset-Button
			   rechtsbuendig anordnen */
			PANEL_EACH_ITEM(subpanel, item)
				if (item != apply_button && item != reset_button)
					xv_set(item,
						XV_X, max_label_breite -
							xv_get(item, PANEL_LABEL_WIDTH),
						0);
			PANEL_END_EACH

			/* Groesse des Teilpanels anpassen */
			window_fit(subpanel);

			/* Position des Apply-/Reset-Buttons anpassen */
			xv_set(apply_button,
				XV_X, (xv_get(subpanel, XV_WIDTH) -
					xv_get(apply_button, PANEL_LABEL_WIDTH) -
					xv_get(reset_button, PANEL_LABEL_WIDTH)) / 3,
				0);
			xv_set(reset_button,
				XV_X, (xv_get(subpanel, XV_WIDTH) -
					xv_get(apply_button, PANEL_LABEL_WIDTH) -
					xv_get(reset_button, PANEL_LABEL_WIDTH)) * 2 / 3 +
					xv_get(apply_button, PANEL_LABEL_WIDTH),
				0);

			/* Groesse des gesamten Fensters fuer properties_kategorie_knopf
			   speichern */
			xv_set(subpanel,
				XV_KEY_DATA, 0, xv_get(subpanel, XV_WIDTH),
				XV_KEY_DATA, 1, xv_get(properties_panel, XV_HEIGHT) +
					xv_get(subpanel, XV_HEIGHT),
				0);
		}

		{
			Panel subpanel;

			/* Groesse des Property-Panels anpassen */
			window_fit(properties_panel);

			/* das Property-Panel reicht immer bis zum
			   rechten Fensterrand */
			xv_set(properties_panel,
				XV_WIDTH, WIN_EXTEND_TO_EDGE,
				0);

			/* aktuell gewaehltes Teilpanel feststellen */
			subpanel = xv_get(category_button, XV_KEY_DATA,
				xv_get(category_button, PANEL_VALUE));

			/* Teilpanel auf sichtbar setzen */
			xv_set(subpanel,
				XV_SHOW, TRUE,
				0);

			/* Groesse des Property-Fensters anpassen */
			xv_set(properties_frame,
				XV_WIDTH, xv_get(subpanel, XV_KEY_DATA, 0),
				XV_HEIGHT, xv_get(subpanel, XV_KEY_DATA, 1),
				0);
		}
	}

	/* Disconnect-Knopf erzeugen */
	xv_create(hauptpanel, PANEL_BUTTON,
		PANEL_LABEL_STRING, "Disconnect",
		/* auszufuehrende Funktion: */
		PANEL_NOTIFY_PROC, disconnect_knopf,
		0);

	/* Fenstergroesse anpassen */
	window_fit_height(hauptpanel);

	{
		Canvas blickfeld_canvas;

		/* Zeichenfeld fuer Hauptfenster initialisieren */
		blickfeld_canvas = xv_create(hauptframe, CANVAS,
			WIN_BELOW, hauptpanel, /* unter Hauptpanel anordnen */
			XV_X, 0,               /* x-Koordinate auf 0 */
			CANVAS_X_PAINT_WINDOW, TRUE,
			CANVAS_FIXED_IMAGE, FALSE,
			CANVAS_RETAINED, FALSE,
			/* Anhaengen von Struktur mit Daten und Funktion, die beim
			   Neuzeichnen erforderlich sind */
			XV_KEY_DATA, 1, &blickfeld_fenster,
			0);

		/* Initialisierung dieser Struktur */
		blickfeld_fenster.daten = &objekte;
		blickfeld_fenster.X_zeichne = X_zeichne_blickfeld;
		blickfeld_fenster.X_init = X_fenster_init;
		blickfeld_fenster.X_freigeben = X_fenster_freigeben;

		/* initialisiert Daten fuer X11-Grafik in blickfeld_fenster */
		fenster_init(blickfeld_canvas);

		xv_set(blickfeld_canvas,
			CANVAS_REPAINT_PROC, repaint_fenster,
			CANVAS_RESIZE_PROC, resize_fenster,
			0);
	}

	/* audio-visuelle Untermalung von Ereignissen initialisieren */
	xv_ereignisse_init(hauptframe, hauptpanel);

	/* Initialisierung der Eingabe-Abfrage */
	xv_eingabe_init(hauptframe, hauptpanel);

	/* bei der Initialisierung ist kein Fehler aufgetreten */
	return 0;
}


/*
** grafik_ende
**  gibt Datenstrukturen frei
*/
void grafik_ende()
{
	/* Eingabe-Routine beenden */
	xv_eingabe_ende();

	/* audio-visuelle Untermalung von Ereignissen beenden */
	xv_ereignisse_ende();

	/* falls es noch einen Haupframe gibt */
	if (hauptframe_gueltig)
		ende_ausloesen();
}


/*
** disconnect_abfrage
**  macht eine Sicherheitsabfrage und beendet bei Zustimmung die Verbindung
*/
void disconnect_abfrage()
{
	static char *meldung[] = { "Do you really want to disconnect?", NULL };

	if (!rueckfrage(meldung, "Sure", "Play on"))
		return;

	/* zur Zeit noch Programmende, spaeter nur disconnect */
	ende_ausloesen();
}


/*
** ueblen_fehler_anzeigen
**  oeffnet eine Hinweisbox mit einer Fehlermeldung und wartet auf Druecken
**  eines Knopfes
**
** Parameter:
**  meldung: Feld von Strings, die die Zeilen des auszugebenden Textes
**           beinhalten
**  knopf: Text der auf dem Knopf stehen soll;
**         falls NULL, so wird ein Standardtext verwendet
*/
void ueblen_fehler_anzeigen(meldung, knopf)
char **meldung, *knopf;
{
	/* xview schon aktiv? */
	if (hauptframe_gueltig)
	{
		int deskriptor; /* Spiel-Deskriptor vom Netzwerkteil */

		/* mit Server synchronisieren */
		xv_set(hauptframe,
			SERVER_SYNC_AND_PROCESS_EVENTS,
			0);

		/* falls es einen Spiel-Deskriptor gibt, Handler loeschen */
		if (verbindung_aktiv && (deskriptor = spiel_deskriptor()) >= 0)
			notify_set_input_func(hauptframe, NULL, deskriptor);
	}

	/* bei ueblem Fehler kann sowieso nicht mehr kommuniziert werden */
	verbindung_aktiv = 0;

	/* Aufruf eines Notice;
	   falls Texte fuer Knopf=NULL, Standardaufschrift einsetzen */
	if (notice_oeffnen(attr_create_list(
		NOTICE_MESSAGE_STRINGS_ARRAY_PTR, meldung,
		NOTICE_BUTTON_YES, knopf == NULL ? "Bad Luck" : knopf,
		0)) == NOTICE_FAILED)
	{
		int i; /* Zaehlvariable fuer Ausgeben der Meldung */

		/* falls Oeffnen des Notice-Fensters fehlgeschlagen,
		   Ausgabe der Meldung auf standard-error */
		for (i = 0; meldung[i] != NULL; i++)
			fprintf(stderr, "iMaze: %s\n", meldung[i]);
	}
}


/*
** rueckfrage
**  oeffnet eine Abfragebox mit einer Rueckfrage und zwei Knoepfen zur
**  Auswahl
**
** Parameter:
**  meldung: Feld von Strings, die die Zeilen des auszugebenden Textes
**           beinhalten
**  weiter_knopf: Text der auf dem positiv-Knopf stehen soll (default);
**                falls NULL, so wird ein Standardtext verwendet
**  abbruch_knopf: Text der auf dem negativ-Knopf stehen soll;
**                 falls NULL, so wird ein Standardtext verwendet
**
**  Rueckgabewert:
**   true, falls positiv-Knopf gewaehlt
**   false, falls negativ-Knopf gewaehlt
*/
int rueckfrage(meldung, weiter_knopf, abbruch_knopf)
char **meldung, *weiter_knopf, *abbruch_knopf;
{
	int wahl; /* Nummer des gewaehlten Knopfes */

	/* Erzeugen eines Notice-Fensters;
	   falls Texte fuer Knoepfe = NULL, Standardaufschrift einsetzen */
	wahl = notice_oeffnen(attr_create_list(
		NOTICE_MESSAGE_STRINGS_ARRAY_PTR, meldung,
		NOTICE_BUTTON_YES, weiter_knopf == NULL ?
			"Continue" : weiter_knopf,
		NOTICE_BUTTON_NO, abbruch_knopf == NULL ?
			"Quit" : abbruch_knopf,
		0));

	/* Oeffnen des Notice-Fensters fehlgeschlagen? */
	if (wahl == NOTICE_FAILED)
	{
		static char *meldung[] = { "iMaze - Fatal Error", "",
			"Couldn't open notice window.", NULL };

		uebler_fehler(meldung, NULL);
	}

	/* Rueckgabe: positiv-Knopf = true; negativ-Knopf = false */
	return wahl == NOTICE_YES;
}
