//  ximage.c: functions to handle XImages, save & load images, palettes
//  a part of PMS-grabber package
// the "GNU Public Lincense" policy applies to all programs of this package
// (c) Wolfgang Koehler, wolf@first.gmd.de, Dec. 1994
//     Kiefernring 15
//     14478 Potsdam, Germany
//     0331-863238

#include "window.h"
#include <X11/Xutil.h>
#include "evaluation.h"
#include "view.h"
#include "ximage.h"
#include "files.h"
#include <errno.h>

#ifdef SPARC

// define swapping big-endian -> little-endian
#define S_SWAB(x) (((x & 0xFF) << 8) | ((x & 0xFF00) >> 8))
#define L_SWAB(x) (((x & 0xFF) << 24) | ((x & 0xFF00) << 8) | \
		   ((x & 0xFF0000) >> 8) | ((x & 0xFF000000) >> 24))

#define L_XCH(x) x = L_SWAB((long) x)  // in place swapping
#define S_XCH(x) x = S_SWAB((short) x)  // in place swapping

// getw and putw are not defined on Suns
int getw(FILE* f) { 
  int w; fread(&w,4,1,f); 
  return (L_SWAB(w));
}

int putw(int w, FILE *f) { int wx = L_SWAB(w); fwrite(&wx,4,1,f); }

void swap_xcolor(XColor &xc) { // swaps the members of XColor struct
  L_XCH(xc.pixel); 
  S_XCH(xc.red); S_XCH(xc.green); S_XCH(xc.blue);
}

#else

#define L_XCH(x) // do nothing
#define S_XCH(x)
void swap_xcolor(XColor &xc) {};

#endif // SPARC 

// load XColor structure from file, if on SPARC swap the bytes
XColor fread_color(FILE *f) {
  XColor xc;
  fread(&xc, sizeof(XColor), 1, f);
  swap_xcolor(xc);
  //  printf(" %x %x %x %x\n",xc.pixel,xc.red,xc.green,xc.blue);
  return xc;
}

void fwrite_color(XColor xc, FILE *f) {
  swap_xcolor(xc);
  fwrite(&xc, sizeof(XColor), 1, f);
}

Bool True_Color; // if T : no coltab mapping !
int image_depth;
static Visual *visual;

void get_visual() {
  visual = DefaultVisual(display,screen);
  int cl = visual->c_class;
  True_Color = (cl == TrueColor);
  printf("Visual Class = %d\n",cl);
  image_depth = DefaultDepth(display,screen);
  printf("depth = %d\n",image_depth);
}

// save captured picture in palettized format (1 byte per pixel = [0..ncols-1])
// together with the pixtab-entries of the XColor palette into the newly 
// (quite similiar to xwd/xwud)

void save_image(XImage *ximage, FILE *fsave,int ncols,unsigned long *pixtab) { 
  if (fsave == NULL) return; // possibly not confirmed
  putw(ncols, fsave);  // = used palette size
  int i;
  for (i=0; i < ncols; i++) { 
    XColor xc = { pixtab[i] }; 
    XQueryColor(display, def_cmap, &xc); // get rgb-values for pixtab-value
    fwrite_color(xc, fsave);
  }
  int ww = ximage->width, hh = ximage->height;
  putw(ww,fsave); putw(hh,fsave);
  fwrite(ximage->data, ww * hh, 1, fsave); // the direct image
  fclose(fsave);
 
}

// loads image to newly created XImage structure, transforms to color cells
// pointed to from pixtab vector
// set color cells ; returns also "ncols" = number of used color cells
void load_image(FILE *fload, int &ncols, unsigned long *pixtab) {
  ncols = getw(fload); 
  int i;
  if (True_Color) {  
    unsigned short col_trans[256]; 
    for (i=0; i < ncols; i++) { 
      XColor xc = fread_color(fload);
      unsigned short pix = xc.pixel;
      if (pix > 256) error("wrong pixel");
      XAllocColor(display, def_cmap, &xc);
      col_trans[pix] = xc.pixel; // the returned value !
    }
    int ww = getw(fload), hh = getw(fload), nn = ww*hh;
    unsigned char bbuf[nn];
    fread(bbuf, nn, 1, fload); 

    create_buffers(ww,hh);
    // transform pixel values to actual pixtab-values
    for (i = 0; i < nn; i++) cap_buf[i] = col_trans[bbuf[i]];

  } else {
    if (ncols > max_cols) error("not enough color cells");
    // trafo between color index from file -> pixtab entries
    unsigned long col_trans[256]; 
    for (i=0; i< 256; i++) col_trans[i] = 0;
    for (i=0; i < ncols; i++) { 
      XColor xc = fread_color(fload);
      if (xc.pixel > 256) error("wrong pixel");
      col_trans[xc.pixel] = pixtab[i]; xc.pixel = pixtab[i];
      XStoreColor(display, def_cmap, &xc);
    
    }
    int ww = getw(fload), hh = getw(fload), nn = ww*hh;
    create_buffers(ww,hh); 
    unsigned char *ibuf =  (unsigned char *) ximage->data; // unsigned !!
    fread(ibuf, nn, 1, fload); 
    // transform pixel values to actual pixtab-values
    for (i = 0; i < nn; i++) ibuf[i] = col_trans[ibuf[i]];
  }
}

// function called from image-view only
void load_pix_file(FILE *fload) { 
  if (fload == NULL) return; 

  load_image(fload,ncols,pixtab);
  printf("loaded file with %d colors\n",ncols);

  // int ww = ximage->width, hh = ximage->height;
  // this simple resizing conflicts with create_buffer
  // it should be disentangled for better usage
  // XResizeWindow(display,mw->Win,ww,hh+41); 
  grab_wi->redraw();
}

void rgb555_to_565(unsigned short &z) {
  // shift bits 14..5 for 1 bit left, let bits 4..0 unchanged
  z = (z & 0x1f) | ((z & 0x7fe0) << 1) ;
}

int rgb_mode(char *fina) { // extract rgb mode from name, eg "555" or "565"
  char *p5 = strrchr(fina,'5')-2; // last !! char 5 in fina -2
  // printf("%s\n",p5);
  if (strncmp(p5,"555",3) == 0) return 555; 
  if (strncmp(p5,"565",3) == 0) return 565;
  return 0;
}

void load_capture_file(FILE *fload, char *fina) { 
  if (fload == NULL) return;
  int ww  = getw(fload), hh = getw(fload), nn = ww*hh;
  if ((ww != biWidth) || (hh != biHeight)) adjust_size(ww,hh);  
  fread(cap_buf, 2, nn, fload);
  Bool g6_to_5 = (True_Color && (rgb_mode(fina) == 555));
  if (g6_to_5) for (int i=0; i< nn; i++) rgb555_to_565(cap_buf[i]);
  map_and_redraw();
  printf("loaded capture from %s size =  %d x %d\n",
	  fina,biWidth,biHeight);
}


void write_ppm_file_15(unsigned short *data,int width,int height, FILE *f){
  int total = width * height;
  fprintf(f,"P6\n%d %d 255\n",width,height);
  while (total--) {
    putc((*data >> 10) << 3,f);
    putc((*data >> 5) << 3,f);
    putc((*data & 31) << 3,f);
    data++;
  }
}

void write_ppm_file_16(unsigned short *data,int width,int height,FILE *f) {
  int total = width * height;
  fprintf(f,"P6\n%d %d 255\n",width,height);
  while (total--) {
    putc((*data >> 11) << 3,f);
    putc((*data >> 5) << 2,f);
    putc((*data & 31) << 3,f);
    data++;
  }
}

// ****** simple callback functions for pulldown menus *******

void save_pix() {   
  char *defname = "default.pix", fina[80];
  FILE *fsave = open_file_interactive(defname,fina,MODE_WRITE);
  if (fsave == NULL) return;
  save_image(ximage,fsave,ncols,pixtab); 
  int ww = ximage->width, hh = ximage->height;
  printf("saved image to %s with %d colors, size = (%d x %d) %d byte\n",
       fina,ncols,ww,hh,ww*hh);
}

void save_ppm() {   
  char *defname = "default.ppm", fina[80];
  FILE *fsave = open_file_interactive(defname,fina,MODE_WRITE);
  if (fsave == NULL) return;
  int ww = ximage->width, hh = ximage->height;
  if (True_Color) write_ppm_file_16(cap_buf,ww,hh,fsave); 
  else write_ppm_file_15(cap_buf,ww,hh,fsave); 
  printf("saved image in ppm format to %s with size = (%d x %d) %d byte\n",
	 fina,ww,hh,3*ww*hh);
  fclose(fsave);
}

void load_pix_interactive() {  
  char *defname = "*.pix", fina[80]; 
  FILE *fload = open_file_interactive(defname,fina,MODE_READ);
  if (fload == NULL) return;

  load_image(fload,ncols,pixtab);
  fclose(fload);

  int ww = ximage->width, hh = ximage->height;
  printf("loaded image from %s with %d colors, size = (%d x %d) %d byte\n",
	 fina,ncols,ww,hh,ww*hh);

  grab_wi->redraw();
}

// save captured image in direct color rgb555 format
void save_capture() {
  char *defname, fina[80];
  defname = (True_Color) ? "default.cap565" : "default.cap555";
  FILE *fsave = open_file_interactive(defname,fina,MODE_WRITE);
  if (fsave == NULL) return;
  putw(biWidth,fsave); putw(biHeight,fsave);
  fwrite(cap_buf, 2, biWidth*biHeight, fsave);
  fclose(fsave);  
  printf("saved capture to %s size = %d byte = %d x %d\n",
	 fina,2*biWidth*biHeight,biWidth,biHeight);
}

// load captured image in direct color rgb555 format
void load_capture_interactive() {
  char *defname = (True_Color) ? "*.cap565" : "*.cap555", fina[80]; 
  FILE *fload = open_file_interactive(defname,fina,MODE_READ);
  if (fload == NULL) return; 
  load_capture_file(fload, fina);  
  fclose(fload);  
}

// save coltab55
void save_palette() {  
  char *defname = "default.pal565", fina[80];
  FILE *fsave = open_file_interactive(defname,fina,MODE_WRITE);
  if (fsave == NULL) return;

  putw(ncols, fsave); int i;  
  for (i=0; i < ncols; i++) { 
    XColor xc = { pixtab[i] }; 
    XQueryColor(display, def_cmap, &xc); // get rgb-values for pixtab-value
    fwrite_color(xc, fsave);
  }
  fwrite(coltab555, 32*32*32, 1, fsave);
  fclose(fsave);   
  printf("palette with %d colors saved to file %s\n",ncols,fina);
}

// load saved palette, the only problem is that the pixel values of the saved
// palette are not nessecarily free on the display they are loaded to
// fina only for display needed
void load_palette_file(FILE *fload) {
  if (fload == NULL) return;
  ncols = getw(fload); 
  if (ncols > max_cols) error("not enough color cells");
  unsigned long col_trans[256]; 
  int i;
  // compute translation of old -> new pixel values
  for (i=0; i< 256; i++) col_trans[i] = 0;
  for (i=0; i < ncols; i++) { 
    XColor xc = fread_color(fload);
    if (xc.pixel > 256) error("wrong pixel");
    col_trans[xc.pixel] = pixtab[i]; xc.pixel = pixtab[i];
    XStoreColor(display, def_cmap, &xc);    
  }
  fread (coltab555, 32*32*32, 1, fload);
  fclose(fload); 
  for (i=0; i < 32*32*32; i++) coltab555[i] = col_trans[coltab555[i]];
}

// interface function for menu
void load_palette_interactive() {
  char *defname = "*.pal555", fina[80]; 
  FILE *fload = open_file_interactive(defname,fina,MODE_READ);
  load_palette_file(fload);
  printf("palette with %d colors loaded from file %s\n",ncols,fina);
  map_and_redraw();
}

#ifndef SEEK_CUR
#define SEEK_CUR 1
#endif

// mini_window : display a compressed picture
class mini_window : public window {
  mapped_image *visimg;
  char *fina;
public:
  mini_window(window &parent, int w, int h, int x, int y, char *fina) :
    window(parent,w,h,x,y,0), fina(fina) {
      visimg = new mapped_image(w,h,True_Color,False); // don't use SHM
    }
  ~mini_window() { delete visimg; }

  // read fload into picture buffer, compressing, ww = width of stored image
  read_file(FILE *fload) {
    int ww  = getw(fload), hh = getw(fload);
    int xoff = ww/width, yoff = hh/height; // offsets
    unsigned short *bptr = visimg->buf_16; // to copy to 
    int x,y;
    for (y = 0; y < height; y++) {
      unsigned short pline[ww], *plptr = pline;
      fread(pline, 2, ww, fload);
      for (x = 0; x < width; x++) { *bptr++ = *plptr; plptr += xoff; }  
      fseek(fload, 2*ww*(yoff-1), SEEK_CUR); // skip yoff-1 lines  
    }
    if (rgb_mode(fina) == 555) 
      for (int i=0; i< width*height; i++) rgb555_to_565(visimg->buf_16[i]);
    visimg->map_buffer(); // member function
  }
  virtual void redraw() {
    visimg->PutImage(Win,0,0,width,height);
  }
};
  
// combine mini_window and name_window in one, 
// implement BPRess_action : load the capture-file 
class mini_with_name : public window {
public:
  char *fina;
  char *dirname; // directory path to capture file
  mini_window *mini;
  mini_with_name(window &parent, int w, int h, int x, int y, 
		 char *name, char *dirname) :
    window(parent,w,h,x,y,1), dirname(dirname) {
    fina = new char[strlen(name)+1];
    strcpy(fina,name); // must be stored here
    mini = new mini_window(*this,width,height-15,0,0,fina);
    new text_win(*this,fina,width,15,0,height-15);
    selection_mask |= ButtonPressMask;
  }
  ~mini_with_name()  { delete fina; }

  virtual void BPress_1_CB(XButtonEvent) { 
    char path[200]; 
    strcpy(path,dirname); strcat(path,"/"); strcat(path,fina);
    // printf("%s\n",path);
    FILE *fload = fopen(path,"r");
    if (fload == NULL) return;
    load_capture_file(fload, path);  
    fclose(fload); 
  }
};

// a main_window with up to 50 mini_windows to select one 
class mini_selector : public main_window {
  mini_with_name *minis[50]; // contains max 50 mini_windows
  int wx,wyy,wy,nx; // wyy = height + bottom text, wy = eff. height
public:
  char *dirname;
  mini_selector(char* dirname, int nx, int ny, int wx, int wyy) : 
    wx(wx), wyy(wyy), wy(wyy-15), nx(nx), dirname(dirname),
    main_window(dirname, wx*nx, wyy*ny + 20) { 
    int i; for (i=0; i < 50; i++) minis[i] = NULL; // initialize with 0
    new unmap_button(*this,"close",80,20,(width-80)/2,height-20);
  }

  load_minis(int ncf, char *fitems[]) {
    int i, ix = 0, iy = 0;
    for (i = 0; i < ncf; i++) {
      char *fname = fitems[i];
      printf("%s\n",fname);
      FILE *fload = fopen(fname,"r");        
      if (minis[i] == 0)  { // create windows only once !
	// set width & height + 1, so borders can overlap
	minis[i] = new mini_with_name(*this, wx+1, wyy+1, ix*wx, iy*wyy, 
				      fname, dirname);
	minis[i]->mini->read_file(fload);
	ix++; if (ix == nx) { ix = 0; iy++; }
	fclose(fload);
      }
    }
  }
};

#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>

// compare function for qsort, special for filenames like "12-3.cap555"
//  compares strings pointed to, not the pointers
int comp_capname(char **e1, char **e2) {
  unsigned u1,p1,u2,p2; // ux = first part, px = second part of number
  sscanf(*e1,"%u-%u",&u1,&p1); sscanf(*e2,"%u-%u",&u2,&p2);
  if (u1 == u2) return (p1 - p2); else return (u1 - u2);
}


typedef int (*IVPVP)(const void *,const void *); // the correct type for qsort

// show all files *.cap555 in dir in compressed format to select one
// dirname must be a subdirectory (or later an absolute path)
void show_capture_dir(char *dirname) {
  static mini_selector *selwin[4] = { 0,0,0,0 }; // max 4 per application !
  int is; // look if dirname is already loaded as mini_selector
  for (is = 0; is < 4; is++) { 
    if (selwin[is] == 0) break;
    if (strcmp(selwin[is]->dirname, dirname) == 0) { // this is i
      selwin[is]->RealizeChildren(); return;
    }
  } 
  if (is >= 4) error("too many mini_selectors");
  DIR *dirp = opendir(dirname); 
  if (dirp == NULL) { printf("error reading directory \n"); return; }
  
  chdir(dirname);
  if (! True_Color) {
    FILE *fpal = fopen("def.pal555","r");
    load_palette_file(fpal); // if present use the default palette
  }
  // count all capture files, sort them alphabetically
  int ncf = 0; // number of capture files in dirp
  struct dirent *dp;  
  char strings[10000]; // all filenames in one vector
  char *fitems[200]; // max 100 entries in this vector
  int nstrp = 0;
  for (dp = readdir(dirp); dp != NULL; dp = readdir(dirp)) {
    struct stat st; 
    int ret = stat(dp->d_name, &st); 
    char *ppos = strchr(dp->d_name,'.');     // find "." in the filename
    if ((ret == 0) && (st.st_mode & S_IFREG) // regular file
	&& (strcmp(ppos,".cap555") == 0)) { // match string  
      int nl = strlen(dp->d_name) + 1;
      char *strent = strings + nstrp; // entry point in strings;
      strcpy(strent,dp->d_name);
      fitems[ncf++] = strent; nstrp += nl; 
    } 
  }
  qsort(fitems, ncf, 4, (IVPVP) comp_capname);

  int nx = 6, ny = (ncf-1)/nx + 1 ; // number of columns and rows of minis 
  int wx = 72, wy = 54, wyy = wy+15; // size of miniwins, wyy + name_win

  selwin[is] = new mini_selector(dirname,nx,ny,wx,wyy);
  selwin[is]->load_minis(ncf,fitems);
  chdir(".."); // go back

  selwin[is]->RealizeChildren();

}

// class mapped_image : an image buffer (16bit), and associated Ximage
// which can be mapped directly, or with map_buffer
// shm = True means : if possible use SHM
mapped_image::mapped_image(int w, int h, Bool direct_map, Bool shm) : 
direct_map(direct_map) {
  nn = w * h;

#ifdef MIT_SHM
  if (direct_map && shm) { // shm can only be used for case direct_map !
    static Status st = XShmQueryExtension(display);
    if (!st) {
      printf("no Shared Memory Extension on this display !\n");
      use_shm = False; 
    } else {
      use_shm = True;
      ximage = XShmCreateImage(display,visual,image_depth,ZPixmap,0,
			       &shminfo,w, h);
      int id = shmget(IPC_PRIVATE,ximage->bytes_per_line*ximage->height,
		      IPC_CREAT|0777);
      if (id < 0) error("shmghet");
      shminfo.shmid = id;
      char *addr = shmat(shminfo.shmid,0,0);
      if (addr == (char *) (-1)) error("shmat failed");
      shminfo.shmaddr = ximage->data = addr;
      shminfo.readOnly = False;
      XShmAttach(display, &shminfo);
      // printf("ximage = %x addr = %x id = %d\n",ximage,addr,id);
      buf_16 = (unsigned short*) ximage->data;
    }
  } else use_shm = False;
#else
  use_shm = False;
#endif
  if (! use_shm) {
    buf_16 = new unsigned short[nn]; 
    char *image_buf;
    if (! direct_map) {
      image_buf = new char[nn];
    if (image_buf == NULL) error("no room for image");
    } else 
      image_buf = (char *) buf_16; // use direct map !
    ximage = XCreateImage(display,visual,image_depth, ZPixmap,0,
			  image_buf, w, h, image_depth,0);
  }
}

mapped_image::~mapped_image() { 
  if (! direct_map) delete buf_16;
  XDestroyImage(ximage);

#ifdef MIT_SHM 
  if (use_shm) {
    XShmDetach(display,&shminfo); 
    XSync(display,False);  
    int ret;
    ret = shmdt(shminfo.shmaddr);
    if (ret < 0) printf("error shmdt %d %d %s ",ret,errno,strerror(errno));

    // check attached nr
    struct shmid_ds shds;
    ret = shmctl(shminfo.shmid, IPC_STAT, &shds);
    if (ret < 0) printf(" %d %d %s ",ret,errno,strerror(errno));
    // else printf("natt = %d", shds.shm_nattch); // should be == 0

    ret = shmctl(shminfo.shmid, IPC_RMID, 0);
    if (ret < 0) printf(" %d %d %s ",ret,errno,strerror(errno));
  }
#endif

}

void mapped_image::map_buffer() {
  if (direct_map) return;
  unsigned char *imb =  (unsigned char *) ximage->data; 
  unsigned short *bptr = buf_16; 
  for (int i=0; i < nn; i++) *imb++ = coltab555[*bptr++];
}

void  mapped_image::PutImage(Drawable Win, int x, int y, int ww, int wh) {
#ifdef MIT_SHM
  if (use_shm) 
    XShmPutImage(display,Win,gc_copy,ximage,0,0,x,y,ww,wh,False);  
  else
#endif
    XPutImage(display,Win,gc_copy,ximage,0,0,x,y,ww,wh);  
}


