/* Linux stuff */
#include <sys/types.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <stdio.h>
#include <linux/soundcard.h>

/* misc defines */
#define MIXERDEVICE     "/dev/mixer"

/* stereo_flag */
#define BOTH			0
#define LEFT 			1
#define RIGHT			2

/* types */
#define RECORD_SOURCE	0	
#define SAVE_PRESET		1
#define LOAD_PRESET		2
#define HELP			3
#define LOUDNESS		4
#define ENHANCE			5
#define MUTE			6
#define LEVEL			7

/* Prototypes */
void Messages(), Quit(), read_mixer_values();
int atoi(char a[]), zindex(char a[], char b[]), isdigit(char a),\
toupper(char a);

/* SOUND_DEVICE_NAMES comes from <linux/soundcard.h> */
char *soundDeviceNames[] = SOUND_DEVICE_NAMES;
char *soundDeviceLabels[] = SOUND_DEVICE_LABELS;
char *progname;
int supported, stereodevs, recmask;
int mixer_fd;
int nr_of_devices;
int error_field;

int mixer_to_disk(char filename[])
{
int rec_src, value;
int i;
char total_filename[256];
FILE *presetfile;
char user_responce;

sprintf(total_filename, "%s.cmix0.3.preset", filename); 
presetfile = fopen(total_filename, "rb");
if(presetfile)
	{
	fclose(presetfile);
	fprintf(stderr,\
	"\nPreset file %s already exists, overwrite (y / n <ENTER>) ? ",\
	total_filename);
	user_responce = getc(stdin);
/*	fscanf(stdin, "%s", user_responce);
*/
	if(toupper(user_responce) != 'Y')
		{
		fprintf(stdout, "\nSave command cancelled\n");
		exit(1);
		}
	}
presetfile = fopen(total_filename, "wb");
if(! presetfile)
	{
	fprintf(stdout, "\nCannot openn %s for write\n", total_filename);
	exit(1);
	}

/* read record source */
if (ioctl(mixer_fd, SOUND_MIXER_READ_RECSRC, &rec_src) == -1)
	{
	fprintf(stderr,"\n%s: Error reading recording source\n", progname);
	if(presetfile)fclose(presetfile);
	exit(1);
	}
else
	{
	fprintf(presetfile, "%d\n", rec_src);
	}

/* read all normal device settings and write to file */
for (i = 0; i < SOUND_MIXER_NRDEVICES; i++)
	{
	if (((1 << i) & supported) != 0)
		{
		if (ioctl(mixer_fd,MIXER_READ(i), &value) == -1)
			{
			fprintf(stderr,"\n%s: Error reading %s in %s\n",
			progname, soundDeviceNames[i], MIXERDEVICE);
			error_field = (error_field & ~(1 << i));
			if (error_field == 0)
				{
				fprintf(stderr,"FATAL: your sound board probably does not have a mixer device\n");
				if(presetfile)fclose(presetfile);
				exit(-1);
				}
			}	
		else
			{
			fprintf(presetfile, "%d\n", value);
 			}/* end device is supported ( ioctl ok)*/
		}/* end supported */
	}/* end i for all normal devices */
/* read special devices and write to disk */	
if (ioctl(mixer_fd,MIXER_READ(28), &value) == -1)
	{
	fprintf(stderr,"\n%s: Error reading %s in %s\n",
	progname, soundDeviceNames[28], MIXERDEVICE);
	if(presetfile)fclose(presetfile);
	exit(1);
	}
else	
	{
	fprintf(presetfile, "%d\n", value);
	}
if (ioctl(mixer_fd,MIXER_READ(29),&value) == -1)
	{
	fprintf(stderr,"\n%s: Error reading %s in %s\n",
	progname, soundDeviceNames[28], MIXERDEVICE);
	if(presetfile)fclose(presetfile);
	exit(1);
	}
else	
	{
	fprintf(presetfile, "%d\n", value);
	}
if (ioctl(mixer_fd,MIXER_READ(30),&value) == -1)
	{
	fprintf(stderr,"\n%s: Error reading %s in %s\n",
	progname, soundDeviceNames[28], MIXERDEVICE);
	if(presetfile)fclose(presetfile);
	exit(1);
	}
else	
	{
	fprintf(presetfile, "%d\n", value);
	}
fclose(presetfile);
return(0);
}/* end functiom mixer_to_disk */

int mixer_from_disk(char filename[])
{
int rec_src, value;
int i;
char total_filename[256];
FILE *presetfile;

sprintf(total_filename, "%s.cmix0.3.preset", filename); 
presetfile = fopen(total_filename, "rb");
if(! presetfile)
	{
	fprintf(stdout, "\nCannot openn %s for read\n", total_filename);
	exit(1);
	}

/* get record source */
fscanf(presetfile, "%d\n", &rec_src);
/* write record source */
if (ioctl(mixer_fd, SOUND_MIXER_WRITE_RECSRC, &rec_src) == -1)
	{
	fprintf(stderr,"\n%s: Error reading writing source\n", progname);
	if(presetfile)fclose(presetfile);
	exit(1);
	}

/* read all normal device settings and write to file */
for (i = 0; i < SOUND_MIXER_NRDEVICES; i++)
	{
	if (((1 << i) & supported) != 0)
		{
		/* get values for device */
		fscanf(presetfile, "%d\n", &value);
		if (ioctl(mixer_fd,MIXER_WRITE(i), &value) == -1)
			{
			fprintf(stderr,"\n%s: Error writing %s in %s\n",
			progname, soundDeviceNames[i], MIXERDEVICE);
			error_field = (error_field & ~(1 << i));
			if (error_field == 0)
				{
				fprintf(stderr,"FATAL: your sound board probably does not have a mixer device\n");
				if(presetfile)fclose(presetfile);
				exit(-1);
				}
 			}/* end device is supported ( ioctl ok)*/
		}/* end supported */
	}/* end i for all normal devices */

/* read special devices and write to disk */	
fscanf(presetfile, "%d\n", &value);
if (ioctl(mixer_fd,MIXER_WRITE(28), &value) == -1)
	{
	fprintf(stderr,"\n%s: Error writing %s in %s\n",
	progname, soundDeviceNames[28], MIXERDEVICE);
	if(presetfile)fclose(presetfile);
	exit(1);
	}
fscanf(presetfile, "%d\n", &value);
if (ioctl(mixer_fd,MIXER_WRITE(29),&value) == -1)
	{
	fprintf(stderr,"\n%s: Error writing %s in %s\n",
	progname, soundDeviceNames[28], MIXERDEVICE);
	if(presetfile)fclose(presetfile);
	exit(1);
	}
fscanf(presetfile, "%d\n", &value);
if (ioctl(mixer_fd,MIXER_WRITE(30),&value) == -1)
	{
	fprintf(stderr,"\n%s: Error writing %s in %s\n",
	progname, soundDeviceNames[28], MIXERDEVICE);
	if(presetfile)fclose(presetfile);
	exit(1);
	}
fclose(presetfile);
return(0);
}


void mixerInit()
/* init mixer and get some useful constants */
{
mixer_fd = open (MIXERDEVICE, O_RDWR, 0);
if (mixer_fd < 0 )
	{
	fprintf (stderr, "\n%s: Error opening mixer device\n", progname);
	fprintf (stderr, "FATAL: %s probably does not exist or has no r/w permissions\n", MIXERDEVICE);
	exit (-1);
	}
	if (ioctl(mixer_fd, SOUND_MIXER_READ_DEVMASK, &supported) == -1)
	supported = ((1 << (SOUND_MIXER_NRDEVICES))-1);

	if (ioctl(mixer_fd, SOUND_MIXER_READ_STEREODEVS, &stereodevs) == -1)
	stereodevs = ((1 << (SOUND_MIXER_NRDEVICES))-1);

	if (ioctl(mixer_fd, SOUND_MIXER_READ_RECMASK, &recmask) == -1)
	stereodevs = ((1 << (SOUND_MIXER_NRDEVICES))-1);
	error_field = supported;
}/* end function mixerInit */

void get_it(int i, int record_source)
{
int value, lvalue, rvalue;
char rec_sym[20];

if( record_source  &  (1 << i)  )
	{
	strcpy(rec_sym,"rec   ");
	}
else strcpy(rec_sym, "      ");
	
printf("\n%s%s\t\t",rec_sym, soundDeviceNames[i]);
if (((1 << i) & supported) != 0)
	{
	if (ioctl(mixer_fd,MIXER_READ(i), &value) == -1)
		{
		fprintf(stderr,"\n%s: Error reading %s in %s\n",
		progname, soundDeviceNames[i], MIXERDEVICE);
		error_field = (error_field & ~(1 << i));
		if (error_field == 0)
			{
			fprintf(stderr,"FATAL: your sound board probably does not have a mixer device\n");
			exit(-1);
			}
		}
	else
		{
		int left_value, right_value;
		left_value = value & 0x00FF;
		right_value = (value & 0xFF00) >> 8;
		printf("left=%3d\tright=%3d", left_value, right_value);
		lvalue = value & 0x00FF;
		rvalue = (value & 0xFF00) >> 8;
 		}/* end device is supported ( ioctl ok)*/
	}/* end all supported reported present */
}/* end function get_it */

void read_mixer_values()
/* this procedure reads the values from the mixer device */
{
int i, value, left_vol, right_vol, rec_src;
	
left_vol = 100;
right_vol = 100;

if (ioctl(mixer_fd, SOUND_MIXER_READ_RECSRC, &rec_src) == -1)
	{
	fprintf(stderr,"\n%s: Error reading recording source\n", progname);
	}
        
for (i = 0; i < SOUND_MIXER_NRDEVICES; i++)
	{
	get_it(i, rec_src);
    }/* end for all i */           
if (ioctl(mixer_fd,MIXER_READ(28), &value) == -1)
	{
	fprintf(stderr,"\n%s: Error reading %s in %s\n",
	progname, soundDeviceNames[28], MIXERDEVICE);
	exit(1);
	}
else	
	{
	printf("\n      mute\t\t%d\t\t(use 0, 1, on, off)", value);
	}
if (ioctl(mixer_fd,MIXER_READ(29),&value) == -1)
	{
	fprintf(stderr,"\n%s: Error reading %s in %s\n",
	progname, soundDeviceNames[29], MIXERDEVICE);
	exit(1);
	}
else
	{
	printf("\n      enhance\t\t%d\t\t(use 0, 40, 60, 80)", value);
	}
if (ioctl(mixer_fd,MIXER_READ(30),&value) == -1)
	{
	fprintf(stderr,"\n%s: Error writing %s in %s\n",
	progname, soundDeviceNames[30], MIXERDEVICE);
	exit(1);
	}
else	
	{
	printf("\n      loudness\t\t%d\t\t(use 0, 1, on, off)", value);
	}
}/* end function read_mixer_values */



void info()
{
fprintf(stdout,"\nPanteltje (c)  cmix v0.3 1996\n");
fprintf(stdout,"             *The command line mixer*\n"); 
}


void main(int argc, char **argv)
{
int device, value, leftvalue, rightvalue, readvalue, record_source, type;
char label[40];
int i, j;
int a;
int stereo_flag;
char filename[256];

/* init mixer */
mixerInit();
stereo_flag = BOTH;
value = -1;
device = -1;
type = -1;
filename[0] = 0;/* mark empty */
if(argc > 1)
	{
	for(j = 1; j < argc; j++)/* for all command line arguments */
		{
		strcpy(label, argv[j]);
		if( isdigit(label[0]) )
			{
			value = atoi(label);
			}
		else
			{
			for(i = 0; i < 12; i++)
				{
				if(zindex(soundDeviceNames[i], label) == 0)
					{
					device = i;
					break;
					}
				}/* end for all labels */
			if(zindex("left", label) == 0)stereo_flag = LEFT;
			if(zindex("right", label) == 0)stereo_flag = RIGHT;
			if(zindex("mute", label) == 0)device = 28;
			if(zindex("enhance", label) == 0)device = 29;
			if(zindex("loudness", label) == 0)device = 30;
			if(zindex("off", label) == 0)value = 0;
			if(zindex("on", label) == 0)value = 1;
			if(zindex("max", label) == 0)value = 100;
			if(zindex("min", label) ==0)value = 0;
			if(zindex("norm", label) == 0)value = 75;

			if(zindex("help", label) == 0)type = HELP;
			if(zindex("-help", label) == 0)type = HELP;
			if(zindex("--help", label) == 0)type = HELP;
			if(zindex("rcs", label) == 0)type = RECORD_SOURCE;
			if(zindex("load", label) == 0)
				{
				type= LOAD_PRESET;
				}
			if(zindex("save", label) == 0)
				{
				type = SAVE_PRESET;
				}
			if( (zindex("save", label) != 0) && (zindex("load", label) != 0) )
				{
				if(device == -1)
					{
					if(filename[0] == 0)strcpy(filename, label);
											/* could be filename */
					}
				}	
			}/* end alpha in label */		
		}/* end for j all command line arguments */
/*
printf("\ndevice=%d type=%d label=%s value=%d filename=%s\n",\
device, type ,label, value, filename);
*/
	if(type == HELP)
		{
		info();
		fprintf(stdout,\
		"\nUsage: cmix [left | right] device on | off | max | min | 0 - 100");
		fprintf(stdout,\
		"\nArguments may be in any order, abbreviations are allowed, but: ");
		fprintf(stdout, "\nbe specific with pcm and pcm2, left and line, etc. ");
		fprintf(stdout, "\nOnly ONE command at the time is executed.");
		fprintf(stdout,\
		"\ncmix normally does not print anything unless an error occurs,");
		fprintf(stdout, "\nthis allows it to be used from within a script.");
		fprintf(stdout,\
		"\nIf no arguments are given, the mixer status is displayed.");
		fprintf(stdout, "\nExamples:");
		fprintf(stdout,\
		"\ncmix vol 60            sets volume to 60 percent");
		fprintf(stdout,\
		"\ncmix rcs mic on        sets record source to microphone on");
		fprintf(stdout,\
		"\ncmix left cd max       sets left channel cd to 100 percent)");
		fprintf(stdout,\
		"\ncmix mute off          sets mute off");
		fprintf(stdout,\
"\ncmix enhanced 80       sets enhanced stereo mode to 80 (use 0 40 60 or 80");
		fprintf(stdout,\
"\ncmix save this         saves mixer settings in file this.cmix0.3.preset");
		fprintf(stdout,\
"\ncmix load that         loads mixer settings from file that.cmix0.3.preset");
		fprintf(stdout,\
		"\ncmix le sy norm        sets left channel synth to 75 percent");
		fprintf(stdout,\
		"\ncmix spe ri max        sets right speaker to 100 percent");
		fprintf(stdout, "\n");
		exit(0);
		}
	if( (type != SAVE_PRESET) && (type != LOAD_PRESET) )
		{
		if(device == -1)
			{
			fprintf(stderr, "\nNo device name specified\n");
			exit(1);
			} 
		}	
	if(type == RECORD_SOURCE)
		{
		if(value == -1)
			{
			fprintf(stderr, "\nNo value specified (use on or off)\n");
			exit(1);
			}
		/* make device identifier */
		a = 1 << device;
		/* get present record source(s) */
		if (ioctl(mixer_fd, SOUND_MIXER_READ_RECSRC, &record_source) == -1)
			{
			fprintf(stderr,"\n%s: Error reading recording source\n", progname);
     		exit(1);
			}
		/* modify */		
		if(value == 1)record_source |= a;
		if(value == 0)record_source &= (0xffff - a);
		/* write new record source(s) */
		if (ioctl(mixer_fd, SOUND_MIXER_WRITE_RECSRC, &record_source) == -1)
			{
			fprintf(stderr,"\n%s: Error writing recording source\n", progname);
     		exit(1);
			}

		info();
	    read_mixer_values();
		printf("\n");
		
		exit(0);
		}/* end type is RECORD_SOURCE */	

	if(type == SAVE_PRESET)
		{
		if(filename[0] == 0)
			{
			fprintf(stdout, "\nNo filename specified for preset file\n");
			exit(1);
			}
printf("\nSaving %s\n", filename);	
		mixer_to_disk(filename);
		exit(0);
		}/* end type is SAVE_PRESET */

	if(type == LOAD_PRESET)
		{
		if(filename[0] == 0)
			{
			fprintf(stdout, "\nNo filename specified for preset outputfile\n");
			exit(1);
			}
		mixer_from_disk(filename);
		exit(0);
		}/* end type is LOAD_PRESET */

		{	
		if(value == -1)
			{
			fprintf(stderr,	"\nNo value specified use ");
			if( (device >= 0) && (device < 12) )
				{
				fprintf(stderr,	"0 - 100, on, off ,max, min\n");
				}
			if(device == 28)fprintf(stderr, "0, 1, off, on\n");				
			if(device == 29)fprintf(stderr, "0, 40, 60, 80\n");				
			if(device == 30)fprintf(stderr, "0, 1, off, on\n");				
			exit(1);
			}/* end value is -1 */
		
		if( (value > 100) || (value < 0) )
			{
			fprintf(stdout, "\nIllegal value\n");
			exit(-1);
			}
		if(device == 28)value = value;
		else if(device == 29)value = value;
		else if(device == 30)value = value;
		else if(stereo_flag == BOTH)value = value + (value << 8); 
		else
			{
			/* get what is there first */
			if (ioctl(mixer_fd,MIXER_READ(device), &readvalue) == -1)
				{
				fprintf(stderr,"\n%s: Error reading %s in %s\n",
				progname, soundDeviceNames[device], MIXERDEVICE);
				exit(1);
				}	
			else
				{
				leftvalue = readvalue & 0xff;
				rightvalue = readvalue & 0xff00;						
				if(stereo_flag == LEFT)
					{
					value = rightvalue | value; 
					}
				if(stereo_flag == RIGHT)
					{
					value = leftvalue | ( (value & 0xff) << 8);
					}
				}/* end readvalue read ok from mixer */
			}/* end user specified left or right */
		/* write back modified value */
/*
		printf("\ndevice=%d, value=%d\n", device, value);
*/
		if (ioctl(mixer_fd, MIXER_WRITE(device), &value) == -1)
			{
			fprintf(stderr,"\n%s: Error writing %s in %s\n",
			progname, soundDeviceNames[device], MIXERDEVICE);
			exit(1);
			}
/*
		printf("\ndevice=%d, value=%d\n", device, value);
*/

		}/* end type is LEVEL */
	}/* end arc > 1 */
else
	{
	info();
	read_mixer_values();
	printf("\n");
	}/* end argc < 2*/

exit (0);
}/* end main */
