/****************************************************************************** 
 *
 * File:        mpu401.c 
 * Version:     $Id: mpu401.c,v 1.45 1995/07/30 01:59:41 burgaard Exp $
 *              $Version: 0.2$
 *
 * Project:     Roland MPU-401 MIDI Interface Device driver.
 * Authors:     Kim Burgaard, <burgaard@daimi.aau.dk>
 * Copyrights:  Copyright (c) 1994, 1995 Kim Burgaard.
 *
 *      This package is free software; you can redistribute it and/or modify it
 *      under the terms of the GNU General Public License as published by the
 *      Free Software Foundation; either version 2, or (at your option) any
 *      later version.
 *
 *      This package is distributed in the hope that it will be useful, but
 *      WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *      Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License along
 *      with this program; see the file COPYING. If not, write to the Free
 *      Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************************************/

/*** INCLUDES & DEFINES *******************************************************/

#include <linux/module.h>
#include <linux/version.h>

#include <fcntl.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/malloc.h>
#include <linux/major.h>
#include <linux/ioctl.h>
#include <linux/ioport.h>
#include <linux/stat.h>
#include <linux/errno.h>
#include <linux/delay.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/segment.h>
#include <asm/system.h>
#include <asm/string.h>

#include "midiconst.h"
#include "miditypes.h"
#include "midiqueue.h"
#include "mpuioctl.h"
#include "mpu401.h"

#ifndef OFF
#define OFF 0
#endif

#ifndef ON
#define ON 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

#ifndef TRUE
#define TRUE 1
#endif

/*** CONSTANTS ****************************************************************/

/* Set kernel version information for module interface */
char kernel_version[] = UTS_RELEASE;

#ifndef MPU_RC_FILE
#define MPU_RC_FILE "/var/lib/mpu401.conf"
#endif

#ifndef MPU_VERSION
#define MPU_VERSION 0
#endif
#ifndef MPU_SUBVERSION
#define MPU_SUBVERSION 2
#endif
#ifndef MPU_REVISION
#define MPU_REVISION " "
#endif

#define IOPORT_RETRY 500000
#define IRQ_PROBE_RETRY 5

#define MPU_SYSEX_BUFFER 30
#define MAX_QUEUE_EVENTS 500

/* This array is indexed after the most significant 4 BITS of a CHANNEL MESSAGE
 *                                         0 1 2 3 4 5 6 7 8 9 A B C D E F */
static const char MIDI_CHMSG_ARGS[0x10] = {0,0,0,0,0,0,0,0,2,2,2,2,1,1,2,0};

/*** ``PRIVATE'' VARIABLES ****************************************************/

static const char driver_version_str[] = "MPU-401 device driver v%hd.%hd.%s\n";

/* Keep track of IRQ request/release */
static int mpu_has_irq = FALSE;

/* MPU hardware and driver information */
static mpu_version_t drv_ver;
static mpu_hardw_t mpu_hw;
static mpu_version_t mpu_ver;

static unsigned long mpu_status; /* status flags */
static enum MPU_DEBUG mpu_debug; /* mpu_debuglevel */
static enum MPU_ERROR mpu_error; /* error flag */

/* Driver queue and blocking information */
static mpu_queue_t mpu_queues;
static mpu_block_t mpu_blocking;

/* MPU settings */
static mpu_metro_t metronome;
static mpu_inisw_t init_switches;
static mpu_chtbl_t channeltables;
static mpu_swtch_t switches;
static mpu_track_t tracks;

static midi_queue mpu_immediate_queue;

/* Running status etc. */
static mpu_runmsg_t runmsg;

#ifdef DEVELOP
static mpu_action_t actionq;
#endif

/*** ACTION QUEUE *************************************************************
 * Keep a log of the last 256 commands executed in the driver.
 * Usefull for a detailed inspection, when problems occur.
 * -- Not used --
 ******************************************************************************/

#ifdef DEVELOP
inline static void mpu_action_queue_reset(mpu_action_t * queue)
{
  if (!queue) return;
  queue->head = queue->tail = queue->count = 0;
}

static void mpu_action_queue_put(mpu_action_t * queue, enum MPU_ACTION act)
{
  byte state = ((mpu_hw.mutex) ? MPU_FLG_IRQ : 0) |
    ((mpu_hw.fake)             ? MPU_FLG_FAK : 0) |
    ((mpu_blocking.demand)     ? MPU_FLG_DEM : 0) |
    ((mpu_blocking.play_end)   ? MPU_FLG_END : 0) |
    ((mpu_blocking.play_delay) ? MPU_FLG_DEL : 0) |
    ((mpu_blocking.record)     ? MPU_FLG_REC : 0) |
    ((mpu_blocking.sync)       ? MPU_FLG_SYN : 0);
  if (!queue) return;
  queue->count++;
  queue->head++;
  queue->head %= MPU_ACTION_SIZE;
  queue->queue[queue->head] = ((queue->head<<24)&0xff000000)|((mpu_error<<16)&0xff0000)|
    ((state<<8)&0xff00)|(act&0xff);
}

static long mpu_action_queue_get(mpu_action_t * queue)
{
  long tmp;
  if (!queue || !queue->count) return 0;
  tmp = queue->queue[queue->tail];
  queue->count--;
  queue->tail++;
  queue->tail %= MPU_ACTION_SIZE;
  return tmp;
}

inline static long mpu_action_queue_peek_head(mpu_action_t * queue)
{
  return (queue && queue->count > 0) ? queue->queue[queue->head] : 0;
}
   
inline static long mpu_action_queue_peek_tail(mpu_action_t * queue)
{
  return (queue && queue->count > 0) ? queue->queue[queue->tail] : 0;
}

#endif

/*** MISC. ********************************************************************/

inline static unsigned long mpu_total_event_cnt(void)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_total_event_cnt);
#endif
  return mpu_queues.voice.count + mpu_queues.control.count + mpu_queues.record.count;
}

inline static void mpu_clear_run(void)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_clear_run);
#endif
  runmsg.play = runmsg.record = 0;
}

inline static void mpu_clear_run_play(void)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_clear_run_play);
#endif
  runmsg.play = 0;
}

/* Delay (msec) routine. Mostly taken from ftape-io.c by Bas Laarhoven */
static void mpu_msdelay(word time)
{
  unsigned long flags;
  int ticks = 1 + (time + (1000/HZ) - 1) / (1000/HZ);

#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_msdelay);
#endif

  if (mpu_hw.mutex)
    {
      printk("MPU-401: PANIC: mpu_msdelay called from interrupt!\n");
      mpu_error = MPU_ERR_DELAY;
      return;
    };

  if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: MILLISECOND DELAY REQUEST %d\n", time);
  
  /* error in range [0..1] (1000/HZ). Time too small for scheduler, do a busy wait!*/
  if (time < (1000/HZ))
    {
      udelay(1000 * time);
    }
  else
    {
      current->timeout = jiffies + ticks;
      current->state = TASK_INTERRUPTIBLE;
      save_flags( flags);
      sti();
      do
	{
	  while (current->state != TASK_RUNNING)
	    {
	      schedule();
	    };
	  if (current->signal & ~current->blocked)
	    {
	      printk("MPU-401: delay sleep awoken by non-blocked signal\n");
	      break; /* exit on signal */
	    };
	}    
      while (current->timeout > 0);
      restore_flags(flags);
    };
}

static char mpu_reset(char update);

static long mpu_ticks_per_beat()
{
  switch (metronome.setup.denominator)
    {
    case 1: return 4*metronome.timebase;
    case 2: return 2*metronome.timebase;
    case 4: return metronome.timebase;
    case 8: return metronome.timebase/2;
    case 16: return metronome.timebase/4;
    case 32: return metronome.timebase/8;
    };
  /* anything under x/32 time signature does not compute integer values */
  return metronome.timebase;
};

/*** ERROR HANDLING ***********************************************************/

/* This function dumps an event to the kernel log -- use rarely, if ever!!! */
static void mpu_printkevent(midi_event *event)
{
  int i;

#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_printkevent);
#endif

  if (mpu_debug< 2) return;
  if (!event)
    {
      printk("MPU-401: Event DUMP - NULL EVENT!\n");
      mpu_error = MPU_ERR_NULL;
      return;
    }
  else
    {
      printk("MPU-401: Event DUMP - Type %hx, time %ld, size %d\n",  event->type&0xff, event->time, event->size);
      printk("         Data [ ");
      for (i = 0; i < event->size; i++) printk("%02hx ", event->data[i]&0xff);
      printk("]\n");
    };
}

static void mpu_data_error(byte data)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_data_error);
#endif
  mpu_clear_run();
  printk("MPU-401: Unknown MPU-401 device message (0x%02x) not processed!\n", data);
  mpu_error = MPU_ERR_DEV_MSG;
  mpu_reset(TRUE);
}

static int mpu_can_store(void)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_can_store);
#endif
  if (mpu_total_event_cnt() > MAX_QUEUE_EVENTS)
    {
      printk("MPU-401: EVENT QUEUE OVERFLOW!!!\n");
      mpu_error = MPU_ERR_Q_OFLO;
      return FALSE;
    };
  return TRUE;
}

/*** MPU-401 DEVICE LOWLEVEL METHODS ******************************************/

inline static char * mpu_kmalloc(size_t size)
{
  char * tmp = kmalloc(size, GFP_ATOMIC);
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_kmalloc);
#endif
  if (!tmp)
    {
      printk("MPU-401: kmalloc(%d) failed!\n", size);
      mpu_error = MPU_ERR_MEM;
    }
  else
    {
      mpu_queues.kmallocs += size;
    };
  return tmp;
}

inline static void mpu_kfree_s(char * ptr, size_t size)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_kfree_s);
#endif
  if (!ptr)
    {
      printk("MPU-401: kfree_s(%d) failed!\n", size);
      mpu_error = MPU_ERR_MEM;
    }
  else
    {
      kfree_s(ptr, size);
      mpu_queues.kfrees += size;
    };
}

inline midi_event * midi_event_alloc(void)
{
  return (midi_event *)mpu_kmalloc(sizeof(midi_event));
}

inline void midi_event_free(midi_event *event)
{
  if (!event) return;
  if (event->data) mpu_kfree_s((char *)event->data, event->size);
  mpu_kfree_s((char *)event, sizeof(midi_event));
}

static int mpu_wait_DRR(void)
{
  long i = IOPORT_RETRY;
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_wait_DRR);
#endif
  while ( (inb(mpu_hw.stp)&MPU_DRR) && (--i > 0) );

  if ((IOPORT_RETRY - i) > mpu_hw.loops_drr) mpu_hw.loops_drr = IOPORT_RETRY - i;
  if (i) return TRUE;

  printk("MPU-401: mpu_wait_DRR() time out on port 0x%03x\n", mpu_hw.stp);
  mpu_hw.timeout_drr++;
  mpu_error = MPU_ERR_DRR;

  return FALSE;
}

static int mpu_wait_DSR(void)
{
  long i = IOPORT_RETRY;
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_wait_DSR);
#endif
  while ( (inb(mpu_hw.stp)&MPU_DSR) && (--i > 0) );

  if ((IOPORT_RETRY - i) > mpu_hw.loops_dsr) mpu_hw.loops_dsr = IOPORT_RETRY - i;
  if (i) return TRUE;

  printk("MPU-401: mpu_wait_DSR() time out on port 0x%03x\n", mpu_hw.stp);
  mpu_hw.timeout_dsr++;
  mpu_error = MPU_ERR_DSR;

  return FALSE;
}

inline static byte mpu_rd_dta(void)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_rd_dta);
#endif
  return inb(mpu_hw.dtp);
}

inline static void mpu_wr_dta(byte dta)
{
  outb(dta, mpu_hw.dtp);
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_wr_dta);
#endif
}

inline static void mpu_wr_cmd(byte cmd)
{
  outb(cmd, mpu_hw.cmp);
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_wr_cmd);
#endif
}

/*** MPU-401 DEVICE INTERNAL METHODS ******************************************/

/* prototype declaration */
static void mpu_irqhandler(int, struct pt_regs *);

inline static byte mpu_RcDta(void)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_RcDta);
#endif
  return (mpu_wait_DSR()) ? mpu_rd_dta() : 0;
}

static int mpu_SnCmd(byte cmd)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_SnCmd);
#endif
  mpu_error = 0;
  cli();
  if (!mpu_wait_DRR())
    {
      sti();
      return FALSE;
    };
  mpu_wr_cmd(cmd);
  while ((mpu_RcDta() != MPU_MSG_ACKNOWLEDGE) && !(mpu_error))
    {
      mpu_hw.fake = TRUE;
      mpu_irqhandler(mpu_hw.irq, NULL);
      mpu_hw.fake = FALSE;
    };
  sti();
  return (mpu_error == MPU_ERR_NONE);
}

static byte mpu_RqDta(byte cmd)
{
  static byte tmp;
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_RqDta);
#endif
  mpu_error = 0;
  cli();
  if (!mpu_wait_DRR())
    {
      sti();
      return FALSE;
    };
  mpu_wr_cmd(cmd);
  while ((mpu_RcDta() != MPU_MSG_ACKNOWLEDGE) && !(mpu_error))
    {
      mpu_hw.fake = TRUE;
      mpu_irqhandler(mpu_hw.irq, NULL);
      mpu_hw.fake = FALSE;
    };
  if (mpu_error == MPU_ERR_NONE) tmp = mpu_RcDta();
  sti();
  return tmp;
}

inline static int mpu_SnDta(byte dta)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_SnDta);
#endif
  if (!mpu_wait_DRR()) return FALSE;
  mpu_wr_dta(dta);
  return TRUE;
}

static int mpu_SnCmdDta(byte cmd, byte dta)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_SnCmdDta);
#endif
  mpu_error = 0;
  if ((cmd < MPU_SET_IMMEDIATE_FIRST) || (cmd > MPU_SET_IMMEDIATE_LAST))
    {
      mpu_error = MPU_ERR_CMD;
      return FALSE;
    };
  return (mpu_SnCmd(cmd)) ? mpu_SnDta(dta) : FALSE;
}

/*** IRQ HANDLING *************************************************************
 *
 * Some precaution should be taken in the following methods, as they are most
 * likely called from the interrupt handler. I.e. *never* call sleep etc.
 *
 ******************************************************************************/

inline static void mpu_disable_irq(void)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_disable_irq);
#endif
  disable_irq(mpu_hw.irq);
}
inline static void mpu_enable_irq(void)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_enable_irq);
#endif
  enable_irq(mpu_hw.irq);
}

inline static void mpu_sleep(struct wait_queue ** sleeper)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_sleep);
#endif
  interruptible_sleep_on(sleeper);
}

static void mpu_wake_up(struct wait_queue ** sleeper, enum MPU_BLOCK msg)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_wake_up);
#endif
  if (!sleeper || !(*sleeper)) return;
  mpu_disable_irq();
  mpu_blocking.bias.message |= msg;
  wake_up_interruptible(sleeper);
  if (sleeper) *sleeper = NULL;
  mpu_enable_irq();
}

static void mpu_midi_in(long time)
{
  byte *buf = NULL;
  byte dta = 0;
  byte siz = 0;
  byte i   = 0;
  
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_midi_in);
#endif

  dta = mpu_RcDta();
  if (dta > MID_EVENT_CHMSG_LAST)
    {
      switch (dta)
	{
	case MPU_MRK_MEASURE:
	  break;
	case MPU_MRK_DATA_END:
	  /* Stop recording */
	  mpu_status &= ~(MPU_STA_REC);
	  mpu_wake_up(&mpu_blocking.record, MPU_BLK_END);
	  mpu_wake_up(&mpu_blocking.demand, MPU_BLK_END);
	  break;
	default:
	  mpu_data_error(dta);
	  break;
	};
    }
  else if (dta >= MID_EVENT_CHMSG_FIRST)
    {
      runmsg.record = dta; /* save voice command for running status */
      siz = MIDI_CHMSG_ARGS[dta>>4] + 1;
      if (mpu_can_store() && (buf = (byte *)mpu_kmalloc(siz)))
	{
	  buf[0] = dta;
	  for (i = 1; i < siz; i++) buf[i] = mpu_RcDta();
	  midi_queue_put_event(&mpu_queues.record, MIDI_TYP_VOICE, time, siz, buf);

	  if (mpu_queues.record.count > mpu_blocking.bias.record)
	    {
	      mpu_wake_up(&mpu_blocking.record, MPU_BLK_RECORD|MPU_BLK_DEMAND);
	      mpu_wake_up(&mpu_blocking.demand, MPU_BLK_RECORD|MPU_BLK_DEMAND);
	    };
	}
      else
	for (i = 1; i < siz; i++) mpu_RcDta(); /* flush rest of incoming data */
    }
  else if (runmsg.record)
    {
      siz = MIDI_CHMSG_ARGS[(runmsg.record>>4)&0x0F] + 1;
      if ( mpu_can_store() && (buf = (byte *)mpu_kmalloc(siz)))
	{
	  buf[0] = runmsg.record;
	  buf[1] = dta;
	  for (i = 2; i < siz; i++) buf[i] = mpu_RcDta();
	  midi_queue_put_event(&mpu_queues.record, MIDI_TYP_VOICE, time, siz, buf);

	  if (mpu_queues.record.count > mpu_blocking.bias.record)
	    {
	      mpu_wake_up(&mpu_blocking.record, MPU_BLK_RECORD|MPU_BLK_DEMAND);
	      mpu_wake_up(&mpu_blocking.demand, MPU_BLK_RECORD|MPU_BLK_DEMAND);
	    };
	}
      else
	for (i = 2; i < siz; i++) mpu_RcDta(); /* flush rest of incoming data */
    }
  else
    {
      printk("MPU-401: Illegal running status received from MIDI IN (0x%02hx)\n", dta);
      return;
    };
}

static void mpu_midi_out(byte track)
{
  midi_event * event = NULL;
  long dt = 0;
  word i = 0;

#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_midi_out);
#endif

  if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: MIDI OUT (%d)\n", track);
  if (track == MPU_TRK_TIMER) /* metronome/sync ``timer'' */
    {
      event = midi_queue_peek(&mpu_immediate_queue);
      if (event) /* forget everything about time, play it right away */
	{
	  mpu_SnDta(1);
	  for (i = 0; i < event->size; i++) mpu_SnDta(event->data[i]);
	  midi_queue_remove_event(&mpu_immediate_queue, event);
	  metronome.time += 1;
	  metronome.subtime += 1;
	  mpu_clear_run_play();
	}
      else if (mpu_status & (MPU_STA_PLAY|MPU_STA_REC))
	{
	  mpu_SnDta(1);
	  mpu_SnDta(MPU_MRK_NO_OP);
	  metronome.time += 1;
	  metronome.subtime += 1;
	}
      else
	{
	  mpu_SnDta(0);
	  mpu_SnDta(MPU_MSG_DATA_END);
	};
      return;
    }
  else if (track != MPU_TRK_VOICE)
    {
      mpu_SnDta(0);
      mpu_SnDta(MPU_MSG_DATA_END);
      return;
    };

  event = midi_queue_peek(&mpu_queues.voice);
  if (!event)
    {
      if (mpu_queues.control.count > 0)
	{
	  /* still some ctl messages left to play... */
	  mpu_SnDta(MPU_MSG_MAX_TIME);
	  mpu_SnDta(MPU_MRK_NO_OP);
	  mpu_queues.nop_cnt++;
	  mpu_queues.voice_time += MPU_MSG_MAX_TIME;
	  
	  if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: NOP, CTL LEFT\n");
	}
      else if (mpu_blocking.play_delay)
	{
	  mpu_SnDta(0);
	  mpu_SnDta(MPU_MRK_NO_OP);
	  mpu_queues.nop_cnt++;
	  mpu_wake_up(&mpu_blocking.play_delay, MPU_BLK_DELAY|MPU_BLK_VOICE);
	}
      else if (mpu_blocking.demand)
	{
	  mpu_SnDta(0);
	  mpu_SnDta(MPU_MRK_NO_OP);
	  mpu_queues.nop_cnt++;

	  mpu_wake_up(&mpu_blocking.demand, MPU_BLK_DEMAND|MPU_BLK_VOICE);
	  if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: NOP, PLAY/OVERDUB BLOCKED\n");
	}
      else if (mpu_status & MPU_STA_REC)
	{
	  /* just idle, we're still recording */
	  mpu_SnDta(MPU_MSG_MAX_TIME);
	  mpu_SnDta(MPU_MRK_NO_OP);
	  mpu_queues.voice_time += MPU_MSG_MAX_TIME;
	  mpu_queues.nop_cnt++;
	}
      else
	{
	  /* otherwise stop replaying */
	  mpu_SnDta(0);
	  mpu_SnDta(MPU_MSG_DATA_END);
	  mpu_queues.nop_cnt++;
	  
	  mpu_wake_up(&mpu_blocking.play_end, MPU_BLK_END);
	  if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: MIDI STOP\n");
	};
      return;
    };
  
  dt = event->time - mpu_queues.voice_time;
  if (mpu_debug==MPU_DBG_ALL) printk("MPU-401: OUT TIME, DELTATIME: %ld, %ld\n",event->time,dt);

  /* if delta time < 0 then we're behind schedule :-( i.e. hurry up! :-) */
  if (dt < 0) dt = 0;
  if (dt < MPU_MSG_MAX_TIME)
    {
      if (event->data[0] == runmsg.play)
	{
	  mpu_queues.voice_hits++;
	  i = 1;
	}
      else
	{
	  runmsg.play = event->data[0];
	  i = 0;
	};
      mpu_SnDta(dt);
      for (; i < event->size; i++) mpu_SnDta(event->data[i]);

      mpu_queues.voice_time += dt;
      mpu_queues.voice_cnt++;

      midi_queue_remove_event(&mpu_queues.voice, event);

      if ((event = midi_queue_peek(&mpu_queues.voice)) && (event->time != mpu_queues.voice_time))
	{
	  mpu_wake_up(&mpu_blocking.play_delay, MPU_BLK_DELAY|MPU_BLK_VOICE);
	};
      if (mpu_queues.voice.count < mpu_blocking.bias.voice)
	{
	  mpu_wake_up(&mpu_blocking.demand, MPU_BLK_DEMAND|MPU_BLK_VOICE);
	};

      if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: SEND MIDI OUT VOICE EVENT\n");
    }
  else
    {
      mpu_SnDta(MPU_MSG_MAX_TIME);
      mpu_SnDta(MPU_MRK_NO_OP);

      mpu_queues.nop_cnt++;
      mpu_queues.voice_time += MPU_MSG_MAX_TIME;

      if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: NOP, SCHEDULED\n");
    };
}

inline static void mpu_store_control_event(midi_event * event)
{ 
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_store_control_event);
#endif
  if (!event) return;
  memcpy(&runmsg.control_data[0], event->data, runmsg.control_size = event->size);
  runmsg.control = TRUE;
}

/* The method for sending conductor data is a little bit hairy at a first glance.
 * (but it isn't :-) */
static void mpu_control_out(void)
{
  midi_event * event = NULL;
  long dt = 0;
  long old_ticks_meas, new_ticks_meas, j, k;
  word i = 0;

#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_control_out);
#endif

  if (runmsg.control)
    {
      /* send current control data */
      for (i = 0; i < runmsg.control_size; i++) mpu_SnDta(runmsg.control_data[i]);
      runmsg.control = FALSE;
      mpu_clear_run_play();
    };

  event = midi_queue_peek(&mpu_queues.control);
  if (!event)
    {
      runmsg.control_size = 0;
      runmsg.control = FALSE;

      if (mpu_queues.voice.count > 0)
	{
	  /* still some voice messages left to play... */
	  mpu_SnDta(MPU_MSG_MAX_TIME);
	  mpu_SnDta(MPU_MRK_NO_OP);
	  mpu_queues.nop_cnt++;
	  mpu_queues.control_time += MPU_MSG_MAX_TIME;
	  
	  if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: NOP, VOICE LEFT\n");
	}
      else if (mpu_blocking.play_delay)
	{
	  mpu_SnDta(0);
	  mpu_SnDta(MPU_MRK_NO_OP);
	  mpu_queues.nop_cnt++;
	  mpu_wake_up(&mpu_blocking.play_delay, MPU_BLK_DELAY|MPU_BLK_CONTROL);
	}
      else if (mpu_blocking.demand)
	{
	  mpu_SnDta(0);
	  mpu_SnDta(MPU_MRK_NO_OP);
	  mpu_queues.nop_cnt++;

	  mpu_wake_up(&mpu_blocking.demand, MPU_BLK_DEMAND|MPU_BLK_CONTROL);
	  if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: NOP, PLAY/OVERDUB BLOCKED\n");
	}
      else if (mpu_status & MPU_STA_REC)
	{
	  /* just idle, we're still recording */
	  mpu_SnDta(MPU_MSG_MAX_TIME);
	  mpu_SnDta(MPU_MRK_NO_OP);
	  mpu_queues.control_time += MPU_MSG_MAX_TIME;
	  mpu_queues.nop_cnt++;
	}
      else
	{
	  /* otherwise stop replaying */
	  mpu_SnDta(0);
	  mpu_SnDta(MPU_MSG_DATA_END);
	  mpu_queues.nop_cnt++;
	  
	  mpu_wake_up(&mpu_blocking.play_end, MPU_BLK_END);
	  if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: MIDI STOP\n");
	};

      return;
    };
  
  /* if delta time < 0 then we're behind schedule :-( i.e. hurry up! :-) */
  dt = event->time - mpu_queues.control_time;
  if (dt < 0) dt = 0;
  if (dt < MPU_MSG_MAX_TIME)
    {
      mpu_clear_run_play(); 
      mpu_queues.control_time += dt;

      switch (event->type)
	{
	case MIDI_TYP_META:
	  switch (event->data[0])
	    {
	    case MID_META_SET_TEMPO:
	      if (event->size != 4)
		{
		  mpu_SnDta(dt);
		  mpu_SnDta(MPU_MRK_NO_OP);
		  if (mpu_debug > MPU_DBG_NONE) printk("MPU-401: Invalid META TEMPO DATA\n");
		  mpu_printkevent(event);
		  break;
		};

	      metronome.mysectempo = (event->data[1]<<16)|(event->data[2]<<8)|event->data[3];
	      metronome.bpm = (byte)((60.0*1000000.0)/(float)(metronome.mysectempo));

	      mpu_SnDta(dt);
	      mpu_SnDta(MPU_SET_TEMPO);
	      mpu_SnDta(metronome.bpm);

	      mpu_queues.meta_cnt++;
	      break;

	    case MID_META_TIME_SIG:
	      if (event->size < 1)
		{
		  mpu_SnDta(dt);
		  mpu_SnDta(MPU_MRK_NO_OP);
		  if (mpu_debug > MPU_DBG_NONE) printk("MPU-401: Invalid META TIME SIG DATA\n");
		  mpu_printkevent(event);
		  break;
		};

	      old_ticks_meas = metronome.setup.numerator * mpu_ticks_per_beat();

	      metronome.setup.numerator = event->data[1];
	      metronome.setup.denominator = 1;
	      for (i = 0; i < event->data[2]; i++) metronome.setup.denominator *= 2;
	      metronome.midi_metro = event->data[3];

	      mpu_SnDta(dt);
	      mpu_SnDta(MPU_SET_METRO_MEAS);
	      mpu_SnDta(metronome.setup.numerator);

	      /* Reset the metronome (sub)time in order to insure correct measure counting:
	       * Case 1: old time sig is. ``greater'' than new time sig., e.g. from 4/4 to 3/4
	       * | old time sig. | new time sig | --> time
	       * |----j----| |-k-|
	       *           | ^this would wrongly look like end of measure
	       *           ^event possibly comes before end of measure.
	       *
	       * Case 2: old time sig is. ``lesser'' than new time sig., e.g. from 4/4 to 5/4
	       * | old time sig. | new time sig | --> time
	       * |-----j-----|   |-k-|
	       *             |       ^this would wrongly look like end of measure
	       *             ^event possibly comes before end of measure. */
	      j = metronome.subtime % old_ticks_meas; /* number of ticks into current measure */
	      new_ticks_meas = metronome.setup.numerator*mpu_ticks_per_beat();
	      k = new_ticks_meas - old_ticks_meas;
	      /* pretend the last measure is in the new time sig. by compensating */
	      metronome.subtime = j - k;
	      mpu_queues.meta_cnt++;
	      break;

	    default:
	      mpu_SnDta(dt);
	      mpu_SnDta(MPU_MRK_NO_OP);

	      mpu_queues.nop_cnt++;
	      if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: NOP\n");
	      break;
	    };

	  midi_queue_remove_event(&mpu_queues.control, event);
	  runmsg.control_size = 0;
	  runmsg.control = FALSE;
	  break;
	case MIDI_TYP_SYSEX:
	  mpu_SnDta(dt);
	  mpu_SnDta(MPU_SND_SYSEX);
	  mpu_store_control_event(event);
	  midi_queue_remove_event(&mpu_queues.control, event);
	  mpu_queues.sysex_cnt++;
	  break;
	default:
	  mpu_SnDta(dt);
	  mpu_SnDta(MPU_MRK_NO_OP);
	  mpu_queues.nop_cnt++;

	  printk("MPU-401: control running event is invalid\n");
	  mpu_printkevent(event);
	  midi_queue_remove_event(&mpu_queues.control, event);
	  return;
	  break;
	};

      if ((event=midi_queue_peek(&mpu_queues.control)) && (event->time != mpu_queues.control_time))
	{
	  mpu_wake_up(&mpu_blocking.play_delay, MPU_BLK_DELAY|MPU_BLK_CONTROL);
	};
      if (mpu_queues.control.count < mpu_blocking.bias.control)
	{
	  mpu_wake_up(&mpu_blocking.demand, MPU_BLK_DEMAND|MPU_BLK_CONTROL);
	};
    }
  else
    {
      mpu_SnDta(MPU_MSG_TIME_OVERFLOW);

      mpu_queues.control_time += MPU_MSG_MAX_TIME;
      mpu_queues.nop_cnt++;
    };
}

/* The metronome operates according to the following rules:
 *
 * a. First beat in measure: Play meas_click_key.
 *
 * b. Else if the nominator is even and we're exactly halfway through a measure,
 *    acc_click_key will be played.
 *
 * c. Else a click_key will be played if we are positioned at a denominator subdivision.
 *
 * Examples: 4/4 results in DING! tick TICK tick | DING! ...
 *           3/4 results in DING! tick tick | DING! ...
 *           6/8 results in DING! tick tick TICK tick tick | DING! ...
 *           7/8 results in DING! tick tick tick tick tick tick | DING! ...
 */
static void mpu_metronome(long time, long beat)
{
  byte * data = NULL;

#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_metronome);
#endif

  if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: METRONOME\n");

  /* check for first beat in bar */
  if ( !(time % (beat*metronome.setup.numerator)) )
    {
      if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: METRONOME MEASURE\n");
      data = (byte *)mpu_kmalloc(3);
      data[0] = MID_EVENT_NOTE_ON | metronome.setup.channel;
      data[1] = metronome.setup.meas_click_key;
      data[2] = metronome.setup.meas_click_velo;
      midi_queue_put_event(&mpu_immediate_queue, MIDI_TYP_VOICE, 0, 3, data);

      data = (byte *)mpu_kmalloc(3);
      data[0] = MID_EVENT_NOTE_OFF | metronome.setup.channel;
      data[1] = metronome.setup.meas_click_key;
      data[2] = metronome.setup.meas_click_velo;
      midi_queue_put_event(&mpu_immediate_queue, MIDI_TYP_VOICE, 0, 3, data);
    }
  else if ( !(metronome.setup.numerator % 2) &&
	    !(time % (beat * (metronome.setup.numerator/2))) )
    {
      if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: METRONOME ACCENTED\n");
      data = (byte *)mpu_kmalloc(3);
      data[0] = MID_EVENT_NOTE_ON | metronome.setup.channel;
      data[1] = metronome.setup.acc_click_key;
      data[2] = metronome.setup.acc_click_velo;
      midi_queue_put_event(&mpu_immediate_queue, MIDI_TYP_VOICE, 0, 3, data);

      data = (byte *)mpu_kmalloc(3);
      data[0] = MID_EVENT_NOTE_OFF | metronome.setup.channel;
      data[1] = metronome.setup.acc_click_key;
      data[2] = metronome.setup.acc_click_velo;
      midi_queue_put_event(&mpu_immediate_queue, MIDI_TYP_VOICE, 0, 3, data);
    }
  else 
    {
      if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: METRONOME CLICK\n");
      data = (byte *)mpu_kmalloc(3);
      data[0] = MID_EVENT_NOTE_ON | metronome.setup.channel;
      data[1] = metronome.setup.click_key;
      data[2] = metronome.setup.click_velo;
      midi_queue_put_event(&mpu_immediate_queue, MIDI_TYP_VOICE, 0, 3, data);

      data = (byte *)mpu_kmalloc(3);
      data[0] = MID_EVENT_NOTE_OFF | metronome.setup.channel;
      data[1] = metronome.setup.click_key;
      data[2] = metronome.setup.click_velo;
      midi_queue_put_event(&mpu_immediate_queue, MIDI_TYP_VOICE, 0, 3, data);
    };
}

static void mpu_irqhandler(int irq, struct pt_regs * regs)
{
  static long oldtime = 0; /* prevent double metronome/sync triggers */
  byte *buf = NULL;
  byte *buf2 = NULL;
  byte dta  = 0;
  word cnt  = 0;
  word size = 0;
  long time = 0;
  long beat = 0;

#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_irqhandler);
#endif

  /* If DSR = 1, then IRQ was not from MPU. */
  if (inb(mpu_hw.stp) & MPU_DSR) return;
  dta = mpu_rd_dta();

  (!mpu_hw.fake) ? mpu_hw.real_irq_cnt++ : mpu_hw.fake_irq_cnt++;

  mpu_hw.mutex = TRUE;

  if (dta <= MPU_MSG_MAX_TIME)
    {
      if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: MIDI IN\n");
      mpu_queues.record_time += dta;
      mpu_midi_in(mpu_queues.record_time);
    }
  else if (dta < MPU_MSG_TIME_OVERFLOW)
    {
      if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: MIDI OUT\n");
      mpu_midi_out(dta&0x0F);
    }
  else switch (dta)
    {
    case MPU_MSG_TIME_OVERFLOW:
      mpu_queues.record_time += 240;
      if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: TIMING OVERFLOW\n");
      break;

    case MPU_REQ_CONDUCTOR:
      if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: CONDUCTOR\n");
      mpu_control_out();
      break;

    case MPU_MSG_ALL_END:
      mpu_status &= ~(MPU_STA_PLAY);
      if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: ALL END\n");

      runmsg.control = FALSE;
      mpu_clear_run_play();

      mpu_wake_up(&mpu_blocking.record, MPU_BLK_END);
      mpu_wake_up(&mpu_blocking.play_end, MPU_BLK_END);
      mpu_wake_up(&mpu_blocking.play_delay, MPU_BLK_END);
      mpu_wake_up(&mpu_blocking.demand, MPU_BLK_END);
      mpu_wake_up(&mpu_blocking.sync, MPU_BLK_END);
      oldtime = -1;
      break;

    case MPU_MSG_CLOCK_TO_HOST:
      break;

    case MPU_MSG_ACKNOWLEDGE:
      if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: ACKNOWLEDGE\n");
      break;

    case MPU_MSG_MIDI_SYS_MSG:
      size = MPU_SYSEX_BUFFER;
      cnt = 0;
      mpu_clear_run_play();
      buf = (byte *)mpu_kmalloc(size);
      while ( (dta = mpu_RcDta()) != MID_SYS_EOX )
	{
	  if (!buf) continue;
	  if (cnt >= size) /* dynamically expand sysex data buffer */
	    {
	      size += MPU_SYSEX_BUFFER;
	      if ( (buf2 = (byte *)mpu_kmalloc(size)) )
		{
		  memmove(buf2, buf, cnt);
		  mpu_kfree_s(buf, cnt);
		  buf = buf2;
		  buf2 = NULL;
		}
	      else break;
	    };
	  buf[cnt++] = dta;
	};
      if ((cnt > 0) && (cnt < size)) /* did we allocate too much? (probably) */
	{
	  buf2 = mpu_kmalloc(cnt);
	  memmove(buf2, buf, cnt);
	  mpu_kfree_s(buf, size);
	  buf = buf2;
	  buf2 = NULL;
	};
      size = cnt;
      time = mpu_queues.record_time + mpu_RqDta(MPU_REQ_REC_CNT);
      if (size)
	{
	  midi_queue_put_event(&mpu_queues.record, MIDI_TYP_SYSEX, time, size, buf);
	  if (mpu_queues.record.count > mpu_blocking.bias.record)
	    {
	      mpu_wake_up(&mpu_blocking.record, MPU_BLK_DEMAND|MPU_BLK_RECORD);
	      mpu_wake_up(&mpu_blocking.demand, MPU_BLK_DEMAND|MPU_BLK_RECORD);
	    };

	  if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: MIDI IN SYSEX\n");
	};
      break;

    default:
      mpu_data_error(dta);
      break;
    };

  /* metronome and sync gets called on every beat */
  beat = mpu_ticks_per_beat();    
  if ( (mpu_status & (MPU_STA_PLAY|MPU_STA_REC)) && (!(metronome.subtime%beat)) &&
       (oldtime != metronome.subtime) )
    {
      metronome.subtime %= (beat * metronome.setup.numerator);
      oldtime = metronome.subtime;
      if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: VIRTUAL CLOCK TO HOST\n");
      if (metronome.status) mpu_metronome(metronome.subtime, beat);
      if (mpu_blocking.bias.sync) mpu_wake_up(&mpu_blocking.demand, MPU_BLK_SYNC);
      mpu_wake_up(&mpu_blocking.sync, MPU_BLK_SYNC);
    };
  
  mpu_hw.mutex = FALSE;
}

/*** KERNEL RESOURCE MANAGMENT ************************************************
 *
 ******************************************************************************/

static int mpu_irq_disabled = 0;

static int mpu_request_irq(void)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_request_irq);
#endif
  if (mpu_debug > MPU_DBG_NONE) printk("MPU-401: mpu_request_irq()\n");
  if (mpu_has_irq)
    {
      printk("MPU-401: Driver has already got IRQ %0d!\n", mpu_hw.irq);
      return FALSE;
    };
  if ( request_irq(mpu_hw.irq, mpu_irqhandler, SA_INTERRUPT, "MPU-401") )
    {
      printk("MPU-401: mpu_request_irq(%02d) failed.\n", mpu_hw.irq);
      return FALSE;
    };
  mpu_has_irq = TRUE;
  mpu_irq_disabled = 0;
  return TRUE;
}

static void mpu_free_irq(void)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_free_irq);
#endif
  if (mpu_debug > MPU_DBG_NONE) printk("MPU-401: mpu_free_irq(%02d)\n", mpu_hw.irq);
  if (!mpu_has_irq)
    {
      printk("MPU-401: IRQ %0d already freed!\n", mpu_hw.irq);
      return;
    };
  mpu_irq_disabled = 0;
  mpu_has_irq = FALSE;
  free_irq(mpu_hw.irq);
}

static int mpu_request_ioports(void)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_request_ioports);
#endif
  if (mpu_debug > MPU_DBG_NONE) printk("MPU-401: mpu_request_ioports()\n");
  if ( check_region(mpu_hw.ioport, 2) )
    {
      printk("MPU-401: Unable to request I/O Port address 0x%03hx to 0x%03hx",
	     mpu_hw.ioport, mpu_hw.ioport+1);
      return FALSE;
    };
  request_region(mpu_hw.ioport, 2, "mpu401");
  return TRUE;
}

static void mpu_release_ioports(void)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_release_ioports);
#endif
  if (mpu_debug > MPU_DBG_NONE) printk("MPU-401: mpu_release_ioports()\n");
  release_region(mpu_hw.ioport, 2);
}

/*** GENERAL MIDI/ROLAND GS ``MACROS'' ****************************************
 *
 * This will be moved out of the driver in time. Either to a piggybacked module
 * or replaced with an string that can be set at runtime.
 *
 ******************************************************************************/

static const byte gm_reset[] = 
{
  0xf0, 0x7e, 0x7f, 0x09, 0x01, 0xf7
};

static const byte gs_reset1[] = 
{
  0xf0, 0x41, 0x10, 0x42, 0x12, 0x40, 0x00, 0x7f, 0x00, 0x41, 0xf7
};

static const byte gs_reset2[] = 
{
  0xf0, 0x41, 0x10, 0x42, 0x12, 0x40, 0x01, 0x10, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x00,
  0x00, 0x00, 0x00, 0x02, 0x02, 0x02, 0x02, 0x02, 0x17, 0xf7
};

static void mpu_send_sysex(int size, const byte * msg)
{
  byte * data = NULL;
  int i = 0;

#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_send_sysex);
#endif

  if (mpu_debug > MPU_DBG_SPARSE) printk("MPU-401: mpu_send_sysex()\n");

  if (mpu_status & (MPU_STA_PLAY|MPU_STA_REC))
    {
      mpu_disable_irq();
      data = (byte *)mpu_kmalloc(size);
      memcpy(data, msg, size);
      midi_queue_put_event(&mpu_immediate_queue, MIDI_TYP_SYSEX, 0, size, data);
      mpu_enable_irq();
    }
  else
    {
      mpu_disable_irq();
      if (!mpu_SnCmd(MPU_SND_SYSEX)) return;
      for (i = 0; i < size; i++) mpu_SnDta(msg[i]);
      mpu_clear_run_play();
      mpu_enable_irq();
    };

  return;
}

static void mpu_kill_synth()
{
  byte i, j;

  if (mpu_status & (MPU_STA_PLAY|MPU_STA_REC)) return;

  mpu_disable_irq();

  for (j = 0; j < 16; j++)
    {
      /* send ALL SOUNDS OFF control event */
      mpu_SnCmd(MPU_SND_DATA | MPU_TRK_IMMEDIATE);
      mpu_SnDta(MID_EVENT_CONTROL | j);
      mpu_SnDta(MID_CTRL_ALL_SOUNDS_OFF);
      mpu_SnDta(0);

      /* send ALL NOTES OFF control event */
      mpu_SnCmd(MPU_SND_DATA | MPU_TRK_IMMEDIATE);
      mpu_SnDta(MID_EVENT_CONTROL | j);
      mpu_SnDta(MID_CTRL_ALL_NOTES_OFF);
      mpu_SnDta(0);

      /* send RESET ALL CONTROLLERS control event */
      mpu_SnCmd(MPU_SND_DATA | MPU_TRK_IMMEDIATE);
      mpu_SnDta(MID_EVENT_CONTROL | j);
      mpu_SnDta(MID_CTRL_RESET_ALL_CTRL);
      mpu_SnDta(0);

      /* send manual NOTE OFF to all notes */
      for (i = 0; i < 127; i++)
	{
	  mpu_SnCmd(MPU_SND_DATA | MPU_TRK_IMMEDIATE);
	  mpu_SnDta(MID_EVENT_NOTE_OFF | j);
	  mpu_SnDta(i);  /* key number */
	  mpu_SnDta(64); /* default note off velocity */
	};
    };

  mpu_clear_run_play();
  mpu_enable_irq();
}

/*** MPU-401 DEVICE INTERFACE METHODS *****************************************
 *
 ******************************************************************************/

static void mpu_flush_queues(void)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_flush_queues);
#endif
  midi_queue_flush(&mpu_queues.control);
  midi_queue_flush(&mpu_queues.voice);
  midi_queue_flush(&mpu_queues.record);
  midi_queue_flush(&mpu_immediate_queue);
  mpu_queues.voice_time = mpu_queues.control_time = mpu_queues.record_time = 0;
  metronome.time = 0;
  metronome.subtime = 0;
}

/* This function resets all the drivers variables to their default value.
 * NEVER CALL THIS FUNCTION execpt from insmod!!! 
 * See mpu401.h and the MPU-401 Tech. Ref. for an explanation of the values used
 */
static void mpu_reset_driver(void)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_reset_driver);
#endif
  if (mpu_debug > MPU_DBG_SPARSE) printk("MPU-401: mpu_reset_driver()\n");

  mpu_has_irq = FALSE;

  mpu_hw.irq = 5;
  mpu_hw.ioport = mpu_hw.dtp = 0x330;
  mpu_hw.cmp = mpu_hw.stp = mpu_hw.ioport + 1;
  mpu_hw.real_irq_cnt = mpu_hw.fake_irq_cnt = 0;
  mpu_hw.mutex = mpu_hw.fake = FALSE;
  mpu_hw.timeout_drr = mpu_hw.timeout_dsr = mpu_hw.loops_drr = mpu_hw.loops_dsr = 0;

  mpu_ver.version = 1;
  mpu_ver.subversion = 5;
  mpu_ver.revision[0] = 0;

  drv_ver.version = MPU_VERSION;
  drv_ver.subversion = MPU_SUBVERSION;
  strncpy(drv_ver.revision, MPU_REVISION, sizeof(drv_ver.revision)-1);

  mpu_status = MPU_STA_NONE;
  mpu_debug = MPU_DBG_NONE;
  mpu_error = MPU_ERR_NONE;

  midi_queue_reset(&mpu_queues.voice);
  midi_queue_reset(&mpu_queues.control);
  midi_queue_reset(&mpu_queues.record);
  midi_queue_reset(&mpu_immediate_queue);

  mpu_queues.voice_time = mpu_queues.record_time = mpu_queues.control_time = 0;
  mpu_queues.kmallocs = mpu_queues.kfrees = 0;
  mpu_queues.voice_cnt = mpu_queues.sysex_cnt = mpu_queues.meta_cnt = 0;
  mpu_queues.nop_cnt = mpu_queues.voice_hits = 0;

  mpu_blocking.play_end = mpu_blocking.play_delay = NULL;
  mpu_blocking.record = mpu_blocking.demand = mpu_blocking.sync= NULL;
  mpu_blocking.bias.voice = 0;
  mpu_blocking.bias.control = 0;
  mpu_blocking.bias.record = 0;
  mpu_blocking.bias.sync = OFF;
  mpu_blocking.bias.message = MPU_BLK_NONE;

  metronome.status = OFF;

  metronome.setup.numerator = 4;
  metronome.setup.denominator = 4;
  metronome.setup.channel = 9; /* yes, it *is* channel 10 ;-) */
  metronome.setup.click_key = 33; /* GS metronome click */
  metronome.setup.click_velo = 100;
  metronome.setup.acc_click_key = 33; /* GS metronome click */
  metronome.setup.acc_click_velo = 127;
  metronome.setup.meas_click_key = 34; /* GS metronome bell */
  metronome.setup.meas_click_velo = 127;

  metronome.time = 0;
  metronome.subtime = 0;
  metronome.timebase = 120;
  metronome.bpm = 100;
  metronome.mysectempo = 60000;
  metronome.rel_bpm = 0x40;
  metronome.grad_bpm = 00;
  metronome.midi_metro = 24;
  metronome.int_clk2host = 240;

  init_switches.all_notes_off = ON;
  init_switches.no_real_time = OFF;
  init_switches.all_thru = ON;
  init_switches.dummy_time = ON;
  init_switches.mode_mess = OFF;
  init_switches.SysEx_thru = ON;
  init_switches.common2host = OFF;
  init_switches.realtime2host = OFF;
  init_switches.uart_mode = OFF;

  channeltables.a_ch = 0;
  channeltables.b_ch = 1;
  channeltables.c_ch = 2;
  channeltables.d_ch = 3;

  switches.int_clk = ON;
  switches.fsk_clk = switches.mid_clk = OFF;

  switches.metronome = OFF;
  switches.bender = ON;
  switches.midi_thru = ON;
  switches.step_rec = OFF;
  switches.send_meas = ON;
  switches.conductor = ON;
  switches.realtime = OFF;
  switches.fsk_to_midi = OFF;
  switches.clk2host = OFF;
  switches.SysEx2host = OFF;

  switches.ch_ref_a = switches.ch_ref_b = switches.ch_ref_c = switches.ch_ref_d = OFF;

  tracks.channelmap = 0xffff;
  tracks.send_play_cnt = ON;

  runmsg.play = 0;
  runmsg.record = 0;
  runmsg.control_size = 0;
  runmsg.control = FALSE;
}

static char mpu_reset(char update)
{
  char isopen = mpu_status & MPU_STA_OPEN;
  word i = 0;

#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_reset);
#endif

  if (mpu_debug > MPU_DBG_SPARSE) printk("MPU-401: mpu_reset()\n");

  mpu_wake_up(&mpu_blocking.record, MPU_BLK_END);
  mpu_wake_up(&mpu_blocking.play_end, MPU_BLK_END);
  mpu_wake_up(&mpu_blocking.play_delay, MPU_BLK_END);
  mpu_wake_up(&mpu_blocking.demand, MPU_BLK_END);
  mpu_wake_up(&mpu_blocking.sync, MPU_BLK_END);

  mpu_disable_irq();

  mpu_blocking.bias.voice = 0;
  mpu_blocking.bias.control = 0;
  mpu_blocking.bias.record = 0;
  mpu_blocking.bias.sync = OFF;
  mpu_blocking.bias.message = MPU_BLK_NONE;

  if (!mpu_SnCmd(MPU_CLR_RESET))
    {
      printk("MPU-401: reset command failed.\n");
      return FALSE;
    };

  mpu_status = (isopen) ? MPU_STA_OPEN : MPU_STA_NONE;
  mpu_error = MPU_ERR_NONE;

  runmsg.control = FALSE;
  mpu_clear_run();

  if (update)
    {
      /*******************************************************
       * !setup the MPU according the settings in mpu_setup! *
       *******************************************************/

      /* Persistent (until next MPU_RESET that is) MPU switches */
      if (!init_switches.all_notes_off) mpu_SnCmd(MPU_INI_NOTES_OFF_ALL);
      if (init_switches.no_real_time) mpu_SnCmd(MPU_INI_NO_REAL_TIME);
      if (!init_switches.all_thru) mpu_SnCmd(MPU_INI_ALL_THRU_OFF);
      if (init_switches.dummy_time) mpu_SnCmd(MPU_INI_TIMING_BYTE_ON);
      if (init_switches.mode_mess) mpu_SnCmd(MPU_INI_MODE_MESS_ON);
      if (init_switches.SysEx_thru) mpu_SnCmd(MPU_INI_EXL_THRU_ON);
      if (init_switches.common2host) mpu_SnCmd(MPU_INI_COMMON_TO_HOST);
      if (init_switches.realtime2host) mpu_SnCmd(MPU_INI_REAL_TIME);
      if (init_switches.uart_mode) mpu_SnCmd(MPU_INI_UART_MODE);

      /* MPU-401 run time switches */
      if (switches.int_clk)      mpu_SnCmd(MPU_CLK_INT_CLOCK);
      else if (switches.fsk_clk) mpu_SnCmd(MPU_CLK_FSK_CLOCK);
      else if (switches.mid_clk) mpu_SnCmd(MPU_CLK_MIDI_CLOCK);

      mpu_SnCmd(MPU_SET_METRO_OFF); /* always turn off the internal MPU metronome */
      mpu_SnCmd(MPU_SWI_BENDER + switches.bender);
      mpu_SnCmd(MPU_SWI_MIDI_THRU + switches.midi_thru);
      mpu_SnCmd(MPU_SWI_DATA_IN_STOP + switches.step_rec);
      mpu_SnCmd(MPU_SWI_SEND_MEAS_END + switches.send_meas);
      mpu_SnCmd(MPU_SWI_CONDUCTOR + switches.conductor);
      mpu_SnCmd(MPU_SWI_REAL_TIME_AFF + switches.realtime);
      mpu_SnCmd(MPU_CLK_FSK_TO_INT + switches.fsk_to_midi);
      mpu_SnCmd(MPU_SWI_CLK_TO_HOST/*+ switches.clk2host*/); /* always OFF */
      mpu_SnCmd(MPU_SWI_EXL_TO_HOST + switches.SysEx2host);

      /* channel tables */
      mpu_SnCmd(MPU_SET_CH_REF_TBL_A + switches.ch_ref_a);
      mpu_SnCmd(MPU_SET_CH_REF_TBL_B + switches.ch_ref_b);
      mpu_SnCmd(MPU_SET_CH_REF_TBL_C + switches.ch_ref_c);
      mpu_SnCmd(MPU_SET_CH_REF_TBL_D + switches.ch_ref_d);

      /* metronome */
      mpu_SnCmdDta(MPU_SET_TEMPO, metronome.bpm);
      mpu_SnCmdDta(MPU_SET_METRO_MEAS, metronome.setup.numerator);
      mpu_SnCmdDta(MPU_SET_REL_TEMPO, metronome.rel_bpm);
      mpu_SnCmdDta(MPU_SET_GRADUATION, metronome.grad_bpm);
      mpu_SnCmdDta(MPU_SET_MIDI_METRO, metronome.midi_metro);
      mpu_SnCmdDta(MPU_SET_INT_HOST, metronome.int_clk2host);

      /* setup timebase */
      i = metronome.timebase;
      if (i % 24 || (i < 48) || (i > 192))
	{
	  i = 120;
	  printk("MPU-401: Invalid timebase %d\n", metronome.timebase);
	};
      mpu_SnCmd( MPU_SET_TIME_BASE + (i / 24) - 2 );

      /* channel and track management */
      mpu_SnCmdDta(MPU_SET_CH_1_8, tracks.channelmap&0xff);
      mpu_SnCmdDta(MPU_SET_CH_9_16, (tracks.channelmap>>8)&0xff);
    };

  mpu_enable_irq();
  return TRUE;
}

/* detect calls sleep, so *never* call it from an interrupt! */
static char mpu_detect(void)
{
  unsigned long ic = 0;
  int i = 0;

#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_detect);
#endif

  if (mpu_debug > MPU_DBG_NONE) printk("MPU-401: mpu_detect()\n");

  /* First see if a RESET commands behaves well (verify I/O port setting) */
  if (!mpu_reset(FALSE)) return FALSE;

  /* Get MPU-401 version information */
  if (mpu_debug > MPU_DBG_SPARSE) printk("         getting mpu version\n");
  if ( (i = mpu_RqDta(MPU_REQ_VERSION)) < 0 )
    {
      printk("MPU-401: request version (%02x) failed.\n", MPU_REQ_VERSION);
      return FALSE;
    };
  if (i < 0x14)
    {
      printk("MPU-401: This MPU-401 device is older than the 1.4 version\n");
      printk("         and may therefore not work reliably in all situations\n");
    };
  mpu_ver.version = (i&0xf0) >> 4;
  mpu_ver.subversion = i&0x0f;
  if (mpu_debug > MPU_DBG_SPARSE) printk("         getting mpu revision\n");
  if ( (i = mpu_RqDta(MPU_REQ_REVISION)) < 0)
    {
      printk("MPU-401: Request revision (%02x) failed.\n", MPU_REQ_REVISION);
      return FALSE;
    };
  mpu_ver.revision[0] = (char)((i) ? i-1+'A' : 0);
  mpu_ver.revision[1] = (char)0;

  /* Verify IRQ setting */
  if (mpu_debug > MPU_DBG_SPARSE) printk("         verifying irq setting\n");
  if (!mpu_reset(TRUE)) return FALSE;

  mpu_SnCmdDta(MPU_SET_TRACK_ACTIVE, 1);
  mpu_SnCmd(MPU_CLR_PLAY_CNT);
  mpu_SnCmd(MPU_SEQ_START_PLAY);

  mpu_status |= MPU_STA_PLAY;

  ic = mpu_hw.real_irq_cnt;
  for (i = 0; i < IRQ_PROBE_RETRY; i++)
    {
      if (mpu_debug > MPU_DBG_SPARSE) printk("         irq test loop #%d\n", i+1);
      mpu_msdelay(20); /* sleep 20 milliseconds, hopefully we've triggered an interrupt! */
      if (ic < mpu_hw.real_irq_cnt) break;
    };

  mpu_status &= ~MPU_STA_PLAY;
  mpu_SnCmd(MPU_CLR_RESET);

  if (ic == mpu_hw.real_irq_cnt)
    {
      printk("MPU-401: Wrong IRQ setting (%hd)\n", mpu_hw.irq);
      return FALSE;
    };

  printk("MPU-401 v%hd.%hd%s detected at IRQ 0x%02x I/O port 0x%04x\n",
	 mpu_ver.version, mpu_ver.subversion, mpu_ver.revision, mpu_hw.irq, mpu_hw.ioport);

  mpu_reset(TRUE);
  return TRUE;
}

/*** DEVICE INTERFACE *********************************************************
 *
 * int mpu_open(struct inode * inode, struct file * filp)
 *    Opens and initializes the MPU-401 interface and returns succes conforming to 
 *    <linux/errno.h>.
 *
 * void mpu_release(struct inode * inode, struct file * filp)
 *    Resets and releases the MPU-401 interface.
 *
 * int mpu_ioctl(struct inode * inode, struct file * filp, unsigned int cmd, char * argp)
 *    Tell the driver what's going on ;-)
 *
 * int mpu_register_device()
 *    This routine registers the character device used for communication with the
 *    device driver.
 *
 * void mpu_unregister_device()
 *    Take a guess ;).
 *
 ******************************************************************************/

static int mpu_ioctl_guest(struct inode * inode, struct file * filp, word request, unsigned long argp)
{
  mpu_status_t mpu_stat;

#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_ioctl_guest);
#endif

  if ((char)IOCGROUP(request) != 'M')
    {
      printk("MPU-401: Unknown ioctl request, 0x%08x (bad ``magic'' ID)\n", request);
      return -EINVAL;
    };

  switch (request)
    {
    case MPUIOC_BLOCK_SYNC: /* _IOR long */
      if (!argp || !(mpu_status&(MPU_STA_PLAY|MPU_STA_REC))) return -EINVAL;
      mpu_sleep(&mpu_blocking.sync);
      put_fs_word(metronome.time, (long *)argp);
      break;

    case MPUIOC_GET_STATUS: /* _IOR mpu_setup_t */
      if (!argp) return -EINVAL;

      mpu_stat.hardware = mpu_hw;
      mpu_stat.mpu_ver = mpu_ver;
      mpu_stat.drv_ver = drv_ver;

      mpu_stat.status = mpu_status;
      mpu_stat.debug = mpu_debug;
      mpu_stat.error = mpu_error;

      mpu_stat.queues = mpu_queues;
      mpu_stat.blocking = mpu_blocking;

      mpu_stat.metronome = metronome;
      mpu_stat.init_switches = init_switches;
      mpu_stat.switches = switches;

      memcpy_tofs((mpu_status_t *)argp, &mpu_stat, sizeof(mpu_status_t));
      break;

    case MPUIOC_SET_DEBUG_LVL: /* _IOW('M', 0xf1, word) */
      if (!argp) return -EINVAL;
      mpu_debug = get_fs_word((word *)argp);
      break;

    case MPUIOC_GET_DEBUG_LVL: /* _IOR('M', 0xf2, word) */
      if (!argp) return -EINVAL;
      put_fs_word(mpu_debug, (word *)argp);
      break;

    default:
      return -EINVAL;
    };
  return 0;
}

static int mpu_ioctl(struct inode * inode, struct file * filp, word request, unsigned long argp)
{
  static midi_event event;
  static mpu_status_t mpu_stat;

  midi_event * pevent = NULL; 
  midi_event * tevent = NULL;
  byte * data = NULL;
  int res = 0;
  word i = 0;
  long j = 0;

#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_ioctl);
#endif

  if ( MINOR(inode->i_rdev)==MPU_DEV_STATUS ) return mpu_ioctl_guest(inode, filp, request, argp);
  if ((char)IOCGROUP(request) != 'M')
    {
      printk("MPU-401: Unknown ioctl request, 0x%08x (bad ``magic'' ID)\n", request);
      return -EINVAL;
    };

  switch (request)
    {
    case MPUIOC_RESET: /* _IO */
      mpu_disable_irq();
      mpu_flush_queues();
      mpu_reset(TRUE);
      mpu_enable_irq();
      break;

      /*** SEQUENCER CONTROL ***/

    case MPUIOC_START_PLAY: /* _IOW long */
      if (!argp) return -EINVAL;

      if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: MPUIOC_START_PLAY\n");

      if (switches.step_rec)
	{
	  switches.step_rec = OFF;
	  mpu_SnCmd(MPU_SWI_DATA_IN_STOP + switches.step_rec);
	  mpu_status &= ~(MPU_STA_STEPREC);
	};

      if ( !(mpu_status & MPU_STA_REC) ) mpu_reset(TRUE);
      mpu_disable_irq();

      j = get_fs_long((long *)argp);
      if (j % metronome.timebase) j -= j % metronome.timebase;
      if (j < 0) j = 0;
      mpu_queues.voice_time = mpu_queues.control_time = j;
      mpu_queues.record_time = metronome.time = metronome.subtime = j;

      mpu_SnCmdDta(MPU_SET_TRACK_ACTIVE, (1<<MPU_TRK_VOICE)|(1<<MPU_TRK_TIMER));
      /* mpu_SnCmdDta(MPU_SET_SEND_PLAY_CNT, (1<<MPU_TRK_VOICE)|(1<<MPU_TRK_TIMER)); */
      mpu_SnCmd(MPU_CLR_PLAY_CNT);

      if (mpu_status & MPU_STA_REC) mpu_SnCmd(MPU_SEQ_START_OVERDUB);
      else mpu_SnCmd(MPU_SEQ_START_PLAY);

      mpu_status |= MPU_STA_PLAY;
      mpu_enable_irq();
      break;

    case MPUIOC_STOP_PLAY: /* _IO */
      mpu_disable_irq();

      mpu_status &= ~(MPU_STA_PLAY);
      mpu_SnCmdDta(MPU_SET_TRACK_ACTIVE, 0);

      if (mpu_status & MPU_STA_REC) mpu_SnCmd(MPU_SEQ_START_REC);
      else mpu_SnCmd(MPU_SEQ_STOP_PLAY);
      mpu_enable_irq();
      break;

    case MPUIOC_START_OVERDUB: /* _IOW long */
      if (!argp) return -EINVAL;

      if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: MPUIOC_START_OVERDUB\n");

      if (switches.step_rec)
	{
	  switches.step_rec = OFF;
	  mpu_SnCmd(MPU_SWI_DATA_IN_STOP + switches.step_rec);
	  mpu_status &= ~(MPU_STA_STEPREC);
	};

      if ( !(mpu_status & (MPU_STA_PLAY|MPU_STA_REC)) ) mpu_reset(TRUE);
      mpu_disable_irq();

      j = get_fs_long((long *)argp);
      if (j % metronome.timebase) j -= j % metronome.timebase;
      if (j < 0) j = 0;
      mpu_queues.voice_time = mpu_queues.control_time = j;
      mpu_queues.record_time = metronome.time = metronome.subtime = j;

      mpu_SnCmdDta(MPU_SET_TRACK_ACTIVE, (1<<MPU_TRK_VOICE)|(1<<MPU_TRK_TIMER));
      mpu_SnCmd(MPU_CLR_PLAY_CNT);
      mpu_SnCmd(MPU_SEQ_START_OVERDUB);

      mpu_status |= MPU_STA_PLAY|MPU_STA_REC;
      mpu_enable_irq();
      break;

    case MPUIOC_STOP_OVERDUB: /* _IO */
      mpu_disable_irq();
      mpu_status &= ~(MPU_STA_PLAY|MPU_STA_REC);
      mpu_SnCmdDta(MPU_SET_TRACK_ACTIVE, 0);
      mpu_SnCmdDta(MPU_SET_SEND_PLAY_CNT, 0);
      mpu_SnCmd(MPU_SEQ_STOP_OVERDUB);
      mpu_enable_irq(); 
      break;

    case MPUIOC_START_RECORD: /* _IOW long */
      if (!argp) return -EINVAL;

      if (mpu_debug == MPU_DBG_ALL) printk("MPU-401: MPUIOC_START_RECORD\n");

      if (switches.step_rec)
	{
	  switches.step_rec = OFF;
	  mpu_SnCmd(MPU_SWI_DATA_IN_STOP + switches.step_rec);
	  mpu_status &= ~(MPU_STA_STEPREC);
	};

      if ( !(mpu_status & MPU_STA_PLAY) ) mpu_reset(TRUE);
      mpu_disable_irq();

      j = get_fs_long((long *)argp);
      if (j % metronome.timebase) j -= j % metronome.timebase;
      if (j < 0) j = 0;
      mpu_queues.voice_time = mpu_queues.control_time = j;
      mpu_queues.record_time = metronome.time = metronome.subtime = j;

      if ( !(mpu_status & MPU_STA_PLAY) )
	{
	  mpu_SnCmdDta(MPU_SET_TRACK_ACTIVE, 1<<MPU_TRK_TIMER);
	  mpu_SnCmd(MPU_CLR_PLAY_CNT);
	}
      else
	{
	  mpu_SnCmdDta(MPU_SET_TRACK_ACTIVE, (1<<MPU_TRK_VOICE)|(1<<MPU_TRK_TIMER));
	  mpu_SnCmd(MPU_CLR_PLAY_CNT);
	};
      mpu_SnCmd(MPU_SEQ_START_OVERDUB);

      mpu_enable_irq();
      break;

    case MPUIOC_STOP_RECORD: /* _IO */
      mpu_disable_irq();

      mpu_status &= ~(MPU_STA_REC);
      if (mpu_status & MPU_STA_PLAY) mpu_SnCmd(MPU_SEQ_START_PLAY);
      else mpu_SnCmd(MPU_SEQ_STOP_OVERDUB);

      mpu_enable_irq(); 
      break;

    case MPUIOC_START_STEPREC: /* _IO */
      if (mpu_status & (MPU_STA_PLAY|MPU_STA_REC)) return -EINVAL;
      mpu_disable_irq();
      switches.step_rec = ON;
      mpu_SnCmd(MPU_SWI_DATA_IN_STOP + switches.step_rec);
      mpu_status |= MPU_STA_STEPREC;
      mpu_enable_irq();
      break;
    case MPUIOC_STOP_STEPREC: /* _IO */
      if (!(mpu_status & MPU_STA_STEPREC)) return -EINVAL;
      mpu_disable_irq();
      switches.step_rec = OFF;
      mpu_SnCmd(MPU_SWI_DATA_IN_STOP + switches.step_rec);
      mpu_status &= ~(MPU_STA_STEPREC);
      mpu_enable_irq();
      break;

    case MPUIOC_STOP_ALL: /* _IO */
      mpu_reset(TRUE);
      break;

    case MPUIOC_BLOCK_PLAY_END: /* _IO */
      mpu_blocking.bias.message = MPU_BLK_NONE;
      if (mpu_status & MPU_STA_PLAY) mpu_sleep(&mpu_blocking.play_end);
      else return -EINVAL;
      break;

    case MPUIOC_BLOCK_DELAY: /* _IO */
      mpu_blocking.bias.message = MPU_BLK_NONE;
      if (mpu_status & (MPU_STA_PLAY|MPU_STA_REC)) mpu_sleep(&mpu_blocking.play_delay);
      else return -EINVAL;
      break;

    case MPUIOC_BLOCK_DEMAND: /* _IOWR mpu_demand_t */
      if (!argp) return -EINVAL;

      mpu_disable_irq();
      memcpy_fromfs(&mpu_blocking.bias, (mpu_demand_t *)argp, sizeof(mpu_demand_t));
      mpu_blocking.bias.message = MPU_BLK_NONE;
      mpu_enable_irq();

      /* if all blocking limits are zero, then we don't need to block */
      if (!(mpu_blocking.bias.voice || mpu_blocking.bias.control ||
	    mpu_blocking.bias.record || mpu_blocking.bias.sync))
	{
	  return -EINVAL;
	};

      /* determine what kind of blocking strategy to apply */
      if (mpu_status&(MPU_STA_PLAY|MPU_STA_REC)) mpu_sleep(&mpu_blocking.demand);
      else if ((mpu_status & MPU_STA_REC) && mpu_blocking.bias.record)
	{
	  mpu_sleep(&mpu_blocking.record);
	}
      else return -EINVAL;
      mpu_blocking.bias.sync = OFF; /* this must be turned on explicitly */
      memcpy_tofs((mpu_demand_t *)argp, &mpu_blocking.bias, sizeof(mpu_demand_t));
      break;

    case MPUIOC_BLOCK_SYNC: /* _IOR long */
      if (!argp || !(mpu_status&(MPU_STA_PLAY|MPU_STA_REC))) return -EINVAL;
      mpu_sleep(&mpu_blocking.sync);
      put_fs_word(metronome.time, (long *)argp);
      break;

    case MPUIOC_METRONOME_ON: /* _IO */
      metronome.status = ON;
      break;

    case MPUIOC_METRONOME_OFF: /* _IO */
      metronome.status = OFF;
      break;

    case MPUIOC_SET_METRONOME: /* _IOW mpu_metrosetup_t */
      if (!argp) return -EINVAL;
      memcpy_fromfs(&metronome.setup, (mpu_metrosetup_t *)argp, sizeof(mpu_metrosetup_t));

      /* validate the data parameters */
      if (metronome.setup.numerator < 1 || metronome.setup.numerator > 32) 
	metronome.setup.numerator = 4;
      switch (metronome.setup.denominator)
	{
	case 1:
	case 2:
	case 4:
	case 8:
	case 16:
	case 32:
	  break;
	default:
	  metronome.setup.denominator = 4;
	};
      if (metronome.setup.channel > 15) metronome.setup.channel = 9;
      if (metronome.setup.click_key > 127) metronome.setup.click_key = 33;
      if (metronome.setup.click_velo > 127) metronome.setup.click_key = 100;
      if (metronome.setup.acc_click_key > 127) metronome.setup.click_key = 33;
      if (metronome.setup.acc_click_velo > 127) metronome.setup.click_key = 127;
      if (metronome.setup.meas_click_key > 127) metronome.setup.click_key = 34;
      if (metronome.setup.meas_click_velo > 127) metronome.setup.click_key = 127;
      break;

    case MPUIOC_GET_METRONOME:/* _IOR mpu_metrosetup_t */
      if (!argp) return -EINVAL;
      memcpy_tofs((mpu_metrosetup_t *)argp, &metronome.setup, sizeof(mpu_metrosetup_t));
      break;

    case MPUIOC_SET_TIMEBASE: /* _IOW word */
      if (!argp) return -EINVAL;

      /* setup timebase */
      metronome.timebase = get_fs_word((word *)argp);
      i = metronome.timebase;
      if (i % 24 || (i < 48) || (i > 192))
	{
	  printk("MPU-401: Invalid timebase %d\n", metronome.timebase);
	  return -EINVAL;
	};
      mpu_SnCmd( MPU_SET_TIME_BASE + (i / 24) - 2 );
      break;

    case MPUIOC_GET_TIMEBASE:/* _IOR word */
      if (!argp) return -EINVAL;
      put_fs_word(metronome.timebase, (word *)argp);
      break;

    case MPUIOC_SET_NUMERATOR: /* _IOW byte */
      if (!argp) return -EINVAL;
      metronome.setup.numerator = get_fs_byte((byte *)argp);
      break;

    case MPUIOC_GET_NUMERATOR:/* _IOR byte */
      if (!argp) return -EINVAL;
      put_fs_word(metronome.setup.numerator, (byte *)argp);
      break;

    case MPUIOC_SET_DENOMINATOR: /* _IOW byte */
      if (!argp) return -EINVAL;
      metronome.setup.denominator = get_fs_byte((byte *)argp);
      break;

    case MPUIOC_GET_DENOMINATOR:/* _IOR byte */
      if (!argp) return -EINVAL;
      put_fs_word(metronome.setup.denominator, (byte *)argp);
      break;

    case MPUIOC_SET_TEMPO: /* _IOW byte */
      if (!argp) return -EINVAL;
      metronome.bpm = get_fs_byte((byte *)argp);
      break;

    case MPUIOC_GET_TEMPO:/* _IOR byte */
      if (!argp) return -EINVAL;
      put_fs_word(metronome.bpm, (byte *)argp);
      break;

      /*** EVENT CONTROL ***/
	
    case MPUIOC_PUT_IMMEDIATE: /* _IOW midi_event */
      if ( !argp ) return -EINVAL;
	
      memcpy_fromfs(&event, (midi_event *)argp, sizeof(midi_event));
	
      if ( !event.size ) return -EINVAL;
      if ( !event.data ) return -EINVAL;

      data = mpu_kmalloc(event.size);
      if (!data) return -EINVAL;
      memcpy_fromfs(data, event.data, event.size);
      res = 0;

      mpu_disable_irq();

      if (!(mpu_status & (MPU_STA_PLAY|MPU_STA_REC)))
	{
	  mpu_clear_run_play();
	  mpu_SnCmd(MPU_CLR_PLAY_MAP);
	  if ( event.type == MIDI_TYP_VOICE ) mpu_SnCmd(MPU_SND_DATA | 1<<MPU_TRK_IMMEDIATE);
	  else if ( event.type == MIDI_TYP_SYSEX ) mpu_SnCmd(MPU_SND_SYSEX);
	  else
	    {
	      printk("MPU-401: Attempt to send CONTROL event immediate\n");
	      res = -EINVAL;
	    };
	  if (!res) for (i = 0; i < event.size; i++) mpu_SnDta(data[i]);
	  mpu_kfree_s(data, event.size);
	}
      else
	{
	  midi_queue_put_event(&mpu_immediate_queue, event.type, 0, event.size, data);
	};
      mpu_enable_irq();
      return res;

    case MPUIOC_PUT_EVENT: /* _IOW midi_event */
      if ( !argp ) return -EINVAL;
      if ( !mpu_can_store() ) return -EBUSY;

      memcpy_fromfs(&event, (midi_event *)argp, sizeof(midi_event));
	
      if ( !event.size ) return -EINVAL;
      if ( !event.data ) return -EINVAL;
	
      if ( !(data = (byte *)mpu_kmalloc(event.size)) ) return -ENOMEM;
      else memcpy_fromfs(data, event.data, event.size);

      mpu_queues.voice.current = NULL;
      mpu_queues.control.current = NULL;
      mpu_queues.record.current = NULL;

      if (mpu_debug == MPU_DBG_ALL)
	printk("MPU-401: PUT EVENT TYPE %d TIME %ld\n", event.type, event.time);
	
      mpu_disable_irq();
      if ( event.type == MIDI_TYP_VOICE )
	midi_queue_put_event(&mpu_queues.voice, event.type, event.time, event.size, data);
      else
	midi_queue_put_event(&mpu_queues.control, event.type, event.time, event.size, data);
      mpu_enable_irq();
      break;

    case MPUIOC_GET_EVENT_SIZE: /* _IOR long */
      if (!argp) return -EINVAL;

      if ((pevent = midi_queue_peek(&mpu_queues.record)))
	put_fs_long(pevent->size, (long *)argp);
      else
	put_fs_long(0, (long *)argp);
      break;
	
    case MPUIOC_GET_EVENT: /* _IOWR midi_event */
      if ( !argp ) return -EINVAL;
	
      pevent = (midi_event *)argp; /* DESTINATION */
      memcpy_fromfs(&event, pevent, sizeof(midi_event)); /* DESTINATION */
      tevent = midi_queue_peek(&mpu_queues.record); /* SOURCE */
	
      if ( !mpu_queues.record.count || !tevent || (tevent->size > event.size) )
	{
	  put_fs_word(0, &pevent->size);
	  return 0;
	};

      mpu_disable_irq();
      tevent = midi_queue_get(&mpu_queues.record); /* SOURCE */

      event.type = tevent->type;
      event.time = tevent->time;
      event.size = tevent->size;
      memcpy_tofs(event.data, tevent->data, tevent->size);
      memcpy_tofs(pevent, &event, sizeof(midi_event));
      midi_event_free(tevent);
      mpu_enable_irq();
      break;

    case MPUIOC_GM_ON: /* _IO */
      runmsg.control = FALSE;
      mpu_clear_run_play();

      mpu_send_sysex(sizeof(gm_reset), gm_reset);
      break;
    case MPUIOC_GS_RESET: /* _IO */
      runmsg.control = FALSE;
      mpu_clear_run_play();

      mpu_send_sysex(sizeof(gs_reset1), gs_reset1);
      mpu_send_sysex(sizeof(gs_reset2), gs_reset2);
      break;

    case MPUIOC_GET_STATUS: /* _IOR mpu_setup_t */
      if (!argp) return -EINVAL;

      mpu_stat.hardware = mpu_hw;
      memcpy(&mpu_stat.mpu_ver, &mpu_ver, sizeof(mpu_ver));
      memcpy(&mpu_stat.drv_ver, &drv_ver, sizeof(drv_ver));

      mpu_stat.status = mpu_status;
      mpu_stat.debug = mpu_debug;
      mpu_stat.error = mpu_error;

      mpu_stat.queues = mpu_queues;
      mpu_stat.blocking = mpu_blocking;

      mpu_stat.metronome = metronome;
      mpu_stat.init_switches = init_switches;
      mpu_stat.switches = switches;

      memcpy_tofs((mpu_status_t *)argp, &mpu_stat, sizeof(mpu_status_t));
      break;

    case MPUIOC_SET_DEBUG_LVL: /* _IOW word */
      if (!argp) return -EINVAL;
      mpu_debug = get_fs_word((word *)argp);
      break;

    case MPUIOC_GET_DEBUG_LVL: /* _IOR word */
      if (!argp) return -EINVAL;
      put_fs_word(mpu_debug, (word *)argp);
      break;

    default:
      printk("MPU-401: Unknown MPU-401 device driver ioctl request 0x%04x.\n", request);
      return -EINVAL;
    };
  
  return 0;
}

static int mpu_open(struct inode * inode, struct file * filp)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_open);
#endif
  if ( MINOR(inode->i_rdev) == MPU_DEV_RW )
    {
      if ( mpu_status & MPU_STA_OPEN ) return -EBUSY;
      else if (!mpu_request_irq()) return -EBUSY;
      else if (!mpu_reset(TRUE))
	{
	  mpu_free_irq();
	  return -EIO;
	}
      else
	{
	  mpu_status = MPU_STA_OPEN;
	  MOD_INC_USE_COUNT;
	};
    }
  else if ( MINOR(inode->i_rdev) == MPU_DEV_STATUS )
    {
      /* anonymous login. Restrictions apply ;-) */
      MOD_INC_USE_COUNT;
    }
  else
    return -ENXIO;

  return 0;
}

static void mpu_release(struct inode * inode, struct file * filp)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_release);
#endif
  if ( MINOR(inode->i_rdev) == MPU_DEV_RW )
    {
      mpu_reset(TRUE);
      mpu_kill_synth();

      mpu_free_irq();
      mpu_status &= ~MPU_STA_OPEN;
    };
  MOD_DEC_USE_COUNT;
}

#ifdef DEVELOP
static int mpu_read(struct inode * inode, struct file * file, char * buf, int count)
{
  char didwrite = 0;
  long i = 0;
  int k = 0;
  byte l = 0;
  static int j = 0;
  static char s[160];

  if ( !inode || !file || !buf || !count ) return -EINVAL;
  if ( (MINOR(inode->i_rdev) != MPU_DEV_STATUS) ) return -EINVAL;

  if (j)
    {
      k = strlen(&s[j]);
      memcpy_tofs(buf, &s[j], (k > count) ? count : k);
       if (k > count)
	{
	  j += count;
	  file->f_pos += count;
	  return count;
	}
      j = 0;
      count = k;
      file->f_pos += k;
      didwrite = 1;
    };

  if (!j && (i = mpu_action_queue_get(&actionq)) != 0 )
    {
      l = (i >> 8) & 0xff;
      sprintf(&s[0], "%20s %c%c%c%c%c%c%c %20s %03d\n",
	      mpu_action_names[i&0xff],
	      (l & MPU_FLG_IRQ) ? 'I' : '-', /* irq */
	      (l & MPU_FLG_FAK) ? 'F' : '-', /* fake irq */
	      (l & MPU_FLG_DEM) ? 'D' : '-', /* demand block */
	      (l & MPU_FLG_END) ? 'E' : '-', /* play end block */
	      (l & MPU_FLG_DEL) ? 'D' : '-', /* play delay block */
	      (l & MPU_FLG_REC) ? 'R' : '-', /* record block */
	      (l & MPU_FLG_SYN) ? 'S' : '-', /* sync block */
	      mpu_error_names[(i>>16)&0xff],
	      (i >> 24) & 0xff
	      );
    };

  if (!didwrite)
    {
      j = 0;
      k = strlen(&s[j]);
      memcpy_tofs(buf, &s[j], (k > count) ? count : k);
      if (k > count) j += count;
      else count = k;
    };
  return count;
}
#endif

static struct file_operations mpu_fops = 
{
  NULL,        /* mpu_lseek */ 
#ifdef DEVELOP
  mpu_read,
#else
  NULL,        /* mpu_read */
#endif
  NULL,        /* mpu_write */
  NULL,        /* mpu_readdir */
  NULL,        /* mpu_select */
  mpu_ioctl,
  NULL,        /* mpu_mmap */
  mpu_open,
  mpu_release
};

static int mpu_register_device(void)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_register_device);
#endif
  if (mpu_debug > MPU_DBG_NONE) printk("MPU-401: mpu_register_device()\n");
  if (register_chrdev(MPU_DEV_MAJOR, "mpu401", &mpu_fops))
    {
      printk("MPU-401: Unable to register sound character device \"mpu401\"\n");
      return FALSE;
    };
  return TRUE;
}

static void mpu_unregister_device(void)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_unregister_device);
#endif
  if (mpu_debug > MPU_DBG_NONE) printk("MPU-401: mpu_unregister_device()\n");
  if (unregister_chrdev(MPU_DEV_MAJOR, "mpu401")) 
    {
      printk("MPU-401: Unable to unregister sound character device \"mpu401\"\n");
    };
}

/*** RUNTIME CONTROL **********************************************************
 *
 * void mpu_get_rc()
 *
 *    This routine looks after a file named `/var/lib/mpu.conf'
 *
 ******************************************************************************/

static struct inode * mpu_inode = NULL;
static struct file mpu_file;
static unsigned short mpu_fs = 0;

static void mpu_end_read(void)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_end_read);
#endif
  set_fs(mpu_fs);
  if (mpu_inode) iput(mpu_inode);
}

static void mpu_close_read(void)
{
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_close_read);
#endif
  if (mpu_file.f_op->release) mpu_file.f_op->release(mpu_inode, &mpu_file);
  mpu_end_read();
}

static int mpu_get_conf(void)
{
  char buf[8];
    
#ifdef DEVELOP
  mpu_action_queue_put(&actionq, MPU_ACT_get_conf);
#endif

  if (mpu_debug > MPU_DBG_NONE)
    {
      printk("MPU-401: mpu_get_conf() configuration file is %s\n", MPU_RC_FILE);
    };
    
  mpu_fs = get_fs();
    
  if ( (open_namei(MPU_RC_FILE, 1, S_IRUGO, &mpu_inode, NULL)) )
    {
      printk("MPU-401: couldn't open config file\n");
      mpu_inode = NULL;
      mpu_end_read();
      return FALSE;
    };

  if (mpu_inode->i_size != 8)
    {
      printk("MPU-401: Invalid config file format (please use mpuconfig).\n");
      mpu_end_read();
      return FALSE;
    }

  if (!S_ISREG(mpu_inode->i_mode)) /* is it a regular file? */
    {
      printk("MPU-401: config file is not a regular file.\n");
      mpu_end_read();
      return FALSE;
    }

  if (!mpu_inode->i_op || !mpu_inode->i_op->default_file_ops) /* does it have file operations? */
    {
      printk("MPU-401: cannot read config file (i_op is NULL).\n");
      mpu_end_read();
      return FALSE;
    }

  mpu_file.f_mode = 1; /* 1 = read acces */
  mpu_file.f_flags = 0;
  mpu_file.f_count = 1;
  mpu_file.f_inode = mpu_inode;
  mpu_file.f_pos = 0;
  mpu_file.f_reada = 0;
  mpu_file.f_op = mpu_inode->i_op->default_file_ops;

  /* can we open this file? */
  if ( mpu_file.f_op->open && mpu_file.f_op->open(mpu_inode, &mpu_file) )
    {
      printk("MPU-401: config file not found.\n");
      mpu_end_read();
      return FALSE;
    };

  /* does it have file operations? */
  if (!mpu_file.f_op->read)
    {
      printk("MPU-401: cannot read config file (f_op->read is NULL).\n");
      mpu_end_read();
      return FALSE;
    }

  set_fs(KERNEL_DS);

  if ( mpu_file.f_op->read(mpu_inode, &mpu_file, &buf[0], 8) != 8)
    {
      printk("MPU-401: error while reading config file.\n");
      mpu_close_read();
      return FALSE;
    };

  if (buf[0] == 'M' && buf[1] == 'P' && buf[2] == 'U' && buf[3] == '!')
    {
      mpu_hw.irq = buf[4] << 8;
      mpu_hw.irq |= buf[5];
      if ((mpu_hw.irq < 2) || (mpu_hw.irq > 0x10))
	{
	  printk("MPU-401: IRQ request %hd outside range [2;15]\n", mpu_hw.irq);
	  mpu_close_read();
	  return FALSE;
	};

      mpu_hw.ioport = buf[6] << 8;
      mpu_hw.ioport |= buf[7];
      if (mpu_hw.ioport % 2 || (mpu_hw.ioport < 0x200) || (mpu_hw.ioport > 0x400))
	{
	  printk("MPU-401: IO port request 0x%3hx outside range [0x200;0x400]\n", mpu_hw.ioport);
	  mpu_close_read();
	  return FALSE;
	};
    }
  else
    printk("MPU-401: Invalid format in config file (please use mpucondig).\n");

  mpu_close_read();

  mpu_hw.dtp = mpu_hw.ioport;
  mpu_hw.cmp = mpu_hw.ioport + 1;
  mpu_hw.stp = mpu_hw.ioport + 1;

  return TRUE;
}

/*** MODULAR KERNEL INTERFACE *************************************************/

int init_module(void)
{
#ifdef DEVELOP
  mpu_action_queue_reset(&actionq);
#endif

  mpu_reset_driver();
  printk(driver_version_str, drv_ver.version, drv_ver.subversion, drv_ver.revision);

  if (mpu_debug > MPU_DBG_NONE) printk("MPU-401: init_module()\n"); 
  if (!mpu_get_conf()) return -EINVAL;
  if (!mpu_register_device()) return -EBUSY;
  if (!mpu_request_ioports())
    {
      mpu_unregister_device();
      return -EBUSY;
    };
  if (!mpu_request_irq())
    {
      mpu_release_ioports();
      mpu_unregister_device();
      return -EBUSY;
    };
  if (!mpu_detect())
    {
      mpu_free_irq();
      mpu_release_ioports();
      mpu_unregister_device();
      return -ENODEV;
    };

  mpu_free_irq();
  return 0;
}

void cleanup_module(void)
{
  if (mpu_debug > MPU_DBG_NONE) printk("MPU-401: cleanup_module()\n");

  if (MOD_IN_USE)
    {
      printk("MPU-401: Module is in use, removal refused!.\n");
      return;
    };

  mpu_reset(TRUE);
  mpu_kill_synth();

  if (mpu_has_irq) mpu_free_irq();
  mpu_release_ioports();
  mpu_unregister_device();

  mpu_flush_queues();
}

/*** End of File **************************************************************/
/*
 * Local variables:
 *   fill-column: 99
 * End:
 */
