/*
 * Macintosh front end user interface for CSound. The front end provides
 * a somewhat standard Mac interface to set up the CSound files and options
 * and then forms a UNIX command line (argc and argv) to invoke CSound.
 * This module contains all the menu and dialog code, as well as all the code
 * needed to set up the command line.
 *
 * Revision history:
 *  Aug, 1990		Bill Gardner - file creation.
 *  Feb 5, 1992		bg - Adding SSDir menu option, and making minor changes.
 *  Mar 20, 1994	bv - extended the dialog for ALL csound flag options.
 *
 */
#include	<stdio.h>
#include	<stdlib.h>
#include	<stdarg.h>
#include	<console.h>
#include	<ctype.h>
#include	<pascal.h>
#include	"tc_misc.h"

#define FRONT_END_ONLY	FALSE  /* Set TRUE to test CSound front end by itself */

#define APPLE_ID		1		/* MENU ID's */

#define FILE_ID			2		/* File (really "CSound") */
#define F_OPEN			1
#define F_QUIT			2

#define EDIT_ID			3		/* Edit */

#define ABOUT_ID		128		/* DIALOG IDs and ITEM NUMBERS */

#define OS_DLG_ID		129
#define OS_ORC_SEL	4
#define OS_ORC_NAM		5
#define OS_ORC_OPT		6
#define OS_SCO_SEL	8
#define OS_SCO_NAM		9
#define OS_SCO_OPT		10
#define OS_SDIR_SET	11
#define OS_SOUT_CHK	12
#define OS_SOUT_NAM		13
#define OS_SOUT_FMT		14
#define OS_SIN_CHK	15
#define OS_SIN_SEL	16
#define OS_SIN_NAM		17
#define OS_MIDI_CHK	18
#define OS_MIDI_SEL	19
#define OS_MIDI_NAM		20
#define OS_MIDI_OPT		21
#define OS_LST_CHK	22
#define OS_LST_NAM		23
#define OS_PRT_LVL	24
#define OS_DSP_LVL	25
#define OS_DEFAULTS	26
#define OS_VU_CMND	27

#define OOP_DLG_ID		131
#define OOP_OVERRIDE	4
#define OOP_SR_LBL		5
#define OOP_SR			6
#define OOP_KR_LBL		7
#define OOP_KR			8
#define OOP_IONLY		9

#define SOP_DLG_ID		132
#define SOP_SCOT		4
#define SOP_XTR_CHK		5
#define SOP_XTR_SEL		6
#define SOP_XTR_NAM		7
#define SOP_CSCORE		8
#define SOP_TMP_CHK		9
#define SOP_TMP_VAL		10

#define SFD_DLG_ID		133
#define SFD_SELECT		5
#define SFD_PATH		6
#define SSD_SELECT		8
#define SSD_PATH		9
#define SAD_SELECT		11
#define SAD_PATH		12

#define FMT_DLG_ID		134
#define FMT_8			4
#define FMT_16			5
#define FMT_32			6
#define FMT_32F			7
#define FMT_SDII		9
#define FMT_AIFF		10
#define FMT_WAV			11
#define FMT_RAW			12
#define FMT_CR_SD		14
#define FMT_CR_AUD		15
#define FMT_BUFSAMPS	17
#define FMT_RWRTHDR		18
#define FMT_HEARTBT		19

#define MOP_DLG_ID		135
#define MOP_PED_VAL		5
#define MOP_NOTIFY		6
#define MOP_TERMIN		7

#define POP_DLG_ID		137
#define POP_NOTES		5
#define POP_RANGE		6
#define POP_WARNINGS	7
#define POP_VERBOSE		8

#define DOP_DLG_ID		138
#define DOP_NODISP		4
#define DOP_ASCII		5

#define CMD_DLG_ID		139
#define CMD_CMD			3
#define CMD_OKRUN		5

#define ENABLE_CTL		0
#define DISABLE_CTL		255

#define PATH_LEN		128

/* Currently selected files. The score and orchestra files are necessary,
 * and each may come from a different directory. The output file (optional)
 * goes in the "SnDIR" hierarchy (see Manual), established with the SFDIR
 * shell environment variable in UNIX, and here with the sfdir_path[] pathname.
 * SoundIn, MIDI_IN and ListOut are optional and may be in any directory.
 */
typedef struct {
	char sco[PATH_LEN];	/* score file name */
	short sco_vrn;		/* score volume reference number */
	short sco_vld;
	char orc[PATH_LEN];	/* orchestra file name */
	short orc_vrn;		/* orchestra volume reference number */
	short orc_vld;
	char sout[PATH_LEN];/* output sound file name */
	short sout_vrn;
	char sin[PATH_LEN];	/* soundin file name */
	short sin_vrn;		/* soundin volume reference number */
	short sin_vld;
	char midi[PATH_LEN];/* MIDI file name */
	short midi_vrn;		/* MIDI file volume refernce number */
	short midi_vld;
	char lst[PATH_LEN];	/* listing (stdout) file name */
	short lst_vrn;		/* listing file volume reference number */
	short lst_vld;
	short sout_chk;		/* check-box status */
	short sin_chk;
	short midi_chk;
	short lst_chk;
} SO_FILE;

typedef struct {
	int  override;
	long sr;
	long kr;
	int  ionly;
} ORCHOPS;

typedef struct {
	int scot;
	int xtrchk;
	char xtract[PATH_LEN];
	short xtr_vrn;
	int xtr_vld;
	int Cscore;
	int tmpchk;
	int tempo;
} SCOROPS;

typedef struct {
	int pedthresh;
	int notify;
	int termin;
} MIDIOPS;

typedef struct {
	int msglevel;
	int odebug;
} PRNTOPS;

typedef struct {
	int nodisplays;
	int asciigraphs;
} DISPOPS;

typedef struct {
	int samptype;	/* sample format type - same as dialog item number */
	int filetype;	/* output file type - same as dialog item number */
	int creator;
	int bufsamps;	/* output blocksize in sample frames */
	int	rwrthdr;
	int	heartbt;
} OUT_FMT;

#define OF_NOTES		0x0001 	/* msglevel option flags	*/
#define OF_RANGE		0x0002
#define OF_WARNINGS		0x0004

static SO_FILE so_file;
static ORCHOPS orchopts;
static SCOROPS scoropts;
static MIDIOPS midiopts;
static PRNTOPS prntopts;
static DISPOPS dispopts;
static OUT_FMT out_fmt;

char sfdir_path[PATH_LEN];	/* Set in the S_DIRS dialog	*/
char ssdir_path[PATH_LEN];
char sadir_path[PATH_LEN];

/*
 * Configuration resource structure. Everything that gets remembered
 * between invocations must go here.
 */
#define	CONFIG_TYPE	((long)'CSND')
#define CONFIG_ID	128

typedef struct {			/* delete proj.rsrc if changing this struct */
	SO_FILE so_file;
	ORCHOPS orchopts;
	SCOROPS scoropts;
	MIDIOPS midiopts;
	PRNTOPS prntopts;
	DISPOPS dispopts;
	OUT_FMT out_fmt;
	char sfdir_path[PATH_LEN];
	char ssdir_path[PATH_LEN];
	char sadir_path[PATH_LEN];
} CONFIG;

CONFIG **Config;

static SO_FILE so_dflts = {	"", 0, 0,
							"", 0, 0,
							"", 0,
							"", 0, 0,
							"", 0, 0,
							"", 0, 0,
							0, 0, 0, 0	};
static ORCHOPS or_dflts = {	0, 0, 0, 0	};
static SCOROPS sc_dflts = {	0, 0, "", 0, 0, 0, 0, 0	};
static MIDIOPS mi_dflts = {	127, 0, 0	};
static PRNTOPS pr_dflts = {	7, 0	};
static DISPOPS di_dflts = {	0, 0	};
static OUT_FMT ou_dflts = {	FMT_16, FMT_SDII, FMT_CR_SD, 1024, 0, 0 };

extern getFile(char *, short *, int,...); /* required prototype */

static char cmd_line[512] = "";	/* current command line */

#define MAX_ARG		32
static char *argv[MAX_ARG];	/* argv argc for passing to Csound main() */
static int argc;

			/* Text that appears in "About CSound..." dialog. */
unsigned char *info[] = {
"\p Copyright 1994 by the Massachusetts Institute of Technology.",
"\p                    All rights reserved.",
"\p",
"\p Developed by Barry Vercoe at the Music and Cognition Group,",
"\p		Media Laboratory, M.I.T., Cambridge, Massachusetts,",
"\p with partial support from the System Development Foundation",
"\p and from the National Science Foundation Grant #IRI-8704665.",
"\p",
"\p		Macintosh version assisted by Dan Ellis, Bill Gardner,",
"\p				and Richard Boulanger."
};
#define N_INFO (sizeof(info) / sizeof(char *))

doAboutBox()			/* ABOUT CSOUND.. DIALOG */
{
DialogPtr dp;
short itemHit;
short itemType;
Rect itemBox;
Handle item;
FontInfo fi;
int height;
int i;
	dp = GetNewDialog(ABOUT_ID, 0, (DialogPtr) -1);
	SetPort(dp);
	TextFont(monaco);
	TextSize(9);
	GetFontInfo(&fi);
	height = fi.ascent + fi.descent + fi.leading;
	GetDItem(dp,2,&itemType,&item,&itemBox);
	for (i = 0; i < N_INFO; i++) {
		MoveTo(itemBox.left,itemBox.top + (i + 1) * height);
		DrawString(info[i]);
	}
	TextFont(systemFont);
	TextSize(12);
	ModalDialog(0, &itemHit);
	DisposDialog(dp);
}

/* called from main for encoded selection of FILE OPEN menu item */
long fileOpenSelect()
{
	return (((long) FILE_ID) << 16) | F_OPEN;
}

DoMenuClick(select)				/*** PROCESS MENU EVENT ***/
long select;
{
int menu;				/* menu ID */
int item;				/* menu item */
char itemName[64];		/* item name for desk acc. */
GrafPtr port;
	
	if (select == 0) return;
	
	menu = HiWord(select);
	item = LoWord(select);
	
	HiliteMenu(menu);
	switch(menu) {
	case APPLE_ID:
		if (item > 1) {
			GetPort(&port);
			GetItem(GetMenu(APPLE_ID), item, itemName);
			OpenDeskAcc(itemName);
			SetPort(port);
		}
		else {
			GetPort(&port);
			doAboutBox();
			SetPort(port);
		}
		break;	

	case FILE_ID:
		cursWait();
		switch(item) {
			int stat;
		case F_OPEN:		/* call toplevel ORCH/SCORE dialog */
			if ((stat = dlgFileOpen())) {
				if (stat != CMD_OKRUN)
					formCmdLine();
				parseCmdLine();
				saveConfig();		/*** save CONFIG, ***/
				callCSound();		/*** then call CSOUND ***/
			}
			break;
		default:
		case F_QUIT:
			ExitToShell();
			break;
		}
		cursPop();
		break;

	case EDIT_ID:
		switch(item) {
		case 1: case 3: case 4: case 5: case 6:
			SystemEdit(item-1);
			break;
		}
		break;
	default:
		break;
	}
	HiliteMenu(0);
}

setupMenuBar()			/* INITIALIZE MENUS */
{
	SetMenuBar(GetNewMBar(1));
	AddResMenu(GetMHandle(1), 'DRVR');
	DrawMenuBar();
}

/*
 * Set the contents of a dialog text item via sprintf.
 */
SetDlgStrItem(DialogPtr dp, short itemHit, char *fmt,...)
{
short itemType;
Rect itemBox;
Handle item;
char buf[256];
va_list args;
	va_start(args,fmt);
	vsprintf(buf,fmt,args);
	va_end(args);
	GetDItem(dp,itemHit,&itemType,&item,&itemBox);
	SetIText(item,CtoPstr(buf));
}

/*
 * Scan the contents of a dialog text item via sscanf.
 */
GetDlgStrItem(dp,itemHit,fmt,p)
DialogPtr dp;
short itemHit;
char *fmt;
char *p;
{
short itemType;
Rect itemBox;
Handle item;
char buf[256];
	GetDItem(dp,itemHit,&itemType,&item,&itemBox);
	GetIText(item,buf);
	PtoCstr(buf);
	sscanf(buf,fmt,p);
}

/*
 * Get the contents of a dialog string item. Note that
 * you can't use GetDlgStrItem() if the text contains spaces.
 */
GetDlgString(dp,itemHit,p)
DialogPtr dp;
short itemHit;
char *p;			/* C string */
{
short itemType;
Rect itemBox;
Handle item;
	GetDItem(dp,itemHit,&itemType,&item,&itemBox);
	GetIText(item,p);
	PtoCstr(p);
}

addFileExt(s,x)		/* REPLACE ".suffix" with x */
char *s;
char *x;
{
char *t;
	for (t = s; *t && *t != '.'; t++);
	strcpy(t, x);
}

static int
exists(fname,vrefnum)	/* CHK FILE EXISTS, return TRUE/FALSE */
char *fname;
short vrefnum;
{
char path[256];
	return getPath(path,fname,vrefnum) == noErr;
}

static void
formOutputFile(dp,p)	/* FORM SOUNDOUT FILE NAME */
DialogPtr dp;		/* called after ScoreFile setup */
SO_FILE *p;
{
	strcpy(p->sout,p->sco);
	addFileExt(p->sout,".snd");
	if (dp) {
		SetDlgStrItem(dp,OS_SOUT_NAM,"%s",p->sout);
		SelIText(dp,OS_SOUT_NAM,0,32767);
	}
}

static void
setScoreFile(dp,p)	/* FORM probable ORCHESTRA FILE NAME */
DialogPtr dp;		/* If exists, make orch file valid */
SO_FILE *p;			/* Called after ScoreFile setup */
{
	p->sco_vld = 1;
	if (!(p->orc_vld)) {
		strcpy(p->orc,p->sco);
		p->orc_vrn = p->sco_vrn;
		addFileExt(p->orc,".orc");
		if (exists(p->orc,p->orc_vrn))
			p->orc_vld = 1;
	}
	formOutputFile(dp,p);
}

static void
setOrchestraFile(dp,p)	/* FORM probable SCORE FILE NAME */
DialogPtr dp;			/* If exists, make score file valid */
SO_FILE *p;				/* Called after OrchFile setup */
{
	p->orc_vld = 1;
	if (!(p->sco_vld)) {
		/*
		 * Synthesize probable name of score file and test for
		 * existence. If found, make score file valid.
		 */
		strcpy(p->sco,p->orc);
		p->sco_vrn = p->orc_vrn;
		addFileExt(p->sco,".sco");
		if (exists(p->sco,p->sco_vrn)) {
			p->sco_vld = 1;
			formOutputFile(dp,p);
		}
	}
}

static int ok_enabled;	/* set TRUE if allow \r to mean OK button */

			/* Event filter for dlgFileOpen() modal dialog. */
			/* Include to allow \r to mean OK when enabled */
static pascal Boolean myFilter(dp,ev,itemHit)
DialogPtr dp;
EventRecord *ev;
int *itemHit;
{
#define ENTER 0x03
	if (ev->what == keyDown)
		switch(ev->message & charCodeMask) {
		case '\r':
		case ENTER:
			*itemHit = OK;
			return ok_enabled;
		default:
			return FALSE;
		}
	else return FALSE;
}

#define OKorCAN					\
		case OK:				\
			stat = TRUE;		\
			InDialog = FALSE;	\
			break;				\
		case Cancel:			\
			stat = FALSE;		\
			InDialog = FALSE;	\
			break;

static void dlgFileOrchOpts();
static void dlgFileScorOpts();
static void dlgFileMidiOpts();
static void dlgFilePrntOpts();
static void dlgFileDispOpts();
static void dlgFileOutputFmt();

int dlgFileOpen()		/***** TOP_LEVEL ORCH & SCORE DIALOG *****/
{
DialogPtr dp;
short itemHit, itemType;
Rect itemBox;
Handle item;
int InDialog = TRUE;
GrafPtr port;
int stat, chkval, dispFlag = TRUE;
char noname[] = "";
SO_FILE tmp = so_file;	/* copy struct to temporary */

	GetPort(&port);
	if ((dp = GetNewDialog(OS_DLG_ID, NULL, (DialogPtr)-1)) == NULL)
		return FALSE;
	SetPort(dp);

	if (tmp.sout[0]) {
		SetDlgStrItem(dp,OS_SOUT_NAM,"%s",tmp.sout);
		SelIText(dp,OS_SOUT_NAM,0,32767);
	}

	cursNorm();
	ok_enabled = FALSE;
	while(InDialog) {		
		if (!dispFlag) goto modal;
		
				/* frame rectangles around static text items */
		GetDItem(dp,OS_ORC_NAM,&itemType,&item,&itemBox);
		InsetRect(&itemBox,-2,-2);
		FrameRect(&itemBox);
		GetDItem(dp,OS_SCO_NAM,&itemType,&item,&itemBox);
		InsetRect(&itemBox,-2,-2);
		FrameRect(&itemBox);
		GetDItem(dp,OS_SIN_NAM,&itemType,&item,&itemBox);
		InsetRect(&itemBox,-2,-2);
		FrameRect(&itemBox);
		GetDItem(dp,OS_MIDI_NAM,&itemType,&item,&itemBox);
		InsetRect(&itemBox,-2,-2);
		FrameRect(&itemBox);
				
				/* display current parameters */
		SetDlgStrItem(dp,OS_ORC_NAM,tmp.orc_vld ? "%s":"",tmp.orc);
		SetDlgStrItem(dp,OS_SCO_NAM,tmp.sco_vld ? "%s":"",tmp.sco);
	
 		chkval = tmp.sout_chk;
		GetDItem(dp,OS_SOUT_CHK,&itemType,&item,&itemBox);
		SetCtlValue(item, chkval);
		GetDItem(dp,OS_SOUT_FMT,&itemType,&item,&itemBox);
		HiliteControl(item, chkval ? ENABLE_CTL : DISABLE_CTL);
		SetDlgStrItem(dp,OS_SOUT_NAM, chkval ? "%s":"",tmp.sout);
		
 		chkval = tmp.sin_chk;
		GetDItem(dp,OS_SIN_CHK,&itemType,&item,&itemBox);
		SetCtlValue(item, chkval);
		GetDItem(dp,OS_SIN_SEL,&itemType,&item,&itemBox);
		HiliteControl(item, chkval ? ENABLE_CTL : DISABLE_CTL);
		chkval = chkval && tmp.sin_vld;
		SetDlgStrItem(dp,OS_SIN_NAM, chkval ? "%s":"",tmp.sin);

 		chkval = tmp.midi_chk;
		GetDItem(dp,OS_MIDI_CHK,&itemType,&item,&itemBox);
		SetCtlValue(item, chkval);
		GetDItem(dp,OS_MIDI_SEL,&itemType,&item,&itemBox);
		HiliteControl(item, chkval ? ENABLE_CTL : DISABLE_CTL);
		GetDItem(dp,OS_MIDI_OPT,&itemType,&item,&itemBox);
		HiliteControl(item, chkval ? ENABLE_CTL : DISABLE_CTL);
		chkval = chkval && tmp.midi_vld;
		SetDlgStrItem(dp,OS_MIDI_NAM, chkval ? "%s":"",tmp.midi);

		chkval = tmp.lst_chk;
		GetDItem(dp,OS_LST_CHK,&itemType,&item,&itemBox);
		SetCtlValue(item, chkval);
		chkval = chkval && tmp.lst_vld;
		SetDlgStrItem(dp,OS_LST_NAM, chkval ? "%s":"",tmp.lst);

		/* Enable OK RUN button only if orc file is valid.  This needed 
		 * since ModalDialog doesn't check for OK button being disabled. */
		ok_enabled = tmp.orc_vld;
		GetDItem(dp,OK,&itemType,&item,&itemBox);
		HiliteControl(item, tmp.orc_vld ? ENABLE_CTL: DISABLE_CTL);

modal:	dispFlag = TRUE;
	  	ModalDialog(myFilter,&itemHit);
			switch(itemHit) {
			OKorCAN
			case OS_ORC_SEL:
				SetVol(noname,tmp.orc_vrn);
				if (getFile(tmp.orc,&tmp.orc_vrn,1,(long)'TEXT'))
					setOrchestraFile(dp,&tmp);
				break;
			case OS_ORC_OPT:
				dlgFileOrchOpts();
				break;
			case OS_SCO_SEL:
				SetVol(noname,tmp.sco_vrn);
				if (getFile(tmp.sco,&tmp.sco_vrn,1,(long)'TEXT'))
					setScoreFile(dp,&tmp);
				break;
			case OS_SCO_OPT:
				dlgFileScorOpts();
				break;
			case OS_SDIR_SET:		/* set SFDir and/or SSDir */
				dlgFileSDIRs();
				break;
			case OS_SOUT_CHK:
				tmp.sout_chk ^= 1;
				break;
			case OS_SOUT_NAM:
				dispFlag = FALSE;	/* prevent flickering while typing */
	/*			GetDlgStrItem(dp,OS_SOUT_NAM,"%s",tmp.sout);(can't delete last)	*/
				GetDlgString(dp,OS_SOUT_NAM,tmp.sout);
				break;
			case OS_SOUT_FMT:
				dlgFileOutputFmt();
				break;
			case OS_SIN_CHK:
				tmp.sin_chk ^= 1;
				break;
			case OS_SIN_SEL:
				if (getFile(tmp.sin,&tmp.sin_vrn,1,(long)'TEXT'))
					tmp.sin_vld = 1;
				else tmp.sin_vld = 0;
				break;
			case OS_MIDI_CHK:
				tmp.midi_chk ^= 1;
				break;
			case OS_MIDI_SEL:
				if (getFile(tmp.midi,&tmp.midi_vrn,0))
					tmp.midi_vld = 1;
				else tmp.midi_vld = 0;
				break;
			case OS_MIDI_OPT:
				dlgFileMidiOpts();
				break;
			case OS_LST_CHK:
				tmp.lst_chk ^= 1;
				if (!tmp.lst_chk)
					break;
				if (tmp.orc_vld && !tmp.lst_vld) {
					strcpy(tmp.lst,tmp.orc);
					addFileExt(tmp.lst,".lst");
					tmp.lst_vrn = tmp.orc_vrn;
					tmp.lst_vld = 1;
				}
					/* FALLS THROUGH! */
			case OS_LST_NAM:
				if (putFile(NULL,tmp.lst,&tmp.lst_vrn))
					tmp.lst_vld = tmp.lst_chk = 1;
				else tmp.lst_vld = tmp.lst_chk = tmp.lst_vrn = 0;
				break;
			case OS_PRT_LVL:
				dlgFilePrntOpts();
				break;
			case OS_DSP_LVL:
				dlgFileDispOpts();
				break;
			case OS_VU_CMND:
				so_file = tmp;
				if ((stat = dlgFileViewCmnd()))	/* CMD_OKRUN ? */
					InDialog = 0;
				break;
			case OS_DEFAULTS:
				setGlobDflts();
				tmp = so_file;
				break;
			default:
				break;
			}
	}
	cursPop();

	if (stat && tmp.sout_chk && !tmp.sout[0]) {
		GetDlgString(dp,OS_SOUT_NAM,tmp.sout);
		formOutputFile(NULL,&tmp);
	}
	if (stat && tmp.lst_chk && !tmp.lst[0])
		GetDlgString(dp,OS_LST_NAM,tmp.lst);
	so_file = tmp;		/* always save results of dialog */
	DisposDialog(dp);
	SetPort(port);
	return stat;
}

static void dlgFileOrchOpts()			/*** ORCH_OPTIONS DIALOG ***/
{
DialogPtr dp;
short itemHit, itemType;
Rect itemBox;
Handle item;
int InDialog = TRUE;
GrafPtr port;
int stat;
long sr, kr;
ORCHOPS tmp = orchopts;	/* copy current to temporary */

	GetPort(&port);
	if ((dp = GetNewDialog(OOP_DLG_ID, NULL, (DialogPtr)-1)) == NULL)
	    return;
	SetPort(dp);
	
	sr = tmp.sr;
	kr = tmp.kr;
	SetDlgStrItem(dp,OOP_SR, sr ? "%ld":"",sr);
	SetDlgStrItem(dp,OOP_KR, kr ? "%ld":"",kr);
	
	cursNorm();
	while(InDialog) {
				/* display chkbox flags */
		GetDItem(dp,OOP_OVERRIDE,&itemType,&item,&itemBox);
		SetCtlValue(item, tmp.override);
		GetDItem(dp,OOP_IONLY,&itemType,&item,&itemBox);
		SetCtlValue(item, tmp.ionly);
		
		ModalDialog(NULL,&itemHit);
		switch(itemHit) {
		OKorCAN
		case OOP_OVERRIDE:
			tmp.override ^= 1;
			if (tmp.override) {
				SetDlgStrItem(dp,OOP_SR, sr ? "%ld":"",sr);
				SetDlgStrItem(dp,OOP_KR, kr ? "%ld":"",kr);
			}
			else {
				GetDlgStrItem(dp,OOP_SR,"%ld",(char *)&sr);
				GetDlgStrItem(dp,OOP_KR,"%ld",(char *)&kr);
				SetDlgStrItem(dp,OOP_SR,"");
				SetDlgStrItem(dp,OOP_KR,"");
			}				
			break;
		case OOP_IONLY:
			tmp.ionly ^= 1;
			break;
		default:
			break;
		}
	}
	cursPop();

	if (stat) {
		if (tmp.override) {
			GetDlgStrItem(dp,OOP_SR,"%ld",(char *)&tmp.sr);
			GetDlgStrItem(dp,OOP_KR,"%ld",(char *)&tmp.kr);
		}
		orchopts = tmp;		/* copy temp back into scoropts */
	}
	DisposDialog(dp);
	SetPort(port);
}

static void dlgFileScorOpts()			/*** SCORE_OPTIONS DIALOG ***/
{
DialogPtr dp;
short itemHit, itemType;
Rect itemBox;
Handle item;
int InDialog = TRUE;
GrafPtr port;
int stat, chkval, xtr_vld, tempo;
SCOROPS tmp = scoropts;	/* copy current to temporary */

	GetPort(&port);
	if ((dp = GetNewDialog(SOP_DLG_ID, NULL, (DialogPtr)-1)) == NULL)
	    return;
	SetPort(dp);
	
	xtr_vld = tmp.xtr_vld;
	tempo = tmp.tempo;
	SetDlgStrItem(dp,SOP_TMP_VAL, tempo ? "%d":"",tempo);
	SelIText(dp,SOP_TMP_VAL,0,1024);

	cursNorm();
	while(InDialog) {
				/* display flags in chkboxes */
		GetDItem(dp,SOP_SCOT,&itemType,&item,&itemBox);
		SetCtlValue(item, tmp.scot);
		GetDItem(dp,SOP_XTR_CHK,&itemType,&item,&itemBox);
		SetCtlValue(item, tmp.xtrchk);
		GetDItem(dp,SOP_CSCORE,&itemType,&item,&itemBox);
		SetCtlValue(item, tmp.Cscore);
		GetDItem(dp,SOP_TMP_CHK,&itemType,&item,&itemBox);
		SetCtlValue(item, tmp.tmpchk);
		
				/* display file select buttons & values */
		GetDItem(dp,SOP_XTR_SEL,&itemType,&item,&itemBox);
		HiliteControl(item, tmp.xtrchk ? ENABLE_CTL : DISABLE_CTL);
		chkval = tmp.xtrchk && tmp.xtr_vld; /* also used at exit */
		SetDlgStrItem(dp,SOP_XTR_NAM, chkval ? "%s":"",tmp.xtract);
		
		ModalDialog(NULL,&itemHit);
		switch(itemHit) {
		OKorCAN
		case SOP_SCOT:
			tmp.scot ^= 1;
			break;
		case SOP_XTR_CHK:
			tmp.xtrchk ^= 1;
			if (tmp.xtrchk)
				tmp.xtr_vld = xtr_vld;
			else {
				xtr_vld = tmp.xtr_vld;
				tmp.xtr_vld = 0;
			}
			break;
		case SOP_XTR_SEL:
			if (getFile(tmp.xtract,&tmp.xtr_vrn,1,(long)'TEXT'))
				xtr_vld = tmp.xtr_vld = 1;
			break;
		case SOP_CSCORE:
			tmp.Cscore ^= 1;
			break;
		case SOP_TMP_CHK:
			tmp.tmpchk ^= 1;
			if (tmp.tmpchk)
				SetDlgStrItem(dp,SOP_TMP_VAL, tempo ? "%d":"",tempo);
			else {
				GetDlgStrItem(dp,SOP_TMP_VAL,"%d",(char *)&tempo);
				SetDlgStrItem(dp,SOP_TMP_VAL,"");
			}
			break;
		default:
			break;
		}
	}
	cursPop();

	if (stat) {
		if (!chkval)
			tmp.xtract[0] = '\0';
		if (tmp.tmpchk)
			GetDlgStrItem(dp,SOP_TMP_VAL,"%d",(char *)&tmp.tempo);
		scoropts = tmp;		/* copy temp back into scoropts */
	}
	DisposDialog(dp);
	SetPort(port);
}

static void dlgFileMidiOpts()		/*** MIDI_OPTIONS DIALOG ***/
{
DialogPtr dp;
short itemHit, itemType;
Rect itemBox;
Handle item;
int InDialog = TRUE;
GrafPtr port;
int stat;
MIDIOPS tmp = midiopts;	/* copy current to temporary */

	GetPort(&port);
	if ((dp = GetNewDialog(MOP_DLG_ID, NULL, (DialogPtr)-1)) == NULL) return;
	SetPort(dp);

	SetDlgStrItem(dp,MOP_PED_VAL,"%d",tmp.pedthresh);
	SelIText(dp,MOP_PED_VAL,0,1024);

	cursNorm();
	while(InDialog) {
			/* display values and chks */
		GetDItem(dp,MOP_NOTIFY,&itemType,&item,&itemBox);
		SetCtlValue(item, tmp.notify);
		GetDItem(dp,MOP_TERMIN,&itemType,&item,&itemBox);
		SetCtlValue(item, tmp.termin);

		ModalDialog(NULL,&itemHit);
		switch(itemHit) {
		OKorCAN
		case MOP_NOTIFY:
			tmp.notify ^= 1;
			break;
		case MOP_TERMIN:
			tmp.termin ^= 1;
			break;
		default:
			break;
		}
	}
	cursPop();
	if (stat) {
		GetDlgStrItem(dp,MOP_PED_VAL,"%d",(char *)&tmp.pedthresh);
		midiopts = tmp;
	}
	DisposDialog(dp);
	SetPort(port);
}

static void dlgFilePrntOpts()			/*** PRINT_OPTIONS DIALOG ***/
{
DialogPtr dp;
short itemHit, itemType;
Rect itemBox;
Handle item;
int InDialog = TRUE;
GrafPtr port;
int stat;
PRNTOPS tmp = prntopts;	/* copy current to temporary */

	GetPort(&port);
	if ((dp = GetNewDialog(POP_DLG_ID, NULL, (DialogPtr)-1)) == NULL) return;
	SetPort(dp);

	cursNorm();
	while(InDialog) {
		/* display flags in checkboxes */
		GetDItem(dp,POP_NOTES,&itemType,&item,&itemBox);
		SetCtlValue(item,(tmp.msglevel & OF_NOTES) ? 1 : 0);
		GetDItem(dp,POP_RANGE,&itemType,&item,&itemBox);
		SetCtlValue(item,(tmp.msglevel & OF_RANGE) ? 1 : 0);
		GetDItem(dp,POP_WARNINGS,&itemType,&item,&itemBox);
		SetCtlValue(item,(tmp.msglevel & OF_WARNINGS) ? 1 : 0);
		GetDItem(dp,POP_VERBOSE,&itemType,&item,&itemBox);
		SetCtlValue(item, tmp.odebug);

		ModalDialog(NULL,&itemHit);
		switch(itemHit) {
		OKorCAN
		case POP_NOTES:
			tmp.msglevel ^= OF_NOTES;
			break;
		case POP_RANGE:
			tmp.msglevel ^= OF_RANGE;
			break;
		case POP_WARNINGS:
			tmp.msglevel ^= OF_WARNINGS;
			break;
		case POP_VERBOSE:
			tmp.odebug ^= 1;
			break;
		default:
			break;
		}
	}
	cursPop();
	if (stat)  prntopts = tmp;
	DisposDialog(dp);
	SetPort(port);
}

static void dlgFileDispOpts()		/*** DISPLAY_OPTIONS DIALOG ***/
{
DialogPtr dp;
short itemHit,itemType;
Rect itemBox;
Handle item;
int InDialog = TRUE;
GrafPtr port;
int stat;
DISPOPS tmp = dispopts;	/* copy current to temporary */

	GetPort(&port);
	if ((dp = GetNewDialog(DOP_DLG_ID, NULL, (DialogPtr)-1)) == NULL) return;
	SetPort(dp);

	cursNorm();
	while(InDialog) {
		/* display flags in checkboxes */
		GetDItem(dp,DOP_NODISP,&itemType,&item,&itemBox);
		SetCtlValue(item, tmp.nodisplays);
		GetDItem(dp,DOP_ASCII,&itemType,&item,&itemBox);
		SetCtlValue(item, tmp.asciigraphs);

		ModalDialog(NULL,&itemHit);
		switch(itemHit) {
		OKorCAN
		case DOP_NODISP:
			tmp.nodisplays ^= 1;
			break;
		case DOP_ASCII:
			tmp.asciigraphs ^= 1;
			break;
		default:
			break;
		}
	}
	cursPop();
	if (stat)  dispopts = tmp;
	DisposDialog(dp);
	SetPort(port);
}

static void dlgFileOutputFmt()		/*** SOUND-OUT_FORMAT DIALOG ***/
{
DialogPtr dp;
short itemHit, itemType;
Rect itemBox;
Handle item;
int InDialog = TRUE;
GrafPtr port;
int stat;
OUT_FMT tmp = out_fmt;	/* copy current to temporary */

	GetPort(&port);
	if ((dp = GetNewDialog(FMT_DLG_ID, NULL, (DialogPtr)-1)) == NULL)
	    return;
	SetPort(dp);

	/* display current parameters */
	SetDlgStrItem(dp,FMT_BUFSAMPS,"%d",tmp.bufsamps);
	SelIText(dp,FMT_BUFSAMPS,0,1024);

	cursNorm();
	while(InDialog) {
		int s = tmp.samptype;
		int f = tmp.filetype;
		int c = tmp.creator;

					/* display sample format buttons */
		GetDItem(dp,FMT_8,&itemType,&item,&itemBox);
		SetCtlValue(item, s==FMT_8);

		GetDItem(dp,FMT_16,&itemType,&item,&itemBox);
		SetCtlValue(item, s==FMT_16);

		GetDItem(dp,FMT_32,&itemType,&item,&itemBox);
		SetCtlValue(item, s==FMT_32);

		GetDItem(dp,FMT_32F,&itemType,&item,&itemBox);
		if (f==FMT_SDII || f==FMT_AIFF || f==FMT_WAV) {
			HiliteControl(item, DISABLE_CTL);
			if (s==FMT_32F) s = 0;
		}				
		else HiliteControl(item, ENABLE_CTL);
		SetCtlValue(item, s==FMT_32F);
						
					/* display file type buttons */
		GetDItem(dp,FMT_SDII,&itemType,&item,&itemBox);
		if (s==FMT_32F) {
			HiliteControl(item, DISABLE_CTL);
			if (f==FMT_SDII) f = 0;
		}				
		else HiliteControl(item, ENABLE_CTL);
		SetCtlValue(item, f==FMT_SDII);

		GetDItem(dp,FMT_AIFF,&itemType,&item,&itemBox);
		if (s==FMT_32F) {
			HiliteControl(item, DISABLE_CTL);
			if (f==FMT_AIFF) f = 0;
		}				
		else HiliteControl(item, ENABLE_CTL);
		SetCtlValue(item, f==FMT_AIFF);

		GetDItem(dp,FMT_WAV,&itemType,&item,&itemBox);
		if (s==FMT_32F) {
			HiliteControl(item, DISABLE_CTL);
			if (f==FMT_WAV) f = 0;
		}				
		else HiliteControl(item, ENABLE_CTL);
		SetCtlValue(item, f==FMT_WAV);

		GetDItem(dp,FMT_RAW,&itemType,&item,&itemBox);
		SetCtlValue(item, f==FMT_RAW);
		
				/* display creator buttons */
		GetDItem(dp,FMT_CR_SD,&itemType,&item,&itemBox);
		SetCtlValue(item, c==FMT_CR_SD);
		
		GetDItem(dp,FMT_CR_AUD,&itemType,&item,&itemBox);
		SetCtlValue(item, c==FMT_CR_AUD);
		
				/* display flags in checkboxes */
		GetDItem(dp,FMT_RWRTHDR,&itemType,&item,&itemBox);
		SetCtlValue(item, tmp.rwrthdr);
		GetDItem(dp,FMT_HEARTBT,&itemType,&item,&itemBox);
		SetCtlValue(item, tmp.heartbt);
		
		ModalDialog(NULL,&itemHit);
		switch(itemHit) {
		OKorCAN
		case FMT_8:
			tmp.samptype = FMT_8;
			break;
		case FMT_16:
			tmp.samptype = FMT_16;
			break;
		case FMT_32:
			tmp.samptype = FMT_32;
			break;
		case FMT_32F:
			tmp.samptype = FMT_32F;
			break;
		case FMT_SDII:
			tmp.filetype = FMT_SDII;
			break;
		case FMT_AIFF:
			tmp.filetype = FMT_AIFF;
			break;
		case FMT_WAV:
			tmp.filetype = FMT_WAV;
			break;
		case FMT_RAW:
			tmp.filetype = FMT_RAW;
			break;
		case FMT_CR_SD:
			tmp.creator = FMT_CR_SD;
			break;
		case FMT_CR_AUD:
			tmp.creator = FMT_CR_AUD;
			break;
		case FMT_RWRTHDR:
			tmp.rwrthdr ^= 1;
			break;
		case FMT_HEARTBT:
			tmp.heartbt ^= 1;
			break;
		default:
			break;
		}
	}
	cursPop();

	if (stat) {
		GetDlgStrItem(dp,FMT_BUFSAMPS,"%d",(char *)&tmp.bufsamps);
		out_fmt = tmp;		/* copy temp back into out_fmt */
	}
	DisposDialog(dp);
	SetPort(port);
}

/*	if (dlgFilePath(sadir_path,sadir_prompt,"\pSelect SADir:") == SFD_SAVE)
 *		saveConfig();
 */

dlgFileSDIRs()				/*** SET S_DIRs DIALOG ***/
{
DialogPtr dp;
short itemHit;
int InDialog = TRUE;
GrafPtr port;
int stat;
short vrefnum;
char buf[256];
char anyname[] = "Filename doesn't matter";

	GetPort(&port);
	if ((dp = GetNewDialog(SFD_DLG_ID, NULL,(DialogPtr)-1)) == NULL) return FALSE;
	SetPort(dp);

	SetDlgStrItem(dp,SFD_PATH,"%s",sfdir_path);
	SetDlgStrItem(dp,SSD_PATH,"%s",ssdir_path);
	SetDlgStrItem(dp,SAD_PATH,"%s",sadir_path);
	SelIText(dp,SFD_PATH,0,32767);

	cursNorm();
	while(InDialog) {

		ModalDialog(NULL,&itemHit);
		switch(itemHit) {
		OKorCAN
		case SFD_SELECT:
			strcpy(buf,anyname);
			if (putFile("\pSelect SFDIR:",buf,&vrefnum)) {
				getPath(buf,NULL,vrefnum);
				SetDlgStrItem(dp,SFD_PATH,"%s",buf);
				SelIText(dp,SFD_PATH,0,32767);
			}
			break;
		case SSD_SELECT:
			strcpy(buf,anyname);
			if (putFile("\pSelect SSDIR",buf,&vrefnum)) {
				getPath(buf,NULL,vrefnum);
				SetDlgStrItem(dp,SSD_PATH,"%s",buf);
				SelIText(dp,SSD_PATH,0,32767);
			}
			break;
		case SAD_SELECT:
			strcpy(buf,anyname);
			if (putFile("\pSelect SADIR:",buf,&vrefnum)) {
				getPath(buf,NULL,vrefnum);
				SetDlgStrItem(dp,SAD_PATH,"%s",buf);
				SelIText(dp,SAD_PATH,0,32767);
			}
			break;
		default:
			break;
		}
	}
	cursPop();
	if (stat) {
		GetDlgString(dp,SFD_PATH,sfdir_path);
		GetDlgString(dp,SSD_PATH,ssdir_path);
		GetDlgString(dp,SAD_PATH,sadir_path);
	}
	DisposDialog(dp);
	SetPort(port);
}

freeArgs()		/* Free any existing args in argv[]. */
{
int i;
	for (i = 0; i < argc; i++) {
		free(argv[i]);
		argv[i] = 0;
	}
	argc = 0;
}

appendArg(s)	/* Append an arg to the cur argv[] array */
char *s;
{
char *arg;
	if (argc < MAX_ARG) {
		if ((arg = malloc((size_t)strlen(s)+1)) == NULL) {
			SysBeep(10L);
			return;
		}
		strcpy(arg,s);
		argv[argc++] = arg;
	}
	else SysBeep(10L);
}

int spaceInStr(s)	/* does string contains a space ? */
char *s;
{
	while (*s) {
		if (isspace(*s)) return TRUE;
		s++;
	}
	return FALSE;
}

/* Append the (fmt, args) to the current command line using sprintf */

static void appendCmd(char *fmt,...)
{
va_list args;
	va_start(args,fmt);
	vsprintf(&cmd_line[strlen(cmd_line)],fmt,args);
	va_end(args);
}

static int  lstoutreq;
static char lstoutnam[64];	/* lst fname before concat */

/* Using current options and files, create a UNIX-style command line
	to invoke CSound. If pathnames contain spaces, double-quote them  */
formCmdLine()
{
int m, n;
char path[256];
SO_FILE *p = &so_file;
#define MSG_DEFAULT	7
#define BUFSMPS_DFLT 1024
#define PEDTHRSH_DFLT 128

	cmd_line[0] = 0;
	appendCmd("Csound");
	if (p->sout_chk && p->sout[0]) {
		switch (out_fmt.filetype) {
		case FMT_SDII:
			appendCmd(" -D");
			break;
		case FMT_AIFF:
			appendCmd(" -A");
			break;
		case FMT_WAV:
			appendCmd(" -W");
			break;
		case FMT_RAW:
			appendCmd(" -h");
			break;
		}
		switch (out_fmt.samptype) {
		case FMT_8:
			appendCmd("c");
			break;
		case FMT_16:
			appendCmd("s");
			break;
		case FMT_32:
			appendCmd("l");
			break;
		case FMT_32F:
			appendCmd("f");
			break;
		}
		m = out_fmt.rwrthdr;	n = out_fmt.heartbt;
		if (m || n) appendCmd(" -%s%s", m ? "R":"", n ? "H":"");
		if ((m = out_fmt.bufsamps) != BUFSMPS_DFLT)
			appendCmd(" -b%d", m);
	}
	else appendCmd(" -n");

	if (orchopts.override)
		appendCmd(" -r%ld -k%ld", orchopts.sr, orchopts.kr);
	if (orchopts.ionly) appendCmd(" -I");

	if (scoropts.scot) appendCmd(" -S");
	if (scoropts.xtrchk) {
		getPath(path, scoropts.xtract, scoropts.xtr_vrn);
		if (spaceInStr(path)) appendCmd(" -x\"%s\"", path);
		else appendCmd(" -x%s", path);
	}
	if (scoropts.Cscore) appendCmd(" -C");
	if (scoropts.tmpchk) appendCmd(" -t%d", scoropts.tempo);
	
	if (prntopts.odebug) appendCmd(" -v");
	if ((m = prntopts.msglevel) != MSG_DEFAULT) appendCmd(" -m%d", m);
	m = dispopts.nodisplays;	n = dispopts.asciigraphs;
	if (m || n) appendCmd(" -%s%s", m ? "d":"", n ? "g":"");

	if (p->sout_chk && p->sout[0]) {
		strcpy(path, p->sout);
		if (spaceInStr(path)) appendCmd(" \"-o%s\"",path);
		else appendCmd(" -o%s",path);
	}
	if (p->sin_chk && p->sin[0]) {
		strcpy(path, p->sin);
		if (spaceInStr(path)) appendCmd(" \"-i%s\"",path);
		else appendCmd(" -i%s",path);
	}
	if (p->midi_chk && p->midi_vld) {
		getPath(path, p->midi, p->midi_vrn);
		if (spaceInStr(path)) appendCmd("\r\"-F%s\"",path);
		else appendCmd("\r-F%s",path);
		if ((m = midiopts.pedthresh) != PEDTHRSH_DFLT)
			appendCmd(" -P%d", m);
		m = midiopts.notify;	n = midiopts.termin;
		if (m || n) appendCmd(" -%s%s", m ? "N":"", n ? "T":"");
	}
	if (p->orc_vld) {
		getPath(path, p->orc, p->orc_vrn);
		if (spaceInStr(path)) appendCmd("\r\"%s\"",path);
		else appendCmd("\r%s",path);
	}
	if (p->sco_vld) {
		getPath(path, p->sco, p->sco_vrn);
		if (spaceInStr(path)) appendCmd("\r\"%s\"",path);
		else appendCmd("\r%s",path);
	}
	if ((lstoutreq = p->lst_chk && p->lst_vld))
		strcpy(lstoutnam, p->lst);
}

/*
 * Uses state machine to parse command lines into space separated arguments.
 * Handles double quotes to enclose spaces within an argument.
 */
parseCmdLine()
{
enum {
	read_spaces,
	read_chars,
	read_quotes
} state = read_spaces;
char *t;
char *s;
char buf[256];
int running = TRUE;
	freeArgs();
	s = cmd_line;
	while (running) {
		switch (state) {
		case read_spaces:
			if (*s == 0) running = FALSE;
			else if (isspace(*s)) s++;
			else {
				t = buf;
				*t = 0;
				state = read_chars;
			}
			break;
		case read_chars:
			if (isspace(*s) || *s == 0) {
				*t = 0;
				appendArg(buf);
				state = read_spaces;
			}
			else if (*s == '"') {
				s++;
				state = read_quotes;
			}
			else *t++ = *s++;
			break;
		case read_quotes:
			if (*s == 0) state = read_chars;
			else if (*s == '"') {
				s++;
				state = read_chars;
			}
			else *t++ = *s++;
			break;
		}
	}
}

dlgFileViewCmnd()			/*** VIEW COMMAND_LINE DIALOG ***/
{					/* & optionally run Csound directly */
DialogPtr dp;
short itemHit;
int InDialog = TRUE;
GrafPtr port;
int stat;
char vu_line[512];

	GetPort(&port);
	if ((dp = GetNewDialog(CMD_DLG_ID, NULL, (DialogPtr)-1)) == NULL) return FALSE;
	SetPort(dp);

					/* form, copy and display command line */
	formCmdLine();
	strcpy(vu_line,cmd_line);
	SetDlgStrItem(dp,CMD_CMD,"%s",vu_line);
	SelIText(dp,CMD_CMD,32767,32767);

	ok_enabled = FALSE;  /* so CR can be entered into text */
	cursNorm();
	while(InDialog) {

		ModalDialog(myFilter,&itemHit);
		switch(itemHit) {
		case CMD_OKRUN:		/* != OK so CR can be entered into text */
			stat = CMD_OKRUN;
			InDialog = FALSE;
			break;
		case Cancel:
			stat = FALSE;
			InDialog = FALSE;
			break;
		case CMD_CMD:
			GetDlgStrItem(dp,CMD_CMD,"%s",vu_line);
			break;
		default:
			break;
		}
	}
	cursPop();

	if (stat)	/* if OKrun, send (modified) preview to cmd_line */
		GetDlgString(dp,CMD_CMD,cmd_line);
	DisposDialog(dp);
	SetPort(port);
	return stat;	/* stat=0 or OKrun on return to top-level */
}

#if FRONT_END_ONLY
tc_main(argc,argv)			/*** CSOUND TC_MAIN TEST STUB ***/
int argc;
char **argv;
{
int i;
	printf("argc %d\n",argc);
	for (i = 0; i < argc; i++) printf("argv[%d] '%s'\n",i,argv[i]);
}
#endif


static int theyHavePaused = 0;
static void MaybePause()		/* plugged into AtExit list */
	{
	if(!theyHavePaused)
		{
		cursNorm();
		SysBeep(10L);
		printf("\n*** PRESS MOUSE BUTTON TO EXIT ***\n");
		while (!Button() && !isAbortEvent());
		}
	}
		
callCSound()			/* REDIRECT OUTPUT, AND CALL CSOUND */
{
	int main_st;
	
	atexit(MaybePause);		/* set up trap for Csound's 'exit()'s */

		/* Redirect stdout to listing file if necessary. Change file types
		 * of input files. Change cursor to wristwatch. Call CSound main().	*/
	if (lstoutreq) {
		SetVol(0L, so_file.lst_vrn);
		if (freopen(lstoutnam,"w",stdout) == NULL)
			SysBeep(10L);
	}
	cursWait();

    DisableItem(GetMenu(APPLE_ID),1);  /* these added for tc_backgrd.c */
	DisableItem(GetMenu(FILE_ID),0);
	DisableItem(GetMenu(EDIT_ID),0);

	main_st = tc_main(argc,argv);
	cursPop();
	cursNorm();
					/* If listing file, close it & reopen stdout console */
	if (lstoutreq && freopenc(stderr,stdout) == NULL)
		SysBeep(10L);
				/* Because CSound main() is not re-entrant, we need to quit.
	if (!main_st)
		theyHavePaused = 1;		/* set semaphore for fn called by exit() */
/*	ExitToShell(); */
	exit(0);
}

static long getSFType()
{
	switch(out_fmt.filetype) {
	case FMT_AIFF:	return ((long)'AIFF');
	case FMT_WAV:	return ((long)'WAV ');
	case FMT_SDII:
	default:		return ((long)'Sd2f');
	}
}

static long getSFCreator()
{	return (out_fmt.creator == FMT_CR_SD ? (long)'Sd2a':(long)'Sd2b');	}


SetMacCreator(ofname)		/* set creator & file type	*/
char *ofname;		/* called from writeheader w. ofname now full path	*/
{
	char filnam[PATH_LEN];
	short vRefNum;
	
	getFV(ofname, filnam, &vRefNum);
	if (chgFInfo(filnam, vRefNum, getSFType(), getSFCreator()) == 0)
		printf("WARNING: SetMacCreator: chgFInfo() returned 0\n");
}

setGlobDflts()
{
	so_file  = so_dflts;
	orchopts = or_dflts;
	scoropts = sc_dflts;
	midiopts = mi_dflts;
	prntopts = pr_dflts;
	dispopts = di_dflts;
	out_fmt  = ou_dflts;
	strcpy(sfdir_path,"");
	strcpy(ssdir_path,"");
	strcpy(sadir_path,"");
}

static void fullPath(fname, vrefnum)
char *fname;
short vrefnum;
{
	char buf[PATH_LEN];
	
	if (vrefnum) {
		getPath(buf, fname, vrefnum);
		strcpy(fname, buf);
	}
}

static void unPath(fname, vrefnum)
char *fname;
short *vrefnum;
{
	char buf[PATH_LEN];
	
	if (*fname == '\0')
		*vrefnum = 0;
	else {
		getFV(fname, buf, vrefnum);
		strcpy(fname, buf);
	}
}

initConfig()	/*** Read config resource, create it if not found. ***/
{								/* called from tc_main */
	if ((Config = (CONFIG **)GetResource(CONFIG_TYPE,CONFIG_ID)) == NULL) {
								/* no resource, so create one */
		if ((Config = (CONFIG **)NewHandle((long)sizeof(CONFIG))) == NULL) {
			SysBeep(10L);
			ExitToShell();
		}
		HLock(Config);
		AddResource(Config,CONFIG_TYPE,CONFIG_ID,"\pCSound settings");
		HUnlock(Config);
		setGlobDflts(); 	/* Init globals to default values */
		saveConfig();		/*	 and write the values to disk */
	}
				/* Copy from disk Config to globals */
	HLock(Config);
	so_file = (*Config)->so_file;
	orchopts = (*Config)->orchopts;
	scoropts = (*Config)->scoropts;
	midiopts = (*Config)->midiopts;
	prntopts = (*Config)->prntopts;
	dispopts = (*Config)->dispopts;
	out_fmt = (*Config)->out_fmt;
	strcpy(sfdir_path,(*Config)->sfdir_path);
	strcpy(ssdir_path,(*Config)->ssdir_path);
	strcpy(sadir_path,(*Config)->sadir_path);
	HUnlock(Config);
	unPath(so_file.orc, &so_file.orc_vrn);
	unPath(so_file.sco, &so_file.sco_vrn);
	unPath(so_file.sin, &so_file.sin_vrn);
	unPath(so_file.midi,&so_file.midi_vrn);
	unPath(so_file.lst, &so_file.lst_vrn);
}

saveConfig()		/*** Copy globals back to Config ***/
{
	fullPath(so_file.orc, so_file.orc_vrn);
	fullPath(so_file.sco, so_file.sco_vrn);
	fullPath(so_file.sin, so_file.sin_vrn);
	fullPath(so_file.midi,so_file.midi_vrn);
	fullPath(so_file.lst, so_file.lst_vrn);
	HLock(Config);
	(*Config)->so_file = so_file;
	(*Config)->orchopts = orchopts;
	(*Config)->scoropts = scoropts;
	(*Config)->midiopts = midiopts;
	(*Config)->prntopts = prntopts;
	(*Config)->dispopts = dispopts;
	(*Config)->out_fmt = out_fmt;
	strcpy((*Config)->sfdir_path,sfdir_path);
	strcpy((*Config)->ssdir_path,ssdir_path);
	strcpy((*Config)->sadir_path,sadir_path);
	ChangedResource(Config);
	if (ResError() != noErr)
		SysBeep(10L);		/* disk locked */
	else
		WriteResource(Config);
	HUnlock(Config);
}

#define PERIOD_KEY_BIT	47
#define COMMAND_KEY_BIT	55

int tstKey(keys,bit)
KeyMap keys;
int bit;
{
	return !!(keys[bit >> 5] & (1L << (bit & 0x1F)));
}

int isAbortEvent()
{
KeyMap keys;
	GetKeys(&keys);
	return (tstKey(keys,PERIOD_KEY_BIT) && tstKey(keys,COMMAND_KEY_BIT));
}
