/*	CHED -- Sunview version of "ched", MIDI Chart Editor, psl 12/85  */

#include	<suntool/sunview.h>
#include	<suntool/canvas.h>
#include	<suntool/panel.h>
#include	<midi.h>
#undef d
#include	<stdio.h>

#define	PIX_INV	PIX_NOT(PIX_DST)
#define	PIX_OR	(PIX_SRC|PIX_DST)

#define	ROP(OX,OY,CX,CY,OP)	pw_rop(Dpw,OX,OY,CX-(OX),CY-(OY),OP,0,0,0)
#define	ROPS(OX,OY,CX,CY,OP,S)	pw_rop(Dpw,OX,OY,CX-(OX),CY-(OY),OP,S,0,0)
#define	TREPL(X,Y,W,H,OP,T)	pw_replrop(Dpw,X,Y,W,H,OP,T,((X)&15),((Y)&15))
#define	RECTOP(R,OP)		ROP((R).ox,(R).oy,(R).cx,(R).cy,OP)
#define	RINV(OX,OY,CX,CY)	pw_rop(Dpw,OX,OY,CX-(OX),CY-(OY),PIX_INV,0,0,0)
#define	RECTINV(R)		RINV((R).ox,(R).oy,(R).cx,(R).cy)

#define	CPM			1000
	/* CPM is Clocks Per Moment (arbitrarily a thousand clocks) */
#define	MAXCHAN			16
#define	MINKEY			0
#define	MAXKEY			128
#define	KRANGE		(MAXKEY - MINKEY)
#define	MIDVEL			0x60	/* NOTEHH width velocity */
#define	MAXBL			1024	/* maximum number of TCWMEs */
#define	VOLRES			1200	/* maximum display width, pixels */
#define	MAXNOTES		1024	/* maximum number of NOTEs in Note[] */

#define	MSBH		34	/* minimum hor scroll bar height */
#define	MAXCMINK	42	/* always show staves initially */
#define	MINCMAXK	78
#define	NOTEHH		(Ppk >> 1)	/* note half height */
#define	VSAWIDTH	16	/* width of Vert Scroll Area */
#define	VKAWIDTH	25	/* width of Vert Kbd Area */
#define	CAMARGIN	4	/* minimum space around Chart Area */

#define	FONTPATH	"/usr/lib/fonts/fixedwidthfonts/screen.r.14"
#define	GSTR(M,B,L)	gstring(M,B,L,Dframe,Fontp)

/* values for Mymouse */
#define	VUMOUSE		1
#define	HSMOUSE		2
#define	VSMOUSE		3
#define	CAMOUSE		4
/* values for Bmode */
#define	BM_SWEEP	0
#define	BM_DRAG		1
/* values for Gmode */
#define	GM_SEL		0
#define	GM_UNSEL	1
/* special value for M1mode */
#define	MM_OFF		0
/* values for Smode */
#define	SM_STATES	0
#define	SM_EVENTS	1
/* values for Vmode */
#define	VM_OFF		0
#define	VM_ON		1
/* modes for openclose() */
#define	OPEN		1	/* open selected time */
#define	CLOSE		2	/* close selected time */
/* dummy file args for selpart() */
#define	NO_FILE		((char *) 0)	/* discard sel or unsel area */
#define	NOTES		((char *) 1)	/* save sel or unsel data in Note[] */
/* values for "which" in Note[] */
#define	NBEG		1
#define	NEND		2
#define	NBOTH		3

/* chart area macros -- these depend on Ppm, Ppk, Cmint, & Cmink */
#define	CAX_T(X)	(1 + Cmint + (((X) - Ca.ox) * CPM - 1) / Ppm)
#define	CAT_X(T)	(Ca.ox + (((T) - Cmint) * Ppm) / CPM)
/****#define	CAX_T(X)	(Cmint + (((X) - Ca.ox) * CPM + Ppm / 2) / Ppm)
/****#define	CAX_T(X)	(Cmint + (((X) - Ca.ox) * CPM) / Ppm)
/****#define	CAT_X(T)	(Ca.ox + (((T) - Cmint) * Ppm + CPM / 2) / CPM)
/****/
#define	CAK_Y(K)	(Ca.cy - ((K) - Cmink) * Ppk)
#define	CAK_YA(K)	(CAK_Y(K) - Ppk + 1)
#define	CAK_YB(K)	(CAK_Y(K) + Ppk - 1)
#define	CAY_K(Y)	(Cmink + (Ca.cy - (Y) + NOTEHH) / Ppk)
#define	CAY_KA(Y)	(CAY_K(Y) + 1)
#define	CAY_KB(Y)	(CAY_K(Y) - 1)

/* horizontal scroll area macros */
#define	HSAX_T(X)	(1 + Mint + (((X) - Hsa.ox) * Ranget - 1) / Hswid)
#define	HSAT_X(T)	(Hsa.ox + (((T) - Mint) * Hswid) / Ranget)
#define	HSAK_Y(K)	(Hsa.cy - (((K) - Mink) * Hsht) / (Maxk-Mink))
#define	VOLT_I(T)	(((T) * Hswid) / Ranget)

/* vertical scroll area macros */
#define	VSAY_K(Y)	(1 + (((Vsa.cy - (Y)) * KRANGE - 1) / Vspix))
#define	VSAK_Y(K)	(Vsa.cy - ((K) * Vspix) / KRANGE)

#define	C_OFF	0x0000		/* channel is off */
#define	C_ON	0x0001		/* channel is on (play & display) */
#define	C_GREY	0x0002		/* channel gets highlight (display in grey) */

#ifdef	LINT
short	Ltgrey_bits[], Grey_bits[], Dkgrey_bits[];
Icon	*Chedicon;
Pixrect	Vuoff, Vuon, Vugoff, Vugon;
#else	~LINT
short	Ltgrey_bits[]	= {
	0x0000, 0x5555, 0x0000, 0xAAAA,
	0x0000, 0x5555, 0x0000, 0xAAAA,
	0x0000, 0x5555, 0x0000, 0xAAAA,
	0x0000, 0x5555, 0x0000, 0xAAAA,
};
short	Grey_bits[]	= {
	0xAAAA, 0x5555, 0xAAAA, 0x5555,
	0xAAAA, 0x5555, 0xAAAA, 0x5555,
	0xAAAA, 0x5555, 0xAAAA, 0x5555,
	0xAAAA, 0x5555, 0xAAAA, 0x5555,
};
short	Dkgrey_bits[]	= {
	0xFFFF, 0x5555, 0xFFFF, 0xAAAA,
	0xFFFF, 0x5555, 0xFFFF, 0xAAAA,
	0xFFFF, 0x5555, 0xFFFF, 0xAAAA,
	0xFFFF, 0x5555, 0xFFFF, 0xAAAA,
};
static	short Chedicon_bits[] = {
/* Format_version=1, Width=64, Height=64, Depth=1, Valid_bits_per_item=16
 */
	0xFFFF,0xFFFF,0xFFFF,0xFFFF,0x8000,0x0000,0x0000,0x0001,
	0x8000,0x0000,0x0000,0x0001,0x87FF,0xFFFF,0xFFFF,0xFFC1,
	0x8400,0x0000,0x0000,0x0041,0x84EE,0xEEEE,0xEEEE,0xEE41,
	0x84EE,0xEEEE,0xEEEE,0xEE41,0x84EE,0xEEEE,0xEEEE,0xEE41,
	0x8400,0x0000,0x0000,0x0041,0x87FF,0xFFFF,0xFFFF,0xFFC1,
	0x8000,0x0000,0x0000,0x0001,0x8000,0x0000,0x0000,0x0001,
	0x8000,0x0000,0x0000,0x0001,0x8000,0x0000,0x0000,0x0001,
	0x8000,0x0000,0x0040,0x0001,0x8000,0x0000,0x0040,0x0001,
	0x9FFF,0xFFFF,0xFFFF,0xFFF9,0x9401,0x0000,0x2040,0x0011,
	0x9401,0x0000,0x2040,0x0011,0x9401,0x0002,0x2040,0x0011,
	0x9401,0x0002,0x2040,0x0011,0x9401,0x0002,0x2040,0x0011,
	0x9FFF,0xFFFF,0xFFFF,0xFFF9,0x9401,0x0002,0x23C0,0x0011,
	0x9401,0x0002,0x26C0,0x0011,0x9401,0x0202,0x2440,0x0011,
	0x940F,0x0202,0x26C1,0xF811,0x941F,0x0202,0x23C1,0xF811,
	0x9FFF,0xFFFF,0xFFFF,0xFFF9,0x941F,0x021E,0x2040,0x0011,
	0x940E,0x023E,0x2040,0x0011,0x9400,0x023E,0x2040,0x0011,
	0x9400,0x023E,0x23C0,0x0011,0x9400,0x021C,0x26C0,0x0011,
	0x9FFF,0xFFFF,0xFFFF,0xFFF9,0x9400,0x1E00,0x26C0,0x0011,
	0x9400,0x3600,0x23C0,0x0011,0x9400,0x2200,0x2040,0x0011,
	0x9400,0x3600,0x23C0,0x0011,0x9400,0x1C00,0x26C0,0x0011,
	0x9FFF,0xFFFF,0xFFFF,0xFFF9,0x8000,0x0000,0x06C0,0x0001,
	0x8000,0x0000,0x0380,0x0001,0x8000,0x0000,0x0000,0x0001,
	0xAAAA,0xAAAA,0xAAAA,0xAAA9,0x9555,0x5555,0x5555,0x5555,
	0xAAAA,0xAAAA,0xAAAA,0xAAA9,0x9555,0x5555,0x5555,0x5555,
	0xAAAA,0x8288,0x8082,0xAAA9,0x9555,0x1111,0x1511,0x5555,
	0xAAAA,0x8888,0x8A88,0xAAA9,0x9555,0x1511,0x1511,0x5555,
	0xAAAA,0x8A80,0x8288,0xAAA9,0x9555,0x1511,0x1511,0x5555,
	0xAAAA,0x8888,0x8A88,0xAAA9,0x9555,0x1111,0x1511,0x5555,
	0xAAAA,0x8288,0x8082,0xAAA9,0x9555,0x5555,0x5555,0x5555,
	0xAAAA,0xAAAA,0xAAAA,0xAAA9,0x9555,0x5555,0x5555,0x5555,
	0xAAAA,0xAAAA,0xAAAA,0xAAA9,0x9555,0x5555,0x5555,0x5555,
	0x8000,0x0000,0x0000,0x0001,0xFFFF,0xFFFF,0xFFFF,0xFFFF
};
DEFINE_ICON_FROM_IMAGE(Chedicon, Chedicon_bits);
static	short Vuoff_bits[] = {
	0xffff, 0xffff, 0xd557, 0xebeb, 0xdd77, 0xfaab, 0xfd5f, 0xeeab, 
	0xd757, 0xebab, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 
};
static	short Vuon_bits[] = {
	0xffff, 0xffff, 0xc003, 0xc3c3, 0xcc33, 0xd00b, 0xe017, 0xc023, 
	0xc043, 0xc183, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 
};
static	short Vugoff_bits[] = {
	0xffff, 0xffff, 0xd557, 0xebeb, 0xdd77, 0xfaab, 0xfd5f, 0xeeab, 
	0xd757, 0xffff, 0xd113, 0xc447, 0xd113, 0xc447, 0xd113, 0xffff, 
};
static	short Vugon_bits[] = {
	0xffff, 0xffff, 0xc003, 0xc3c3, 0xcc33, 0xd00b, 0xe017, 0xc023, 
	0xc043, 0xffff, 0xd113, 0xc447, 0xd113, 0xc447, 0xd113, 0xffff, 
};
mpr_static(Vuoff, 16, 16, 1, Vuoff_bits);
mpr_static(Vuon, 16, 16, 1, Vuon_bits);
mpr_static(Vugoff, 16, 16, 1, Vugoff_bits);
mpr_static(Vugon, 16, 16, 1, Vugon_bits);
#endif	~LINT

typedef	struct	recta	{
	int	ox, oy, cx, cy;
} RECTA;
typedef	struct	notestr	{
	long	beg, end;	/* times of key-on & key-off in input */
	short	bx, ex, y;	/* pixel addresses of key-on & key-off */
	char	chn, key, vel;	/* channel, key number & velocity */
	char	which;		/* NBOTH, NBEG, or NEND */
} NOTE;

char	Oldfile[128];		/* where the data comes from for "read" */
char	Newfile[128];		/* where the data goes for "write" */
char	Workfile[128];		/* where it lives while we work */
char	Flbuf[128];		/* frame label buffer */
char	*Selfile = "/tmp/chedsel";	/* where the selected piece goes */
char	*Playfile = "/tmp/chedplay";	/* where stuff to be played goes */
char	*Playcmd = "play";
char	*Mergecmd = "merge";
char	*Tshiftcmd = "tshift";
int	M1args[]	= {	/* must agree with M1choic item */
	1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 12, 16,
};
int	M2args[]	= {	/* must agree with M2choic item */
	1, 2, 3, 4, 6, 8, 12, 16,
};
int	Tclefk[]	= { 77, 74, 71, 67, 64, };
int	Bclefk[]	= { 57, 53, 50, 47, 43, };
int	Chans[MAXCHAN];		/* mode of channel, on/off, black/grey */
int	Key[MAXCHAN][MAXKEY];	/* key-on count, used in several routines */
int	Vel[MAXCHAN][MAXKEY];	/* vels playing, used in several routines */
int	Vol[VOLRES];		/* key velocities (volumes) */
int	Kiu[MAXKEY];		/* keys in use (in current data) */
int	Barline[MAXBL];		/* where the TCWMEs are */
int	Nbl	= 0;		/* index into MAXBL */
int	Debug	= 0;		/* for extra debugging output */
int	Dirty	= 0;		/* changed but unwritten */
int	Tgrid	= 0;		/* grid ticks per bar */
int	Mymouse	= 0;		/* who "owns" the mouse at the moment */
int	Bmode = BM_SWEEP;	/* are we sweeping areas or dragging notes? */
int	Gmode = GM_UNSEL;	/* where non-note data (garbage) goes */
int	M1mode = MM_OFF;	/* metronome beats per bar */
int	M2mode = 0;		/* metronome beat length */
int	Smode	= SM_STATES;	/* does selpart() select states or events */
int	Vmode = VM_OFF;		/* whether to show velocities */
int	Barlen	= 480;		/* MIDI clocks per bar, 0=> use TCWMEs */
int	Bpb	= 4;		/* beats per bar */
int	Baroff	= -2;		/* set to first TCWME (% Barlen) by process() */
int	Tempo	= 100;		/* basis for play -t argument */
int	Margin	= 10;		/* guaranteed empty space around score */
int	Ppm;			/* pixels per moment (see CPM def) */
int	Ppk;			/* pixels per note (vertically) */
int	Playpid	= 0;		/* >0 ==> something is playing */
int	Mink, Maxk;		/* min and max+1 key values in input */
int	Rangek;			/* range of key values in input */
int	Cmink, Cmaxk;		/* min and max+1 key values to display */
int	Crangek;		/* range of key values to display */
int	Smink, Smaxk;		/* min, max+1 key values in sel [selpart()] */
int	Skmin, Skmax;		/* min, max+1 possible keys in Sel */
int	Hswid;			/* width of Hsa, pixels */
int	Hsht;			/* height of Hsa, pixels */
int	Hshht;			/* half height of Hsa, pixels */
int	Vspix;			/* height of Vsa & Vka, pixels */
int	Csizx, Csizy;		/* max width & height of Ca, pixels */
int	Nnotes;			/* number of notes in Note[] */
long	Mint, Maxt;		/* min and max+1 times in input (Mint=0) */
long	Ranget;			/* range of time in input */
long	Cmint, Cmaxt;		/* min and max+1 times to display */
long	Cranget;		/* range of time to display */
long	Smint, Smaxt;		/* min, max+1 times in sel [selpart()] */
long	Stmin, Stmax;		/* min, max+1 possible times in Sel */
FILE	*Wfp;			/* slightly preprocessed data */
RECTA	Da;			/* entire drawing area (minus Margin) */
RECTA	Ca;			/* chart area */
RECTA	Hsa;			/* horizontal scroll area */
RECTA	Vsa;			/* vertical scroll area */
RECTA	Vka;			/* vertical keyboard area */
RECTA	Sel;			/* selected area */
NOTE	Note[MAXNOTES];		/* used for grab/drag mode (Bmode) */
Pixrect	*Ltgrey;		/* light grey texture */
Pixrect	*Grey;			/* grey texture */
Pixrect	*Dkgrey;		/* dark grey texture */
Pixrect	*Vupr[]	= { &Vuoff, &Vuon, &Vugoff, &Vugon, };
Frame	Dframe;			/* whole display's frame */
Canvas	Dcanvas;		/* whole display's canvas */
#ifdef	LINT
Pixwin	*Dpw;			/* whole canvas pixwin */
#else
Pixwin	*Dpw	= (Pixwin *) 0;	/* whole canvas pixwin */
#endif
Pixfont	*Fontp;			/* the font we use (almost everywhere) */
int	Fontwidth;		/* the width of chars in *Fontp */
Panel	Controls;		/* control panel */
int	Cntlshow = TRUE;	/* whether to show control panel */
Panel_item Bslide;		/* id for clocks/bar slider */
Panel_item Oslide;		/* id for offset slider */
Panel_item Gslide;		/* id for ticks/bar slider */
Panel_item Tslide;		/* id for tempo slider */
Panel_item Cminmsg;		/* id for Cmint message */
Panel_item Cmaxmsg;		/* id for Cmint message */
Panel_item Bchoic;		/* id for Bmode choice */
Panel_item Gchoic;		/* id for Gmode choice */
Panel_item M1choic;		/* id for M1mode choice */
Panel_item M2choic;		/* id for M1mode choice */
Panel_item Schoic;		/* id for Smode choice */
Panel_item Vchoic;		/* id for Vmode choice */
Panel_item Vuchoic[MAXCHAN];	/* id for Vu choice */
Menu	B2mp;			/* normal middle button menu */
Menu	B2emp;			/* no selected area middle button menu */
Menu	B3mp;			/* normal right button menu */
Menu	B3emp;			/* right button menu for scroll areas*/
Cursor	Curs[2];		/* cursors for BM_SWEEP & BM_DRAG */

/* Button 2 menu - operations on the selected area */
#define	BBOX2	1
#define	CLOSE2	2
#define	CUTBB2	3
#define	CUT2	4
#define	FILTER2	5
#define	GLOM2	6
#define	IPROC2	7
#define	INFO2	8
#define	OPEN2	9
#define	PASTE2	10
#define	PLAY2	11
#define	WRITE2	12
#define	ZOOMI2	13
#define	ZOOMO2	14

/* Button 3 menu - operations on the entire file */
#define	CHDIR3	1
#define	DRAG3	2
#define	FILTER3	3
#define	IPROC3	4
#define	PLAY3	5
#define	QUIT3	6
#define	READ3	7
#define	SWEEP3	8
#define	WRITE3	9

char	*clk2c();
long	mmputm();
FILE	*nopen();

main(argc, argv)
char	*argv[];
{
	int i, barspec;
	extern FILE *sopen();

	setbuf(stderr, 0);
	barspec = 0;
	for (i = 1; i < argc; i++) {
	    if (argv[i][0] == '-') {
		switch (argv[i][1]) {
		case 'B':
		    Barlen = atoi(&argv[i][2]);
		    break;
		case 'b':
		    barspec = atoi(&argv[i][2]);
		    break;
		case 'd':
		    Debug++;
		    break;
		case 'm':
		    Margin = atoi(&argv[i][2]);
		    break;
		case 't':
		    Tgrid = atoi(&argv[i][2]);
		    break;
		default:
		    goto syntax;
		}
	    } else if (*Oldfile == 0)
		sprintf(Oldfile, "%s", argv[i]);
	    else {
syntax:
		fprintf(stderr, "Usage: %s [options] [file]\n", argv[0]);
		fprintf(stderr, "-B120\tSet bar length to 120 MPU clocks\n");
		fprintf(stderr, "-b12\tShow the first 12 bars\n");
		fprintf(stderr, "-d\tTurn Debug on\n");
		fprintf(stderr, "-m20\tProvide a 20 pixel margin\n");
		fprintf(stderr, "-t4\tSet ticks/bar to 4\n");
		exit(2);
	    }
	}
	if (!*Oldfile || !initfile(Oldfile)) {		/* no file read */
	    Mint = 0;
	    if (Barlen)
		Maxt = 2 * Barlen;
	    else
		Maxt = 4 * MPU_CLOCK_PERIOD;
	    Mink = MAXCMINK;
	    Maxk = MINCMAXK;
	    Wfp = 0;
	}
	Cmink = min(Mink - 2, MAXCMINK);
	Cmaxk = max(Maxk + 9, MINCMAXK);
	Cmint = Mint;
	Cmaxt = Maxt;
	if (barspec > 0) {
	    if (Barlen)
		Cmaxt = Cmint + barspec * Barlen + 1;
	    else if (Nbl >= barspec)
		Cmaxt = Barline[barspec] + 1;
	}
	if (!(Fontp = pf_open(FONTPATH)))
	    Fontp = pf_default();
	Fontwidth = Fontp->pf_defaultsize.x;
/****
	if (fork() > 0)
	    exit(0);
****/
	miscinits();
	window_main_loop(Dframe);
	unlink(Workfile);
	exit(0);
}

miscinits()	/* one-time SunView inits */
{
	Rect tmp;
	void resized(), canvasevent();

	B2mp = menu_create(
	    MENU_INITIAL_SELECTION,		MENU_SELECTED,
	    MENU_INITIAL_SELECTION_SELECTED,	FALSE,
	    MENU_NCOLS,				2,
	    MENU_TITLE_ITEM,	"-= SELECTED",
	    MENU_TITLE_ITEM,	"AREA  OPS =-",
	    MENU_STRING_ITEM,	"PLAY",		PLAY2,
	    MENU_STRING_ITEM,	"BB CUT",	CUTBB2,
	    MENU_STRING_ITEM,	"GLOM",		GLOM2,
	    MENU_STRING_ITEM,	"PASTE",	PASTE2,
	    MENU_STRING_ITEM,	"BBOX",		BBOX2,
	    MENU_STRING_ITEM,	"CUT",		CUT2,
	    MENU_STRING_ITEM,	"OPEN",		OPEN2,
	    MENU_STRING_ITEM,	"CLOSE",	CLOSE2,
	    MENU_STRING_ITEM,	"INFO",		INFO2,
	    MENU_STRING_ITEM,	"FILTER",	FILTER2,
	    MENU_STRING_ITEM,	"IPROC",	IPROC2,
	    MENU_STRING_ITEM,	"WRITE",	WRITE2,
	    MENU_STRING_ITEM,	"ZOOM IN",	ZOOMI2,
	    MENU_STRING_ITEM,	"ZOOM OUT",	ZOOMO2,
	0);
	B2emp = menu_create(MENU_STRINGS,
	    "No area selected.", "Sweep an area with the left button", 0,
	0);
	B3mp = menu_create(
	    MENU_INITIAL_SELECTION, MENU_SELECTED,
	    MENU_INITIAL_SELECTION_SELECTED, TRUE,
	    MENU_TITLE_ITEM,	"-= GLOBAL =-",
	    MENU_STRING_ITEM,	"CHDIR",	CHDIR3,
	    MENU_STRING_ITEM,	"FILTER",	FILTER3,
	    MENU_STRING_ITEM,	"IPROC",	IPROC3,
	    MENU_STRING_ITEM,	"SWEEP",	SWEEP3,
	    MENU_STRING_ITEM,	"DRAG",		DRAG3,
	    MENU_STRING_ITEM,	"PLAY",		PLAY3,
	    MENU_STRING_ITEM,	"READ",		READ3,
	    MENU_STRING_ITEM,	"WRITE",	WRITE3,
	    MENU_STRING_ITEM,	"QUIT",		QUIT3,
	0);
	B3emp = menu_create(MENU_STRINGS,
	    "Use the left button to sweep a range.",
	    "Use the middle button to move the range.",
	    0,
	0);
	Curs[BM_SWEEP] = cursor_create(
	    CURSOR_SHOW_CROSSHAIRS,	TRUE,
	    CURSOR_CROSSHAIR_GAP,	0,
	0);
	Curs[BM_DRAG] = cursor_create(
	    CURSOR_SHOW_CROSSHAIRS,	TRUE,
	    CURSOR_CROSSHAIR_GAP,	5,
	0);
	tmp.r_left = 8;
	tmp.r_top = 64;
	tmp.r_width = 1136;	/* max width - 16 */
	tmp.r_height = 660;	/* max height - 240 */
	sprintf(Flbuf, "-=[ CHED ]=-   src:%s   dst:%s", Oldfile, Newfile);
	Dframe = window_create(NULL, FRAME,
	    FRAME_LABEL,	Flbuf,
	    FRAME_ICON,		&Chedicon,
	    WIN_FONT,		Fontp,
	    FRAME_OPEN_RECT,	&tmp,
	0);
	gencntl();		/* called here to reserve its space */
	Dcanvas = window_create(Dframe, CANVAS,
	    CANVAS_FIXED_IMAGE,	FALSE,
	    CANVAS_RESIZE_PROC,	resized,
	    WIN_CURSOR,		Curs[BM_SWEEP],
	0);
	window_set(Dcanvas, WIN_CONSUME_PICK_EVENTS,
	    WIN_NO_EVENTS,
	    LOC_WINENTER, LOC_WINEXIT, LOC_MOVE, WIN_MOUSE_BUTTONS, 0,
	0);			/* set the canvas defaults */
	/* set non-defaults that we need */
	window_set(Dcanvas, WIN_CONSUME_KBD_EVENTS, WIN_ASCII_EVENTS, 0, 0);
	window_set(Dcanvas, WIN_CONSUME_PICK_EVENTS, LOC_DRAG, 0, 0);
	window_set(Dcanvas, WIN_EVENT_PROC, canvasevent, 0);
	Dpw = canvas_pixwin(Dcanvas);
	Ltgrey = mem_point(16, 16, 1, Ltgrey_bits);
	Grey = mem_point(16, 16, 1, Grey_bits);
	Dkgrey = mem_point(16, 16, 1, Dkgrey_bits);
}

gencntl()	/* generate control panel; called also to re-generate */
{
	char tbuf[8], ibuf[8];
	int swidth, fwidth, vuwidth, i, x, y;
	void cntlchng(), panelevent(), vuevent();

	if (Controls) {
	    window_destroy(Controls);
	    Controls = 0;
	}
	fwidth = (int) window_get(Dframe, WIN_WIDTH);
	if (fwidth > 495) {
	    swidth = (fwidth - 475) | 1;
	    vuwidth = 30;
	} else {
	    swidth = fwidth / 10;
	    vuwidth = fwidth / MAXCHAN;
	}
	Controls = window_create(Dframe, PANEL,
	    WIN_X,			0,
	    WIN_Y,			0,
	    WIN_WIDTH,			WIN_EXTEND_TO_EDGE,
	    WIN_SHOW,			Cntlshow,
	    WIN_FONT,			Fontp,
	    PANEL_SHOW_MENU,		FALSE,
	    PANEL_LABEL_BOLD,		TRUE,
	    PANEL_ITEM_X_GAP,		25,
	    PANEL_BACKGROUND_PROC,	panelevent,
	0);
	i = max(0, Barlen - (3 * swidth) / 4);
	Bslide = panel_create_item(Controls, PANEL_SLIDER,
	    PANEL_LABEL_STRING,		"CLOX/BAR",
	    PANEL_SLIDER_WIDTH,		swidth,
	    PANEL_VALUE,		Barlen,
	    PANEL_MIN_VALUE,		i,
	    PANEL_MAX_VALUE,		i + swidth - 1,
	    PANEL_NOTIFY_PROC,		cntlchng,
	    PANEL_EVENT_PROC,		panelevent,
	0);
	Baroff = Barlen? Baroff : 0;
	Oslide = panel_create_item(Controls, PANEL_SLIDER,
	    PANEL_LABEL_STRING,		"  OFFSET",
	    PANEL_SLIDER_WIDTH,		swidth,
	    PANEL_VALUE,		Baroff,
	    PANEL_MIN_VALUE,		0,
	    PANEL_MAX_VALUE,		Barlen? Barlen - 1 : 0,
	    PANEL_NOTIFY_PROC,		cntlchng,
	    PANEL_EVENT_PROC,		panelevent,
	0);
	Tgrid = Barlen? Tgrid : 0;
	Gslide = panel_create_item(Controls, PANEL_SLIDER,
	    PANEL_LABEL_STRING,		"TICKS/BAR ",
	    PANEL_SLIDER_WIDTH,		swidth,
	    PANEL_VALUE,		Tgrid,
	    PANEL_MIN_VALUE,		0,
	    PANEL_MAX_VALUE,		Barlen? 48 : 0,
	    PANEL_NOTIFY_PROC,		cntlchng,
	    PANEL_EVENT_PROC,		panelevent,
	0);
	Tslide = panel_create_item(Controls, PANEL_SLIDER,
	    PANEL_LABEL_STRING,		"   TEMPO",
	    PANEL_SLIDER_WIDTH,		swidth,
	    PANEL_VALUE,		Tempo,
	    PANEL_MIN_VALUE,		10,
	    PANEL_MAX_VALUE,		250,
	    PANEL_NOTIFY_PROC,		cntlchng,
	    PANEL_EVENT_PROC,		panelevent,
	0);
	Gchoic = panel_create_item(Controls, PANEL_CYCLE,
	    PANEL_ITEM_Y,		4,
	    PANEL_LABEL_STRING,		"Non-key data:",
	    PANEL_CHOICE_STRINGS,	"IGNORED",
					"SELECTED",
					"UNSELECTED",
					"BOTH",
					0,
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_MENU_TITLE_STRING, 	"PROG. CHANGE, CONTROLLERS, ETC.",
	    PANEL_MENU_CHOICE_STRINGS,	"Ignore all but key-on & key-off",
					"Leave non-key data in selected area",
					"Leave non-key data in u selected area",
					"Leave non-key data in both",
					0,
	    PANEL_VALUE,	Gmode,
	    PANEL_NOTIFY_PROC,	cntlchng,
	0);
	x = (int) panel_get(Gchoic, PANEL_ITEM_X);
	M1choic = panel_create_item(Controls, PANEL_CHOICE,
	    PANEL_ITEM_X,		x,
	    PANEL_ITEM_Y,		22,
	    PANEL_DISPLAY_LEVEL,	PANEL_CURRENT,
	    PANEL_FEEDBACK,		PANEL_NONE,
	    PANEL_LABEL_STRING,	"Metronome:",
	    PANEL_CHOICE_STRINGS,
		"OFF", " 1", " 2", " 3", " 4", " 5", "6",
		" 7", " 8", " 9", "10", "12", "16",
		0,
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_MENU_TITLE_STRING, 	"METRONOME SETTINGS",
	    PANEL_MENU_CHOICE_STRINGS,
		"No Metronome", "1 / measure", "2 / measure", "3 / measure",
		"4 / measure", "5 / measure", "6 / measure", "7 / measure",
		"8 / measure", "9 / measure", "10 / measure", "12 / measure",
		"16 / measure",
		0,
	    PANEL_VALUE,	M1mode,
	    PANEL_NOTIFY_PROC,	cntlchng,
	0);
	M2choic = panel_create_item(Controls, PANEL_CHOICE,
	    PANEL_ITEM_X,		x + 120,
	    PANEL_ITEM_Y,		22,
	    PANEL_DISPLAY_LEVEL,	PANEL_CURRENT,
	    PANEL_FEEDBACK,		PANEL_NONE,
	    PANEL_LABEL_STRING,		"/",
	    PANEL_CHOICE_STRINGS,
		"1", "2", "3", "4", "6", "8", "12", "16",
		0,
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_MENU_TITLE_STRING, 	"BEAT LENGTH",
	    PANEL_MENU_CHOICE_STRINGS,
		"whole note", "half note", "1/3 note", "quarter note",
		"quarter triplet", "eighth note", "eighth triplet",
		"sixteenth note",
		0,
	    PANEL_VALUE,	M2mode,
	    PANEL_NOTIFY_PROC,	cntlchng,
	0);
	Vchoic = panel_create_item(Controls, PANEL_CYCLE,
	    PANEL_ITEM_X,	x,
	    PANEL_ITEM_Y,	40,
	    PANEL_LABEL_STRING,	"Velocity:",
	    PANEL_CHOICE_STRINGS,	"NOT DISPLAYED",
					"NOTE WIDTH",
					0,
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_MENU_TITLE_STRING, 	"KEY VELOCITY DISPLAY",
	    PANEL_MENU_CHOICE_STRINGS,	"Bar thickness held constant",
					"Bar thickness indicates key velocity",
					0,
	    PANEL_VALUE,	Vmode,
	    PANEL_NOTIFY_PROC,	cntlchng,
	0);
	Schoic = panel_create_item(Controls, PANEL_CYCLE,
	    PANEL_ITEM_X,	x,
	    PANEL_ITEM_Y,	58,
	    PANEL_LABEL_STRING,	"Select:",
	    PANEL_CHOICE_STRINGS,	"NOTES",
					"KEY EVENTS",
					0,
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_MENU_TITLE_STRING, 	"SELECTION & EDITING MODE",
	    PANEL_MENU_CHOICE_STRINGS,	"Edit notes (cut can ADD key events)",
					"Edit events (cut only deletes events)",
					0,
	    PANEL_VALUE,	Smode,
	    PANEL_NOTIFY_PROC,	cntlchng,
	0);
	Bchoic = panel_create_item(Controls, PANEL_CYCLE,
	    PANEL_ITEM_X,	x,
	    PANEL_ITEM_Y,	76,
	    PANEL_LABEL_STRING,	"Left Button:",
	    PANEL_CHOICE_STRINGS,	"SWEEP",
					"DRAG",
					0,
	    PANEL_SHOW_MENU,		TRUE,
	    PANEL_MENU_TITLE_STRING, 	"LEFT & MIDDLE MOUSE BUTTON OPS",
	    PANEL_MENU_CHOICE_STRINGS,	"Sweep with left; menu with middle",
					"Grab and move with left & middle",
					0,
	    PANEL_VALUE,	Bmode,
	    PANEL_NOTIFY_PROC,	cntlchng,
	0);
	y = (int) panel_get(Tslide, PANEL_ITEM_Y);
	Cminmsg = panel_create_item(Controls, PANEL_MESSAGE,
	    PANEL_LABEL_STRING,	clk2c(Cmint, 0),
	    PANEL_LABEL_X,	4,
	    PANEL_LABEL_Y,	y + 32,
	    PANEL_LABEL_BOLD,	TRUE,
	0);
	Cmaxmsg = panel_create_item(Controls, PANEL_MESSAGE,
	    PANEL_LABEL_STRING,	clk2c(Cmaxt-1, 0),
	    PANEL_LABEL_X,	fwidth - 11 * 10,
	    PANEL_LABEL_Y,	y + 32,
	    PANEL_LABEL_BOLD,	TRUE,
	0);
	for (i = 0; i < MAXCHAN; i++) {
	    x = (fwidth + (i + i - 15) * vuwidth) / 2;
	    sprintf(ibuf, "%d", i + 1);
	    sprintf(tbuf, "Chan %d", i + 1);
	    Vuchoic[i] = panel_create_item(Controls, PANEL_CHOICE,
		PANEL_MENU_TITLE_STRING, tbuf,
		PANEL_LABEL_STRING,	ibuf,
		PANEL_LABEL_X,		x - (i < 9? 12 : 16),
		PANEL_LABEL_Y,		y + 18,
		PANEL_VALUE_X,		x - vuwidth / 2,
		PANEL_VALUE_Y,		y + 32,
		PANEL_DISPLAY_LEVEL,	PANEL_CURRENT,
		PANEL_CHOICE_IMAGES,	&Vuoff, &Vuon, &Vugoff, &Vugon, 0,
		PANEL_SHOW_MENU,	TRUE,
		PANEL_MENU_CHOICE_STRINGS, "OFF","ON","OFF grey","ON grey",0,
		PANEL_FEEDBACK,		PANEL_NONE,
		PANEL_VALUE,		Chans[i],
		PANEL_NOTIFY_PROC,	cntlchng,
		PANEL_EVENT_PROC,	vuevent,
	    0);
	}
	window_fit_height(Controls);
	if (Cntlshow)
	    window_set(Controls, WIN_SHOW, Cntlshow, 0);
}

void
cntlchng(item, value, event)
Panel_item item;
Event	*event;
{
	register int change;

	change = 0;
	if (item == Bslide) {
	    if (Barlen != value) {
		Barlen = value;
		gencntl();
		change++;
	    }
	} else if (item == Oslide) {
	    if (Baroff != value) {
		Baroff = value;
		change++;
	    }
	} else if (item == Gslide) {
	    if (Tgrid != value) {
		Tgrid = value;
		change++;
	    }
	} else if (item == Tslide) {
	    Tempo = value;
	} else if (item == Bchoic) {
	    if ((Bmode = value) == BM_DRAG)
		change++;
	    window_set(Dcanvas, WIN_CURSOR, Curs[Bmode], 0);
	} else if (item == Gchoic) {
	    Gmode = value;
	} else if (item == M1choic) {
	    M1mode = value;
	} else if (item == M2choic) {
	    M2mode = value;
	} else if (item == Schoic) {
	    Smode = value;
	} else if (item == Vchoic) {
	    if (Vmode != value) {
		Vmode = value;
		change++;
	    }
	} else
	    change = (vucheck(item, value) == 1);
	if (change)
	    redraw();
}

void
canvasevent(window, event, arg)
Window	window;
Event	*event;
{
	register int i, change;

	i = event_id(event);
	if (i != MS_LEFT && i != MS_MIDDLE && i != MS_RIGHT && i != LOC_DRAG)
	    return;
	if (Mymouse == HSMOUSE		/* cursor in horizontal scroll area */
	 || (Mymouse == 0 && einr(event, Hsa))) {
	    Mymouse = HSMOUSE;
	    change = hsamouse(window, event);
	} else if (Mymouse == VSMOUSE	/* cursor in vertical scroll area */
	 || (Mymouse == 0 && einr(event, Vsa))) {
	    Mymouse = VSMOUSE;
	    change = vsamouse(window, event);
	} else if (Mymouse == CAMOUSE	/* cursor in chart area */
	 || Mymouse == 0) {
	    Mymouse = CAMOUSE;
	    change = camouse(window, event);
	}
	if (change)
	    redraw();
}

void
panelevent(object, event)	/* panel area events where RIGHT => menu3ca */
Panel	object;			/* could be Panel_item, also */
Event	*event;
{
	register int i;

	i = event_id(event);
	if (i == MS_RIGHT && menu3ca(Controls, event))
	    redraw();
	else
	    panel_default_handle_event(object, event);
}

void
vuevent(item, event)		/* vu meter events */
Panel_item	item;
Event	*event;
{
	register int i, value, nv;

	i = event_id(event);
	if (event_is_down(event) && (i == MS_LEFT || i == MS_MIDDLE)) {
	    value = (int) panel_get_value(item);
	    nv = value ^ (i == MS_LEFT? 1 : 2);
	    panel_set_value(item, nv);
	    if ((i = vucheck(item, nv)) >= 0) {
		if (i == 1)
		    redraw();
		return;
	    }
	}
	panel_default_handle_event(item, event);
}

vucheck(item, nv)	/* ret -1 for not vu item, else 1 if redraw needed */
Panel_item item;
{
	register int i, ov;

	for (i = MAXCHAN; --i >= 0; ) {
	    if (Vuchoic[i] == item) {
		if ((ov = Chans[i]) != nv) {
		    Chans[i] = nv;
		    if ((ov & 1) || (nv & 1))
			return(1);
		}
		return(0);
	    }
	}
	return(-1);
}

void
resized(canvas, width, height)	/* called when canvas is resized */
Canvas	canvas;
{
	if (canvas != Dcanvas)
	    return;		/* oops */
	Da.ox = Da.oy = Margin;
	Da.cx = width - Margin;
	Da.cy = height - Margin;	/* entire canvas area (-borders) */
	redraw();
}

redraw()	/* called whenever resized or cmd changes view */
{		/* needs values for Da set by resized() */
	register int x, y;

	if (Wfp)
	    fclose(Wfp);
	if ((Wfp = nopen(Workfile, "r")) == (FILE *) NULL)
	    exit(1);
	/* recalc Ca, Csizx, Csizy, Ppm, caoff, etc. when scaling occurs */
	/* after this, CAT_X(), CAY_K(), CAX_T(), etc. are usable */
	Ca.ox = Da.ox + VSAWIDTH + CAMARGIN;	/* maximum chart area rect */
	Ca.cx = Da.cx - CAMARGIN - VKAWIDTH;
	Ca.oy = Da.oy + CAMARGIN;
	Ca.cy = Da.cy - MSBH - CAMARGIN;
	Csizx = Ca.cx - Ca.ox;		/* maximum chart size */
	Csizy = Ca.cy - Ca.oy;
	Cranget = Cmaxt - Cmint;
	Crangek = Cmaxk - Cmink;
	Ppm = (Csizx * CPM) / Cranget;
	Ppk = Csizy / Crangek;
	x = Csizx - (Cranget * Ppm) / CPM;	/* adjust to make both */
	y = Csizy - Crangek * Ppk;		/* Ppm and Ppk exact */
	Ca.ox += x / 2;			/* center in maximum area */
	Ca.oy += y / 2;
	Ca.cx -= x / 2;
	Ca.cy -= y / 2;
	Csizx = Ca.cx - Ca.ox;		/* real chart area in use */
	Csizy = Ca.cy - Ca.oy;
	Hsa.ox = Ca.ox;			/* horizontal scroll area */
	Hsa.cx = Ca.cx;
	Hsa.oy = Ca.cy + CAMARGIN;
	Hsa.cy = Da.cy;
	Hswid = Hsa.cx - Hsa.ox;
	Hshht = (Hsa.cy - Hsa.oy) >> 1;
	Hsht = Hshht << 1;
	Vsa.ox = Da.ox;			/* vertical scroll area */
	Vsa.cx = Ca.ox - CAMARGIN;
	Vsa.oy = Ca.oy;
	Vsa.cy = Ca.cy;
	Vspix = Vsa.cy - Vsa.oy;
	Vka.ox = Ca.cx + CAMARGIN;		/* vertical keyboard area */
	Vka.cx = Da.cx;
	Vka.oy = Ca.oy;
	Vka.cy = Ca.cy;
	if (Skmax != Skmin) {
	    Sel.ox = CAT_X(Stmin);
	    Sel.cx = CAT_X(Stmax);
	    Sel.oy = CAK_YA(Skmax);
	    Sel.cy = CAK_YB(Skmin);
	}
	pw_writebackground(Dpw, 0, 0, Da.cx+Margin, Da.cy+Margin, PIX_CLR);
	process(Wfp);
	drawstaves();
	drawhs();
	drawvs();
	RECTINV(Sel);
	hsaselinv();
}

initfile(file)		/* set up for file; label window, call copyfile() */
char	file[];
{
	int i;
	FILE *ifp;

	Sel.ox = Sel.oy = Sel.cx = Sel.cy = 0;
	Stmin = Stmax = 0;
	Skmin = Skmax = 0;
	Baroff = -2;		/* so process() will look for first TCWME */
	if (*file) {
	    if ((ifp = sopen(file, "r")) == NULL) {
		perror(file);
		return(0);
	    }
	    sprintf(Newfile, "%s", file);
	} else {
	    if (*Newfile == '\0')
		sprintf(Newfile, "ched.new");
	    return(0);
	}
	sprintf(Flbuf, "-=[ CHED ]=-   src:%s   dst:%s", file, Newfile);
	window_set(Dframe, FRAME_LABEL, Flbuf, 0);
	sprintf(Workfile, "/tmp/ched%d", getpid());
	if ((Wfp = nopen(Workfile, "w")) == (FILE *) NULL)
	    exit(1);
	for (i = MAXCHAN; --i >= 0; Chans[i] = C_OFF);
	copyfile(ifp, Wfp);
	if (ifp != stdin)
	    fclose(ifp);
	fclose(Wfp);
	if ((Wfp = nopen(Workfile, "r")) == (FILE *) NULL)
	    exit(1);
	Dirty = 0;
	return(1);
}

copyfile(ifp, ofp)	/* measure file for MINk, Maxk, Mint, Maxt, etc. */
FILE	*ifp, *ofp;	/* and copy into ofp (workfile) */
{
	register int status, mode, chan;
	long now;
	MCMD *mp;

	mmputm((FILE *) 0, (MCMD *) 0);
	for (now = 0L; mp = getmcmd(ifp, now); now = mmputm(ofp, mp)) {
	    status = *mp->cmd;
	    mode = (status & M_CMD_MASK);
	    chan = (status & M_CHAN_MASK);
	    if (mode == CH_KEY_ON || mode == CH_KEY_OFF) {
		if (mode == CH_KEY_OFF) {	/* turn key_off into key_on */
		    mp->cmd[0] = CH_KEY_ON | chan;
		    mp->cmd[2] = 0;
		} else if (mp->cmd[2] != 0) {
		    if ((Chans[chan] & C_ON) != C_ON) {
			Chans[chan] |= C_ON;
			panel_set_value(Vuchoic[chan], Chans[chan]);
		    }
		}
	    }
	}
	mmputm(ofp, (MCMD *) 0);
}

process(ifp)	/* set Baroff to -2 to look for first TCWME */
FILE	*ifp;
{
	register int k, v, status, mode;
	int i, chan;
	long now, start[MAXCHAN][MAXKEY];
	MCMD *mp;

	if (Dpw)
	    pw_batch_on(Dpw);
	Nnotes = 0;
	for (i = sizeof Vol / sizeof Vol[0]; --i >= 0; Vol[i] = 0);
	for (i = sizeof Kiu / sizeof Kiu[0]; --i >= 0; Kiu[i] = 0);
	for (chan = 0; chan < MAXCHAN; chan++)
	    if (Chans[chan] & C_ON)
		for (k = Mink; k < Maxk; k++)
		    Key[chan][k] = 0;
	ROP(Hsa.ox-1, Hsa.oy-1, Hsa.cx+1, Hsa.cy+1, PIX_SET);
	ROP(Hsa.ox+1, Hsa.oy+1, Hsa.cx-1, Hsa.cy-1, PIX_CLR);
	Nbl = 0;
	for (now = 0L; mp = getmcmd(ifp, now); ) {
	    now = mp->when;
	    status = mp->cmd[0];
	    if (status == RT_TCWME) {
		Barline[Nbl++] = now;
		if (Baroff < 0)
		    Baroff = now;
	    }
	    mode = (status & M_CMD_MASK);
	    if (mode != CH_KEY_ON)
		continue;
	    chan = status & M_CHAN_MASK;
	    if ((Chans[chan] & C_ON) == 0)
		continue;
	    k = mp->cmd[1];
	    v = mp->cmd[2];
	    if (v == 0) {			/* key-off */
		--Key[chan][k];
		if (Key[chan][k] < 0) {		/* extra key-off */
		    Key[chan][k] = 0;
		    continue;
		}
		if (Key[chan][k] == 0) {	/* end of note */
		    plotnote(start[chan][k], k, now, Vel[chan][k], chan);
		    addvol(start[chan][k], now, Vel[chan][k]);
		}
	    } else {				/* key-on */
		if (Key[chan][k] != 0) {
		    plotnote(start[chan][k], k, now, Vel[chan][k], chan);
		    addvol(start[chan][k], now, Vel[chan][k]);
		}
		start[chan][k] = now;
		Vel[chan][k] = v;
		Key[chan][k]++;
	    }
	}
	for (chan = 0; chan < MAXCHAN; chan++) {   /* check notes left on */
	    if (Chans[chan] & C_ON) {
		for (k = Cmink; k < Cmaxk; k++) {
		    if (Key[chan][k]) {
			plotnote(start[chan][k], k, now, Vel[chan][k], chan);
			addvol(start[chan][k], now, Vel[chan][k]);
		    }
		}
	    }
	}
	if (Dpw)
	    pw_batch_off(Dpw);			/* for deferred update */
	if (Baroff == -2)			/* no TCWME found */
	    Baroff = 0;
	else if (Barlen > 0)
	    Baroff %= Barlen;
	panel_set_value(Oslide, Baroff);
}

/* plot notes, Vmode=VM_OFF => update Hsa, Bmode=BM_DRAG => update Note[] */
plotnote(bt, k, et, v, chn)
long	bt, et;
{
	int bx, ex, y, dy;

	if (Vmode == VM_OFF && (Chans[chn] & C_ON)) { /* draw Hor scroll data */
	    bx = HSAT_X(bt);
	    ex = HSAT_X(et);
	    if (ex == bx)
		ex++;
	    y = HSAK_Y(k);
	    if (Hsa.oy < y && y < Hsa.cy)
		pw_vector(Dpw, bx, y, ex, y, PIX_SET, 1);
	}
	Kiu[k]++;
	if (et < Mint || Maxt <= bt || k < Mink || Maxk <= k
	 || (bx = CAT_X(bt)) >= Ca.cx
	 || (ex = CAT_X(et)) < Ca.ox)
	    return;
	bx = max(bx, Ca.ox);
	if (ex == bx)
	    ex++;
	else
	    ex = min(ex, Ca.cx);
	y = CAK_Y(k);
	if (Ca.oy < (y - NOTEHH) && (y + NOTEHH) < Ca.cy) {
	    dy = Vmode==VM_ON? (v * NOTEHH) / MIDVEL : NOTEHH;
	    if (dy <= 0)
		dy = 1;
	    if (Chans[chn] & C_GREY)
		TREPL(bx, y - dy, ex - bx, 2 * dy, PIX_SRC, Grey);
	    else
		ROP(bx, y - dy, ex, y + dy, PIX_SET);
	    if (bx != Ca.ox)
		pw_vector(Dpw, bx, y - dy - 2, bx, y, PIX_SET, 1);
	    if (ex != Ca.cx)
		pw_vector(Dpw, ex - 1, y, ex - 1, y + dy + 2, PIX_SET, 1);
	}
	if (Bmode == BM_DRAG) {
	    if (Nnotes >= MAXNOTES) {
		fprintf(stderr, "Too many notes in display for DRAG mode\n");
		panel_set_value(Bchoic, Bmode = BM_SWEEP);
	    } else {
		Note[Nnotes].beg = bt;
		Note[Nnotes].end = et;
		Note[Nnotes].chn = chn;
		Note[Nnotes].key = k;
		Note[Nnotes].vel = v;
		Note[Nnotes].bx = bx;
		Note[Nnotes].ex = ex;
		Note[Nnotes].y = y;
		Nnotes++;
	    }
	}
}

addvol(bt, et, v)
long bt, et;
register int v;
{
	register int i, ei;

	i = VOLT_I(bt);
	ei = VOLT_I(et);
	if (ei == i)
	    ei++;
	for (; i < ei; i++) {
	    Vol[i] += v;
	    v = (36 * v + 18) / 37;
	}
}

drawhs()	/* display horiz. scroll area */
{
	register int i, x, y, dy, maxvol;
	long t;

	if (Vmode == VM_ON) {	/* if !Vmode, data drawn in plotnote() */
	    maxvol = 0;
	    for (i = 0; i < Hswid; i++)
		if (Vol[i] > maxvol)
		    maxvol = Vol[i];
	    maxvol++;
	    y = (Hsa.oy + Hsa.cy) >> 1;
	    for (i = 0; i < Hswid; i++) {
		x = Hsa.ox + i;
		dy = (Vol[i] * Hshht) / maxvol;
		pw_vector(Dpw, x, y + dy, x, y - dy, PIX_INV, 1);
	    }
	}
	dy = Hshht / 3;
	if (Barlen) {			/* bar line ticks */
	    for (t = Baroff + (Baroff==0? Barlen : 0); t < Maxt; t += Barlen) {
		x = HSAT_X(t);
		pw_vector(Dpw, x, Hsa.oy, x, Hsa.oy + dy, PIX_INV, 1);
		pw_vector(Dpw, x, Hsa.cy, x, Hsa.cy - dy, PIX_INV, 1);
	    }
	} else {
	    for (i = 0; i < Nbl; i++) {
		x = HSAT_X(Barline[i]);
		pw_vector(Dpw, x, Hsa.oy, x, Hsa.oy + dy, PIX_INV, 1);
		pw_vector(Dpw, x, Hsa.cy, x, Hsa.cy - dy, PIX_INV, 1);
	    }
	}
	fliphs();
}

RECTA
hsrect()
{
	RECTA q;

	q.ox = HSAT_X(Cmint);
	q.oy = Hsa.oy;
	q.cx = HSAT_X(Cmaxt);
	q.cy = Hsa.cy;
	return(q);
}

fliphs()
{
	RECTA q;

	q = hsrect();
	RECTINV(q);
}

drawvs()	/* display vert. scroll area */
{
	register int i, x, y, dx, dy;

	ROP(Vsa.ox-1, Vsa.oy-1, Vsa.cx+1, Vsa.cy+1, PIX_SET);
	ROP(Vsa.ox+1, Vsa.oy+1, Vsa.cx-1, Vsa.cy-1, PIX_CLR);
	for (i = 5; --i >= 0; ) {	/* horizontal staff lines */
	    y = VSAK_Y(Tclefk[i]);
	    pw_vector(Dpw, Vsa.ox, y, Vsa.cx, y, PIX_INV, 1);
	    y = VSAK_Y(Bclefk[i]);
	    pw_vector(Dpw, Vsa.ox, y, Vsa.cx, y, PIX_INV, 1);
	}
	x = (Vsa.ox + Vsa.cx) / 2;
	dx = (Vsa.cx - Vsa.ox) / 6;
	dy = (VSAK_Y(0) - VSAK_Y(1)) / 2;
	for (i = sizeof Kiu / sizeof Kiu[0]; --i >= 0; ) {
	    if (Kiu[i]) {
		y = VSAK_Y(i);
		ROP(x - dx, y - dy, x + dx, y + dy, PIX_SET);
	    }
	}
	flipvs();
}

RECTA
vsrect()	/* generate vertical scroll rectangle */
{
	RECTA q;

	q.ox = Vsa.ox;
	q.oy = VSAK_Y(Cmaxk-1)+1;
	q.cx = Vsa.cx;
	q.cy = VSAK_Y(Cmink);
	return(q);
}

flipvs()
{
	RECTA q;

	q = vsrect();
	RECTINV(q);
}

einr(e, r)	/* true if event e is in rectangle r */
Event	*e;
RECTA	r;
{
	register int x, y;

	x = e->ie_locx;
	y = e->ie_locy;
	return (r.ox <= x && x <= r.cx && r.oy <= y && y <= r.cy);
}

hsamouse(window, event)		/* mouse event in horizontal scroll area */
Window	window;
Event	*event;
{
	int id, x, y;
	long t, q;
	RECTA hsrect(), oldR;
	static int state, onend, dt;
	static long lastt, ocmaxt, ocmint;

	id = event_id(event);
	oldR = hsrect();
	if (state == 0) {		/* waiting for button down */
	    ocmaxt = Cmaxt;
	    ocmint = Cmint;
	    lastt = -1;
	    t = HSAX_T(event_x(event));
	    if (id == MS_LEFT) {
		onend = HSAX_T(event_x(event));
		Cmint = Mint;
		Cmaxt = Maxt;
		panel_set(Cminmsg, PANEL_LABEL_STRING, clk2c(Cmint, 0), 0);
		panel_set(Cmaxmsg, PANEL_LABEL_STRING, clk2c(Cmaxt, 0), 0);
		uprect(oldR, hsrect());
		state = id;
	    } else if (id == MS_MIDDLE) {
		dt = Cmaxt - Cmint;
		if (Cmint <= t && t <= Cmaxt) {
		    x = (oldR.ox + oldR.cx) >> 1;
		    y = (oldR.oy + oldR.cy) >> 1;
		    window_set(window, WIN_MOUSE_XY, x, y, 0);
		}
		state = id;
	    } else if (id == MS_RIGHT)
		menu_show(B3emp, window, event, 0);
	} else if (id == LOC_DRAG) {
	    if (state == MS_LEFT) {
		t = HSAX_T(event_x(event));
		t = Maxt < t? Maxt : t;
		t = Mint > t? Mint : t;
		if (t != lastt) {
		    Cmaxt = onend > t? onend : t;
		    Cmint = onend < t? onend : t;
		    if (Cmaxt == Cmint) {
			Cmaxt = Maxt;
			Cmint = Mint;
		    }
		    panel_set(Cminmsg, PANEL_LABEL_STRING, clk2c(Cmint, 0), 0);
		    panel_set(Cmaxmsg, PANEL_LABEL_STRING, clk2c(Cmaxt, 0), 0);
		    uprect(oldR, hsrect());
		    lastt = t;
		}
	    } else if (state == MS_MIDDLE) {
		t = HSAX_T(event_x(event));
		t = (q = Maxt - dt / 2) < t? q : t;
		t = (q = Mint + dt / 2) > t? q : t;
		if (t != lastt) {
		    Cmint = t - dt / 2;
		    Cmaxt = Cmint + dt;
		    panel_set(Cminmsg, PANEL_LABEL_STRING, clk2c(Cmint, 0), 0);
		    panel_set(Cmaxmsg, PANEL_LABEL_STRING, clk2c(Cmaxt, 0), 0);
		    uprect(oldR, hsrect());
		    lastt = t;
		}
	    }
	} else if (event_is_up(event)) {			/* button up */
	    state = 0;
	    Mymouse = 0;			/* give up the mouse */
	    return(Cmaxt != ocmaxt || Cmint != ocmint);
	}
	return(0);
}

vsamouse(window, event)		/* mouse event in vertical scroll area */
Window	window;
Event	*event;
{
	register int k, id, x, y, t;
	RECTA vsrect(), oldR;
	static int state, onend, lastk, dk, ocmaxk, ocmink;

	id = event_id(event);
	oldR = vsrect();
	if (state == 0) {		/* waiting for button down */
	    ocmaxk = Cmaxk;
	    ocmink = Cmink;
	    lastk = -1;
	    k = VSAY_K(event_y(event));
	    if (id == MS_LEFT) {
		onend = k + 1;
		Cmink = MINKEY;
		Cmaxk = MAXKEY;
		uprect(oldR, vsrect());
		state = id;
	    } else if (id == MS_MIDDLE) {
		dk = Crangek;
		if (Cmink <= k && k <= Cmaxk) {
		    x = (oldR.ox + oldR.cx) >> 1;
		    y = (oldR.oy + oldR.cy) >> 1;
		    window_set(window, WIN_MOUSE_XY, x, y, 0);
		}
		state = id;
	    } else if (id == MS_RIGHT)
		menu_show(B3emp, window, event, 0);
	} else if (id == LOC_DRAG) {
	    if (state == MS_LEFT) {
		k = VSAY_K(event_y(event));
		k = min(MAXKEY, k);
		k = max(MINKEY, k);
		if (k != lastk) {
		    Cmink = min(onend, k);
		    Cmaxk = max(onend, k);
		    if (Cmaxk == Cmink) {
			Cmink = MINKEY;
			Cmaxk = MAXKEY;
		    }
		    uprect(oldR, vsrect());
		    lastk = k;
		}
	    } else if (state == MS_MIDDLE) {
		k = VSAY_K(event_y(event));
		t = MAXKEY - dk / 2;
		k = min(t, k);
		t = MINKEY + dk / 2;
		k = max(t, k);
		if (k != lastk) {
		    Cmink = k - dk / 2;
		    Cmaxk = Cmink + dk;
		    uprect(oldR, vsrect());
		    lastk = k;
		}
	    }
	} else if (event_is_up(event)) {		/* button up */
	    state = 0;
	    Mymouse = 0;			/* give up the mouse */
	    return(Cmink != ocmink || Cmaxk != ocmaxk);
	}
	return(0);
}

camouse(window, event)		/* mouse event in chart area */
Window	window;
Event	*event;
{
	register int i, change;

	change = 0;
	i = event_id(event);
	if (i == MS_LEFT
	 || (i == LOC_DRAG && window_get(Dcanvas, WIN_EVENT_STATE, MS_LEFT)))
	    if (Bmode == BM_SWEEP)
		selectca(window, event);
	    else
		change = dragca(window, event);
	else if (i == MS_MIDDLE)
	    change = menu2ca(window, event);
	else if (i == MS_RIGHT)
	    change = menu3ca(window, event);
	return(change);
}

dragca(window, event)		/* drag notes around in selected area */
Window	window;
Event	*event;
{
	register int x, y, i, retval, k;
	char *file;
	RECTA new;
	static int bi, dx, lastk;
	static RECTA old;

	retval = 0;
	x = event_x(event);
	y = event_y(event);
	if (event_id(event) == MS_LEFT) {
	    if (event_is_down(event)) {			/* button down */
		for (i = 0; i < Nnotes; i++)
		    if (Note[i].bx < x && Note[i].y - NOTEHH < y
		    && x < Note[i].ex && y < Note[i].y + NOTEHH)
			break;
		if (i >= Nnotes)
		    bi = -1;
		else {
		    bi = i;
		    old.ox = Note[bi].bx;
		    old.oy = Note[bi].y - NOTEHH;
		    old.cx = Note[bi].ex;
		    old.cy = Note[bi].y + NOTEHH;
		    dx = old.cx - old.ox;
		    lastk = Note[bi].key;
		    if (Smode == SM_STATES) {
			Note[bi].which = NBOTH;
			x = (old.ox + old.cx) / 2;
		    } else {
			i = (x - Note[bi].bx) < (Note[bi].ex - x)? NBEG : NEND;
			Note[bi].which = i;
			x = (i == NBEG)? old.ox : old.cx;
		    }
		    y = Note[bi].y;
		    window_set(window, WIN_MOUSE_XY, x, y, 0);
		    RECTINV(old);
		}
	    } else if (bi >= 0) {			/* button up */
		RECTINV(old);
		i = bi? 0 : 1;
		Note[i] = Note[bi];
		Note[i].beg = CAX_T(old.ox);
		Note[i].end = CAX_T(old.cx);
		Note[i].key = lastk;
		if (movenote(Workfile, file = "/tmp/chedtmp", bi, i)) {
		    unlink(Workfile);
		    link(file, Workfile);
		    unlink(file);
		    retval = Dirty = 1;
		}
		bi = -1;
		Mymouse = 0;			/* give up the mouse */
	    }
	} else if (bi >= 0 && (x != old.ox || y != old.oy)
	 && event_id(event) == LOC_DRAG) {
	    if (Note[bi].which == NBOTH) {
		new.ox = x - dx / 2;
		new.cx = new.ox + dx;
	    } else if (Note[bi].which == NBEG) {
		new.ox = x;
		new.cx = old.cx;
	    } else {
		new.ox = old.ox;
		new.cx = x;
	    }
	    k = CAY_K(y);
	    if (new.ox != old.ox || new.cx != old.cx || k != lastk) {
		new.oy = CAK_Y(k) - NOTEHH;
		new.cy = new.oy + Ppk;
		if (k == lastk)
		    uprect(old, new);
		else {
		    RECTINV(new);
		    RECTINV(old);
		}
		lastk = k;
		old = new;
	    }
	}
	return(retval);
}

selectca(window, event)		/* define selected area in chart area */
Window	window;
Event	*event;
{
	register int x, y;
	static int firstx, firsty, lastx, lasty;
	RECTA old;

	x = event_x(event);
	y = event_y(event);
	if (event_id(event) == MS_LEFT) {
	    if (event_is_down(event)) {			/* button down */
		RECTINV(Sel);
		hsaselinv();
		Sel.ox = Sel.cx = firstx = lastx = x;
		Sel.oy = Sel.cy = firsty = lasty = y;
	    } else {					/* button up */
		Stmin = CAX_T(Sel.ox);
		Stmax = CAX_T(Sel.cx) + 1;
		Skmin = CAY_KA(Sel.cy);
		Skmax = CAY_KB(Sel.oy) + 1;
		firstx = firsty = lastx = lasty = 0;
		Mymouse = 0;			/* give up the mouse */
		hsaselinv();
	    }
	} else if (event_id(event) == LOC_DRAG && (x != lastx || y != lasty)) {
	    old = Sel;
	    Sel.ox = min(firstx, x);
	    Sel.oy = min(firsty, y);
	    Sel.cx = max(firstx, x);
	    Sel.cy = max(firsty, y);
	    lastx = x;
	    lasty = y;
	    uprect(old, Sel);
	}
}

menu2ca(window, event)
Window	window;
Event	*event;
{
	register int i, change;
	char *file;

	if (Stmin == Stmax - 1) {
	    menu_show(B2emp, window, event, 0);
	    return(0);
	}
	change = 0;
	i = (int) menu_show(B2mp, window, event, 0);
	Mymouse = 0;			/* give up the mouse */
	if (i <= 0)
	    return(0);
	RECTINV(Ca);
	switch(i) {
	case BBOX2:
	    bbox();
	    break;
	case CLOSE2:
	    if (openclose(file = "/tmp/chedtmp", CLOSE)) {
		unlink(Workfile);
		link(file, Workfile);
		unlink(file);
		change = 1;
		Dirty++;
	    }
	    break;
	case CUTBB2:			/* falls through */
	    bbox();
	case CUT2:
	    if (selpart(Selfile, file = "/tmp/chedtmp")) {
		unlink(Workfile);
		link(file, Workfile);
		unlink(file);
		change = 1;
		Dirty++;
	    }
	    break;
	case IPROC2:
	    change = iproc(0);
	    break;
	case FILTER2:
	    if (change = filter(0))
		Dirty++;
	    break;
	case GLOM2:
	    selpart(Selfile, NO_FILE);
	    break;
	case INFO2:
	    RECTINV(Ca);
	    infomenu(window, event);
	    RECTINV(Ca);
	    break;
	case OPEN2:
	    if (openclose(file = "/tmp/chedtmp", OPEN)) {
		unlink(Workfile);
		link(file, Workfile);
		unlink(file);
		change = 1;
		Dirty++;
	    }
	    break;
	case PASTE2:
	    if (change = paste(Selfile, Workfile))
		Dirty++;
	    break;
	case PLAY2:
	    change = play(0);
	    break;
	case WRITE2:
	    change = writefile(0);
	    break;
	case ZOOMI2:
	    change = zoomin();
	    break;
	case ZOOMO2:
	    change = zoomout();
	    break;
	}
	if (change == 0)
	    RECTINV(Ca);
	return(change);
}

bbox()
{
	selpart(NO_FILE, NO_FILE);	/* set Smink, Smint, etc. */
	RECTINV(Sel);
	hsaselinv();
	if (Smint < Smaxt && Smink < Smaxk) {
	    Sel.ox = CAT_X(Stmin = Smint) - 1;
	    Sel.cx = CAT_X((Stmax = Smaxt) - 1) + 1;
	    Sel.cy = CAK_YB(Skmin = Smink);
	    Sel.oy = CAK_YA((Skmax = Smaxk) - 1);
	    RECTINV(Sel);
	    hsaselinv();
	} else {
	    Sel.ox = Sel.cx = Sel.oy = Sel.cy = 0;
	    Stmin = Stmax = 0;
	    Skmax = Skmin = 0;
	}
}

typedef	struct	tstr	{
	long	bar;
	int	beat;
	int	clock;
} Tstr;

infomenu(window, event)
Window	window;
Event	*event;
{
	char buf[6][32];
	int i;
	Menu m;
	char *key2name();

	m = menu_create(MENU_STRINGS, "SELECTED AREA", 0, 0);
	sprintf(buf[0], "Beg:%s", clk2c(Stmin, 0));
	menu_set(m, MENU_STRING_ITEM, buf[0], 0, 0);
	sprintf(buf[1], " raw clocks: %d", Stmin);
	menu_set(m, MENU_STRING_ITEM, buf[1], 0, 0);
	sprintf(buf[2], "End:%s", clk2c(Stmax - 1, 0));
	menu_set(m, MENU_STRING_ITEM, buf[2], 0, 0);
	sprintf(buf[3], " raw clocks: %d", Stmax - 1);
	menu_set(m, MENU_STRING_ITEM, buf[3], 0, 0);
	i = Skmax - 1;
	sprintf(buf[4], "Top key: %d (X%x) %s", i, i, key2name(i));
	menu_set(m, MENU_STRING_ITEM, buf[4], 0, 0);
	sprintf(buf[5], "Bot key: %d (X%x) %s", Skmin, Skmin, key2name(Skmin));
	menu_set(m, MENU_STRING_ITEM, buf[5], 0, 0);
	menu_show(m, window, event, 0);
	menu_destroy(m);
}

char	*
clk2c(clox, bp)
long	clox;
char	*bp;
{
	Tstr t;
	static char buf[128];

	clk2t(clox, &t);
	sprintf(bp? bp : (bp = buf), "%3d+%d+%03d", t.bar, t.beat, t.clock);
	return(bp);
}

clk2t(clox, tp)
long	clox;
Tstr	*tp;
{
	int i;

	if (Barlen) {
	    tp->bar = clox / Barlen;
	    clox -= tp->bar * Barlen;
	    tp->beat = (clox * Bpb) / Barlen;
	    tp->clock = clox - (tp->beat * Barlen) / Bpb;
	} else {		/* using TCWME */
	    for (i = Nbl; --i >= 0 && Barline[i] > Stmin; );
	    tp->bar = i;
	    if (i < 0) {
	    } else {
		clox -= Barline[i];
		if (i == Nbl - 1) {
		    tp->beat = 0;
		} else {
		     i = (Barline[i + 1] - Barline[i]);
		     tp->beat = (clox * Bpb) / i;
		     tp->clock = clox - (tp->beat * i) / Bpb;
		}
	    }
	}
}

menu3ca(window, event)
Window	window;		/* may be a Panel, if called from panelevent() */
Event	*event;
{
	register int i, change;
	char dirbuf[32];

	change = 0;
	i = (int) menu_show(B3mp, window, event, 0);
	Mymouse = 0;			/* give up the mouse */
	if (i <= 0)
	    return(0);
	RECTINV(Ca);
	switch(i) {
	case CHDIR3:
	    dirbuf[0] = '.';
	    dirbuf[1] = '\0';
	    i = GSTR("Change to directory: ", dirbuf, sizeof dirbuf);
	    if (i && *dirbuf)
		chdir(dirbuf);
	    break;
	case DRAG3:
	    if (Bmode != BM_DRAG) {
		panel_set_value(Bchoic, Bmode = BM_DRAG);
		change++;
		window_set(Dcanvas, WIN_CURSOR, Curs[Bmode], 0);
	    }
	    break;
	case FILTER3:
	    if (change = filter(1))
		Dirty++;
	    break;
	case IPROC3:
	    change = iproc(1);
	    break;
	case PLAY3:
	    change = play(1);
	    break;
	case QUIT3:
	    change = quit();
	    break;
	case SWEEP3:
	    if (Bmode != BM_SWEEP) {
		panel_set_value(Bchoic, Bmode = BM_SWEEP);
		window_set(Dcanvas, WIN_CURSOR, Curs[Bmode], 0);
	    }
	    panel_set_value(Bchoic, Bmode = BM_SWEEP);
	    break;
	case READ3:
	    change = readfile();
	    break;
	case WRITE3:
	    change = writefile(1);
	    break;
	}
	if (change == 0)
	    RECTINV(Ca);
	return(change);
}

play(entire)
{
	register int i;
	char *file, buf[2][16];

	while (Playpid > 0)
	    if ((i = wait(0)) == -1 || i == Playpid)
		Playpid = 0;
	if (entire)
	    file = Workfile;
	else
	    selpart(file = Playfile, NO_FILE);
	if (M1mode == MM_OFF) {
	    sprintf(buf[0], "-t%d", Tempo);
	    Playpid = forkexec(-1, -1, Playcmd, buf[0], file, 0, 0, 0);
	} else {
	    sprintf(buf[0], "-m%d/%d", M1args[M1mode], M2args[M2mode]);
	    sprintf(buf[1], "-t%d", Tempo);
	    Playpid = forkexec(-1, -1, Playcmd, buf[0], buf[1], file, 0, 0);
	}
	return(0);
}

quit()
{
	register int i;
	char buf[32];

	if (Dirty) {
	    sprintf(buf, "y");
	    i = GSTR("Unwritten changes!  Okay? ", buf, sizeof buf);
	    if (i == 0 || *buf != 'y')
		return(0);
	}
	while (Playpid > 0)
	    if ((i = wait(0)) == -1 || i == Playpid)
		Playpid = 0;
	if (Workfile && *Workfile)
	    unlink(Workfile);
	if (Playfile && *Playfile)
	    unlink(Playfile);
	if (Selfile && *Selfile)
	    unlink(Selfile);
	exit(0);
	/*NOTREACHED*/
}

readfile()
{
	char buf[32];
	int i, oldbo;

	if (Dirty) {
	    sprintf(buf, "y");
	    i = GSTR("UNWRITTEN CHANGES!  Okay? ", buf, sizeof buf);
	    if (i == 0 || *buf != 'y')
		return(0);
	}
	i = GSTR("File: ", Oldfile, sizeof Oldfile);
	if (i == 0 || *Oldfile == '\0')
	    return(0);
	oldbo = Baroff;
	Baroff = -2;
	if (initfile(Oldfile) == 0)
	    return(0);
	if (Baroff != oldbo)
	    gencntl();
	if (Cmint >= Maxt) {
	    Cmint = Mint;
	    panel_set(Cminmsg, PANEL_LABEL_STRING, clk2c(Cmint, 0), 0);
	}
	if (Maxt < Cmaxt) {
	    Cmaxt = Maxt;
	    panel_set(Cmaxmsg, PANEL_LABEL_STRING, clk2c(Cmaxt, 0), 0);
	}
	return(1);
}

/* Merge "part" (offset by Stmin) and "whole" into "Workfile" */
/* return 1 for success, 0 for failure */
paste(part, whole)
char	*part, *whole;
{
	char buf[512], *tmp1, *tmp2;
	int change;
	FILE *ifp, *tfp;

	change = 1;		/* assume we'll change something */
	tmp1 = "/tmp/chedpaste1";
	tmp2 = "/tmp/chedpaste2";
	ifp = tfp = (FILE *) NULL;
	sprintf(buf, "%s -c%d <%s >%s; %s %s %s >%s",
	 Tshiftcmd, Stmin, part, tmp1,
	 Mergecmd, whole, tmp1, tmp2);
	if (system(buf)) {
	    perror(buf);
	    change = 0;
	}
	if (change && (ifp = nopen(tmp2, "r")) == (FILE *) NULL)
	    change = 0;
	if (change && (tfp = nopen(Workfile, "w")) == (FILE *) NULL)
	    change = 0;
	if (change)
	    copyfile(ifp, tfp);
	if (ifp)
	    fclose(ifp);
	if (tfp)
	    fclose(tfp);
	unlink(tmp1);
	unlink(tmp2);
	if (!change)
	    fprintf(stderr, "Aborting\n");
	return(change);
}

/* Copy selected part of Workfile to selfile, the rest to unselfile. */
/* If either is NO_FILE, discard that data. */
/* Return 1 for success, 0 for failure. */
selpart(selfile, unselfile)
char	*selfile, *unselfile;
{
	register int k, v, c, status, mode;
	unsigned char mbuf[16];
	int chan, syet, sg, ug;
	int skey[MAXCHAN][MAXKEY];	/* key-on count in selected area */
	long now;
	FILE *ifp, *sfp, *ufp;
	MCMD m, *mp;

	m.cmd = mbuf;
	if ((ifp = nopen(Workfile, "r")) == NULL)
	    return(0);
	if (selfile == NO_FILE)
	    sfp = 0;
	else if ((sfp = nopen(selfile, "w")) == NULL) {
	    fclose(ifp);
	    return(0);
	}
	if (unselfile == NO_FILE)
	    ufp = 0;
	else if ((ufp = nopen(unselfile, "w")) == NULL) {
	    fclose(ifp);
	    if (sfp)
		fclose(sfp);
	    return(0);
	}
	Smink = 999;					/* we set these */
	Smaxk = 0;
	Smint = Stmax;
	Smaxt = Stmin;
	for (c = 0; c < MAXCHAN; c++)		/* only check selected notes */
	    if (Chans[c] & C_ON)
		for (k = Skmin; k < Skmax; k++)
		    Key[c][k] = skey[c][k] = 0;
	syet = 0;
	if (ufp)
	    iputmcmds(0, ufp, 0L);
	if (sfp)
	    iputmcmds(1, sfp, Stmin);
	for (now = 0L; mp = getmcmd(ifp, now); ) {
	    now = mp->when;
	    status = mp->cmd[0];
	    mode = status & M_CMD_MASK;
	    chan = status & M_CHAN_MASK;
	    if (syet == 0 && now >= Stmin		/* entering sel time */
	     && !sfp && !ufp)				/* but no output */
		syet = 1;
	    else if (syet == 0 && now >= Stmin) {	/* entering sel time */
		m.when = Stmin;				/* notes in play */
		m.len = 3;
		for (c = 0; c < MAXCHAN; c++) {
		    if (Chans[c] & C_ON) {
			m.cmd[0] = CH_KEY_ON | c;
			for (k = Skmin; k < Skmax; k++) {
			    if (Key[c][k]) {	/* still playing */
				m.cmd[1] = k;	/* already playing */
				if (ufp && Smode != SM_EVENTS) {
				    m.cmd[2] = 0;	/* stop in unsel */
				    putmcmds(0, &m);
				}
				if (sfp) {		/* start in sel */
				    m.cmd[2] = Vel[c][k];
				    putmcmds(1, &m);
				}
			    }
			}
		    }
		}
		syet = 1;
	    }
	    if (syet == 1 && now < Stmax) {		/* in selected time */
		if (mode == CH_KEY_ON && Smode == SM_EVENTS && ufp)
			skey[chan][mp->cmd[1]] += mp->cmd[2]? 1 : -1;
	    } else if (syet == 1 && now >= Stmax	/* leaving sel time */
	     && !sfp && !ufp) {				/* but no output */
		syet = 2;
	    } else if (syet == 1 && now >= Stmax) {	/* leaving sel time */
		m.when = Stmax - 1;
		m.len = 3;
		for (c = 0; c < MAXCHAN; c++) {
		    if (Chans[c] & C_ON) {
			m.cmd[0] = CH_KEY_ON | c;
			for (k = Skmin; k < Skmax; k++) {
			    if (Key[c][k]) {
				m.cmd[1] = k;
				if (sfp) {
				    m.cmd[2] = 0;
				    putmcmds(1, &m);
				}
				if (ufp && Smode != SM_EVENTS) {
				    m.cmd[2] = Vel[c][k];
				    putmcmds(0, &m);
				}
			    }
			    if (Smode == SM_EVENTS && skey[c][k]) {
				m.cmd[1] = k;
				m.cmd[2] = skey[c][k] > 0? Vel[c][k] : 0;
				putmcmds(0, &m);
			    }
			}
		    }
		}
		if (sfp) {
		    m.when = Stmax;
		    m.len = 1;
		    m.cmd[0] = MPU_NO_OP;
		    putmcmds(1, &m);
		    fclose(sfp);
		    sfp = 0;
		}
		if (!ufp)
		    break;
		syet = 2;
	    }
	    if (mode != CH_KEY_ON) {
		/* decide what to do with non-note "garbage" */
		sg = (syet == 1) && (Gmode & GM_SEL) && sfp;
		ug = (Gmode & GM_UNSEL) && ufp;
		if (sg)
		    putmcmds(1, mp);
		if (ug)
		    putmcmds(0, mp);
		continue;
	    }					/* only key-on/off after here */
	    k = mp->cmd[1];
	    v = mp->cmd[2];
	    if (v == 0) {			/* key-off */
		--Key[chan][k];
		if (Key[chan][k] < 0)		/* extra key-off */
		    Key[chan][k] = 0;
	    } else {				/* key-on */
		Vel[chan][k] = v;
		Key[chan][k]++;
	    }
	    if (syet == 1 && Skmin <= k && k < Skmax	/* in selected area */
	     && (Chans[chan] & C_ON)) {
/****/fprintf("sfp=0x%x (should not be 0)\n", sfp);
		putmcmds(1, mp);
		if (v == 0) {
		    if (now >= Smaxt)
			Smaxt = now + 1;
		} else {
		    if (now < Smint)
			Smint = now;
		}
		if (k < Smink)
		    Smink = k;
		if (k >= Smaxk)
		    Smaxk = k + 1;
	    } else {				/* outside selected area */
/****/fprintf("ufp=0x%x (should not be 0)\n", ufp);
		putmcmds(0, mp);
	    }
	}
	fclose(ifp);
	if (sfp) {		/* we're still in the selected area */
	    m.when = now;
	    for (chan = 0; chan < MAXCHAN; chan++) {
		if (Chans[chan] & C_ON) {
		    m.cmd[0] = CH_KEY_ON | chan;
		    for (k = Skmin; k < Skmax; k++) {
			if (Key[chan][k]) {
			    m.cmd[1] = k;
			    putmcmds(1, &m);
			}
		    }
		}
	    }
	    fclose(sfp);
	}
	if (ufp)
	    fclose(ufp);
	return(1);
}

/* Open or close selected duration (Stmax - Stmin) at Stmin */
/* Return 1 for success, 0 for failure */
openclose(ofile, op)
char	*ofile;
{
	register int syet;
	long inow, oto, dt;
	FILE *ifp, *ofp;
	MCMD *mp;

	if ((dt = Stmax - 1 - Stmin) <= 0)
	    return(0);
	if ((ifp = nopen(Workfile, "r")) == NULL)
	    return(0);
	if ((ofp = nopen(ofile, "w")) == NULL) {
	    fclose(ifp);
	    return(0);
	}
	syet = oto = 0;
	putmcmd((FILE *) 0, (MCMD *) 0);
	for (inow = 0L; mp = getmcmd(ifp, inow); ) {
	    inow = mp->when;
	    if (syet == 0 && inow >= Stmin) {	/* just entered selected time */
		oto = (op == OPEN)? dt : -dt;
		syet = 1;
	    }
	    mp->when += oto;
	    if (oto < 0 && mp->when < Stmin)	/* CLOSE to start of Sel */
		mp->when = Stmin;
	    putmcmd(ofp, mp);
	}
	fclose(ifp);
	fclose(ofp);
	if (op == CLOSE)
	    Stmax = Stmin + 1;
	Maxt += oto;
	return(1);
}

/* copy ifile into ofile, deleting Note[on] and inserting Note[nn] */
/* N.B.: resets Mink, Maxk, Rangek, Mint, Maxt, Ranget, but no others */
/* Return 1 for success, 0 for failure */
movenote(ifile, ofile, on, nn)
char	*ofile;
{
	register int onstat;
	u_char mbuf[4];
	long now;
	FILE *ifp, *ofp;
	MCMD *mp, m;

	if (nn == on || on < 0 || nn < 0)
	    return(0);
	if ((ifp = nopen(ifile, "r")) == NULL)
	    return(0);
	if ((ofp = nopen(ofile, "w")) == NULL) {
	    fclose(ifp);
	    return(0);
	}
	m.cmd = mbuf;
	m.len = 3;
	onstat = (CH_KEY_ON | Note[on].chn);
	mmputm((FILE *) 0, (MCMD *) 0);
	for (now = 0L; mp = getmcmd(ifp, now); ) {
	    now = mp->when;
	    if (Note[nn].beg >= 0 && now > Note[nn].beg) {
		m.when = Note[nn].beg;
		m.cmd[0] = CH_KEY_ON | Note[nn].chn;
		m.cmd[1] = Note[nn].key;
		m.cmd[2] = Note[nn].vel;
		mmputm(ofp, &m);
		Note[nn].beg = -1;
	    }
	    if (Note[nn].end >= 0 && now > Note[nn].end) {
		m.when = Note[nn].end;
		m.cmd[0] = CH_KEY_ON | Note[nn].chn;
		m.cmd[1] = Note[nn].key;
		m.cmd[2] = 0;
		mmputm(ofp, &m);
		Note[nn].end = -1;
	    }
	    if (now == Note[on].beg
	     && mp->cmd[0] == onstat
	     && mp->cmd[1] == Note[on].key
	     && mp->cmd[2] == Note[on].vel) {
		Note[on].beg = -1;
		continue;
	    }
	    if (now == Note[on].end && mp->cmd[1] == Note[on].key
	     && (mp->cmd[0] & M_CHAN_MASK) == Note[on].chn
	     && ((mp->cmd[0] & M_CMD_MASK) == CH_KEY_OFF
	      || (mp->cmd[0] & M_CMD_MASK) == CH_KEY_ON && mp->cmd[2] == 0)) {
		Note[on].end = -1;
		continue;
	    }
	    mmputm(ofp, mp);
	}
	mmputm(ofp, (MCMD *) 0);
	fclose(ifp);
	fclose(ofp);
	return(1);
}

long
mmputm(ofp, mp)		/* put out MPU event, setting mins & maxs */
FILE	*ofp;		/* if ofp==0, just initialize */
MCMD	*mp;		/* if mp==0, finish up */
{
	static long last;

	if (!ofp) {
	    Maxk = 0;
	    Mink = 99999;
	    Mint = 0;
	    putmcmd((FILE *) 0, (MCMD *) 0);
	    last = 0;
	} else if (!mp) {
	    Maxt = last + 1;
	    Ranget = Maxt - Mint;
	    Rangek = Maxk - Mink;
	} else {
	    if ((mp->cmd[0] & M_CMD_MASK) == CH_KEY_ON) {
		if (mp->cmd[1] < Mink)
		    Mink = mp->cmd[1];
		if (mp->cmd[1] >= Maxk)
		    Maxk = mp->cmd[1] + 1;
	    }
	    last = putmcmd(ofp, mp);
	}
	return(last);
}

forkexec(in, out, p, a1, a2, a3, a4, a5)
char	*p;
{
	register int pid;

	switch (pid = fork()) {
	case -1:
	    perror("fork()");
	    break;
	case 0:
	    close(0);
	    if (in >= 0)
		dup2(0, in);
	    close(1);
	    if (out >= 0)
		dup2(1, out);
	    execlp(p, p, a1, a2, a3, a4, a5, 0);
	    perror(p);
	    exit(1);
	}
	return(pid);
}

writefile(entire)
{
	char *file, buf[512];
	int i, ifh, ofh;

	if (entire) {
	    i = GSTR("Write entire file to: ", Newfile, sizeof Newfile);
	    if (i == 0 || *Newfile == '\0')
		return(0);
	    file = Workfile;
	    Dirty = 0;
	} else {
	    i = GSTR("Write selected area to: ", Newfile, sizeof Newfile);
	    if (i == 0 || *Newfile == '\0')
		return(0);
	    selpart(file = "/tmp/chedtmp", NO_FILE);
	}
	sprintf(Flbuf, " CHED    src:%s   dst:%s", Oldfile, Newfile);
	window_set(Dframe, FRAME_LABEL, Flbuf, 0);
	if ((ifh = open(file, 0)) < 0)
	    perror(file);
	else if ((ofh = creat(Newfile, 0644)) < 0)
	    perror(Newfile);
	else {
	    while ((i = read(ifh, buf, sizeof buf)) > 0)
		write(ofh, buf, i);
	    close(ofh);
	}
	close(ifh);
	printf("%s written\n", Newfile);
	return(0);
}

zoomin()
{
	if (Stmin >= Stmax || Skmin >= Skmax)
	    return(0);
	Cmint = Stmin - 1;
	Cmaxt = Stmax + 1;
	Cmink = Skmin - 1;
	Cmaxk = Skmax + 1;
	panel_set(Cminmsg, PANEL_LABEL_STRING, clk2c(Cmint, 0), 0);
	panel_set(Cmaxmsg, PANEL_LABEL_STRING, clk2c(Cmaxt, 0), 0);
	return(1);
}

zoomout()
{
	int lx, hx, dx, ly, hy, dy, dy2, omink, omaxk, nmink, nmaxk;
	long omint, omaxt, nmint, nmaxt;

	omint = Cmint;
	omaxt = Cmaxt;
	omink = Cmink;
	omaxk = Cmaxk;
	dx = Sel.cx - Sel.ox;
	if (dx == 0) {
	    Cmint = Mint;
	    Cmaxt = Maxt;
	} else {
	    lx = Sel.ox - Vsa.cx;
	    hx = Sel.cx - Vsa.cx;
	    nmint = (omint * hx - omaxt * lx) / dx;
	    nmaxt = nmint + (Csizx * (omaxt - omint)) / dx;
	    Cmint = nmint < Mint? Mint : nmint;
	    Cmaxt = nmaxt > Maxt? Maxt : nmaxt;
	}
	dy = Sel.cy - Sel.oy;
	if (dy == 0) {
	    Cmink = MINKEY;
	    Cmaxk = MAXKEY;
	} else {
	    ly = Hsa.oy - Sel.oy;
	    hy = Hsa.oy - Sel.cy;
	    dy2 = dy >> 1;
	    nmink = (omink * ly - omaxk * hy + dy2) / dy;
	    nmaxk = nmink + (Csizy * (omaxk - omink) + dy2) / dy;
	    Cmink = nmink < MINKEY? MINKEY : nmink;
	    Cmaxk = nmaxk > MAXKEY? MAXKEY : nmaxk;
	}
	Stmin = omint;
	Stmax = omaxt;
	Skmin = omink;
	Skmax = omaxk;
	panel_set(Cminmsg, PANEL_LABEL_STRING, clk2c(Cmint, 0), 0);
	panel_set(Cmaxmsg, PANEL_LABEL_STRING, clk2c(Cmaxt, 0), 0);
	return(1);
}

iproc(entire)
{
	char buf[256];
	int i;
	static char cmd[128] = "da";

	i = GSTR("Command: ", cmd, sizeof cmd);
	if (i == 0 || *cmd == '\0')
	    return(0);
	if (entire) {
	    sprintf(buf, "cat %s | %s", Workfile, cmd);
	    if (system(buf))
		perror(buf);
	} else {
	    selpart(Selfile, NO_FILE);
	    sprintf(buf, "cat %s | %s", Selfile, cmd);
	    if (system(buf))
		perror(buf);
	}
	return(0);
}

filter(entire)
{
	char *therest, buf[256];
	int i;
	static char cmd[128];

	i = GSTR("Command: ", cmd, sizeof cmd);
	if (i == 0 || *cmd == '\0')
	    return(0);
	if (entire) {
	    unlink(Selfile);
	    link(Workfile, Selfile);
	    unlink(Workfile);
	    sprintf(buf, "%s <%s >%s", cmd, Selfile, Workfile);
	    if (system(buf)) {
		perror(buf);
		unlink(Workfile);
		link(Selfile, Workfile);
		unlink(Selfile);
	    }
	} else {
	    if (!selpart(Selfile, therest = "/tmp/chedtmp"))
		return(0);
	    sprintf(buf, "%s <%s >%s", cmd, Selfile, "/tmp/ched2");
	    if (system(buf)) {
		perror(buf);
		return(0);
	    }
	    i = paste("/tmp/ched2", therest);
	    unlink(therest);
	    if (!i)
		return(0);
	}
	return(1);
}

drawstaves()
{
	int i, lx, hx, ly, hy, bly, bhy, imax, k, mx, y;
	long t, dt;
	Pixrect *g;

	if (Dpw)
	    pw_batch_on(Dpw);
	lx = CAT_X(Cmint);			/* horizontal staff lines */
	lx = max(lx, Ca.ox);
	hx = CAT_X(Cmaxt);
	for (i = 5; --i >= 0; ) {
	    ly = CAK_Y(Tclefk[i]);
	    if (Ca.oy < ly && ly < Ca.cy)
		pw_vector(Dpw, lx, ly, hx, ly, PIX_SET, 1);
	    ly = CAK_Y(Bclefk[i]);
	    if (Ca.oy < ly && ly < Ca.cy)
		pw_vector(Dpw, lx, ly, hx, ly, PIX_SET, 1);
	}
	lx = Vka.ox;				/* keyboard */
	hx = Vka.cx;
	mx = (lx + lx + hx + hx + hx) / 5;
	pw_vector(Dpw, lx, Vka.oy, lx, Vka.cy, PIX_SET, 1);
	pw_vector(Dpw, hx, Vka.oy, hx, Vka.cy, PIX_SET, 1);
	for (k = Cmink; k <= Cmaxk; k++) {
	    y = CAK_Y(k);
	    ly = y + Ppk / 2;
	    i = k % 12;
	    i = i >= 5? i - 5: i;
	    if (i == 0)
		pw_vector(Dpw, lx, ly, hx, ly, PIX_SET, 1);
	    else if (i & 1) {
		hy = y - Ppk / 2;
		ROP(lx, hy, mx, ly, PIX_SET);
		pw_vector(Dpw, mx, y, hx, y, PIX_SET, 1);
	    }
	}
	ly = Ca.oy;				/* various vertical lines */
	hy = Ca.cy;
	if (Tgrid) {				/* grid ticks */
	    if (Barlen) {
		dt = Barlen / Tgrid;
		imax = (Cmaxt - Baroff + dt - 1) / dt;
		for (i = (Cmint - Baroff + dt - 1) / dt; i < imax; i++) {
		    t = Baroff + (i/Tgrid)*Barlen + (i%Tgrid) * dt;
		    lx = CAT_X(t);
		    g = Ltgrey;
		    if ((i % Tgrid) << 1 == Tgrid)
			g = Dkgrey;
		    else if (((i << 1) % Tgrid) << 1 == Tgrid)
			g = Grey;
		    TREPL(lx, ly, 1, hy - ly, PIX_OR, g);
		}
	    } else {
		/* this needs to be written */
	    }
	}
	ly = CAK_Y(Tclefk[0]);
	ly = max(ly, Ca.oy);
	hy = CAK_Y(Tclefk[4]);
	hy = min(hy, Ca.cy);
	bly = CAK_Y(Bclefk[0]);
	bly = max(bly, Ca.oy);
	bhy = CAK_Y(Bclefk[4]);
	bhy = min(bhy, Ca.cy);
	if (Barlen) {			/* vertical staff (bar) lines */
	    dt = Barlen;
	    for (t = Baroff + dt * ((Mint + dt - 1) / dt); t < Cmaxt; t += dt) {
		if ((lx = CAT_X(t)) > Ca.cx)
		    break;
		if (lx >= Ca.ox) {
		    pw_vector(Dpw, lx, ly, lx, hy, PIX_SET, 1);
		    pw_vector(Dpw, lx, bly, lx, bhy, PIX_SET, 1);
		}
	    }
	} else {
	    for (i = 0; i < Nbl; i++) {
		t = Barline[i];
		lx = CAT_X(t) - 1;
		if (lx > Ca.cx)
		    break;
		if (lx >= Ca.ox)
		    pw_vector(Dpw, lx, ly, lx, hy, PIX_SET, 1);
	    }
	}
	if (Dpw)
	    pw_batch_off(Dpw);			/* for deferred update */
}

hsaselinv()			/* invert selected area in the Hsa */
{
	register int t;
	RECTA hsasel;

	hsasel.ox = HSAT_X(Stmin);
	hsasel.cx = HSAT_X(Stmax);
	if (Vmode == VM_OFF) {
	    t = HSAK_Y(Skmax);
	    hsasel.oy = max(t, Hsa.oy);
	    t = HSAK_Y(Skmin);
	    hsasel.cy = min(t, Hsa.cy);
	} else {
	    hsasel.oy = Hsa.oy;
	    hsasel.cy = Hsa.cy;
	}
	RECTINV(hsasel);
}

uprect(old, new)		/* min changes to make new rect out of old */
RECTA	old, new;
{
	if (new.ox < old.ox) {
	    RINV(new.ox, new.oy, old.ox, new.cy);
	    new.ox = old.ox;
	} else if (old.ox < new.ox) {
	    RINV(old.ox, old.oy, new.ox, old.cy);
	    old.ox = new.ox;
	}
	if (new.oy < old.oy) {
	    RINV(new.ox, new.oy, new.cx, old.oy);
	    new.oy = old.oy;
	} else if (old.oy < new.oy) {
	    RINV(old.ox, old.oy, old.cx, new.oy);
	    old.oy = new.oy;
	}
	if (new.cx > old.cx) {
	    RINV(old.cx, new.oy, new.cx, new.cy);
	    new.cx = old.cx;
	} else if (old.cx > new.cx) {
	    RINV(new.cx, old.oy, old.cx, old.cy);
	    old.cx = new.cx;
	}
	if (new.cy > old.cy) {
	    RINV(new.ox, old.cy, new.cx, new.cy);
	    new.cy = old.cy;
	} else if (old.cy > new.cy) {
	    RINV(old.ox, new.cy, old.cx, old.cy);
	    old.cy = new.cy;
	}
}

FILE	*
nopen(file, mode)			/* noisy open */
char	*file, *mode;
{
	FILE *fp;

	if ((fp = fopen(file, mode)) == (FILE *) NULL)
	    perror(file);
	return(fp);
}
