/*
 * ma (Midi to Adagio)
 *
 * Convert a MIDI file to an Adagio score
 *
 * This program is largely derived from a program
 * 'mftext' by Tim Thompson, and a program 'transcribe'
 * by Roger Dannenberg.
 *
 *		Greg Lee, lee@uhunix.uhcc.hawaii.edu
 *		4/1/89
 */


/*
 * mftext
 * 
 * Convert a MIDI file to verbose text.
 */

#include <stdio.h>
#include <ctype.h>
#include "midifile.h"
#include "midi.h"

static FILE *F;
/*************************/

#include "cext.h"
#include <malloc.h>
#include "ma.h"
#include "midicode.h"

/* the free space counter */
long space = MAXSPACE;

int debug_rec = false;	/* verbose debug flag for this module */
int verbose = false;	/* verbose flag for this module */
int max_notes = -1;	/* -1 is flag that space must be allocated */

/****************************************************************
* data structure notes: the midi stream is stored as an array 
* of 4-byte records, each of which is either a time or midi
* data.	 Midi data always begins with a control byte (high
* order bit set), and it is assumed times are positive (high
* order bit clear), so the two are easy to distinguish
* IF THE COMPILER PUTS THESE BITS IN THE SAME PLACE.  It looks
* like the high order byte of the time lines up with the last
* byte of a 4 byte array, so we will always set the high order
* bit of the last array byte when the first 3 bytes are filled
* with MIDI data.  This is refered to as the "tag" bit.
* WARNING: Lattice C longs are UNSIGNED, therefore always
* positive.  Test the high order bit with a mask.
****************************************************************/

#define MIDI_CMD_BIT		0x80
#define HIGH_BIT		0x80000000
#define istime(note) (!(((note)->when) & HIGH_BIT))

#define ndsw 2
private char *dsw[ndsw] = { "-d", "-debug" };

typedef union note_struct {
	byte n[4];
	long when;
} *note_type, note_node;

private note_type event_buff;	/* pointer to allocated buffer */
private note_type next;	/* pointer to next entry in buffer */
private note_type last;	/* pointer to last entry in buffer */
private int pile_ups;	/* inner loop iteration count */
private int max_pile;	/* maximum of pile_ups */

private int m_division = 96;
private long m_tempo = 500000;

/****************************************************************************
*	Routines local to this module
****************************************************************************/
private void	bend_filter();
private void	byteorder();
private void	ctrl_filter();
private int	event_bend();
private void	filter();
private long	getdur();
private long	getnext();
private long	diffnext();
private char	map_ctrl();
private void	output();
private void	put_pitch();
private void	rec_init();
private void	stowevent();
private void	rec_final();
private long	m_time();
/*************************/

filegetc()
{
	return(getc(F));
}

main(argc,argv)
int argc;
char **argv;
{	int k, files = 0;
	FILE *efopen();

	for (k = 1; k < argc; k++) {
		if (argv[k][0] == '-')
			switch (argv[k][1]) {
				case 't': debug_rec = 1; break;
				case 'v': verbose = 1; break;
				default: break;
			}
		else {
			if (!files) files = k;
			else {
				fprintf(stderr,"only one file, please\n");
				exit(1);
			}
		}
	}


	if ( files )
		F = efopen(argv[files],"r");
	else
		F = stdin;

 	rec_init();
	initfuncs();
	Mf_getc = filegetc;
	midifile();
	fclose(F);
	rec_final(true);	/* write out recorded data, */
				/* suppress time of first event*/
	exit(0);
}

FILE *
efopen(name,mode)
char *name;
char *mode;
{
	FILE *f;
	extern int errno;
	extern char *sys_errlist[];
	extern int sys_nerr;
	char *errmess;

	if ( (f=fopen(name,mode)) == NULL ) {
		(void) fprintf(stderr,"*** ERROR *** Cannot open '%s'!\n",name);
		if ( errno <= sys_nerr )
			errmess = sys_errlist[errno];
		else
			errmess = "Unknown error!";
		(void) fprintf(stderr,"************* Reason: %s\n",errmess);
		exit(1);
	}
	return(f);
}

error(s)
char *s;
{
	fprintf(stderr,"Error: %s\n",s);
}

txt_header(format,ntrks,division)
{
	m_division = division;
	if (!verbose) return;
	printf("*Header format=%d ntrks=%d division=%d\n",format,ntrks,division);
}

txt_trackstart()
{
	Mf_currtime = 0;
	if (!verbose) return;
	printf("*Track start\n");
}

txt_trackend()
{
	if (!verbose) return;
	printf("*Track end\n");
}

txt_noteon(chan,pitch,vol)
{
	stowevent(NOTEON, chan, pitch, vol);
}

txt_noteoff(chan,pitch,vol)
{
	stowevent(NOTEON, chan, pitch, 0);
}

txt_pressure(chan,pitch,press)
{
	stowevent(PRESSURE, chan, pitch, press);
}

txt_parameter(chan,control,value)
{
	stowevent(CONTROLLER, chan, control, value);
}

txt_pitchbend(chan,msb,lsb)
{
	stowevent(PITCHBEND, chan, msb, lsb);
}

txt_program(chan,program)
{
	stowevent(PROGRAM, chan, program, 0);
}

txt_chanpressure(chan,press)
{
	stowevent(CHANPRESSURE, chan, press, 0);
}

txt_sysex(leng,mess)
char *mess;
{
	if (!verbose) return;
	prtime();
	printf("*Sysex, leng=%d\n",leng);
}

txt_metamisc(type,leng,mess)
char *mess;
{
	if (!verbose) return;
	prtime();
	printf("*Meta event, unrecognized, type=0x%02x leng=%d\n",type,leng);
}

txt_metaspecial(type,leng,mess)
char *mess;
{
	if (!verbose) return;
	prtime();
	printf("*Meta event, sequencer-specific, type=0x%02x leng=%d\n",type,leng);
}

txt_metatext(type,leng,mess)
char *mess;
{
	static char *ttype[] = {
		NULL,
		"Text Event",		/* type=0x01 */
		"Copyright Notice",	/* type=0x02 */
		"Sequence/Track Name",
		"Instrument Name",	/* ...       */
		"Lyric",
		"Marker",
		"Cue Point",		/* type=0x07 */
		"Unrecognized"
	};
	int unrecognized = (sizeof(ttype)/sizeof(char *)) - 1;
	register int n, c;
	register char *p = mess;

	if (!verbose) return;
	if ( type < 1 || type > unrecognized )
		type = unrecognized;
	prtime();
	printf("*Meta Text, type=0x%02x (%s)  leng=%d\n",type,ttype[type],leng);
	printf("*     Text = <");
	for ( n=0; n<leng; n++ ) {
		c = *p++;
		printf( (isprint(c)||isspace(c)) ? "%c" : "\\0x%02x" , c);
	}
	printf(">\n");
}

txt_metaseq(num)
{
	if (!verbose) return;
	prtime();
	printf("*Meta event, sequence number = %d\n",num);
}

txt_metaeot()
{
	if (!verbose) return;
	prtime();
	printf("*Meta event, end of track\n");
}

txt_keysig(sf,mi)
{
	if (!verbose) return;
	prtime();
	printf("*Key signature, sharp/flats=%d  minor=%d\n",sf,mi);
}

txt_tempo(tempo)
long tempo;
{
	m_tempo = tempo;
	if (!verbose) return;
	prtime();
	printf("*Tempo, microseconds-per-MIDI-quarter-note=%d\n",tempo);
}

txt_timesig(nn,dd,cc,bb)
{
	int denom = 1;

	if (!verbose) return;
	while ( dd-- > 0 )
		denom *= 2;
	prtime();
	printf("*Time signature=%d/%d  MIDI-clocks/click=%d  32nd-notes/24-MIDI-clocks=%d\n",
		nn,denom,cc,bb);
}

txt_smpte(hr,mn,se,fr,ff)
{
	if (!verbose) return;
	prtime();
	printf("*SMPTE, hour=%d minute=%d second=%d frame=%d fract-frame=%d\n",
		hr,mn,se,fr,ff);
}

txt_arbitrary(leng,mess)
char *mess;
{
	if (!verbose) return;
	prtime();
	printf("*Arbitrary bytes, leng=%d\n",leng);
}

prtime()
{
	printf("*Time=%ld (%ldcs) ", Mf_currtime,m_time());
}

initfuncs()
{
	Mf_error = error;
	Mf_header =  txt_header;
	Mf_trackstart =  txt_trackstart;
	Mf_trackend =  txt_trackend;
	Mf_noteon =  txt_noteon;
	Mf_noteoff =  txt_noteoff;
	Mf_pressure =  txt_pressure;
	Mf_parameter =  txt_parameter;
	Mf_pitchbend =  txt_pitchbend;
	Mf_program =  txt_program;
	Mf_chanpressure =  txt_chanpressure;
	Mf_sysex =  txt_sysex;
	Mf_metamisc =  txt_metamisc;
	Mf_seqnum =  txt_metaseq;
	Mf_eot =  txt_metaeot;
	Mf_timesig =  txt_timesig;
	Mf_smpte =  txt_smpte;
	Mf_tempo =  txt_tempo;
	Mf_keysig =  txt_keysig;
	Mf_seqspecific =  txt_metaspecial;
	Mf_text =  txt_metatext;
	Mf_arbitrary =  txt_arbitrary;
}
/****************************/

/* record.c -- keyboard to adagio recorder
 *
 * the interface consists of three routines:
 *	    rec_init()		-- initialization
 *	    stowevent()		-- store each midi event as encountered
 *	    rec_final()		-- called to finish up
 */

/*****************************************************************************
*	    Change Log
*  Date	    | Change
*-----------+-----------------------------------------------------------------
* 27-Feb-86 | Created changelog
*	    | Use pedal information when computing durations (code taken
*	    |  from transcribe.c)
* 23-Mar-86 | Determine size of transcription when rec_init is called.
* 21-May-86 | Major rewrite to use continuous controls (code taken 
*	    |  from transcribe.c)
*****************************************************************************/

private void stowevent(command, chan, s1, s2)
int command;
{
	next->when = m_time();
	next++;
	next->n[0] = command | chan;
	next->n[1] = s1;
	next->n[2] = s2;
	next->n[3] = MIDI_CMD_BIT;	/* set tag bit */
	next++;

	if (next >= last) {
		fprintf(stderr,"out of space\n");
		exit(1);
	}
}

/****************************************************************************
*				bend_filter
* Inputs:
*	note_type note: the current note
*	note_type last: the last recorded event
*	long now: the current time
* Effect:
*	remove pitch bend events in same 0.01 sec time slot
* Implementation:
*	If the current event is a pitch bend that bends again
*	in the same time slot, make it a no-op by replacing it with
*	the time.
****************************************************************************/

private void bend_filter(note, last, now)
    note_type note;	/* current note */
    note_type last;	/* the last recorded event */
    long now;		/* the current time */
{
    /* first see if there is another bend in this time
     * slot.
     */
    note_type note2 = note + 1;
    while (note2 < last) {
	if (istime(note2) && (note2->when > now)) {
	    break; /* new time slot */
	} else if (note->n[0] == note2->n[0]) {
	    note->when = now;
	    return; /* found another bend */
	}
	note2++;
    }
}

/****************************************************************************
*				byteorder
* Effect: 
*	check out assumptions about byte order and placement
****************************************************************************/

private void byteorder()
{
    if ((sizeof(event_buff[0]) != 4) ||
	(sizeof(event_buff[0].when) != 4) ||
	(sizeof(event_buff[0].n[0]) != 1)) {
	fprintf(stderr, "implementation error: size problem\n");
	exit(1);
    }
    event_buff[0].n[0] = 0x12;
    event_buff[0].n[1] = 0x34;
    event_buff[0].n[2] = 0x56;
    event_buff[0].n[3] = 0x78;
if (debug_rec)
	printf("layout is 0x%lx\n", event_buff[0].when);
    if ((event_buff[0].when != 0x78563412) &&
	(event_buff[0].when != 0x12345678)) {
	fprintf(stderr, "implementation error: layout problem\n");
	exit(1);
    }
}

/****************************************************************************
*				ctrl_filter
* Inputs:
*	note_type note: the current note
*	note_type last: the last recorded event
*	long now: the current time
* Effect:
*	remove ctrl change events in same 0.01 sec time slot
* Implementation:
*	If the current event is a control change that changes again
*	in the same time slot, make it a no-op by replacing it with
*	the time.
****************************************************************************/

private void ctrl_filter(note, last, now)
    note_type note;	/* the current note */
    note_type last;	/* the last recorded event */
    long now;		/* the current time */
{
    /* see if there is another control change in this time
     * slot.
     */
    note_type note2 = note+1;
    while (note2 < last) {
	if (istime(note2) && (note2->when > now)) {
	    break;	/* new time slot */
	} else if ((note->n[0] == note2->n[0]) &&
		   (note->n[1] == note2->n[1])) {
	    note->when = now;
	    return; /* found another change */
	}
	note2++;
    }
}

/****************************************************************************
*				event_bend
* Inputs:
*	note_type note: pointer to a pitch bend event
* Outputs:
*	returns int: an 8 bit pitch bend number
****************************************************************************/

private int event_bend(note)
    note_type note;
{
    return (int) (((note->n[1]) >> 6) + ((note->n[2]) << 1)); 
}

/****************************************************************************
*				filter
* Inputs:
*	note_type last: the last note recorded
* Effect: allow only one control change per time slot (0.01 sec)
* Implementation:
*	call ctrl_filter and bend_filter to overwrite control changes with
*	noop data (the current time is used as a noop)
****************************************************************************/

private void filter(last)
    note_type last;
{
    note_type note;	/* loop control variable */
    long now;		/* last time seen */
    int command;	/* command pointed to by note */
    int chan;		/* channel pointed to by note */

    for (note = event_buff; note <= last; note++) {
	if (istime(note)) {
	    now = note->when;
	} else {
	    command = note->n[0] & MIDI_CODE_MASK;
	    chan = note->n[0] & MIDI_CHN_MASK;

	    if (command == MIDI_CTRL &&
		note->n[1] == SUSTAIN) {
		/* do nothing */;
	    } else if (command == MIDI_CTRL) {
		ctrl_filter(note, last, now);
	    } else if (command == MIDI_TOUCH) {
		bend_filter(note, last, now);	/* bend and touch use the */
	    } else if (command == MIDI_BEND) {	/*  same filter routines  */
		bend_filter(note, last, now);
	    }
	}
    }
}


/****************************************************************************
*				getdur
* Inputs:
*	int i: index of the note
*	note_type last: pointer to the last event recorded
*	int ped: true if pedal is down at event i
*	long now: the time at event i
* Outputs:
*	returns long: the duration of note i
* Assumes:
*	assumes i is a note
* Implementation:
*	This is tricky because of pedal messages.  The note is kept on by
*	either the key or the pedal.  Keep 2 flags, key and ped.  Key is
*	turned off when a key is released, ped goes off and on with pedal.
*	Note ends when (1) both key and ped are false, (2) key is
*	pressed (this event will also start another note).
****************************************************************************/

private long getdur(i, last, ped, now)
    int i;
    note_type last;
    int ped;
    long now;
{
    int key = true;	/* flag that says if note is on */
    long start = now;
    int chan = event_buff[i].n[0] & MIDI_CHN_MASK;
    int pitch = event_buff[i].n[1];
    note_type note = &(event_buff[i+1]);
    int noteon; /* true if a noteon message received on chan */
    int keyon;	/* true if noteon message had non-zero velocity */

    /* search from the next event (i+1) to the end of the buffer:
     */
    for (; note < last; note++) {
	if (istime(note)) {
	    now = note->when;
	} else {
	    noteon = keyon = false;
	    if ((note->n[0] & MIDI_CHN_MASK) == chan) {
		noteon = ((note->n[0] & MIDI_CODE_MASK) == MIDI_ON_NOTE) &&
		     (note->n[1] == pitch);
		keyon = noteon && (note->n[2] != 0);
		if ((noteon && (note->n[2] == 0)) ||
		    (((note->n[0] & MIDI_CODE_MASK) == MIDI_OFF_NOTE) &&
		     (note->n[1] == pitch))) key = false;
		if (((note->n[0] & MIDI_CODE_MASK) == MIDI_CTRL) &&
		    note->n[1] == SUSTAIN && note->n[2] == 127) ped = true;
		if (((note->n[0] & MIDI_CODE_MASK) == MIDI_CTRL) &&
		    note->n[1] == SUSTAIN && note->n[2] == 0) ped = false;

		if ((!key && !ped) || keyon)
		    return now - start;
	    }
	}
    }
    return last->when - start;
}

/****************************************************************************
*				getnext
* Inputs:
*	int i: the index of the current note
*	note_type last: pointer to last valid data
*	long now: the current time
* Outputs:
*	returns long: the time of the next note, program, or control change
*		(returns time of last event if nothing else is found)
****************************************************************************/

private long getnext(i, last, now)
    int i;	/* the index of the current note */
    note_type last;	/* pointer to last valid data */
    long now;	/* the current time */
{
    i++;	/* advance to next item */
    for (; event_buff + i < last; i++) {
	note_type note = &(event_buff[i]);
	int cmd = note->n[0] & MIDI_CODE_MASK;

	if (istime(note)) {
	    now = note->when;
	} else if (((cmd == MIDI_ON_NOTE) &&
		    (note->n[2] != 0)) /* note on */ ||
		   (cmd == MIDI_CH_PROGRAM) /* program change */ ||
		   ((cmd == MIDI_CTRL) &&
		    (note->n[1] != SUSTAIN) /* control change */ ) ||
		   (cmd == MIDI_TOUCH) ||
		   (cmd == MIDI_BEND)) {
	    return now;
	}
    }
    return last->when;
}

/****************************************************************************
*				map_ctrl
* Inputs:
*	int control: a midi control number
* Outputs:
*	returns char: an adagio control change command letter, NULL if
*		control change is not one of PORTARATE, PORTASWITCH,
*		MODWHEEL, FOOT
****************************************************************************/

private char map_ctrl(control)
    int control;
{
    switch (control) {
	case PORTARATE:		return 'J';
	case PORTASWITCH:	return 'K';
	case MODWHEEL:		return 'M';
	case FOOT:		return 'X';
	default:		return  0;
    }
}

private long diffnext(i, last, now, nxlie)
    int i;	/* the index of the current note */
    note_type last;	/* pointer to last valid data */
    long now;	/* the current time */
    boolean *nxlie;
{
	now = getnext(i, last, now) - now;
	if (now < 0) {
		*nxlie = true;
		return 0;
	}
	*nxlie = false;
	return now;
}

/****************************************************************************
*				output
* Inputs:
*	FILE *fp: an opened file pointer
*	note_type last: the last data in the buffer
*	boolean absflag: set to true if first line of the adagio score should
*		include the absolute time
* Effect: 
*	write adagio file using data in event_buff
* Implementation:
*	NOTE: put all program changes in rests
*	use N(ext) notation for all timing
*	output no more than one continuous parameter change per
*	clock tick for each continuous change parameter
****************************************************************************/

private void output(last, absflag)
    note_type last;
    boolean absflag;
{
    int i;			/* loop counter */
    int command;		/* the current command */
    int chan;			/* the midi channel of the current event */
    int lastchan = 0;		/* the default adagio channel (1) */
    int ped = false;		/* flag maintains state of pedal */
    int how_many = last - event_buff;
    long now;		/* the time of the next event */
    boolean bad_ctrl_flag = false;	/* set to true if unknown ctrl read */
    boolean nxlie = false;


    /* set the initial absolute time, all other times are relative */
    if (absflag)
	printf("t%ld ", event_buff[0].when);

    for (i = 0; i < how_many; i++) {
	if (debug_rec) {
	    printf("ev %d: %x %x %x (%ld)\n", i, event_buff[i].n[0],
		event_buff[i].n[1], event_buff[i].n[2], event_buff[i].when);
	}
	if (istime(event_buff+i)) {
	    now = event_buff[i].when;
	    if (debug_rec) printf("i = %d, now = %ld\n", i, now);
	} else {
	    command = event_buff[i].n[0] & MIDI_CODE_MASK;
	    chan = event_buff[i].n[0] & MIDI_CHN_MASK;

	    if ( nxlie &&
		 (command != MIDI_ON_NOTE || event_buff[i].n[2] != 0)
		) {
		printf("t%ld ", now);
		nxlie = false;
	    }

	    if (command == MIDI_ON_NOTE && event_buff[i].n[2] != 0) {
/* Why -12?	put_pitch(event_buff[i].n[1] - 12); */
		put_pitch(event_buff[i].n[1]);
		printf(" u%ld l%d n%ld", getdur(i, last, ped, now),
			event_buff[i].n[2], diffnext(i,last,now,&nxlie));
		if (lastchan != chan) {
		    printf(" v%d\n", chan + 1);
		    lastchan = chan;
		} else printf("\n");
	    } else if (command == MIDI_CH_PROGRAM) {
		printf("r z%d n%ld", event_buff[i].n[1] + 1,
			diffnext(i,last,now,&nxlie));
		if (lastchan != chan) {
		    printf(" v%d\n", chan + 1);
		    lastchan = chan;
		} else printf("\n");
	    } else if (command == MIDI_CTRL &&
		       event_buff[i].n[1] == SUSTAIN) {
		ped = (event_buff[i].n[2] != 0);
	    } else if (command == MIDI_CTRL) {
		char c = map_ctrl(event_buff[i].n[1]);
		if (c != 0) {
		    printf("%c%d n%d\n", c,
			event_buff[i].n[2], diffnext(i,last,now,&nxlie));
		} else bad_ctrl_flag = true;
	    } else if (command == MIDI_TOUCH) {
		printf("O%d n%d\n", event_buff[i].n[1],
			diffnext(i,last,now,&nxlie));
	    } else if (command == MIDI_BEND) {
		printf("Y%d n%d", event_bend(&event_buff[i]),
			diffnext(i,last,now,&nxlie));
		if (lastchan != chan) {
		    printf(" v%d\n", chan + 1);
		    lastchan = chan;
		} else printf("\n");
	    } else if (command != MIDI_ON_NOTE) {
		fprintf(stderr, "Command 0x%x ignored\n", command);
	    }
	}
    }
    if (bad_ctrl_flag)
	fprintf(stderr,
	       "Some unrecognized control changes were omitted from file.\n");
}

/****************************************************************************
*				put_pitch
* Inputs:
*	FILE *fp: an open file
*	int p: a pitch number
* Effect: write out the pitch name for a given number
****************************************************************************/

private void put_pitch(p)
    int p;
{
    static char *ptos[] = {"c", "cs", "d", "ef", "e", "f", "fs", "g",
			   "gs", "a", "bf", "b"};
    printf("%s%d", ptos[p % 12], p / 12);
}

/**********************************************************************
*			rec_final
* Inputs:
*	boolean absflag: output absolute time of first note if true
* Effect:
*	Write recorded data to a file
**********************************************************************/

private void rec_final(absflag)
    boolean absflag;
{
    next->when = m_time();
    last = next;
    if (debug_rec) printf("max_pile_up = %d, ", max_pile);
    if (verbose) printf("* %d times and events recorded.\n", last - event_buff);
    filter(last);
    output(last, absflag);
}

/****************************************************************************
*				rec_init
* Inputs:
*	char *file:  pointer to file name from command line (if any)
*	boolean bender: true if pitch bend should be enabled
* Outputs:
*	returns true if initialization succeeds
* Effect:
*	prepares module to record midi input
****************************************************************************/

private void rec_init()
{
#ifndef linux
    char *malloc();	/* memory allocation */
#endif
    pile_ups = 0;
    max_pile = 0;

    if (max_notes == -1) {	/* allocate space 1st time rec_init called */
	max_notes = space/sizeof(note_node);
	event_buff = (note_type) malloc(sizeof(note_node) * max_notes);
	if (event_buff == NULL) {
	    fprintf(stderr, "Internal error allocating record space.");
	    exit(1);
	}
	byteorder();
if (debug_rec)
	printf("Space for %d events has been allocated.\n", max_notes);
    }
    next = event_buff;
    last = event_buff + max_notes - 2;


    return;
}

long
m_time()
{
	float time;

	time = mf_ticks2sec(Mf_currtime,m_division,m_tempo);
	return( (unsigned long) (time * 100));
}
