/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.plugin.insights.core.exporter;

import java.io.IOException;
import java.nio.charset.Charset;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.ExceptionsHelper;
import org.opensearch.ResourceAlreadyExistsException;
import org.opensearch.action.ActionRequest;
import org.opensearch.action.ActionType;
import org.opensearch.action.admin.indices.create.CreateIndexRequest;
import org.opensearch.action.admin.indices.create.CreateIndexResponse;
import org.opensearch.action.admin.indices.delete.DeleteIndexRequest;
import org.opensearch.action.admin.indices.template.get.GetComposableIndexTemplateAction;
import org.opensearch.action.admin.indices.template.put.PutComposableIndexTemplateAction;
import org.opensearch.action.bulk.BulkRequestBuilder;
import org.opensearch.action.bulk.BulkResponse;
import org.opensearch.action.index.IndexRequest;
import org.opensearch.action.support.clustermanager.AcknowledgedResponse;
import org.opensearch.cluster.ClusterState;
import org.opensearch.cluster.metadata.ComposableIndexTemplate;
import org.opensearch.cluster.metadata.Template;
import org.opensearch.cluster.service.ClusterService;
import org.opensearch.common.compress.CompressedXContent;
import org.opensearch.common.settings.Settings;
import org.opensearch.common.unit.TimeValue;
import org.opensearch.common.xcontent.XContentFactory;
import org.opensearch.core.action.ActionListener;
import org.opensearch.core.xcontent.ToXContent;
import org.opensearch.index.IndexNotFoundException;
import org.opensearch.plugin.insights.core.exporter.QueryInsightsExporter;
import org.opensearch.plugin.insights.core.metrics.OperationalMetric;
import org.opensearch.plugin.insights.core.metrics.OperationalMetricsCounter;
import org.opensearch.plugin.insights.core.utils.IndexDiscoveryHelper;
import org.opensearch.plugin.insights.rules.model.SearchQueryRecord;
import org.opensearch.transport.client.Client;

public class LocalIndexExporter
implements QueryInsightsExporter {
    private final Logger logger = LogManager.getLogger();
    private final Client client;
    private final ClusterService clusterService;
    private final String indexMapping;
    private DateTimeFormatter indexPattern;
    private int deleteAfter;
    private final String id;
    private static final int DEFAULT_NUMBER_OF_SHARDS = 1;
    private static final String DEFAULT_AUTO_EXPAND_REPLICAS = "0-2";
    private static final String TEMPLATE_NAME = "query_insights_top_queries_template";
    private long templatePriority;

    public LocalIndexExporter(Client client, ClusterService clusterService, DateTimeFormatter indexPattern, String indexMapping, String id) {
        this.client = client;
        this.clusterService = clusterService;
        this.indexPattern = indexPattern;
        this.indexMapping = indexMapping;
        this.id = id;
        this.deleteAfter = 7;
        this.templatePriority = 1847L;
    }

    @Override
    public String getId() {
        return this.id;
    }

    public DateTimeFormatter getIndexPattern() {
        return this.indexPattern;
    }

    public void setIndexPattern(DateTimeFormatter indexPattern) {
        this.indexPattern = indexPattern;
    }

    @Override
    public void export(List<SearchQueryRecord> records) {
        if (records == null || records.isEmpty()) {
            return;
        }
        try {
            String indexName = this.buildLocalIndexName();
            if (!this.checkIndexExists(indexName)) {
                this.ensureTemplateExists().whenComplete((templateCreated, templateException) -> {
                    if (templateException != null) {
                        this.logger.error("Error ensuring template exists:", templateException);
                        OperationalMetricsCounter.getInstance().incrementCounter(OperationalMetric.LOCAL_INDEX_EXPORTER_EXCEPTIONS);
                    }
                    try {
                        this.createIndexAndBulk(indexName, records);
                    }
                    catch (IOException ioe) {
                        this.logger.error("Error creating index:", (Throwable)ioe);
                        OperationalMetricsCounter.getInstance().incrementCounter(OperationalMetric.LOCAL_INDEX_EXPORTER_EXCEPTIONS);
                    }
                });
            } else {
                this.bulk(indexName, records);
            }
        }
        catch (IOException e) {
            this.logger.error("Unable to export query insights data:", (Throwable)e);
            OperationalMetricsCounter.getInstance().incrementCounter(OperationalMetric.LOCAL_INDEX_EXPORTER_EXCEPTIONS);
        }
    }

    void createIndexAndBulk(final String indexName, final List<SearchQueryRecord> records) throws IOException {
        CreateIndexRequest createIndexRequest = new CreateIndexRequest(indexName);
        createIndexRequest.settings(Settings.builder().put("index.number_of_shards", 1).put("index.auto_expand_replicas", DEFAULT_AUTO_EXPAND_REPLICAS));
        createIndexRequest.mapping(this.readIndexMappings());
        this.client.admin().indices().create(createIndexRequest, (ActionListener)new ActionListener<CreateIndexResponse>(){

            public void onResponse(CreateIndexResponse createIndexResponse) {
                if (createIndexResponse.isAcknowledged()) {
                    try {
                        LocalIndexExporter.this.bulk(indexName, records);
                    }
                    catch (IOException e) {
                        OperationalMetricsCounter.getInstance().incrementCounter(OperationalMetric.LOCAL_INDEX_EXPORTER_EXCEPTIONS);
                        LocalIndexExporter.this.logger.error("Unable to index query insights data: ", (Throwable)e);
                    }
                }
            }

            public void onFailure(Exception e) {
                Throwable cause = ExceptionsHelper.unwrapCause((Throwable)e);
                if (cause instanceof ResourceAlreadyExistsException) {
                    try {
                        LocalIndexExporter.this.bulk(indexName, records);
                    }
                    catch (IOException ioe) {
                        OperationalMetricsCounter.getInstance().incrementCounter(OperationalMetric.LOCAL_INDEX_EXPORTER_EXCEPTIONS);
                        LocalIndexExporter.this.logger.error("Unable to index query insights data: ", (Throwable)ioe);
                    }
                } else {
                    OperationalMetricsCounter.getInstance().incrementCounter(OperationalMetric.LOCAL_INDEX_EXPORTER_EXCEPTIONS);
                    LocalIndexExporter.this.logger.error("Unable to create query insights index: ", cause);
                }
            }
        });
    }

    void bulk(String indexName, List<SearchQueryRecord> records) throws IOException {
        BulkRequestBuilder bulkRequestBuilder = this.client.prepareBulk().setTimeout(TimeValue.timeValueMinutes((long)1L));
        for (SearchQueryRecord record : records) {
            bulkRequestBuilder.add(new IndexRequest(indexName).id(record.getId()).source(record.toXContentForExport(XContentFactory.jsonBuilder(), ToXContent.EMPTY_PARAMS)));
        }
        bulkRequestBuilder.execute((ActionListener)new ActionListener<BulkResponse>(){

            public void onResponse(BulkResponse bulkItemResponses) {
            }

            public void onFailure(Exception e) {
                OperationalMetricsCounter.getInstance().incrementCounter(OperationalMetric.LOCAL_INDEX_EXPORTER_BULK_FAILURES);
                LocalIndexExporter.this.logger.error("Failed to execute bulk operation for query insights data: ", (Throwable)e);
            }
        });
    }

    @Override
    public void close() {
        this.logger.debug("Closing the LocalIndexExporter..");
    }

    String buildLocalIndexName() {
        ZonedDateTime currentTime = ZonedDateTime.now(ZoneOffset.UTC);
        return IndexDiscoveryHelper.buildLocalIndexName(this.indexPattern, currentTime);
    }

    public void setDeleteAfter(int deleteAfter) {
        this.deleteAfter = deleteAfter;
    }

    public int getDeleteAfter() {
        return this.deleteAfter;
    }

    public void deleteSingleIndex(final String indexName, Client client) {
        final Logger logger = LogManager.getLogger();
        client.admin().indices().delete(new DeleteIndexRequest(indexName), (ActionListener)new ActionListener<AcknowledgedResponse>(this){

            public void onResponse(AcknowledgedResponse acknowledgedResponse) {
            }

            public void onFailure(Exception e) {
                Throwable cause = ExceptionsHelper.unwrapCause((Throwable)e);
                if (cause instanceof IndexNotFoundException) {
                    return;
                }
                OperationalMetricsCounter.getInstance().incrementCounter(OperationalMetric.LOCAL_INDEX_EXPORTER_DELETE_FAILURES);
                logger.error("Failed to delete index '{}': ", (Object)indexName, (Object)e);
            }
        });
    }

    boolean checkIndexExists(String indexName) {
        ClusterState clusterState = this.clusterService.state();
        return clusterState.getRoutingTable().hasIndex(indexName);
    }

    String readIndexMappings() throws IOException {
        if (this.indexMapping == null || this.indexMapping.isEmpty()) {
            return "{}";
        }
        if (this.indexMapping.endsWith(".json")) {
            return new String(Objects.requireNonNull(LocalIndexExporter.class.getClassLoader().getResourceAsStream(this.indexMapping)).readAllBytes(), Charset.defaultCharset());
        }
        return this.indexMapping;
    }

    CompletableFuture<Boolean> ensureTemplateExists() {
        final CompletableFuture<Boolean> future = new CompletableFuture<Boolean>();
        GetComposableIndexTemplateAction.Request getRequest = new GetComposableIndexTemplateAction.Request();
        this.client.execute((ActionType)GetComposableIndexTemplateAction.INSTANCE, (ActionRequest)getRequest, (ActionListener)new ActionListener<GetComposableIndexTemplateAction.Response>(){

            public void onResponse(GetComposableIndexTemplateAction.Response response) {
                if (response.indexTemplates().containsKey(LocalIndexExporter.TEMPLATE_NAME) && ((ComposableIndexTemplate)response.indexTemplates().get(LocalIndexExporter.TEMPLATE_NAME)).priority() == LocalIndexExporter.this.templatePriority) {
                    LocalIndexExporter.this.logger.debug("Template [{}] already exists, skipping creation", (Object)LocalIndexExporter.TEMPLATE_NAME);
                    future.complete(true);
                    return;
                }
                LocalIndexExporter.this.createTemplate(future);
            }

            public void onFailure(Exception e) {
                LocalIndexExporter.this.logger.warn("Failed to check if template [{}] exists: {}", (Object)LocalIndexExporter.TEMPLATE_NAME, (Object)e.getMessage());
                LocalIndexExporter.this.createTemplate(future);
            }
        });
        return future;
    }

    void createTemplate(final CompletableFuture<Boolean> future) {
        try {
            CompressedXContent compressedMapping = new CompressedXContent(this.readIndexMappings());
            Template template = new Template(Settings.builder().put("index.number_of_shards", 1).put("index.auto_expand_replicas", DEFAULT_AUTO_EXPAND_REPLICAS).build(), compressedMapping, null);
            ComposableIndexTemplate composableTemplate = new ComposableIndexTemplate(Collections.singletonList("top_queries-*"), template, null, Long.valueOf(this.templatePriority), null, null);
            PutComposableIndexTemplateAction.Request request = new PutComposableIndexTemplateAction.Request(TEMPLATE_NAME).indexTemplate(composableTemplate);
            this.client.execute((ActionType)PutComposableIndexTemplateAction.INSTANCE, (ActionRequest)request, (ActionListener)new ActionListener<AcknowledgedResponse>(){

                public void onResponse(AcknowledgedResponse response) {
                    if (response.isAcknowledged()) {
                        LocalIndexExporter.this.logger.info("Successfully created or updated template [{}] with priority {}", (Object)LocalIndexExporter.TEMPLATE_NAME, (Object)LocalIndexExporter.this.templatePriority);
                        future.complete(true);
                    } else {
                        LocalIndexExporter.this.logger.warn("Failed to create or update template [{}]", (Object)LocalIndexExporter.TEMPLATE_NAME);
                        future.complete(false);
                    }
                }

                public void onFailure(Exception e) {
                    LocalIndexExporter.this.logger.error("Error creating or updating template [{}]", (Object)LocalIndexExporter.TEMPLATE_NAME, (Object)e);
                    OperationalMetricsCounter.getInstance().incrementCounter(OperationalMetric.LOCAL_INDEX_EXPORTER_EXCEPTIONS);
                    future.completeExceptionally(e);
                }
            });
        }
        catch (Exception e) {
            this.logger.error("Failed to manage template", (Throwable)e);
            OperationalMetricsCounter.getInstance().incrementCounter(OperationalMetric.LOCAL_INDEX_EXPORTER_EXCEPTIONS);
            future.completeExceptionally(e);
        }
    }

    public void setTemplatePriority(long templatePriority) {
        this.templatePriority = templatePriority;
    }

    public long getTemplatePriority() {
        return this.templatePriority;
    }
}

