/* pnminterp - scale up portable anymap by interpolating between pixels.
 * Copyright (C) 1998,2000 Russell Marks.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* this is basically a hacked copy of zgv's readpnm.c,
 * with the write_interp_line() function based on the
 * scaling-with-interpolation code from zgv's vgadisp.c.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>


/* prototypes */
void usage(void);
void read_pnm_line(FILE *in,unsigned char *ptr,
	int width,int raw,int sub,int maxval);
void ditch_line(FILE *in);
int read_next_number(FILE *in);
void write_interp_line(unsigned char *linebuf,unsigned char *outbuf,
	int width,int bytepp,int scale);


/* drop one (source) pixel at right/bottom edges. */
#define EDGE_DROP		1

/* interpolate right/bottom edge pixels to black. */
#define EDGE_INTERP_TO_BLACK	2

/* don't interpolate right/bottom edge pixels (default, and what zgv does). */
#define EDGE_NON_INTERP		3

int edge_mode=EDGE_NON_INTERP;



int main(int argc,char *argv[])
{
FILE *in=stdin;
unsigned char *linebuf,*curline,*nextline,*outbuf;
int c,sub,raw,w,h,maxval,bytepp,y;
int scale;
int done=0,left;

opterr=0;

do
  switch(getopt(argc,argv,"bdh"))
    {
    case 'b':	/* interpolate to black at right/bottom edges */
      edge_mode=EDGE_INTERP_TO_BLACK; break;
    case 'd':	/* drop right/bottom edges */
      edge_mode=EDGE_DROP; break;
    case 'h':	/* help on usage */
      usage();		/* doesn't return */
    case '?':
      fprintf(stderr,"pnminterp: unrecognised option `%c'.\n",optopt);
      exit(1);
    case -1:
      done=1;
    }
while(!done);

/* must be at least one arg left, to specify scale,
 * but no more than two args left (scale and filename).
 */
left=argc-optind;
if(left<1 || left>2)
  usage();

scale=atoi(argv[argc-left]);
if(scale<2)
  fprintf(stderr,"pnminterp: scale must be at least 2.\n"),exit(1);

if(left==2 && (in=fopen(argv[argc-1],"rb"))==NULL)
  fprintf(stderr,"pnminterp: couldn't open file.\n"),exit(1);

c=fgetc(in);
sub=fgetc(in)-48;
if(c!='P' || sub<1 || sub>6)
  fprintf(stderr,"pnminterp: not a PNM file.\n"),exit(1);

fgetc(in);

raw=0;
if(sub>3)
  {
  raw=1;
  sub-=3;
  }

bytepp=(sub==3)?3:1;

w=read_next_number(in);
h=read_next_number(in);

if(w==0 || h==0)
  fprintf(stderr,"pnminterp: bad width and/or height.\n"),exit(1);

/* separate error for this, because it's not *bad* as such */
if(w==1 || h==1)
  fprintf(stderr,"pnminterp: width/height must both be >=2.\n"),exit(1);

if(sub==1)
  maxval=1;
else
  if((maxval=read_next_number(in))==0)
    fprintf(stderr,"pnminterp: bad maxval.\n"),exit(1);

/* to interpolate we only actually need the current line and the
 * next line in memory. Also need single-scaled-up-line buffer
 * for interp_line() though.
 */
if((linebuf=malloc(w*bytepp*2))==NULL || (outbuf=malloc(w*bytepp*scale))==NULL)
  fprintf(stderr,"pnminterp: out of memory.\n"),exit(1);

curline=linebuf;
nextline=linebuf+w*bytepp;

/* read in the first line (normally we read in the `next' line;
 * this gives us an initial `current' one).
 */
read_pnm_line(in,curline,w,raw,sub,maxval);

/* usual filter message when reading PBM but writing PGM: */
if(sub==1)
  fprintf(stderr,"pnminterp: promoting from PBM to PGM\n");

printf("P%d\n%d %d\n255\n",5+(bytepp==3),
	(w-(edge_mode==EDGE_DROP))*scale,
	(h-(edge_mode==EDGE_DROP))*scale);

for(y=1;y<=h;y++)
  {
  if(y==h)
    {
    /* last line is about to be output. there is no further `next line'.
     * if EDGE_DROP, we stop here, with output of h-1 rows.
     * if EDGE_INTERP_TO_BLACK we make next line black.
     * if EDGE_NON_INTERP (default) we make it a copy of the current line.
     */
    if(edge_mode==EDGE_DROP)
      break;
    
    if(edge_mode==EDGE_INTERP_TO_BLACK)
      memset(nextline,0,w*bytepp);
    else	/* EDGE_NON_INTERP */
      memcpy(nextline,curline,w*bytepp);
    }
  else
    {
    /* read a line. */
    read_pnm_line(in,nextline,w,raw,sub,maxval);
    }
  
  /* output curline interpolated towards nextline. */
  write_interp_line(linebuf,outbuf,w,bytepp,scale);
  
  /* it would be quicker to flip the curline/nextline pointers rather
   * than copying memory, but write_interp_line effectively needs a
   * two-line image to work on.
   */
  memcpy(curline,nextline,w*bytepp);
  }

free(outbuf);
free(linebuf);
if(in!=stdin) fclose(in);

exit(0);
}


void usage(void)
{
printf("usage: pnminterp [-bdh] N [pnmfile]\n\
\n\
	-b	interpolate to black at right/bottom edges\n\
	-d	drop (scaled-up) pixels at right/bottom edges\n\
	-h	give this usage help\n\
\n\
	N	the amount to scale up the width and height of the\n\
		input image; for example, `pnminterp 2' produces\n\
		a picture with twice the width and twice the height.\n");
exit(0);
}


void read_pnm_line(FILE *in,unsigned char *ptr,
	int width,int raw,int sub,int maxval)
{
int c,i,n,x,red,grn,blu;

for(x=0;x<width;x++)
  {
  if(raw)
    switch(sub)
      {
      case 1: /* PBM */
        c=fgetc(in);
        for(i=0,n=128;i<8 && x+i<width;i++,n>>=1)
          *ptr++=(c&n)?0:255;
        x+=7;
        break;
      case 2: /* PGM */
        *ptr++=(fgetc(in)*255)/maxval;
        break;
      case 3: /* PPM */
        red=(fgetc(in)*255)/maxval;
        grn=(fgetc(in)*255)/maxval;
        blu=(fgetc(in)*255)/maxval;
        *ptr++=red; *ptr++=grn; *ptr++=blu;
      }
  else
    switch(sub)
      {
      case 1:
        do
          if((c=fgetc(in))=='#') ditch_line(in);
        while(c!='0' && c!='1');
        *ptr++=(c=='1')?0:255;
        break;
      case 2:
        *ptr++=(read_next_number(in)*255)/maxval;
        break;
      case 3:
        red=(read_next_number(in)*255)/maxval;
        grn=(read_next_number(in)*255)/maxval;
        blu=(read_next_number(in)*255)/maxval;
        *ptr++=red; *ptr++=grn; *ptr++=blu;
      }
  }
}


void ditch_line(FILE *in)
{
int c;

while((c=fgetc(in))!='\n' && c!=EOF);
}


/* for text-style PNM files, i.e. P[123].
 * we take an extremely generous outlook - anything other than a decimal
 * digit is considered whitespace.
 * and as per p[bgp]m(5), comments can start anywhere.
 */
int read_next_number(FILE *in)
{
int c,num,in_num,gotnum;

num=0;
in_num=gotnum=0;

do
  {
  if(feof(in)) return(0);
  if((c=fgetc(in))=='#')
    ditch_line(in);
  else
    if(isdigit(c))
      num=10*num+c-49+(in_num=1);
    else
      gotnum=in_num;
  }
while(!gotnum);  

return(num);
}


void write_interp_line(unsigned char *linebuf,unsigned char *outbuf,
	int width,int bytepp,int scale)
{
static unsigned char blackcol[]={0,0,0};
int x,y,i;
int inextpix=1;
int a1,a2,a3,a4,in_rg,in_dn,in_dr;
int wp=width*bytepp,sci=scale*inextpix;
int scaleincr=0,subpix_xpos,subpix_ypos,sxmulsiz,symulsiz,simulsiz=0;
int sisize=0,sis2=0;
unsigned char *ptr1,*ptr2,*ptr3,*ptr4,*dst;
int swidth=width*scale,swidth_bytes;

if(edge_mode==EDGE_DROP) swidth-=scale;
swidth_bytes=swidth*bytepp;

sisize=0;
while(sisize<256) sisize+=scale;
scaleincr=sisize/scale;
simulsiz=scaleincr*sisize;
sis2=sisize*sisize;

for(y=0;y<scale;y++)
  {
  /* This has been hacked into unreadability in an attempt to get it
   * as fast as possible.
   * It's still really slow. :-(
   */
  
  in_rg=inextpix*bytepp;
  in_dn=inextpix*wp;
  in_dr=in_dn+in_rg;
  subpix_ypos=(y%scale)*scaleincr;
  subpix_xpos=0;
  
  ptr1=linebuf;
  ptr2=ptr1+in_rg;
  ptr3=ptr1+in_dn;
  ptr4=ptr1+in_dr;
  
  symulsiz=sisize*subpix_ypos;
  sxmulsiz=sisize*subpix_xpos;
  
  dst=outbuf;
  
  for(x=0;x<swidth;x++)
    {
    a3=symulsiz-(a4=subpix_xpos*subpix_ypos);
    a2=sxmulsiz-a4;
    a1=sis2-sxmulsiz-symulsiz+a4;
    
    for(i=0;i<bytepp;i++)
      *dst++=(ptr1[i]*a1+ptr2[i]*a2+
              ptr3[i]*a3+ptr4[i]*a4)/sis2;
    
    subpix_xpos+=scaleincr;
    sxmulsiz+=simulsiz;
    if(subpix_xpos>=sisize)
      {
      subpix_xpos=sxmulsiz=0;
      ptr1+=bytepp; ptr2+=bytepp; ptr3+=bytepp; ptr4+=bytepp;
      }
    
    if(edge_mode!=EDGE_DROP && x>=swidth-sci)
      {
      if(edge_mode==EDGE_INTERP_TO_BLACK)
        ptr2=ptr4=blackcol;
      else	/* EDGE_NON_INTERP */
        ptr2-=bytepp,ptr4-=bytepp;	/* horiz interp with self */
      }
    }
  
  if(fwrite(outbuf,1,swidth_bytes,stdout)!=swidth_bytes)
    fprintf(stderr,"pnminterp: error writing output.\n"),exit(1);
  }
}
