/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */


/* Ship pixels to printer, other associated routines. */

#include <stdio.h>
#include "defs.h"
#include "units.h"
#include "dvitps.h"
#include "extfil.h"
#include "fontd.h"
#include "emit.h"

/*
 * DEBUG_PIXELS: show the pixel array as * and .
 * SAFE: indices checking for PIX. Should be used consistently with
 * the same switch in emitc.c.
 */
#define SAFE

extern void CheckPixelArrayIndices();
extern void FExFlush();
extern char *PIX;
extern int PixelsMaxRow;
extern int PixelsMaxCol;
extern int NumCharsInPixelFonts;
extern EX_FILES Ex_PsOutput;
extern FE_P CurFontPointer;
extern int EstimatedVmUsed;
extern int PssColumn;
extern char *digit;

#ifdef DEBUG_PIXELS
void DumpPixelsIntoDebugFile();
#endif

/*
 * Blacken
 * *******
 * Given the current position (m,n) blacken count number of pixels
 * 	in that row, starting with the current column.
 *
 * n, m: row/column number
 * count: how many pixels to blacken.
 */
void
Blacken (n, m, count)
     int m, n;
     int count;
{
  int k;
  for (k=m; k<m+count; k++) {
#ifdef SAFE
    CheckPixelArrayIndices(n,k/8);
#endif
    PIX_LOAD(n,k/8) |= (1<<(7-(k%8)));
  }
}

/*
 * DuplicateRows
 * *************
 * n: row number
 * w: width in pixels
 * count: generate that many copies of row n and place them
 *        into rows n-1, ..., n-count
 */
void
DuplicateRows(n, w, count)
     int n;
     int w;
     int count;
{
  int i, j;
  for (i=1; i<=count; i++) {
    for (j=0; j<(w+7)/8; j++) {
#ifdef SAFE
      CheckPixelArrayIndices(n-i,j);
      CheckPixelArrayIndices(n,j);
#endif
      PIX_LOAD(n-i,j) = PIX_LOAD(n,j);
    }
  }
}

/*
 * ShipPixelsToPrinter
 * *******************
 * Assume that the Pixel array is loaded. We now will ship those pixels
 * and other font metric information out to the printer.
 * Also discovers slitex characters, characters where the width and height
 * are both zero.
 *
 * c_code: character code
 * w_pixels: width, in pixels
 * h_pixels: height, in pixels
 * x_off:   x offset of the character origin to the
 *	    box origin. If the value is positive, then the
 *	    origin is inside the pixel box. If negative
 *	    then the origin is outside the pixel box.
 * y_off:   y offset of the character origin to the
 *	    box origin. Its the number of rows above the
 *	    pixel line.
 * p_tfmw:  width of the character in pixels, derived originally
 *	    from the tfm width (rounded to the nearest integer pixel)
 */ 
void
ShipPixelsToPrinter(c_code, w_pixels, h_pixels, x_off, y_off, p_tfmw)
     int c_code;
     PXLU w_pixels, h_pixels;
     PXLU x_off, y_off;
     PXLU p_tfmw;
{
  int i, j; /* Loop variables. */
  int cc; /* Counter shipping of the bit patterns to limit line length. */
  int slitex; /* Is the character a slitex character ? */

  /* Check whether character code does not exceed upper limit established
     by NumCharsInPixelFonts! */
  if (c_code >= NumCharsInPixelFonts)
    Fatal3 ("ShipPixelsToPrinter: character code %d exceeds maximum allowable character code of %d.",
	    c_code, NumCharsInPixelFonts);

  slitex = (h_pixels == 0);
#ifdef DEBUG
  fprintf (stderr, "ShipPixelsToPrinter(): WIDTH: %d, HEIGHT: %d, slitex: %d\n",
	   w_pixels, h_pixels);
#endif

  /* Slitex characters: we are done very quickly! */
  if (slitex) {
    fprintf (EX_FP(Ex_PsOutput), "[<00> 1 1 0 0 %d] %d @dc\n", p_tfmw, c_code);
    return;
  }

#ifdef DEBUG_PIXELS
  DumpPixelsIntoDebugFile(c_code, w_pixels, h_pixels, x_off, y_off, p_tfmw);
#endif DEBUG_PIXELS

  /* Send out the hex pattern loaded in Pixels. Proceed down
   * to top. See documentation about PIX. */
  /* (1) open hex pattern */
  fprintf (EX_FP(Ex_PsOutput), "[<\n");
  cc = 0;

  /* (3) Send the character pixels. */
  for (i=0; i<=h_pixels-1; i++) {
    for (j=0; j<(w_pixels+7)/8; j++) {
      PSS_SEND_HEX(PIX_LOAD(i,j));
      if (cc++ > 35) {
	cc = 0;
	EMITC('\n');
      }
    }
  }

  /* (5) Done with the bit patterns. */
  fprintf (EX_FP(Ex_PsOutput), ">\n");
  
  /* This here is a terrible cluge to make $\longleftarrow$ and related
     instructions to work. I have no idea what the problem is! Is it,
     may be, pk file related or what?! */
  if (c_code == 0 && strncmp(CurFontPointer->f_n, "cmsy", strlen("cmsy")) == 0)
    y_off++;
  /* Height, depth (in pixels), x and y offset, spacing width and character code */
  fprintf (EX_FP(Ex_PsOutput), "\t %d %d %d %d %d] %d @dc\n",
	   w_pixels, h_pixels, x_off, y_off, p_tfmw, c_code);

  /* Clean up the pixel array. */
  for (i=0; i<=h_pixels-1; i++) {
    for (j=0; j<(w_pixels+7)/8; j++)
      {
#ifdef SAFE
	CheckPixelArrayIndices(i,j);
#endif
	PIX_LOAD(i,j) = 0;
      } /* for j */
  } /* for i */

#ifdef DEBUG
  FExFlush(&Ex_PsOutput);
#endif
}


#ifdef DEBUG_PIXELS

/*
 * AsterixByte
 * ***********
 * Print a pixel byte as a sequence of 8 '*' or '.' for black
 * or white pixels.
 *
 * f: the file to be used where output occurs to.
 * c: the character
 * b: number of bits to be printed (starting at bit 7, downwards).
 */
void
AsterixByte(f, c, b)
     FILE *f;
     char c;
     int b;
{
  int i;
  for (i=7; i>=8-b; i--) {
    if (c&(1<<i))
      fprintf (f, "*");
    else
      fprintf (f, ".");
  }
}

/*
 * DumpPixelsIntoDebugFile
 * ***********************
 * Argumens are the same as to ShipPixelsToPrinter().
 */
void
DumpPixelsIntoDebugFile(c_code, w_pixels, h_pixels, x_off, y_off, p_tfmw)
     int c_code;
     PXLU w_pixels, h_pixels;
     PXLU x_off, y_off;
     PXLU p_tfmw;
{
  int i, j;
  fprintf (stderr, "%% char: '%o, w: %d, h: %d, x_off: %d, y_off: %d\n",
	   c_code, w_pixels, h_pixels, x_off, y_off);
  fprintf (stderr, "%% tfm width: %d\n", p_tfmw);
  
  fprintf (stderr, "%%     ");
  for (j=0; j<w_pixels; j++)
    fprintf (stderr, "%d", j/10);

  fprintf (stderr, "\n%%     ");
  for (j=0; j<w_pixels; j++)
    fprintf (stderr, "%d", j%10);
  fprintf (stderr, "\n");

  for (i=h_pixels-1; i>=0; i--) {
    fprintf (stderr, "%% %2d: ", i);
    for (j=0; j<(w_pixels+7)/8; j++) {
      if (j != (w_pixels+7)/8 - 1) {
	AsterixByte(stderr, PIX_LOAD(i,j), 8);
      } else {
	if (w_pixels%8 == 0)	
	  AsterixByte(stderr, PIX_LOAD(i,j), 8);
	else
	  AsterixByte(stderr, PIX_LOAD(i,j), w_pixels%8);
      }
    }
    fprintf (stderr, "\n");
  }
}

#endif
