/*
 * Copyright 1993-1996 Johannes Sixt
 *
 * Permission to use, copy, modify and distribute this software and its
 * documentation for any purpose other than its commercial exploitation
 * is hereby granted without fee, provided that the above copyright
 * notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation. The authors
 * make no representations about the suitability of this software for
 * any purpose. It is provided "as is" without express or implied warranty.
 *
 * THE AUTHORS DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 * IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
 * USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: Johannes Sixt <Johannes.Sixt@telecom.at>
 */


#ifndef _OBJECTS_H_
#define _OBJECTS_H_

#define MAX_TEXT_LEN	127
#define MAX_POINTS	3
#define RASTER_TEXT_LEN	10
/* minimum distance of two raster points in DrawRaster, SetRaster */
#define MIN_RASTER	4.0
/* normal dash length of TeXDashedText box in points */
#define DASH_LEN	5.0

typedef enum {
	TeXLine, TeXVector, TeXFramedText, TeXDashedText, TeXFilled,
	TeXCircle, TeXDisc, TeXOval, TeXText, TeXBezier
} TeXPictType;

#define ALIGN_H_LEFT	0x01
#define ALIGN_H_CENTER	0x02
#define ALIGN_H_RIGHT	0x04
#define ALIGN_H_ALL	(ALIGN_H_LEFT | ALIGN_H_CENTER | ALIGN_H_RIGHT)
#define ALIGN_V_TOP	0x10
#define ALIGN_V_CENTER	0x20
#define ALIGN_V_BOTTOM	0x40
#define ALIGN_V_ALL	(ALIGN_V_TOP | ALIGN_V_CENTER | ALIGN_V_BOTTOM)

typedef enum {
	ShadowNone, ShadowFramed, ShadowFilled
} Shadow;

typedef struct _TeXPictObj {
	TeXPictType	type;
	struct _TeXPictObj *next;	/* linked list */
	float	x, y;			/* position */
	union {
		struct {
			float	w, h;		/* dimensions */
			float	xtra;		/* dash length or corner rad */
			unsigned align;
			Shadow	shadow;
			char	text[MAX_TEXT_LEN+1];
		} rectangular;
		float	r;		/* radius */
		struct {
			float	ex, ey;		/* end of line */
		} linear;
		struct {
			float	sx, sy;		/* curvature point */
			float	ex, ey;		/* end of bezier */
		} bezier;
	} data;
} TeXPictObj;

typedef struct {
	float x_min, y_min, x_max, y_max;
} TeXObjExtent;

extern int circle_diameter[];
extern int disc_diameter[];

void ObjInit(Display *disp, Screen *scr, Drawable d,
		XFontStruct *font, Pixel foreground, Pixel background,
		float initRasterHeight);
TeXPictObj *ObjCreate(TeXPictType type);
void ObjDestroy(TeXPictObj *obj);
void ObjCopy(TeXPictObj *fromObj, TeXPictObj *toObj);
void ObjValidate(TeXPictType type, XPoint *points, int n_points, int point);
void ObjStore(TeXPictObj *obj, XPoint *points, int num_points);
void ObjRetrieve(TeXPictObj *obj, XPoint *points, int *num_points);
void ObjDraw(TeXPictObj *obj, Display *disp, Drawable d);
void ObjRubber(TeXPictType type, Display *disp, Drawable d,
		XPoint *points, int num_points);
void ObjSave(TeXPictObj *obj, FILE *file, float max_x, float max_y);
void ObjOffset(TeXPictObj *obj, float x_off, float y_off);
void ObjGetExtent(TeXPictObj *obj, TeXObjExtent *extent);
Boolean ObjMatches(TeXPictObj *obj, float x, float y);
void ObjRescale(TeXPictObj *obj, float zoom);
void SnapPosition(Position *x, Position *y);
void SnapPositionFloat(Position *x, Position *y, float *xf, float *yf, float h);
void ValidRadius(int x, int y, Position *xp, Position *yp, int *d, float zoom);
Boolean SetRaster(float rast_height);
void SetFileScale(float newScale);

extern Boolean unlimitedSlopes;
extern Boolean unlimitedDiameters;
extern Boolean noMinLength;
extern Boolean snap, raster;
extern char rasterHeightStr[RASTER_TEXT_LEN];
extern float zoomFactor;		/* current zoom value */
extern float fileScale;	       /* rounded unit length in pt [\unitlength/pt] */

#endif /* _OBJECTS_H_ */
