/*
 * LaTeD Version 1.1
 * (c) Gene Ressler 1993, 94, 97
 *   de8827@trotter.usma.edu
 *
 * LaTeD is a graphical editor for drawings in the LaTeX "picture" 
 * environment.  It runs under MSDOS or in a Windows DOS box.  The
 * distribution includes full sources, including LaTeX source for 
 * its documentation.
 *
 * No warranty of this software is expressed or implied by the author.
 *
 * Copy and use this program freely for any purpose except for sale
 * of the program (including the source code) itself.  That is, 
 * no one can copy this program for the purpose of providing it to 
 * another person in exchange for money or other compensation, even 
 * if this program is only part of the exchange.
 *
 * All copies of computer source code in this distribution, whether
 * copies in whole or in part, must have this notice attached.
 */

/* MALLOC.C --- Safe replacements for memory-allocating
		library functions.  */

#include <stdio.h>
#include <stdlib.h>
#include <alloc.h>
#include <graphics.h>
#include "window.h"
#include "settings.h"
#include "canvas.h"
#include "env.h"

/* window.h #defines these.  We want the real ones here. */
#undef malloc
#undef realloc
#undef free
#undef fclose

/* Give up when there's a memory problem. */
void die(char *msg)
{
  static char panic_filename[] = "-PANIC-.TEX";
  restorecrtmode();
  fprintf(stderr, 
    "Panic: %s.\n"
    "Attempting to save picture to %s...", 
    msg, panic_filename);
  do_emergency_save(panic_filename);
  fprintf(stderr,
    "done.\n"
    "LaTeD abort. Sorry...\n");

  exit(1);
}

/* ----- Animal verions. -------------------------------------------- */

#ifdef MALLOC_TRACE

#include <stdio.h>
#include <dos.h>

/* File for trace information. */
static FILE *f;

/* Convert a pointer to an absolute (unsigned long) address. */
#define ABS(P) (((unsigned long)FP_SEG((void far*)P) << 4) + FP_OFF((void far*)P))

void *safe_malloc(unsigned n)
{
  void *p;

  if ((p = malloc(n + 2)) == NULL)
    die();

  if (f != NULL)
    fprintf(f, "m%lu %u\n", ABS(p), n + 2);

  *(unsigned*)p = n + 2;
  return (unsigned*)p + 1;
}

void *safe_realloc(void *p, unsigned n)
{
  void *pp;
  unsigned old_size;

  p = (unsigned*)p - 1;
  old_size = *(unsigned*)p;

  if ((pp = realloc(p, n + 2)) == NULL)
    die();

  if (f != NULL) {
    fprintf(f, "f%lu %u\n", ABS(p), old_size);
    fprintf(f, "m%lu %u\n", ABS(pp), n + 2);
  }

  *(unsigned*)pp = n + 2;
  return (unsigned*)pp + 1;
}

void safe_free(void *p)
{
  p = (unsigned*)p - 1;
  if (f != NULL)
    fprintf(f, "f%lu %u\n", ABS(p), *(unsigned*)p);
  free(p);
}

/* Can't trace this here. */
void safe_fclose(void *p)
{
  fclose(p);
}

void open_malloc_trace(void)
{
  f = fopen("malloc.trc", "w");
}

void close_malloc_trace(void)
{
  if (f != NULL)
    fclose(f);
}

#pragma startup open_malloc_trace
#pragma exit close_malloc_trace

/* ----- Production versions ---------------------------------------- */

#elif defined(NDEBUG)

static void *emergency_block;
static int in_panic_p;

static void panic(void)
{
  if (!in_panic_p) {
    free(emergency_block);
    emergency_block = 0;
    in_panic_p = 1;
    message(bit(mOK), root_window,
	    "Low Memory.\n"
	    "Save IMMEDIATELY!");
    in_panic_p = 0;
  }
}

void *safe_malloc(unsigned n)
{
  void *p;

  /* Emergency block must be at least big enough
     to hold saveunder of panic message window. */
  if (!emergency_block && !in_panic_p)
    emergency_block = malloc(50000u);

retry:
		 
  if ((p = malloc(n)) == NULL) {
    if (emergency_block) {
      panic();
      goto retry;
    }
    else
      die("alloc failure");
  }

  return p;
}

void *safe_realloc(void *p, unsigned n)
{
  void *pp;

retry:

  if ((pp = realloc(p, n)) == NULL) {
    if (emergency_block) {
      panic();
      goto retry;
    }
    else
      die("realloc failure");
  }

  return pp;
}

void safe_free(void *p)
{
  free(p);
}

void safe_fclose(void *p)
{
  fclose(p);
}

/* ----- Debugging versions ----------------------------------------- */

#else

/* Use Turbo library functions to 
   check heap for magic value. */
void check_heap(void)
{
  int res = heapcheckfree(0x55);
  if (res < 0)
    switch( res ) {

      case _HEAPCORRUPT:
	die( "block list error" );

      case _BADVALUE:
	die( "write to free space" );

      default:
	die( "bad error" );

    }
}

void *safe_malloc(unsigned n)
{
  void *p;
	       
  check_heap();

  if ((p = calloc(1, n)) == NULL)
    die("alloc failure");

  return p;
}

static void fill_heap_with_magic(void)
{
  if (heapfillfree(0x55) < 0) 
    die("found corrupt heap during fill");
}

void *safe_realloc(void *p, unsigned n)
{
  void *pp;

  check_heap();

  if ((pp = realloc(p, n)) == NULL)
    die("realloc failure");

  fill_heap_with_magic();

  return pp;
}

void safe_free(void *p)
{
  int h = heapchecknode(p);
  if (h != _USEDENTRY) 
    die("bad block to free");
  check_heap();
  free(p);
  fill_heap_with_magic();
}

void safe_fclose(void *p)
{
  fclose(p);
  fill_heap_with_magic();
}

#pragma startup fill_heap_with_magic

#endif
