/*
 * ParticleAnimation
 */

#ifndef animation_h
#define animation_h

#include <InterViews/monoglyph.h>
#include <InterViews/transformer.h>
#include <InterViews/patch.h>

#include <stdlib.h>

class Particle;
class Color;

typedef float RealCoord;
typedef Coord ScreenCoord;

class ParticleAnimation : public Patch {
public:
  friend Particle;

  ParticleAnimation(Glyph* background, 
       RealCoord top, RealCoord bottom,
       RealCoord left, RealCoord right,
       RealCoord floor, RealCoord ceiling
   );
  virtual ~ParticleAnimation();

  virtual void allocate(Canvas*, const Allocation&, Extension&);
  virtual void draw(Canvas*, const Allocation&) const;
  virtual void print(Printer*, const Allocation&) const;
  virtual void pick(Canvas*, const Allocation&, int depth, Hit&);
  
  void allocate(Particle* particle);
  virtual Particle* head() const;

  virtual void remove(GlyphIndex) { }
  virtual void remove();
  virtual void remove(Particle* particle);

  void update_real_size(
       Coord top =-1.0, Coord bottom =-1.0, Coord left =-1.0, Coord right =-1.0, 
       Coord floor =-1.0, Coord ceiling =-1.0);

private:
  virtual void append(Glyph*);
  virtual void append(Particle* particle);

  void move(Particle* particle);

  void xy_real_to_screen(
      RealCoord real_x, RealCoord real_y,
      ScreenCoord& screen_x, ScreenCoord& screen_y
   );

  void xy_screen_to_real(
      ScreenCoord screen_x, ScreenCoord screen_y,
      RealCoord& real_x, RealCoord& real_y
   );


  void dxy_real_to_screen(
      RealCoord real_dx, RealCoord real_dy,
      ScreenCoord& screen_dx, ScreenCoord& screen_dy
   );

  void dxy_screen_to_real(
      ScreenCoord screen_dx, ScreenCoord screen_dy,
      RealCoord& real_dx, RealCoord& real_dy
   );

  ScreenCoord left() const;
  ScreenCoord right() const;
  ScreenCoord bottom() const;
  ScreenCoord top() const;
  ScreenCoord x() const;
  ScreenCoord y() const;
  
  RealCoord _real_top;
  RealCoord _real_bottom;
  RealCoord _real_left; 
  RealCoord _real_right;
  RealCoord _real_ceiling;
  RealCoord _real_floor;

  virtual void update_screen_to_real();
  boolean _real_size_changed;

  Transformer* _xy_screen_to_real;
  Transformer* _dxy_screen_to_real;

  Patch* _patch;
  Particle* _head;
  Canvas* _canvas;
  Allocation _allocation;
  boolean _allocated;
  class ColorScale* _color_scale;
};

inline Particle* ParticleAnimation::head() const
   { return(_head); }

inline void ParticleAnimation::xy_real_to_screen(
      RealCoord real_x, RealCoord real_y,
      ScreenCoord& screen_x, ScreenCoord& screen_y
   )
{
  _xy_screen_to_real->inverse_transform(real_x, real_y, screen_x, screen_y);
}

inline void ParticleAnimation::xy_screen_to_real(
      ScreenCoord screen_x, ScreenCoord screen_y,
      RealCoord& real_x, RealCoord& real_y
   )
{

//  printf("ParticleAnimation::xy_screen_to_real()\n");
//  printf("\t Screen x=%f y=%f z=%f\n", screen_x, screen_y, screen_z);

  _xy_screen_to_real->transform(screen_x, screen_y, real_x, real_y);

//  printf("\tReal x=%f y=%f z=%f\n", real_x, real_y, real_z);  
  
}

inline void ParticleAnimation::dxy_real_to_screen(
      RealCoord real_dx, RealCoord real_dy,
      ScreenCoord& screen_dx, ScreenCoord& screen_dy
   )
{
  _dxy_screen_to_real->inverse_transform(real_dx, real_dy, screen_dx, screen_dy);
}

inline void ParticleAnimation::dxy_screen_to_real(
      ScreenCoord screen_dx, ScreenCoord screen_dy,
      RealCoord& real_dx, RealCoord& real_dy
   )
{
  _dxy_screen_to_real->transform(screen_dx, screen_dy, real_dx, real_dy);
}

#endif
