/* File: simulation_tools.c
 * Created: 4/26/92 by John Butare(jb7p@andrew.cmu.edu)
 * Description:
 *	
 * Modifications:
 */

#include <InterViews/window.h>
#include <InterViews/bitmap.h>
#include <InterViews/stencil.h>
#include <InterViews/tiff.h>
#include <InterViews/raster.h>
#include <InterViews/image.h>
#include <InterViews/background.h>
#include <InterViews/box.h>
#include <InterViews/glue.h>
#include <InterViews/label.h>
#include <InterViews/deck.h>
#include <InterViews/margin.h>
#include <InterViews/style.h>
#include <InterViews/action.h>
#include <InterViews/border.h>
#include <InterViews/center.h>

#include <IV-look/button.h>
#include <IV-look/kit.h>

#include <OS/string.h>

#include <stdio.h>
#include <strings.h>

#include <UI/generic_button.h>
#include <UI/date_editor.h>
#include <UI/radio_button.h>
#include <UI/rbbox.h>
#include <UI/toggle_button.h>
#include <UI/kit.h>
#include <UI/label.h>
#include <UI/text_editor.h>
#include <UI/string_list.h>

#include "gems.h"
#include "particle_tracker.h"
#include "simulation_tools.h"
#include "toolbox_tools.h"

/*
 * SimulationToolBox
 */

declare(ActionCallback, SimulationToolbox);
implement(ActionCallback, SimulationToolbox);

declareList(SimulationToolList, SimulationTool*);
implementList(SimulationToolList, SimulationTool*);

SimulationToolbox::SimulationToolbox(ToolboxListBox* toolbox_list_box)
   : Patch(nil), Adjustable()
{
  Gems* gems = Gems::instance();
  Style* style = gems->style();
  Kit* kit = Kit::instance();
  GemsKit* gemsKit = GemsKit::instance();

  _tool_list = new SimulationToolList(10);
  _toolbox_list_box = toolbox_list_box;

  body(_deck = new Deck());
  _deck->append(_toolbox_list_box->structure());
  _deck->flip_to(0);

  /*
   * Control
   */
  static char buf[20];
  sprintf(buf, "%d of %d", _deck->card()+1, count());

  Glyph* fixed_hsep = new HGlue(2.0, 0.0, 0.0);

  _control = new Deck();
  _control_patch = new Patch(_control);
  _control->flip_to(0);

  _control->append(
       new Overlay(
       	   new VGlue(0.0, fil, fil),
       	   new HGlue(0.0, fil, fil)
       )
   );

  _control->append( 
       new LRBox(
       	   new Overlay(kit->left_mover(this, style), new HGlue(20.0, 0.0, 0.0)),
	   fixed_hsep,
       	   new Overlay(
       	      gemsKit->push_button(
       	       	  _display  = gemsKit->label(buf, style), 
       	       	  style,
       	       	  new ActionCallback(SimulationToolbox)(this, &SimulationToolbox::enable)
       	      ),
       	      new HGlue(20.0, 0.0, 0.0)
       	   ),
	   fixed_hsep,
       	   new Overlay(kit->right_mover(this, style), new HGlue(20.0, 0.0, 0.0))
       )
   );
}

SimulationToolbox::~SimulationToolbox() { }

void SimulationToolbox::enable()
{
  use(_deck->card());
}

SimulationTool* SimulationToolbox::current()
{
  return(_tool_list->item(_deck->card()-1));
}

void SimulationToolbox::scroll_backward(DimensionName)
{
  GlyphIndex cur = _deck->card() + 1;
  use(cur == _deck->count() ? 1 : cur);
}

void SimulationToolbox::scroll_forward(DimensionName)
{
  GlyphIndex cur = _deck->card() - 1;
  use(cur == 0 ? _deck->count() - 1 : cur);
}

void SimulationToolbox::update_display()
{
  static char buf[20];
  sprintf(buf, "%d", _deck->card());
  _display->string(buf);

  printf("SimulationToolbox::update_display()\n");
  printf("\tbuf=%s\n", buf);
}

void SimulationToolbox::use(GlyphIndex index)
{
  printf("SimulationToolbox::use\n");
  printf("\t index=%d\n", index);

  /* if the simulation toolbox is not active then make it active */
  if (_toolbox_list_box->current() != 0) 
    _toolbox_list_box->use(0);

  /* have the current simulation update the toolbox */
  _tool_list->item(index-1)->update_toolbox();
  _deck->flip_to(index);

  update_display();
  redraw();
  reallocate();
  redraw();
}

void SimulationToolbox::append(Glyph*)
{
  Gems::instance()->error("SimulationToolbox::append cannot append a Glyph\n");
}

void SimulationToolbox::append(SimulationTool* g)
{
  MonoGlyph::append(g);
  _tool_list->append(g);

  printf("SimulationToolbox::append\n");
  printf("\t count()=%d\n", count());

  if (count() == 2)
    control(true);

  use(count()-1);
}

void SimulationToolbox::control(boolean on)
{
  if (on)
    _control->flip_to(1);
  else
    _control->flip_to(0);
 
  _control_patch->redraw();
  _control_patch->reallocate();
  _control_patch->redraw();
}

void SimulationToolbox::remove(GlyphIndex index)
{
  GlyphIndex current_simulation = _deck->card();

  MonoGlyph::remove(index);
  _tool_list->remove(index);

  if (current_simulation == index) {
    GlyphIndex num_simulations = _deck->count()-1;
    
    if (num_simulations == 0) {
      control(false);
      _toolbox_list_box->use(2);
    } else {
      GlyphIndex new_simulation = 
       	  (current_simulation > 1 ? current_simulation-1 : num_simulations);
      use(new_simulation);
    }
  } else
    update_display();
}

void SimulationToolbox::remove(Glyph* g)
{
  Gems* gems = Gems::instance();
  
  for (GlyphIndex i = 0 ; i < count() ; ++i)
    if (component(i) == g) {
      remove(i);
      return;
    }
  
  gems->error("SimulationToolbox asked to remove a glyph it does not contain.");
}

/*
 * SimulationTool
 */

SimulationTool::SimulationTool(SimulationToolbox* simulation_toolbox)
{
  _simulation_toolbox = simulation_toolbox;
}

void SimulationTool::quit_cb()
{
  _simulation_toolbox->remove(this);
}

GemsTextEditor* SimulationTool::editor(GlyphIndex i)
{
  return(_simulation_toolbox->toolbox_list_box()->editor(i));
}

/*
 * ChemistrySimulationTool
 */

ChemistrySimulationTool::ChemistrySimulationTool(
       SimulationToolbox* simulation_toolbox
   ) : SimulationTool(simulation_toolbox)
{
  printf("ChemistrySimulationTool::ChemistrySimulationTool()\n");

  _simulation = simulation();

  printf("\t _simulation=%s\n", _simulation);

  chemical1_cb();
  chemical2_cb();
  height_cb();
  date_cb();
}

ChemistrySimulationTool::~ChemistrySimulationTool()
{
}

void ChemistrySimulationTool::update_toolbox()
{
  printf("ChemistrySimulationTool::update_toolbox()\n");

  simulation(_simulation);
  chemical1(_chemical1);
  chemical2(_chemical2);
  height(_height);
  set_date(_month, _day, _year);
}

void ChemistrySimulationTool::update_value(GlyphIndex index)
{
  printf("ChemistrySimulationTool::update_value()\n");
  printf("\t index=%d\n", index);

  switch (index) {
  case 1:
    break;
  case 2:    
    _chemical1 = chemical1();
    chemical1_cb();
    break;
  case 3:
    _chemical2 = chemical2();
    chemical2_cb();
    break;
  case 4:
    _height = height();
    height_cb();
    break;
  case 5:
    get_date(_month, _day, _year);
    date_cb();
    break;
  }
}

void ChemistrySimulationTool::simulation(char* s) 
{ 
  editor(simulation_index)->string(s);
}

char* ChemistrySimulationTool::simulation()
{
  return(editor(simulation_index)->string());
}

void ChemistrySimulationTool::chemical1(char* s)
{ 
  editor(chemical1_index)->string(s);
}

char* ChemistrySimulationTool::chemical1()
{ 
  return(editor(chemical1_index)->string());
}

void ChemistrySimulationTool::chemical2(char* s)
{
  editor(chemical2_index)->string(s);
}

char* ChemistrySimulationTool::chemical2()
{
  return(editor(chemical2_index)->string());
}
  
void ChemistrySimulationTool::height(float v)
{
  char buf[50];
  sprintf(buf, "%2f", v);
  editor(height_index)->string(buf);
}

float ChemistrySimulationTool::height()
{
  return(atof(editor(height_index)->string()));
}

void ChemistrySimulationTool::get_date(int month, int day, int year)
{
  char buf[20];
  
  sprintf(buf, "%d/%d/%d", month, day, year);
  editor(date_index)->string(buf);
}

const void ChemistrySimulationTool::set_date(int& month, int& day, int& year)
{ 
  const char* old_date = editor(date_index)->string();
  char* new_date = new char[strlen(old_date)];
  strcpy(new_date, old_date);
  char* curr = new_date;

  char *s = strchr(curr, '\\');
  if (s == nil) return;
  *s = '\0';
  month = atoi(curr);
  ++curr;

  s = strchr(curr, '\\');
  if (s == nil) return;
  *s = '\0';
  day = atoi(curr);
  ++curr;

  s = strchr(curr, '\\');
  if (s == nil) return;
  *s = '\0';
  year = atoi(curr);

  delete new_date;
}

/*
 * ParticleTrackerTool
 */

declare(ActionCallback, ParticleTrackerTool);
implement(ActionCallback, ParticleTrackerTool);

ParticleTrackerTool::ParticleTrackerTool(
       SimulationToolbox* simulation_toolbox, ParticleTracker* particle_tracker)
     : ChemistrySimulationTool(simulation_toolbox)
{
  Gems* gems = Gems::instance();
  Kit* kit = Kit::instance();
  GemsKit* gemsKit = GemsKit::instance();
  Style* style = gems->style();

  Glyph* stretch_vsep = new VGlue(2.0, fil, 0.0);
  Glyph* fixed_vsep = new VGlue(2.0, 0.0, 0.0);
  Glyph* stretch_hsep = new HGlue(2.0, fil, 0.0);
  Glyph* fixed_hsep = new HGlue(2.0, 0.0, 0.0);

  /*
   * Defaults
   */

  _particle_tracker = particle_tracker;
  _sources_visible_flag = true;

  int month = 8;
  int day = 27;
  int year = 87;

  _particle_tracker->date(month, day, year);

  /*
   * Parameters
   */

  char buf[50];
  
  sprintf(buf, "%d", _particle_tracker->time_step());
  _time_step_editor = gemsKit->text_editor(style, buf,
       new ActionCallback(ParticleTrackerTool)(this, &ParticleTrackerTool::time_step_cb));

  sprintf(buf, "%f", _particle_tracker->move_step());
  _move_step_editor = gemsKit->text_editor(style, buf,
       new ActionCallback(ParticleTrackerTool)(this, &ParticleTrackerTool::move_step_cb));

  sprintf(buf, "%d", _particle_tracker->release_step());
  _release_step_editor = gemsKit->text_editor(style, buf,
       new ActionCallback(ParticleTrackerTool)(this, &ParticleTrackerTool::release_step_cb));

  Glyph* labels =
       new TBBox(
       	   new VGlue(0.0, 0.0, 0.0), 
           new TMargin(new LRBox(stretch_hsep, kit->label("Time Step", style)), 2.0, fil, 0.0),
       	   new TMargin(new LRBox(stretch_hsep, kit->label("Move Step", style)), 2.0, fil, 0.0),
       	   new TMargin(new LRBox(stretch_hsep, kit->label("Release Step", style)), 2.0, fil, 0.0)
       );


  Glyph* editors =   
       new TBBox(
       	   new VGlue(0.0, 0.0, 0.0),
	   new TMargin(_time_step_editor, 2.0, fil, 0.0),
       	   new TMargin(_move_step_editor, 2.0, fil, 0.0),
       	   new TMargin(_release_step_editor, 2.0, fil, 0.0)
       );
   
  Glyph* parameters = 
    new LRBox(
       stretch_hsep,
       labels,
       new HGlue(4.0, 0.0, 0.0),
       editors,
       stretch_hsep
   );

  /*
   * Controls1
   */

  Glyph* sources_clear = gemsKit->push_button("Clear", style,
       new ActionCallback(ParticleTrackerTool)(this, &ParticleTrackerTool::clear_sources_cb)
   );

  Glyph* sources_visible = gemsKit->toggle_button("Visible", style,
       new ActionCallback(ParticleTrackerTool)(this, &ParticleTrackerTool::sources_visible_cb),
       nil, nil, nil, &_sources_visible_flag);

  Glyph* particles_clear = gemsKit->push_button("Clear", style,
       new ActionCallback(ParticleTrackerTool)(this, &ParticleTrackerTool::clear_particles_cb)
   );

  Glyph* controls1  =
       new TBBox(
	   new VGlue(0.0, 0.0, 0.0),
       	   new LRBox(
       	       stretch_hsep,
       	       new TBBox(
	       	   new VGlue(0.0, 0.0, 0.0),
       	       	   new TMargin(kit->label("Sources", style),  2.0, fil, 0.0),
       	       	   new TMargin(kit->label("Particles", style),  2.0, fil, 0.0)  
       	       ),
       	       fixed_hsep,
       	       new TBBox(
       	       	   new VGlue(0.0, 0.0, 0.0),
       	       	   new TMargin(
       	       	       new LRBox(
       	               	   sources_clear,
       	       	       	   new HGlue(2.0, 0.0, 0.0),
       	       	       	   sources_visible,
       	       	       ),
       	       	       2.0, fil, 0.0
       	       	   ),
       	       	   new TMargin(particles_clear, 2.0, fil, 0.0)
       	       ),
       	       stretch_hsep,
       	   )
       );

  /*
   * Status
   */

  particle_tracker->time_display_action(
       new ActionCallback(ParticleTrackerTool)(this, &ParticleTrackerTool::time_display_cb)
   );
  _time_display = gemsKit->label("12:00 am", style);

  Glyph* status = 
    new TMargin(
       new LRBox(
       	   stretch_hsep,
       	   kit->label("Simulation Time", style),
       	   new HGlue(4.0, 0.0, 0.0),
       	   _time_display,
       	   stretch_hsep
       ),
       2.0, fil, 0.0
   );

  /*
   * Controls2
   */

  _controls = new RBBox();

  Glyph* start = _controls->append(
       gemsKit->radio_button("Start", style,
       	   new ActionCallback(ParticleTrackerTool)(this, &ParticleTrackerTool::start_cb)
       )
   );

  Glyph* stop = _controls->append(
       gemsKit->radio_button("Stop", style,
           new ActionCallback(ParticleTrackerTool)(this, &ParticleTrackerTool::stop_cb)
       )
   );

  _pause = gemsKit->toggle_button("Pause", style,
       new ActionCallback(ParticleTrackerTool)(this, &ParticleTrackerTool::pause_cb)
   );

  Glyph* quit = gemsKit->toggle_button("Quit", style,
       new ActionCallback(ParticleTrackerTool)(this, &ParticleTrackerTool::quit_cb)
   );

  _pause->enable(false); // disable pause
  _controls->choose(2);  // choose stop

  Glyph* controls2 =
       new TMargin(
       	   new LRBox(
       	       start,
       	       stretch_hsep,
       	       stop,
       	       stretch_hsep,
       	       _pause,
       	       stretch_hsep,
       	       quit
       	   ),  
       	   2.0, fil, 0.0
       );
       	   
  /*
   * Layout
   */

  body(	   
       new TBBox(
	   new VGlue(0.0, 0.0, 0.0),
       	   parameters,
       	   controls1,
       	   status,
	   controls2
       ),
   );
}

ParticleTrackerTool::~ParticleTrackerTool()
{
  _particle_tracker->visible(false);
  delete _particle_tracker;
}

void ParticleTrackerTool::update_toolbox()
{
  simulation(_simulation);
  height(_height);
  set_date(_month, _day, _year);
}

/*
 * ParticleTrackerTool Callbacks
 */

void ParticleTrackerTool::time_step_cb()
{
  _particle_tracker->time_step(atoi(_time_step_editor->string()));
  
//  printf("ParticleTrackerTool::time_step_cb()\n");
//  printf("\t%d\n", _particle_tracker->time_step_cb());
}

void ParticleTrackerTool::release_step_cb()
{
  _particle_tracker->release_step(atoi(_release_step_editor->string()));

  printf("ParticleTrackerTool::release_step_cb()\n");
  printf("\t%d\n", _particle_tracker->release_step());
}

void ParticleTrackerTool::height_cb()
{
  _particle_tracker->release_height(_height);

  printf("ParticleTrackerTool::release_height()\n");
  printf("\t%d\n", _particle_tracker->release_height());
}

void ParticleTrackerTool::move_step_cb()
{
  _particle_tracker->move_step(atof(_move_step_editor->string()));

  printf("ParticleTrackerTool::move_step_cb()\n");
  printf("\t%f\n", _particle_tracker->move_step());
}

void ParticleTrackerTool::sources_visible_cb()
{
  printf("ParticleTrackerTool::sources_visible_cb()\n");
  _particle_tracker->sources_visible(_sources_visible_flag);
}

void ParticleTrackerTool::start_cb()
{
  printf("ParticleTrackerTool::start_cb()\n");
  _pause->enable(true);
  _pause->choose(false);
  _particle_tracker->start();
}

void ParticleTrackerTool::time_display_cb()
{
//  printf("ParticleTrackerTool::time_display_cb()\n");
  char buf[50];

  int seconds = _particle_tracker->time();
//  printf("\tseconds=%d\n", seconds);
  int minutes = seconds / 60 % 60;
  int hours = seconds / 3600;
  if (hours > 12) (hours -= 12);
  boolean am = (seconds < 43200); /* 60 * 60 * 12 */ 

  if (hours == 0)
    hours = 12;

  sprintf(buf, "%2d:%02d %s", hours, minutes, (am ? "am" : "pm"));
  _time_display->string(buf);
}

void ParticleTrackerTool::stop_cb()
{ 
  printf("ParticleTrackerTool::stop_cb()\n");
  _pause->enable(false);
  _particle_tracker->stop();
}

void ParticleTrackerTool::pause_cb()
{ 
  printf("ParticleTrackerTool::pause()\n");
  _particle_tracker->pause();
}

void ParticleTrackerTool::quit_cb()
{ 
  printf("ParticleTrackerTool::quit_cb()\n");
  _particle_tracker->stop();
//  delete _particle_tracker;
  SimulationTool::quit_cb();
}

void ParticleTrackerTool::clear_particles_cb()
{
  _particle_tracker->clear_particles();
}

void ParticleTrackerTool::clear_sources_cb()
{
  _particle_tracker->clear_sources();
}

void ParticleTrackerTool::date_cb()
{
  _particle_tracker->date(_month, _day, _year);
}
