/*
 * fn.c: functions for adding and deleting functions from the symbol table.
 * Support for signal handlers is also found here.
 */

#include <signal.h>
#include "rc.h"
#include "utils.h"
#include "status.h"
#include "tree.h"
#include "hash.h"
#include "footobar.h"
#include "walk.h"
#include "nalloc.h"


#define NUMOFSIGNALS (sizeof(sigvals) / sizeof(int))

static void fnint(int);

static int sigvals[] = {
	SIGHUP, SIGINT, SIGQUIT, SIGTERM
};
static char *signames[] = {
	"sighup", "sigint", "sigquit", "sigterm"
};
static void (*sigdefaults[])(int) = {
	SIG_DFL, SIG_DFL, SIG_DFL, SIG_DFL
};

static Node *sigvectors[NUMOFSIGNALS];

static boolean handlerset[NUMOFSIGNALS];

static Node *sigexit;
static boolean runexit = FALSE;

void setsigvectors(boolean interactive, boolean debug) {
	if (interactive) {
		signal(SIGINT, sig);
		sigdefaults[1] = sig;
		if (!debug) {
			signal(SIGQUIT, sig);
			sigdefaults[2] = sig;
		}
	}
}

/* only run this in a child process! */

void setsigdefaults(void) {
	int i;

	for (i = 0; i < NUMOFSIGNALS; i++) {
		if (sigdefaults[i] != SIG_DFL || handlerset[i]) {
			handlerset[i] = FALSE;
			sigdefaults[i] = SIG_DFL;
			signal(sigvals[i], SIG_DFL);
		}
	}
	runexit = FALSE; /* No sigexit on subshells */
}

void rc_exit(int stat) {
	if (runexit) {
		walk(sigexit, PARENT);
		exit(getstatus());
	} else {
		exit(stat);
	}
}

static void fnint(int s) {
	int i;

	signal(s, fnint); /* sgi seems to require re-signalling */

	for (i = 0; i < NUMOFSIGNALS; i++)
		if (sigvals[i] == s) {
			walk(sigvectors[i], PARENT);
			return;
		}
	fprint(2,"fnint: unknown signal!\n");
}

void fnassign(char *name, Node *def) {
	Node *newdef = treecpy(def); /* important to do the treecopy first */
	Function *new = get_fn_place(name);
	int i;

	new->def = newdef;
	new->extdef = NULL;

	if (streq(name, "sigexit")) {
		sigexit = new->def;
		runexit = TRUE;
	}

	for (i = 0; i < NUMOFSIGNALS; i++)
		if (streq(signames[i], name)) {
			handlerset[i] = TRUE;
			sigvectors[i] = new->def;
			if (def != NULL)
				signal(sigvals[i], fnint);
			else
				signal(sigvals[i], SIG_IGN);
			break;
		}
}

void fnassign_string(char *extdef) {
	char *name = get_name(extdef+3); /* +3 to skip over "fn_" */
	Function *new;

	if (name == NULL)
		return;

	new = get_fn_place(name);
	new->def = NULL;
	new->extdef = ecpy(extdef);
}

Node *fnlookup(char *name) {
	Function *look = lookup_fn(name);
	static Node *null_function = NULL;
	Node *ret;

	if (look == NULL)
		return NULL; /* not found */
	if (look->def != NULL)
		return look->def;
	if (look->extdef == NULL) { /* function was set to null, e.g., fn foo {} */
	null:	if (null_function == NULL) {
			null_function = ealloc(offsetof(Node, u[2]));
			null_function->type = BODY;
			null_function->u[0].p = null_function->u[1].p = NULL;
		}
		return null_function;
	}

	ret = parse_fn(name, look->extdef);

	if (ret == NULL) {
		look->extdef = NULL;
		goto null;
	} else {
		return look->def = treecpy(ret); /* Need to take it out of talloc space */
	}
}

char *fnlookup_string(char *name) {
	Function *look = lookup_fn(name);

	if (look == NULL)
		return NULL;
	if (look->extdef != NULL)
		return look->extdef;
	return look->extdef = fun2str(name, look->def);
}

void fnrm(char *name) {
	int i;

	for (i = 0; i < NUMOFSIGNALS; i++)
		if (streq(signames[i], name)) {
			handlerset[i] = FALSE;
			signal(sigvals[i], sigdefaults[i]);
		}

	if (streq(name, "sigexit")) {
		runexit = FALSE;
		sigexit = NULL;
	}

	delete_fn(name);
}
