/* hash.c: hash table support for functions and variables. */

/*
 * functions and variables are cached in both internal and external
 * form for performance. Thus a variable which is never "dereferenced"
 * with a $ is passed on to rc's children untouched. This is not so
 * important for variables, but is a big win for functions, where a call
 * to yyparse() is involved.
 */

#include "rc.h"
#include "utils.h"
#include "hash.h"
#include "list.h"
#include "tree.h"
#include "footobar.h"

static boolean exportable(char *);
static int hash(char *, int);
static int find(char *, Htab *, int);
static void free_fn(Function *);
static void free_var(Variable *);

Htab *fp;
Htab *vp;
static int fused, fsize, vused, vsize;
static char **env;
static int bozosize;
static int envsize;
static boolean env_dirty = TRUE;
static char *dead = "";

#define HASHSIZE 64 /* rc was debugged with HASHSIZE == 2; 64 is about right for normal use */

void inithash(void) {
	int i;

	fp = ealloc(sizeof(Htab) * HASHSIZE);
	vp = ealloc(sizeof(Htab) * HASHSIZE);
	fused = vused = 0;
	fsize = vsize = HASHSIZE;

	for (i = 0; i < HASHSIZE; i++)
		vp[i].name = fp[i].name = NULL;
}

#define ADV()   if ((c = *s++) == '\0') break;

static int hash(char *s, int size) { /* hash function courtesy of paul haahr. */
	int n = 0;
	int c;

	while (1) {
		ADV();
		n += (c << 17) ^ (c << 11) ^ (c << 5) ^ (c >> 1);
		ADV();
		n ^= (c << 14) + (c << 7) + (c << 4) + c;
		ADV();
		n ^= (~c << 11) | ((c << 3) ^ (c >> 1));
		ADV();
		n -= (c << 16) | (c << 9) | (c << 2) | (c & 3);
	}

	if (n < 0)
		n = ~n;

	return n & (size - 1); /* need power of 2 size */
}

static boolean rehash(Htab *ht) {
	int i,j,size;
	int newsize,newused;
	Htab *newhtab;

	if (ht == fp) {
		if (fsize > 2 * fused)
			return FALSE;
		size = fsize;
	} else {
		if (vsize > 2 * vused)
			return FALSE;
		size = vsize;
	}


	newsize = 2 * size;
	newhtab = ealloc(newsize * sizeof(Htab));
	for (i = 0; i < newsize; i++)
		newhtab[i].name = NULL;

	for (i = newused = 0; i < size; i++)
		if (ht[i].name != NULL && ht[i].name != dead) {
			newused++;
			j = hash(ht[i].name, newsize);
			while (newhtab[j].name != NULL) {
				j++;
				j &= (newsize - 1);
			}
			newhtab[j].name = ht[i].name;
			newhtab[j].p = ht[i].p;
		}

	if (ht == fp) {
		fused = newused;
		fp = newhtab;
		fsize = newsize;
	} else {
		vused = newused;
		vp = newhtab;
		vsize = newsize;
	}

	efree(ht);
	return TRUE;
}

#define varfind(s) find(s,vp,vsize)
#define fnfind(s) find(s,fp,fsize)

static int find(char *s, Htab *ht, int size) {
	int h = hash(s, size);

	while (ht[h].name != NULL && !streq(ht[h].name,s)) {
		h++;
		h &= size - 1;
	}

	return h;
}

void *lookup(char *s, Htab *ht) {
	int h;

	if (ht == fp)
		h = fnfind(s);
	else
		h = varfind(s);

	if (ht[h].name == NULL)
		return NULL;
	else
		return ht[h].p;
}

Function *get_fn_place(char *s) {
	int h = fnfind(s);

	env_dirty = TRUE;

	if (fp[h].name == NULL) {
		if (rehash(fp))
			h = fnfind(s);
		fused++;
		fp[h].name = ecpy(s);
		fp[h].p = enew(Function);
	} else
		free_fn(fp[h].p);

	return fp[h].p;
}

Variable *get_var_place(char *s, boolean stack) {
	Variable *new;
	int h = varfind(s);

	env_dirty = TRUE;

	if (vp[h].name == NULL) {
		if (rehash(vp))
			h = varfind(s);
		vused++;
		vp[h].name = ecpy(s);
		strcpy(vp[h].name, s);
		vp[h].p = enew(Variable);
		((Variable *)vp[h].p)->n = NULL;
		return vp[h].p;
	} else {
		if (stack) {
			new = enew(Variable);
			new->n = vp[h].p;
			return vp[h].p = new;
		} else {
			free_var(vp[h].p);
			vp[h].p = enew(Variable);
			((Variable *)vp[h].p)->n = NULL;
			return vp[h].p;
		}
	}
}

static void free_var(Variable *p) {
	if (p == NULL)
		return;
	efree(p->extdef);
	listfree(p->def);
	free_var(p->n);
	efree(p);
}

void delete_fn(char *s) {
	int h = fnfind(s);

	if (fp[h].name == NULL)
		return; /* not found */

	env_dirty = TRUE;

	free_fn(fp[h].p);
	efree(fp[h].p);
	efree(fp[h].name);
	if (fp[h+1].name == NULL) {
		--fused;
		fp[h].name = NULL;
	} else {
		fp[h].name = dead;
	}
}

void delete_var(char *s, boolean stack) {
	int h = varfind(s);
	Variable *v;

	if (vp[h].name == NULL)
		return; /* not found */

	env_dirty = TRUE;

	v = vp[h].p;
	efree(v->extdef);
	listfree(v->def);

	if (v->n != NULL) { /* This is the top of a stack */
		if (stack) { /* pop */
			vp[h].p = v->n;
			efree(v);
		} else { /* else just nullify */
			v->extdef = NULL;
			v->def = NULL;
		}
	} else { /* needs to be removed from the hash table */
		efree(v);
		efree(vp[h].name);
		if (vp[h+1].name == NULL) {
			--vused;
			vp[h].name = NULL;
		} else {
			vp[h].name = dead;
		}
	}
}

static void free_fn(Function *f) {
	treefree(f->def);
	efree(f->extdef);
}

void initenv(char **envp) {
	int n;

	for (n = 0; envp[n] != NULL; n++)
		;
	n++; /* one for the null terminator */

	if (n < HASHSIZE)
		envsize = 2 * HASHSIZE;
	else
		envsize = 2 * n;


	env = ealloc(envsize * sizeof (char *));

	for (; *envp != NULL; envp++)
		if (strncmp(*envp, "fn_", 3) == 0)
			fnassign_string(*envp);
		else
			if (!varassign_string(*envp)) /* add to bozo env */
				env[bozosize++] = *envp;
}

static boolean exportable(char *s) {
	int i;
	static char *notforexport[] = {
		"sighup", "sigint", "sigquit", "sigterm",
		"apid", "pid", "apid", "*", "ifs"
	};

	for (i = 0; i < (sizeof(notforexport) / sizeof(char *)); i++)
		if (streq(s,notforexport[i]))
			return FALSE;
	return TRUE;
}

char **makeenv(void) {
	int ep, i;
	char *v;

	if (!env_dirty)
		return env;

	env_dirty = FALSE;
	ep = bozosize;

	if (vsize + fsize + 1 + bozosize > envsize) {
		envsize = 2 * (bozosize + vsize + fsize + 1);
		env = erealloc(env, envsize * sizeof(char *));
	}

	for (i = 0; i < vsize; i++) {
		if (vp[i].name == NULL || !exportable(vp[i].name))
			continue;
		v = varlookup_string(vp[i].name);
		if (v != NULL)
			env[ep++] = v;
	}
	for (i = 0; i < fsize; i++) {
		if (fp[i].name == NULL || !exportable(fp[i].name))
			continue;
		env[ep++] = fnlookup_string(fp[i].name);
	}
	env[ep] = NULL;
	return env;
}

void whatare_all_vars(void) {
	int i;
	List *s,*t;

	for (i = 0; i < vsize; i++)
		if (vp[i].name != NULL && (s = varlookup(vp[i].name))) {
			if (s->n == NULL)
				fprint(1,"%s=", vp[i].name);
			else
				fprint(1,"%s=(", vp[i].name);
			for (t = s; t->n != NULL; t = t->n)
				fprint(1,"%s ",strprint(t->w, FALSE, TRUE));
			if (s->n == NULL)
				fprint(1,"%s\n",strprint(t->w, FALSE, TRUE));
			else
				fprint(1,"%s)\n", strprint(t->w, FALSE, TRUE));
		}
	for (i = 0; i < fsize; i++)
		if (fp[i].name != NULL)
			fprint(1,"fn %s {%s}\n",fp[i].name,ptree(fnlookup(fp[i].name)));
}
