/* input.c: i/o routines for files and pseudo-files (strings) */

#include <errno.h>
#include "rc.h"
#include "input.h"
#include "utils.h"
#include "walk.h"
#include "hash.h"
#include "jmp.h"
#include "lex.h"
#include "open.h"
#include "nalloc.h"

/*
   warning, changes have been made to (fd|string)(gchar|ugchar) so that
   you cannot unget EOF more than once. lex.c never does this, so I'm
   safe, but if you unget EOF more than once, expect to be able to read
   it back only once. The reason is that EOF is an int, whereas the buffers
   are character buffers.
*/

typedef struct Input {
	enum inputtype t;
	char *ibuf;
	int fd;
	int index;
	int read;
	int lineno;
} Input;

#define BUFSIZE 256
#define STACKSIZE 256

static int dead(void);
static int fdgchar(void);
static int stringgchar(void);
static void history(void);
static void popinput(void);
static void ugdead(int);

static int l = -666;
static char *inbuf;
static int istacksize = 0;
static boolean eofread = FALSE;
static Input *istack;
static int i = 2;
static int n = 0;

int (*gchar)(void);
void (*ugchar)(int);
jmp_buf *jb;
Node *parsetree; /* we're not using yyval because bison declares it as a local */

static int dead() {
	return EOF;
}

static void ugdead(int c) {
	return;
}

static int stringgchar() {
	if (eofread) {
		eofread = FALSE;
		return EOF;
	}

	if (inbuf[i] == '\0') {
		popinput();
		return l = EOF;
	} else
		return l = inbuf[i++];
}

static int fdgchar() {
	if (eofread) {
		eofread = FALSE;
		return EOF;
	}
	if (i == n + 2) {
		inbuf[0] = inbuf[n-2]; /* safeguard ungetchar up to two characters */
		inbuf[1] = inbuf[n-1];
	again:	n = read(istack->fd, inbuf + 2, BUFSIZE);
		if (n == 0) {
			popinput();
			return l = EOF;
		} else if (n == -1) {
			if (errno == EINTR) /* Suppose it was interrupted by a signal */
				goto again;
			uerror("gchar");
			rc_exit(1);
		} else {
			i = 2;
			if (dashvee)
				writeall(2,inbuf + 2,n);
			history();
		}
	}
	return l = inbuf[i++];
}

static void realugchar(int c) {
	if (c == EOF)
		eofread = TRUE;
	else
		inbuf[--i] = c;
}

void initinput() {
	istack = ealloc(STACKSIZE * sizeof (Input));
	istack->t = FD;
	istack->fd = -1;
	ugchar = realugchar;
}

void pushinput(enum inputtype t, void *p) {
	int count;
	char **a;

	istack->index = i;
	istack->read = n;
	istack->ibuf = inbuf;
	istack->lineno = lineno;
	istack++, istacksize++;

	if (istacksize >= STACKSIZE)
		rc_error("input stack overflow");

	istack->t = t;
	if (t == FD) {
		istack->fd = *(int *) p;
		gchar = fdgchar;
		inbuf = ealloc(BUFSIZE + 2);
	} else {
		for (count = 0, a = p; *a != NULL; a++)
			count += strlen(*a) + 1;
		count += 3;

		inbuf = ealloc(count);
		sprint(inbuf + 2, "%a", (char **)p);
		gchar = stringgchar;
	}

	ugchar = realugchar;
	i = 2;
	n = 0;
	lineno = 1;
}

static void popinput() {
	if (istack->t == FD)
		close(istack->fd);
	efree(inbuf);

	--istack, --istacksize;

	if (istack->t == STRING)
		gchar = stringgchar;
	else
		gchar = fdgchar;

	if (istack->fd == -1) { /* top of input stack */
		gchar = dead;
		ugchar = ugdead;
	}

	inbuf = istack->ibuf;
	i = istack->index;
	n = istack->read;
	lineno = istack->lineno;
}

void flushu() {
	int c;

	if (l == '\n')  {
		return;
	} else
		while ((c = gchar()) != '\n' && c != EOF)
			;
}

void doit() {
	List *s;
	int last;
	jmp_buf doitbuf;

	if (atthetop) {
		jb = (jmp_buf *) doitbuf;
		if (setjmp(doitbuf) && !atthetop)
			popallinput();
	}

	while (1) {
		if (interactive) {
			s = varlookup("prompt");
			if (s != NULL) {
				fprint(2,"%s",s->w);
				if (s->n != NULL)
					PS2 = s->n->w;
			}
		}
		inityy();
		if (yyparse() < 0) {
			if (!interactive && atthetop) {
				exit(1);
			} else {
				popallinput(); /* return on error in noninteractive shell */
				longjmp(jb, 1);
			}
		}
		last = l; /* l can be clobbered during a walk() */
		if (parsetree != NULL)
			walk(parsetree, PARENT);
		if (last == EOF)
			return;
		nfree();
	}
}

Node *parseline(char *extdef) {
	char *in[2];

	in[0] = extdef;
	in[1] = NULL;
	pushinput(STRING, in);
	inityy();
	yyparse();
	return parsetree;
}

static void history() {
	static char *histstr;
	static int histfd;
	List *histlist;
	int a;

	if (!interactive)
		return;

	if ((histlist = varlookup("history")) == NULL) {
		if (histstr != NULL) {
			efree(histstr);
			close(histfd);
			histstr = NULL;
		}
		return;
	}

	if (histstr == NULL || !streq(histstr, histlist->w)) { /* open new file */
		if (histstr != NULL) {
			efree(histstr);
			close(histfd);
		}
		histstr = ecpy(histlist->w);
		histfd = rc_open(histstr, APPEND);
		if (histfd < 0) {
			uerror(histstr);
			efree(histstr);
			histstr = NULL;
		}
	}

	/*
	   small unix hack: since read() reads only up to a newline from a terminal, then
	   presumably this write() will write at most only one input line at a time.
	*/

	for (a = 2; a < n + 2; a++) {
		if (inbuf[a] != '#' && inbuf[a] != ' ' && inbuf[a] != '\t' && inbuf[a] != '\n')
			break;
		if (inbuf[a] == '#' || inbuf[a] == '\n')
			return;
	}
	writeall(histfd, inbuf + 2, n);
}

void popallinput() {
	if (!atthetop)
		while (istacksize != 1)
			popinput();
	atthetop = 1;
	interactive = toplevinteractive;
}
