/* status.c: functions for printing fancy status messages in rc */

#include "rc.h"
#include "utils.h"
#include "status.h"
#include "nalloc.h"

/* status == the wait() value of the last command in the pipeline, or the last command */

static int status = 0;

static int pipestats[512];
static int pipelength = 0;

static char *verbstat[] = {
	"", "", "", "quit", "illegal instruction", "", "", "", "", "",
	"bus error",
	"segmentation violation",
	"bad argument to system call",
	"", /* broken pipe; removed by request */
	"alarm clock",
	"terminated",
	"urgent condition on i/o channel",
	"stop signal not from tty",
	"stopped",
	"continue",
	"child stop or exit",
	"background tty read",
	"background tty write",
	"i/o possible",
	"exceeded cpu limit",
	"exceeded file size limit",
	"virtual time alarm",
	"profiling time alarm",
	"window changed",
	"resource lost",
	"sigusr1",
	"sigusr2"
};

/*
   According to td, status is true if and only if every pipe-member has an exit status
   of zero.
*/
int istrue(void) {
	int i;

	if (pipelength == 0)
		return status == 0;

	for (i = 0; i < pipelength; i++)
		if (pipestats[i] != 0)
			return FALSE;
	return TRUE;
}

int getstatus(void) {
	return (status & 0xff ? status & 0x7f : status >> 8);
}

void settrue(void) {
	setstatus(0);
}

void setfalse(void) {
	setstatus(0x100); /* exit status 1 shifted left 8 bits */
}

void setpipestatus(int stats[], int num) {
	int i;
	int j = 0;

	for (i = 0; i < num; i++) {
		statprint(stats[i]);
		pipestats[i] = stats[i];
		j |= stats[i] & 0x80; /* did any dump core? */
	}

	pipelength = num;
	status = stats[0];
}

void setstatus(int i) {
	pipelength = 0;
	statprint(i);
	status = i;
}

void statprint(int i) {
	int j;

	if (i & 0xff) {
		j = i & 0x7f;

		if (j < sizeof(verbstat) / sizeof(char *))
			fprint(2,"%s",verbstat[j]);
		if (i & 0x80) {
			if (*verbstat[j] == '\0') /* Is the status a null string? */
				fprint(2,"core dumped");
			else
				fprint(2,"--core dumped");
		}
		if (j < sizeof(verbstat)/sizeof(char *) && *verbstat[j])
			fprint(2,"\n");
	}

	if (i != 0 && dashee)
		rc_exit(getstatus());
}

List *sgetstatus(void) {
	List *r = nalloc(sizeof(List));
	List *q;
	int i = pipelength - 1;
	int s = status;

	r->w = nalloc(16);
	r->m = NULL;
	r->n = NULL;

	do {
		if (s & 0x7f) {
			switch (s & 0x7f) { /* this is good for SunOS; mileage may vary */
			case 1: strcpy(r->w, "sighup"); break;
			case 2: strcpy(r->w, "sigint"); break;
			case 3: strcpy(r->w, "sigquit"); break;
			case 4: strcpy(r->w, "sigill"); break;
			case 10: strcpy(r->w, "sigbus"); break;
			case 11: strcpy(r->w, "sigsegv"); break;
			case 12: strcpy(r->w, "sigsys"); break;
			case 13: strcpy(r->w, "sigpipe"); break;
			case 14: strcpy(r->w, "sigalrm"); break;
			case 15: strcpy(r->w, "sigterm"); break;
			case 16: strcpy(r->w, "sigurg"); break;
			case 17: strcpy(r->w, "sigstop"); break;
			case 18: strcpy(r->w, "sigtstp"); break;
			case 19: strcpy(r->w, "sigcont"); break;
			case 20: strcpy(r->w, "sigchld"); break;
			case 21: strcpy(r->w, "sigttin"); break;
			case 22: strcpy(r->w, "sigttou"); break;
			case 23: strcpy(r->w, "sigio"); break;
			case 24: strcpy(r->w, "sigxcpu"); break;
			case 25: strcpy(r->w, "sigxfsz"); break;
			case 26: strcpy(r->w, "sigvtalrm"); break;
			case 27: strcpy(r->w, "sigprof"); break;
			case 28: strcpy(r->w, "sigwinch"); break;
			case 29: strcpy(r->w, "siglost"); break;
			case 30: strcpy(r->w, "sigusr1"); break;
			case 31: strcpy(r->w, "sigusr2"); break;
			default: sprint(r->w,"-%d",s & 0x7f); /* unknown signals are negated */
			}
		} else {
			sprint(r->w,"%d",s >> 8);
		}

		if (s & 0x80)
			strcat(r->w, "+core");

		if (i > 0) {
			s = pipestats[pipelength - i];
			q = nnew(List);
			q->w = nalloc(16);
			q->m = NULL;
			q->n = r;
			r = q;
		}
	} while (i-- > 0);

	return r;
}
