/* var.c: provide "public" functions for adding and removing variables from the symbol table */
#include "rc.h"
#include "utils.h"
#include "hash.h"
#include "list.h"
#include "footobar.h"
#include "nalloc.h"
#include "status.h"
#include "glom.h"

static void colonassign(char *, List *, boolean);
static void listassign(char *, List *, boolean);
static int hasalias(char *);

static char *aliases[] = {
	"home", "HOME", "path", "PATH", "cdpath", "CDPATH"
};

void varassign(char *name, List *def, boolean stack) {
	Variable *new;
	List *newdef = listcpy(def); /* important to do the listcpy first; get_var_place()
					frees old values */
	new = get_var_place(name, stack);
	new->def = newdef;
	new->extdef = NULL;
}

boolean varassign_string(char *extdef) {
	char *name = get_name(extdef);
	Variable *new;
	int i;
	static boolean aliasset[6] = {
		FALSE, FALSE, FALSE, FALSE, FALSE, FALSE
	};

	if (name == NULL)
		return FALSE; /* add it to bozo env */

	i = hasalias(name);
	if (i >= 0) {
		aliasset[i] = TRUE;
		if ((i & 1 == 0) && aliasset[i^1])
			return TRUE; /* don't alias variables that are already set in upper case */
	}
	new = get_var_place(name, FALSE);
	new->def = NULL;
	new->extdef = ealloc(strlen(extdef) + 1);
	strcpy(new->extdef, extdef);
	if (hasalias(name) != -1)
		alias(name, varlookup(name), FALSE);
	return TRUE;
}

List *varlookup(char *name) {
	Variable *look;
	List *ret, *l;
	List sub;
	char *t;
	boolean num;

	if (streq(name, "status"))
		return sgetstatus();

	for (num = TRUE, t = name; num && *t; t++)
		if (*t < '0' || *t > '9')
			num = FALSE;

	if (num && *name != '\0') {
		if (streq(name, "0")) {
			l = nnew(List);
			l->w = dollarzero;
			l->m = NULL;
			l->n = NULL;
			return l;
		}
		sub.w = name;
		sub.n = NULL;
		return varsub(varlookup("*"), &sub);
	}

	look = lookup_var(name);

	if (look == NULL)
		return NULL; /* not found */
	if (look->def != NULL)
		return look->def;
	if (look->extdef == NULL)
		return NULL; /* variable was set to null, e.g., a=() echo foo */

	ret = parse_var(name, look->extdef);

	if (ret == NULL) {
		look->extdef = NULL;
		return NULL;
	}
	return look->def = ret;
}

char *varlookup_string(char *name) {
	Variable *look;

	look = lookup_var(name);

	if (look == NULL)
		return NULL;
	if (look->extdef != NULL)
		return look->extdef;
	if (look->def == NULL)
		return NULL;
	return look->extdef = list2str(name, look->def);
}

void varrm(char *name, boolean stack) {
	int i = hasalias(name);

	delete_var(name, stack);
	if (i != -1)
		delete_var(aliases[i^1], stack);
}

void arrayassign(char *name, char **a, boolean stack) {
	List *s, *var;

	if (*a == NULL) {
		varassign(name, NULL, stack);
		return;
	}

	s = var = nnew(List);

	while (1) {
		s->w = *a++;
		if (*a == NULL) {
			s->n = NULL;
			break;
		} else {
			s->n = nnew(List);
			s = s->n;
		}
	}
	varassign(name, var, stack);
}

static void colonassign(char *name, List *def, boolean stack) {
	char *colondef;
	List dud;
	int deflen;
	List *r;

	if (def == NULL) {
		varassign(name, NULL, stack);
		return;
	}

	deflen = listlen(def) + 1; /* one for the null terminator */

	colondef = nalloc(deflen);
	strcpy(colondef, def->w);

	for (r = def->n; r != NULL; r = r->n) {
		strcat(colondef, ":");
		strcat(colondef, r->w);
	}

	dud.w = colondef;
	dud.n = NULL;
	varassign(name, &dud, stack);
}

static void listassign(char *name, List *def, boolean stack) {
	List *val, *r;
	char *v, *w;

	if (def == NULL) {
		varassign(name, NULL, stack);
		return;
	}

	v = def->w;

	r = val = enew(List);

	while((w = strchr(v,':')) != NULL) {
		*w = '\0';
		r->w = ecpy(v);
		*w = ':';
		v = w + 1;
		r->n = enew(List);
		r = r->n;
	}
	r->w = ecpy(v);
	r->n = NULL;

	varassign(name, val, stack);
}

static int hasalias(char *name) {
	int i;

	for (i = 0; i < 6; i++)
		if (streq(name, aliases[i]))
			return i;
	return -1;
}

void alias(char *name, List *s, boolean stack) {
	int i = hasalias(name);
	static void (*vectors[])(char *, List *, boolean) = {
		varassign, varassign, colonassign, listassign, colonassign, listassign
	};

	if (i != -1)
		vectors[i](aliases[i^1], s, stack); /* xor hack to reverse case of alias entry */
}
