/* walk.c: walks the parse tree. */

#include <signal.h>
#include "rc.h"
#include "utils.h"
#include "status.h"
#include "exec.h"
#include "walk.h"
#include "glom.h"
#include "hash.h"
#include "glob.h"
#include "lex.h"
#include "open.h"
#include "jmp.h"

jmp_buf *breakbuf = NULL;

static boolean iscase(Node *);
static boolean isallpre(Node *);
static boolean dofork(void);

void walk(Node *n, enum cp chpt) {
	int pid, wpid;
	int sp;
	int i,j,k;
	int pids[512], stats[512];
	int fd_prev = 1;
	int fd_out = 1;
	int p[2];
	Node *r;
	List *v, *l;
	char apid[8];
	void (*handler)(int);
	jmp_buf jbreak;

	if (n == NULL) {
		settrue();
		return;
	}

	switch (n->type) {
	case ARGS:	/* simple command */
	case BACKQ:
	case CONCAT:
	case rCOUNT:
	case rFLAT:
	case LAPPEND:
	case rREDIR:
	case VAR:
	case VARSUB:
	case rWORD:
		exec(glob(glom(n)), chpt);
		return;
	case BODY:
		walk(n->u[0].p, PARENT);
		walk(n->u[1].p, PARENT);
		return;
	case NOWAIT:
		pid = fork();
		if (pid < 0) {
			uerror("fork");
			rc_error(NULL);
		}
		if (pid == 0) {
			setsigdefaults();
			signal(SIGINT, SIG_IGN);
#ifndef NOJOB
			signal(SIGTTOU, SIG_IGN);
			signal(SIGTTIN, SIG_IGN);
			signal(SIGTSTP, SIG_IGN);
			setpgrp(0, getpid()); /* hack to get job to ignore signals from tty on BSD systems */
#endif
			if (isatty(0))
				dup2(rc_open("/dev/null", FROM),0);
			walk(n->u[0].p, CHILD);
			rc_exit(getstatus());
		} else {
			if (interactive)
				fprint(2,"%d\n",pid);
			sprint(apid,"%d",pid);
			varassign("apid",word(apid, NULL),FALSE);
			redirq = NULL; /* kill pre-redir queue */
			fifoq = NULL;
			return;
		}
	case rANDAND:
		if (walk(n->u[0].p, PARENT), istrue())
			walk(n->u[1].p, PARENT);
		return;
	case rOROR:
		if (!(walk(n->u[0].p, PARENT), istrue()))
			walk(n->u[1].p, PARENT);
		return;
	case rBANG:
		if (walk(n->u[0].p, PARENT), istrue())
			setfalse();
		else
			settrue();
		return;
	case rIF:
		if (walk(n->u[0].p, PARENT), istrue()) {
			if (n->u[1].p != NULL && n->u[1].p->type != rELSE)
				walk(n->u[1].p, PARENT);
			else
				walk(n->u[1].p->u[0].p, PARENT);
		} else {
			if (n->u[1].p != NULL && n->u[1].p->type == rELSE)
				walk(n->u[1].p->u[1].p, PARENT);
		}
		return;
	case rWHILE: {
		jmp_buf *oldbuf;
		walk(n->u[0].p, PARENT); /* prevent spurious breaks inside test */
		if (!istrue())
			return;
		oldbuf = breakbuf;
		breakbuf = (jmp_buf *) jbreak;
		if (setjmp(jbreak) == 0) {
			do
				walk(n->u[1].p, PARENT);
			while (walk(n->u[0].p, PARENT), istrue());
		}
		breakbuf = oldbuf;
		return;
		}
	case FORIN: {
		List *a, *b;
		jmp_buf *oldbuf;
		a = glob(glom(n->u[1].p));
		b = glom(n->u[0].p);
 		oldbuf = breakbuf;
		breakbuf = (jmp_buf *) jbreak;
		if (setjmp(jbreak) == 0) {
			for (; a != NULL; a = a->n) {
				assign(b, word(a->w, NULL), FALSE);
				walk(n->u[2].p, PARENT);
			}
		}
		breakbuf = oldbuf;
		return;
		}
	case rSUBSHELL:
		if (dofork()) {
			setsigdefaults();
			walk(n->u[0].p, PARENT);
			rc_exit(getstatus());
		}
		return;
	case ASSIGN:
		if (n->u[0].p == NULL)
			rc_error("null variable name");
		settrue();
		assign(glom(n->u[0].p), glob(glom(n->u[1].p)), FALSE);
		return;
	case rPIPE:
		for (r = n, i = 0; r->type == rPIPE; r = r->u[2].p, i++) {
			if (i > 500) {
				fprint(2,"pipe too long\n");
				rc_exit(1);
			}
			if (pipe(p) < 0) {
				uerror("pipe");
				rc_error(NULL);
			}
			pid = fork();
			if (pid < 0) {
				uerror("fork");
				rc_error(NULL);
			}
			if (pid == 0) {
				setsigdefaults();
				redirq = NULL; /* clear preredir queue */
				fifoq = NULL;
				dup2(p[0],r->u[1].i);
				if (fd_prev != 1) {
					dup2(fd_prev, fd_out);
					close(fd_prev);
				}
				close(p[0]);
				close(p[1]);
				walk(r->u[3].p, CHILD);
				rc_exit(getstatus());
			} else {
				if (fd_prev != 1)
					close(fd_prev); /* parent must close all pipe fd's */
				pids[i] = pid;
				fd_prev = p[1];
				fd_out = r->u[0].i;
				close(p[0]);
			}
		}

		pid = fork();
		if (pid < 0) {
			uerror("fork");
			rc_error(NULL);
		}
		if (pid == 0) {
			setsigdefaults();
			dup2(fd_prev, fd_out);
			close(fd_prev);
			walk(r, CHILD);
			rc_exit(getstatus());
		} else {
			redirq = NULL; /* clear preredir queue */
			fifoq = NULL;
			close(fd_prev);
			pids[i++] = pid;
			handler = signal(SIGINT, SIG_IGN);
			if (handler == SIG_DFL) /* are we non-interactive? if so, don't ignore ^C */
				signal(SIGINT, SIG_DFL);

			/* collect statuses */
			for (k = i; k != 0;) {
				wpid = wait(&sp);
				if (wpid < 0)
					uerror("wait");

				for (j = 0; j < i; j++)
					if (wpid == pids[j]) {
						stats[j] = sp;
						pids[j] = 0;
						--k;
					}
			}
			setpipestatus(stats, i);
			if (handler != SIG_DFL)
				signal(SIGINT, handler);
		}
		return;
	case NEWFN:
		l = glom(n->u[0].p);
		if (l == NULL)
			rc_error("bad function name");
		while (l != NULL) {
			fnassign(l->w, n->u[1].p);
			l = l->n;
		}
		settrue();
		return;
	case RMFN:
		l = glom(n->u[0].p);
		while (l != NULL) {
			fnrm(l->w);
			l = l->n;
		}
		settrue();
		return;
	case rDUP:
		return; /* Null command */
	case MATCH:
		if (lmatch(glob(glom(n->u[0].p)), glom(n->u[1].p)))
			settrue();
		else
			setfalse();
		return;
	case rSWITCH:
		v = glom(n->u[0].p);
		while (1) {
			do {
				n = n->u[1].p;
				if (n == NULL || n->type != BODY)
					return;
			} while (!iscase(n->u[0].p));
			if (lmatch(v, glom(n->u[0].p)->n)) {
				n = n->u[1].p;
				while(n != NULL && !iscase(n->u[0].p)) {
					if (n->type == BODY) {
						walk(n->u[0].p, PARENT);
						n = n->u[1].p;
					} else {
						walk(n, PARENT);
						break; /* special case at the end of BODY subtree */
					}
				}
				return;
			}
		}
	case PRE:
		if (n->u[0].p->type == rREDIR) {
			qredir(n->u[0].p);
			walk(n->u[1].p, PARENT);
			return;
		} else if (n->u[0].p->type == ASSIGN) {
			if (isallpre(n->u[1].p)) {
				walk(n->u[0].p, PARENT);
				walk(n->u[1].p, PARENT);
				return;
			} else {
				v = glom(n->u[0].p->u[0].p);
				assign(v, glob(glom(n->u[0].p->u[1].p)), TRUE);
				walk(n->u[1].p, PARENT);
				varrm(v->w, TRUE);
			}
		} else
			rc_error("walk: node other than assign or redir in PRE. help!");
		return;
	case BRACE:
		if (dofork()) {
			setsigdefaults();
			walk(n->u[1].p, PARENT); /* Do redirections */
			redirq = NULL;   /* Reset redirection queue */
			walk(n->u[0].p, PARENT); /* Do commands */
			rc_exit(getstatus());
		}
		return;
	case EPILOG:
		qredir(n->u[0].p);
		if (n->u[1].p != NULL)
			walk(n->u[1].p, PARENT); /* Do more redirections. */
		else
			doredirs();	/* Okay, we hit the bottom. */
		return;
	case NMPIPE:
		rc_error("named pipes cannot be executed as commands");
	default:
		rc_error("walk: unknown node; this can't happen");
	}
}

static boolean iscase(Node *n) {
	if (n == NULL)
		return FALSE;

	switch (n->type) {
	default:
		return FALSE;
	case ARGS:
	case LAPPEND:
		return iscase(n->u[0].p);
	case rWORD:
		return streq(n->u[0].s, "case");
	}
}

static boolean isallpre(Node *n) {
	if (n == NULL)
		return TRUE;

	switch (n->type) {
	default:
		return FALSE;
	case rREDIR:
	case ASSIGN:
	case rDUP:
		return TRUE;
	case PRE:
		return isallpre(n->u[1].p);
	}
}

static boolean dofork() {
	int pid = fork();
	int sp;

	if (pid < 0) {
		uerror("fork");
		rc_error(NULL);
	}

	if (pid > 0) {
		redirq = NULL; /* clear out the pre-redirection queue in the parent */
		fifoq = NULL;
		while (pid != wait(&sp))
			if (pid < 0)
				uerror("wait");
		setstatus(sp);
		return FALSE;
	}

	return TRUE;
}
