/***[file.c]******************************************************[TAB=4]****\
*                                                                            *
* PHP/FI                                                                     *
*                                                                            *
* Copyright 1995,1996 Rasmus Lerdorf                                         *
*                                                                            *
*  This program is free software; you can redistribute it and/or modify      *
*  it under the terms of the GNU General Public License as published by      *
*  the Free Software Foundation; either version 2 of the License, or         *
*  (at your option) any later version.                                       *
*                                                                            *
*  This program is distributed in the hope that it will be useful,           *
*  but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*  GNU General Public License for more details.                              *
*                                                                            *
*  You should have received a copy of the GNU General Public License         *
*  along with this program; if not, write to the Free Software               *
*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
*                                                                            *
\****************************************************************************/
#include <php.h>
#include <stdlib.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <string.h>
#include <pwd.h>
#include <grp.h>
#include <errno.h>
#include <parse.h>
#if NSAPI
#include "base/util.h"
#include "frame/protocol.h"
#include "base/buffer.h"
#include "frame/log.h"
#endif

static char *CurrentFilename=NULL;
static char *CurrentStatFile=NULL;
static char *CurrentPI=NULL;
static long CurrentFileSize=0L;
static struct stat gsb;

static FpStack *fp_top = NULL;

#if NSAPI
extern Session *gsn;
extern Request *grq;
extern pblock  *gpb;
#endif

void StripLastSlash(char *str) {
	char *s;

	s = strrchr(str,'/');
	if(s) {
		*s='\0';
	}
}
#if NSAPI
SYS_FILE OpenFile(char *filename, int top) {
#else
int OpenFile(char *filename, int top, long *file_size) {
#endif
	char *fn, *pi, *sn, *fn2=NULL, *fn3=NULL;
	char *s=NULL;
	int ret=-1, free_fn=0;
	int no_httpd=0;
#if NSAPI
	SYS_FILE fd;
	struct stat *fi;
#else
	int fd;
#endif

#if DEBUG	
	Debug("OpenFile called with %s\n",filename?filename:"null");
#endif

	if(!filename) {
#if NSAPI
		pi = pblock_findval("path-info", grq->vars);
		fn = filename;
#else
		pi = getenv("PATH_INFO");
#endif
#ifndef PHP_ROOT_DIR
#ifndef NSAPI
		fn = getenv("PATH_TRANSLATED");
#endif
		if(!fn || (fn && !*fn)) { Info(); exit(0); }
#else
		{
		char temp[1024];
		sprintf(temp,"%s%s",PHP_ROOT_DIR,pi);
		fn = temp;
		}
#endif
	} else {
		fn = filename;
		pi = filename;
#ifndef NSAPI
		no_httpd=1;
#endif
	}
	/*
	 * To prevent someone from uploading a file and then running it through php
	 */
	if(!strncmp(fn,"phpfi",5)) {
		return(-1);
	}

#if NSAPI
	ret=1;
	fn = filename;
#else
	fn = (char *)estrdup(FixFilename(fn,top,&ret));
	free_fn=1;
	*file_size = (long)gsb.st_size;
	CurrentFileSize = (long)gsb.st_size;
#endif
#if DEBUG
	Debug("OpenFile: fn = [%s]\n",fn);
#endif
	fd=ret;
#ifndef NSAPI
	if(ret==-1) {
#if DEBUG
		Debug("Unable to find file [%s] - %s",fn,strerror(errno));
#endif
#ifdef VIRTUAL_PATH
		sn  = VIRTUAL_PATH;
#else
		sn  = getenv("SCRIPT_NAME");
#endif
		if(free_fn) efree(fn);
		if(sn) {
			fn = emalloc(sizeof(char) * (strlen(pi)+strlen(sn)+1));
			fn = emalloc(sizeof(char) * (strlen(pi)));
			strcpy(fn,sn);
			s = strrchr(fn,'/');
			if(s) *s='\0';
			strcat(fn,pi);
			fn2 = (char *) estrdup(FixFilename(fn,1,&ret));
#if DEBUG
			Debug("OpenFile trying to open: %s\n",fn2);
#endif
			if(ret==-1) {
#if DEBUG
				Debug("Unable to find file [%s] - %s",fn2,strerror(errno));
#endif
				efree(fn2);
				fn2 = NULL;
			}
		} else {
			fd=-1;
			ret=-1;
		}
	}
#endif
	if(ret!=-1) {
#ifndef NSAPI
		if((gsb.st_mode&S_IFMT) == S_IFDIR) {
			fn3 = emalloc(sizeof(char) * (strlen(fn) + 20));
			strcpy(fn3,fn);
			strcat(fn3,"/index.html");
			fd = stat(fn3,&gsb);
			if(fd==-1) {
				strcpy(fn3,fn);
				strcat(fn3,"/Overview.html");
				fd = stat(fn3,&gsb);
				if(fd==-1) {
					if(fn2) {
						strcpy(fn3,fn2);
						strcat(fn3,"/index.html");
						fd = stat(fn3,&gsb);
						if(fd==-1) {
							strcpy(fn3,fn2);
							strcat(fn3,"/Overview.html");
							fd = stat(fn3,&gsb);
							if(fd==-1) {
								*fn3='\0';
							}
						}
					}
				}
			}
		}
#if DEBUG
		Debug("OpenFile2: fn = [%s]\n",fn);
#endif
		if(fn3 && *fn3) fd = open(fn3,O_RDONLY);
		else if(!fn2) fd = open(fn,O_RDONLY);
		else fd = open(fn2,O_RDONLY);
		SetStatInfo(&gsb);
#else
		if((!(fi = request_stat_path(fn, grq))) || ((fd = system_fopenRO(fn)) == IO_ERROR)) {
			protocol_status(gsn,grq,(file_notfound()?PROTOCOL_NOT_FOUND:PROTOCOL_FORBIDDEN),NULL);
			log_error(LOG_WARN,"php",gsn,grq,"error opening %s (%s)",fn, system_errmsg(0));
			return REQ_ABORTED;
		}
		SetStatInfo(fi);
#endif
		
#if ACCESS_CONTROL
#if NSAPI
		if(!no_httpd) CheckAccess(pi,fi->st_uid);
#else
		if(!no_httpd) CheckAccess(pi,gsb.st_uid);
#endif
#endif
#if DEBUG
		Debug("OpenFile3: fn = [%s]\n",fn);
#endif
#if 0
#if LOGGING
		if(!no_httpd && getlogging()) Log(pi);
#if DEBUG
		Debug("Back from Log()\n");
#endif
#endif
#endif
		if(CurrentPI) {
			efree(CurrentPI);
			CurrentPI=NULL;
		}
		CurrentPI = estrdup(pi);	
		if(fd==-1) return(-1);
		if(CurrentFilename) {
			efree(CurrentFilename);
			CurrentFilename=NULL;
		}
		if(filename) CurrentFilename = (char *)estrdup(filename);
		else {
			if(fn3 && *fn3) {
				CurrentFilename = (char *)estrdup(fn3);
				efree(fn3);
				if(fn2) efree(fn2);
			} else if(!fn2) CurrentFilename = (char *)estrdup(fn);
			else {
				CurrentFilename = (char *)estrdup(fn2);
				efree(fn2);
			}
		}
	}
#if DEBUG
	if(fd>0) Debug("1. CurrentFilename is [%s]\n",CurrentFilename);
#endif
	if(free_fn) efree(fn);
	return(fd);
}

/*
 * if cd = 1, then current directory will be changed to
 * directory portion of the passed path and the PATH_DIR
 * environment variable will be set
 */
char *FixFilename(char *filename, int cd, int *ret) {
	static char temp[1024];
	char path[1024];
	char fn[128], user[128], *s;
	struct passwd *pw=NULL;
	int st=0;
	char o='\0';
	char *env;
	int l=0;

	s = strrchr(filename,'/');
	if(s) {
		strcpy(fn,s+1);
		o=*s;
		*s='\0';
		strcpy(path,filename);
		*s=o;
	} else {
#ifdef PHP_ROOT_DIR
		strcpy(path,PHP_ROOT_DIR);	
#else
		path[0] = '\0';
#endif
		strcpy(fn,filename);
	}
	if(fn && *fn=='~') {
		strcpy(path,fn);
		fn[0]='\0';
	}
	if(*path) {
		if(*path=='~') {
			s = strchr(path,'/');
			if(s) {
				o=*s;
				*s='\0';
			}
			strcpy(user,path+1);
			if(s) {
				*s=o;		
				strcpy(temp,s);
			} else temp[0]='\0';
			if(*user) {
				pw = getpwnam(user);	
				if(pw) sprintf(path,"%s/%s%s",pw->pw_dir,PHP_PUB_DIRNAME,temp);
			}
		} else if(*path=='/' && *(path+1)=='~') {
			s = strchr(path+1,'/');
			if(s) {
				o=*s;
				*s='\0';
			}
			strcpy(user,path+2);
			if(s) {
				*s=o;		
				strcpy(temp,s);
			} else temp[0]='\0';
			if(*user) {
				pw = getpwnam(user);	
				if(pw) sprintf(path,"%s/%s%s",pw->pw_dir,PHP_PUB_DIRNAME,temp);
			}
		}
		temp[0]='\0';
		if(cd) {
			if(chdir(path)<0) {
#if DEBUG
				Debug("%d [%s]",errno,strerror(errno));
#endif
			}
			env = emalloc(sizeof(char) * (strlen(path)+16));
			sprintf(env,"PATH_DIR=%s",path);
			putenv(env);
		}
		if(*fn) {
			sprintf(temp,"%s/%s",path,fn);
			st = stat(temp,&gsb);
			if((st!=-1) && (gsb.st_mode&S_IFMT)==S_IFDIR) {
				env = emalloc(sizeof(char) * (strlen(temp)+16));
				sprintf(env,"PATH_DIR=%s",temp);
				putenv(env);
				sprintf(temp,"%s/%s/index.html",path,fn);
				st = stat(temp,&gsb);
				if(st==-1) {
					sprintf(temp,"%s/%s/Overview.html",path,fn);
					st = stat(temp,&gsb);
				}
				sprintf(path,"%s/%s",path,fn);
			} else if(st==-1) {
				l = strlen(temp);
				if(strlen(fn)>4) {
					if(!strcasecmp(&temp[l-4],"html")) {
						temp[l-1]='\0';	 /* silly .html -> .htm hack */
						st = stat(temp,&gsb);
					}
				}
			}
		} else {
			st = stat(path,&gsb);
			if((st!=-1) && (gsb.st_mode&S_IFMT)==S_IFDIR) {
				sprintf(temp,"%s/index.html",path);
				st = stat(temp,&gsb);
				if(st==-1) {
					sprintf(temp,"%s/Overview.html",path);
					st = stat(temp,&gsb);
				}
			} else strcpy(temp,path);
		}
	} else {
		st = stat(fn,&gsb);
		if((st!=-1) && (gsb.st_mode&S_IFMT)==S_IFDIR) {
			sprintf(temp,"%s/index.html",fn);
			st = stat(temp,&gsb);
			if(st==-1) {
				sprintf(temp,"%s/Overview.html",fn);
				st = stat(temp,&gsb);
			}
		} else strcpy(temp,fn);
	}		
	*ret=st;	
	return(temp);
}

char *GetCurrentFilename(void) {
	char *r;
	
	if(!CurrentFilename) return("");
	r = strrchr(CurrentFilename,'/');
	if(r) return(r+1);
	else return(CurrentFilename);
}


void SetCurrentFilename(char *filename) {
#if DEBUG
	Debug("Setting CurrentFilename to [%s]\n",filename);
#endif
	if(CurrentFilename) efree(CurrentFilename);
	CurrentFilename = estrdup(filename);
}

long GetCurrentFileSize(void) {
	return(CurrentFileSize);
}

void SetCurrentFileSize(long file_size) {
	CurrentFileSize = file_size;
}

char *getfilename(char *path, int ext) {
	static char filename[64];
	char *s;

	s = strrchr(path,'/');
	if(s) {
		strcpy(filename,s);
	} else {
		strcpy(filename,path);
	}

	if(!ext) {
		s = strrchr(filename,'.');
		if(s) *s='\0';
	}
	return(filename);
}	

void FileFunc(int type) {
	Stack *s;
	static struct stat sb;
	char temp[64];

	s = Pop();
	if(!s) {
		switch(type) {
		case 0:
			Error("Stack error in fileperms");
			break;
		case 1:
			Error("Stack error in fileinode");
			break;
		case 2:
			Error("Stack error in filesize");
			break;
		case 3:
			Error("Stack error in fileowner");
			break;
		case 4:
			Error("Stack error in filegroup");
			break;
		case 5:
			Error("Stack error in fileatime");
			break;
		case 6:
			Error("Stack error in filemtime");
			break;
		case 7:
			Error("Stack error in filectime");
			break;
		}
		return;
	}
	if(!CurrentStatFile || (CurrentStatFile && strcmp(s->strval,CurrentStatFile))) {
		if(CurrentStatFile) efree(CurrentStatFile);
		CurrentStatFile = estrdup(s->strval);
		if(stat(CurrentStatFile,&sb)==-1) {
			Push("-1",LNUMBER);
			return;
		}
	}
	switch(type) {
	case 0: /* fileperms */
		sprintf(temp,"%ld",(long)sb.st_mode);
		Push(temp,LNUMBER);
		break;
	case 1: /* fileinode */
		sprintf(temp,"%ld",(long)sb.st_ino);
		Push(temp,LNUMBER);
		break;
	case 2: /* filesize  */
		sprintf(temp,"%ld",(long)sb.st_size);
		Push(temp,LNUMBER);
		break;
	case 3: /* fileowner */
		sprintf(temp,"%ld",(long)sb.st_uid);
		Push(temp,LNUMBER);
		break;
	case 4: /* filegroup */
		sprintf(temp,"%ld",(long)sb.st_gid);
		Push(temp,LNUMBER);
		break;
	case 5: /* fileatime */
		sprintf(temp,"%ld",(long)sb.st_atime);
		Push(temp,LNUMBER);
		break;
	case 6: /* filemtime */
		sprintf(temp,"%ld",(long)sb.st_mtime);
		Push(temp,LNUMBER);
		break;
	case 7: /* filectime */
		sprintf(temp,"%ld",(long)sb.st_ctime);
		Push(temp,LNUMBER);
		break;
	}
}

void TempNam(void) {
	Stack *s;
	char *d;
	char *t;
	char p[32];

	s = Pop();
	if(!s) {
		Error("Stack error in tmpname");
		return;
	}
/*	p = (char *) estrdup(s->strval); */
	strncpy(p,s->strval,31);

	s = Pop();
	if(!s) {
		Error("Stack error in tmpname");
		return;
	}
	d = (char *) estrdup(s->strval);
	
	t = tempnam(d,p);
	Push(t,STRING);
	if(t) efree(t);
	efree(d);
}

void Unlink(void) {
	Stack *s;
	
	s = Pop();
	if(!s) {
		Error("Stack error in unlink");
		return;
	}
	if(!s->strval || (s->strval && !*(s->strval))) {
		Error("Invalid filename in unlink");
		return;
	}
	unlink(s->strval);
}

void Rename(void) {
	Stack *s;
	char *new;
	int ret;
	char temp[4];

	s = Pop();
	if(!s) {
		Error("Stack error in rename");
		return;
	}
	if(!s->strval || (s->strval && !*(s->strval))) {
		Error("Invalid filename in unlink");
		return;
	}
	new = (char *) estrdup(s->strval);
	s = Pop();
	if(!s) {
		Error("Stack error in rename");
		return;
	}
	if(!s->strval || (s->strval && !*(s->strval))) {
		Error("Invalid filename in unlink");
		return;
	}
	ret = rename(s->strval, new);	
	if(ret==-1) {
		Error("%d [%s]",errno, strerror(errno));
	}
	sprintf(temp,"%d",ret);
	Push(temp,LNUMBER);
}

void Sleep(void) {
	Stack *s;

	s = Pop();
	if(!s) {
		Error("Stack error in sleep");
		return;
	}
	sleep(s->intval);
}

void USleep(void) {
#if HAVE_USLEEP
	Stack *s;

	s = Pop();
	if(!s) {
		Error("Stack error in usleep");
		return;
	}
	usleep(s->intval);
#endif
}

int FpPush(FILE *fp, char *fn) {
	FpStack *new = emalloc(sizeof(FpStack));

	new->fp = fp;
	new->filename = (char *)estrdup(fn);
	new->id = fileno(fp);
	new->next = fp_top;
	fp_top = new;
	return(new->id);
}

FILE *FpFind(int id) {
	FpStack *f;

	f = fp_top;
	while(f) {
		if(f->id == id) return(f->fp);
		f=f->next;
	}
	return(NULL);
}	

void FpDel(int id) {
	FpStack *e,*f;

	f = fp_top;
	e = f;
	while(f) {
		if(f->id == id) {
			if(f==fp_top) {
				efree(f->filename);
				fp_top=f->next;
				efree(f);	
				return;
			} else {
				efree(f->filename);
				e->next = f->next;
				efree(f);
				return;
			}
		}
		e = f;
		f = f->next;
	}
}

void Fopen(void) {
	Stack *s;
	char temp[8];
	FILE *fp;
	int id;
	char *p;

	s = Pop();
	if(!s) {
		Error("Stack error in fopen");
		return;
	}
	if(!*(s->strval)) {
		Push("-1",LNUMBER);
		return;
	}
	p = estrdup(s->strval);
	s = Pop();
	if(!s) {
		Error("Stack error in fopen");
		return;
	}
	if(!*(s->strval)) {
		Push("-1",LNUMBER);
		return;
	}
#if DEBUG
	Debug("Opening [%s] with mode [%s]\n",s->strval,p);
#endif
	fp = fopen(s->strval,p);
	if(!fp) {
		Error("fopen(\"%s\",\"%s\") - %s",s->strval,p,strerror(errno));
		Push("-1",LNUMBER);
		return;
	}
	if(p) efree(p);
	id = FpPush(fp,s->strval);
	sprintf(temp,"%d",id);	
	Push(temp,LNUMBER);
}	

void Fclose(void) {
	Stack *s;
	int id;
	FILE *fp;

	s = Pop();
	if(!s) {
		Error("Stack error in fclose");
		return;
	}
	id = s->intval;

	fp = FpFind(id);
	if(!fp) {
		Error("Unable to find file identifier %d",id);
		return;
	}
	fclose(fp);
	FpDel(id);
}

void Fgets(void) {
	Stack *s;
	FILE *fp;
	int id;
	int len;
	char *buf;

	s = Pop();
	if(!s) {
		Error("Stack error in fgets");
		return;
	}
	len = s->intval;

	s = Pop();
	if(!s) {
		Error("Stack error in fgets");
		return;
	}
	id = s->intval;

	fp = FpFind(id);
	if(!fp) {
		Error("Unable to find file identifier %d",id);
		return;
	}
	buf = emalloc(sizeof(char) * (len + 1));
	if(!fgets(buf,len,fp)) {
		Push("",STRING);
		efree(buf);
		return;
	}
	Push(buf,STRING);
}

void Fputs(void) {
	Stack *s;
	FILE *fp;
	int ret,id;
	char *buf;
	char temp[8];

	s = Pop();
	if(!s) {
		Error("Stack error in fputs");
		return;
	}
	if(!*s->strval) {
		return;
	}
	buf = (char *)estrdup(s->strval);
	s = Pop();
	if(!s) {
		Error("Stack error in fputs");
		return;
	}
	id = s->intval;	
	fp = FpFind(id);
	if(!fp) {
		Error("Unable to find file identifier %d",id);
		return;
	}
	ParseEscapes(buf);
	StripSlashes(buf);
	ret = fputs(buf,fp);
	sprintf(temp,"%d",ret);
	Push(temp,STRING);
}	

void Rewind(void) {
	Stack *s;
	int id;
	FILE *fp;

	s = Pop();
	if(!s) {
		Error("Stack error in rewind");
		return;
	}
	id = s->intval;	
	fp = FpFind(id);
	if(!fp) {
		Error("Unable to find file identifier %d",id);
		return;
	}
	rewind(fp);
}

void Ftell(void) {
	Stack *s;
	int id;
	char temp[16];
	long pos;
	FILE *fp;

	s = Pop();
	if(!s) {
		Error("Stack error in ftell");
		return;
	}
	id = s->intval;	
	fp = FpFind(id);
	if(!fp) {
		Error("Unable to find file identifier %d",id);
		return;
	}
	pos = ftell(fp);
	sprintf(temp,"%ld",pos);
	Push(temp,LNUMBER);
}

void Fseek(void) {
	Stack *s;
	int ret,id;
	long pos;
	char temp[8];
	FILE *fp;

	s = Pop();
	if(!s) {
		Error("Stack error in fseek");
		return;
	}
	pos = s->intval;
	s = Pop();
	if(!s) {
		Error("Stack error in fseek");
		return;
	}
	id = s->intval;
	fp = FpFind(id);
	if(!fp) {
		Error("Unable to find file identifier %d",id);
		return;
	}
	ret = fseek(fp,pos,SEEK_SET);
	sprintf(temp,"%d",ret);
	Push(temp,LNUMBER);
}

char *GetCurrentPI(void) {
	return(CurrentPI);
}

void ChMod(void) {
	Stack *s;
	int ret,mode;
	char temp[8];

	OctDec();	
	s = Pop();
	if(!s) {
		Error("Stack error in chmod()");
		return;
	}
	mode = s->intval;
	s = Pop();
	if(!s) {
		Error("Stack error in chmod()");
		return;
	}
	ret = chmod(s->strval,mode);
	sprintf(temp,"%d",ret);
	Push(temp,LNUMBER);
}	

void ChOwn(void) {
	Stack *s;
	int ret;
	char temp[8];
	struct passwd *pw=NULL;

	s = Pop();
	if(!s) {
		Error("Stack error in chown()");
		return;
	}
	pw = getpwnam(s->strval);
	if(!pw) {
		Error("Unable to find entry for %s in passwd file",s->strval);
		return;
	}
	s = Pop();	
	if(!s) {
		Error("Stack error in chown()");
		return;
	}
	ret = chown(s->strval,pw->pw_uid,-1);
	sprintf(temp,"%d",ret);
	Push(temp,LNUMBER);
}	

void ChGrp(void) {
	Stack *s;
	int ret;
	char temp[8];
	struct group *gr=NULL;

	s = Pop();
	if(!s) {
		Error("Stack error in chown()");
		return;
	}
	if(s->intval != -1) gr = getgrnam(s->strval);
	if(!gr) {
		Error("Unable to find entry for %s in groups file",s->strval);
		return;
	}
	s = Pop();	
	if(!s) {
		Error("Stack error in chown()");
		return;
	}
	ret = chown(s->strval,-1,gr->gr_gid);
	sprintf(temp,"%d",ret);
	Push(temp,LNUMBER);
}

void MkDir(void) {
	Stack *s;
	int ret,mode;
	char temp[8];

	OctDec();	
	s = Pop();
	if(!s) {
		Error("Stack error in mkdir()");
		return;
	}
	mode = s->intval;
	s = Pop();
	if(!s) {
		Error("Stack error in mkdir()");
		return;
	}
	ret = mkdir(s->strval,mode);
	sprintf(temp,"%d",ret);
	Push(temp,LNUMBER);
}	
