/* Routines to read and write xBase-files (.dbf)

   By Maarten Boekhold, 29th of oktober 1995
*/
#include <sys/types.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <ctype.h>

#include "dbf.h"
#include "endian.h"

/* open a dbf-file, get it's field-info and store this information */

dbhead *dbf_open(u_char *file, int flags) {
	int				file_no;
	dbhead			*dbh;
	f_descr			*fields;
	dbf_header		*head;
	dbf_field		*fieldc;
	int			t;

	if ((dbh = (dbhead *)malloc(sizeof(dbhead))) == NULL) {
		return (dbhead *)DBF_ERROR;
	}

	if ((head = (dbf_header *)malloc(sizeof(dbf_header))) == NULL) {
		free(dbh);
		return (dbhead *)DBF_ERROR;
	}

	if ((fieldc = (dbf_field *)malloc(sizeof(dbf_field))) == NULL) {
		free(head);
		free(dbh);
		return (dbhead *)DBF_ERROR;
	}

	if ((file_no = open(file, flags)) == -1) {
		free(fieldc);
		free(head);
		free(dbh);
		return (dbhead *)DBF_ERROR;
	}

/* read in the disk-header */

	if (read(file_no, head, sizeof(dbf_header)) == -1) {
		close(file_no);
		free(fieldc);
		free(head);
		free(dbh);
		return (dbhead *)DBF_ERROR;
	}

	if (!(head->dbh_dbt & DBH_NORMAL)) {
                close(file_no);
                free(fieldc);
                free(head);
                free(dbh);
                return (dbhead *)DBF_ERROR;
	}

	dbh->db_fd = file_no;
	if (head->dbh_dbt & DBH_MEMO) {
		dbh->db_memo = 1;
	} else {
		dbh->db_memo = 0;
	}
	dbh->db_year = head->dbh_year;
	dbh->db_month = head->dbh_month;
	dbh->db_day = head->dbh_day;
	dbh->db_hlen = get_short((u_char *)&head->dbh_hlen);
	dbh->db_records = get_long((u_char *)&head->dbh_records);
	dbh->db_currec = 0;
	dbh->db_rlen = get_short((u_char *)&head->dbh_rlen);
	dbh->db_nfields = (dbh->db_hlen - sizeof(dbf_header)) / sizeof(dbf_field);

								/* dbh->db_hlen - sizeof(dbf_header) isn't the
									correct size, cos dbh->hlen is in fact
									a little more cos of the 0x0D (and
									possibly another byte, 0x4E, I have
									seen this somewhere). Because of rounding
									everything turns out right :) */

	if ((fields = (f_descr *)calloc(dbh->db_nfields, sizeof(f_descr)))
		== NULL) {
                close(file_no);
                free(fieldc);
                free(head);
                free(dbh);
                return (dbhead *)DBF_ERROR;
	}

	for (t = 0; t < dbh->db_nfields; t++) {
		read(file_no, fieldc, sizeof(dbf_field));
		strncpy(fields[t].db_name, fieldc->dbf_name,
			strlen(fieldc->dbf_name));
		fields[t].db_type = fieldc->dbf_type;	
		fields[t].db_flen = fieldc->dbf_flen;
		fields[t].db_dec  = fieldc->dbf_dec;
	}

	dbh->db_offset = get_short((u_char *)&head->dbh_hlen);
	dbh->db_fields = fields;

        free(fieldc);
        free(head);
	return dbh;
}

int	dbf_write_head(dbhead *dbh) {
	dbf_header	head;

	if (lseek(dbh->db_fd, 0, SEEK_SET) == -1) {
		return DBF_ERROR;
	}

/* fill up the diskheader */

	head.dbh_dbt = DBH_NORMAL;
	if (dbh->db_memo) head.dbh_dbt = DBH_MEMO;

/* HERE INSERT CORRECT DATE !!!!!!!!!!!! */

	head.dbh_year = 0;
	head.dbh_month = 0;
	head.dbh_day = 0;

	put_long(head.dbh_records, dbh->db_records);
	put_short(head.dbh_hlen, dbh->db_hlen);
	put_short(head.dbh_rlen, dbh->db_rlen);
	
	if (write(dbh->db_fd, &head, sizeof(dbf_header)) == -1 ) {
		return DBF_ERROR;
	}

	return 0;
}

int dbf_put_fields(dbhead *dbh) {
	dbf_field	field;
	u_long		t;
	u_char		end = 0x0D;

	if (lseek(dbh->db_fd, sizeof(dbf_header), SEEK_SET) == -1) {
		return DBF_ERROR;
	}

	for (t = 0; t < dbh->db_nfields; t++) {
		strncpy(field.dbf_name, dbh->db_fields[t].db_name, DBF_NAMELEN);
		field.dbf_type = dbh->db_fields[t].db_type;
		field.dbf_flen = dbh->db_fields[t].db_flen;
		field.dbf_dec = dbh->db_fields[t].db_dec;

		if (write(dbh->db_fd, &field, sizeof(dbf_field)) == -1) {
			return DBF_ERROR;
		}
	}

	if (write(dbh->db_fd, &end, 1) == -1) {
		return DBF_ERROR;
	}

	return 0;
}

int dbf_add_field(dbhead *dbh, u_char *name, u_char type,
								u_char length, u_char dec) {
f_descr	*ptr;
u_long	size, field_no;

	size = (dbh->db_nfields + 1) * sizeof(f_descr);
	if (!(ptr = (f_descr *) realloc(dbh->db_fields, size))) {
		return DBF_ERROR;
	}
	dbh->db_fields = ptr;

	field_no = dbh->db_nfields;
	strncpy(dbh->db_fields[field_no].db_name, name, DBF_NAMELEN);
	dbh->db_fields[field_no].db_type = type;
	dbh->db_fields[field_no].db_flen = length;
	dbh->db_fields[field_no].db_dec = dec;

	dbh->db_nfields++;
	dbh->db_hlen += sizeof(dbf_field);
	dbh->db_rlen += length;

	return 0;
}

dbhead *dbf_open_new(u_char *name, int flags) {
dbhead	*dbh;

	if (!(dbh = (dbhead *)malloc(sizeof(dbhead)))) {
		return (dbhead *)DBF_ERROR;
	}

	if (flags & O_CREAT) {
		if ((dbh->db_fd = open(name, flags, DBF_FILE_MODE)) == -1) {
			free(dbh);
			return (dbhead *)DBF_ERROR;
		}
	} else {
		if ((dbh->db_fd = open(name, flags)) == -1) {
			free(dbh);
			return (dbhead *)DBF_ERROR;
		}
	}
		

	dbh->db_offset = 0;
	dbh->db_memo = 0;
	dbh->db_year = 0;
	dbh->db_month = 0;
	dbh->db_day	= 0;
	dbh->db_hlen = sizeof(dbhead) + 1;
	dbh->db_records = 0;
	dbh->db_currec = 0;
	dbh->db_rlen = 1;
	dbh->db_nfields = 0;
	dbh->db_fields = (f_descr *)0;

	return dbh;
}
	
void dbf_close(dbhead *dbh) {
	int t;

	close(dbh->db_fd);

	for (t = 0; t < dbh->db_nfields; t++) {
		free(&dbh->db_fields[t]);
	}

	free(dbh);
}
	
int dbf_get_record(dbhead *dbh, field *fields,  u_long rec) {
	u_char  *data;
	int     t, i, offset;
	u_char  *dbffield, *end;

/* calculate at which offset we have to read. *DON'T* forget the
   0x0D which seperates field-descriptions from records!

	Note (april 5 1996): This turns out to be included in db_hlen
*/
	offset = dbh->db_hlen + (rec * dbh->db_rlen);

	if (lseek(dbh->db_fd, offset, SEEK_SET) == -1) {
		lseek(dbh->db_fd, 0, SEEK_SET);
		dbh->db_offset = 0;
		return DBF_ERROR;
	}

	dbh->db_offset 	= offset;
	dbh->db_currec	= rec;

    if ((data = (u_char *)malloc(dbh->db_rlen)) == NULL) {
            return DBF_ERROR;
    }

    read(dbh->db_fd, data, dbh->db_rlen);

    if (data[0] == DBF_DELETED) {
            free(data);
            return DBF_DELETED;
    }

    dbffield = &data[1];
    for (t = 0; t < dbh->db_nfields; t++) {
            strncpy(fields[t].db_name, dbh->db_fields[t].db_name, DBF_NAMELEN);
            fields[t].db_type = dbh->db_fields[t].db_type;
            fields[t].db_flen = dbh->db_fields[t].db_flen;
            fields[t].db_dec  = dbh->db_fields[t].db_dec;

            if (fields[t].db_type == 'C') {
				end = &dbffield[fields[t].db_flen - 1 ];
				i = fields[t].db_flen;
                while (( i > 0) && ((*end < 0x21) || (*end > 0x7E))) {
       	          	end--;
					i--;
                }
				strncpy(fields[t].db_contents, dbffield, i);
                fields[t].db_contents[i] = '\0';
            } else {
				end = dbffield;
				i = fields[t].db_flen;
                while (( i > 0) && ((*end < 0x21) || (*end > 0x7E))) {
					end++;
					i--;
				}
				strncpy(fields[t].db_contents, end, i);
				fields[t].db_contents[i] = '\0';
            }

            dbffield += fields[t].db_flen;
	}

    dbh->db_offset += dbh->db_rlen;

	return DBF_VALID;
}

field *dbf_build_record(dbhead *dbh) {
	int t;
	field	*fields;

	if (!(fields = (field *)calloc(dbh->db_nfields, sizeof(field)))) {
		return (field *)DBF_ERROR;
	}
	
	for ( t = 0; t < dbh->db_nfields; t++) {
		if (!(fields[t].db_contents =
			(u_char *)malloc(dbh->db_fields[t].db_flen + 1))) {
			for (t = 0; t < dbh->db_nfields; t++) {
				if (fields[t].db_contents != 0) {
					free(fields[t].db_contents);
					free(fields);
				}
				return (field *)DBF_ERROR;
			}
		}
		strncpy(fields[t].db_name, dbh->db_fields[t].db_name, DBF_NAMELEN);
		fields[t].db_type = dbh->db_fields[t].db_type;
		fields[t].db_flen = dbh->db_fields[t].db_flen;
		fields[t].db_dec  = dbh->db_fields[t].db_dec;
	}

	return fields;
}

void dbf_free_record(dbhead *dbh, field *rec) {
	int t;

	for ( t = 0; t < dbh->db_nfields; t++) {
		free(rec[t].db_contents);
	}

	free(rec);
}

int dbf_put_record(dbhead *dbh, field *rec, u_long where) {
	u_long	offset, new, idx, t, h, length;
	u_char	*data;
	double	fl;
	u_char	foo[128], format[32];

/*	offset:	offset in file for this record
	new:	real offset after lseek
	idx:	index to which place we are inside the 'hardcore'-data for this
			record
	t:		field-counter
	data:	the hardcore-data that is put on disk
	h:		index into the field-part in the hardcore-data
	length:	length of the data to copy
	fl:		a float used to get the right precision with real numbers
	foo:	copy of db_contents when field is not 'C'
	format:	sprintf format-string to get the right precision with real numbers

	NOTE: this declaration of 'foo' can cause overflow when the contents-field
	is longer the 127 chars (which is highly unlikely, cos it is not used
	in text-fields).
*/

	if (where > dbh->db_records) {
		if ((new = lseek(dbh->db_fd, 0, SEEK_END)) == -1) {
			return DBF_ERROR;
		}
		dbh->db_records++;
	} else {
		offset = dbh->db_hlen + (where * dbh->db_rlen);
		if ((new = lseek(dbh->db_fd, offset, SEEK_SET)) == -1) {
			return DBF_ERROR;
		}
	}

	dbh->db_offset = new;

	if (!(data = (u_char *)calloc(dbh->db_rlen, 1))) {
		return DBF_ERROR;
	}

	data[0] = DBF_VALID;

	idx = 1;
	for (t = 0; t < dbh->db_nfields; t++) {
/*	Handle text */
		if (rec[t].db_type == 'C') {
			if (strlen(rec[t].db_contents) > rec[t].db_flen) {
				length = rec[t].db_flen;
			} else {
				length = strlen(rec[t].db_contents);
			}
			strncpy(&data[idx], rec[t].db_contents, length);
		} else {
/* Handle the rest */
/* Numeric is special, because of real numbers */
			if ((rec[t].db_type == 'N') && (rec[t].db_dec != 0)) {
				fl = atof(rec[t].db_contents);
				sprintf(format, "%%.%d", rec[t].db_dec);
				sprintf(foo, format, fl);
			} else {
				strcpy(foo, rec[t].db_contents);
			}
			if (strlen(foo) > rec[t].db_flen) {
				length = rec[t].db_flen;
			} else {
				length = strlen(foo);
			}
			h = rec[t].db_flen - length;
			strncpy(&data[idx+h], foo, length);
		}
		idx += rec[t].db_flen;
	}

	if (write(dbh->db_fd, data, dbh->db_rlen) == -1) {
		return DBF_ERROR;
	}

	dbh->db_offset += dbh->db_rlen;

	return 0;
}
