/* This program reads in an xbase-dbf file and sends 'inserts' to an
   mSQL-server with the records in the xbase-file

   M. Boekhold (boekhold@cindy.et.tudelft.nl)  okt. 1995
*/

#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <msql.h>
#include "dbf.h"

int	verbose = 0, upper = 0, lower = 0, create = 0, fieldlow = 0;
char	primary[11];
char	*host = NULL;
char 	*dbase = "test";
char 	*table = "test";

inline void strtoupper(char *string);
inline void strtolower(char *string);
void do_create(int, char*, dbhead*);
void do_inserts(int, char*, dbhead*);
char *msqlEscape(char*);
int check_table(int, char*);
void usage(void);

inline void strtoupper(char *string) {
	while(*string != '\0') {
		*string++ = toupper(*string);
	}
}

inline void strtolower(char *string) {
	while(*string != '\0') {
		*string++ = tolower(*string);
	}
}

char *msqlEscape(char *string) {
    char    *escaped, *foo, *bar;
    u_long  count = 0;

    foo = string;
    while(*foo != '\0') {
        if (*foo++ == '\'') count++;
    }

    if (!(escaped = (char *)malloc(strlen(string) + count + 1))) {
        return (char *)-1;
    }

    foo = escaped;
    bar = string;
    while (*bar != '\0') {
        if (*bar == '\'') {
            *foo++ = '\\';
        }
        *foo++ = *bar++;
    }
    *foo = '\0';

    return escaped;
}

int check_table(int sock, char *table) {
	m_result	*result;
	m_row		row;

	while ((result = msqlListTables(sock))) {
		row = msqlFetchRow(result);
		if (strcmp((char *) row[0], table)) {
			msqlFreeResult(result);
			return 1;
		}
	}

	return 0;
}

void usage(void){
		printf("dbf2msql %s\n", VERSION);
		printf("usage:\tdbf2msql\t[-u | -l] [-h hostname] [-p primary key]\n");
		printf("\t\t\t-d dbase -t table [-c] [-f] [-v[v]] dbf-file\n");
}

void do_create(int SQLsock, char *table, dbhead *dbh) {
	char		*query;
	char 		t[20];
	int			i, length;

	if (verbose > 1) {
		printf("Building CREATE-clause\n");
	}

	if (!(query = (char *)malloc(
			(dbh->db_records * 40) + 29 + strlen(table)))) {
		fprintf(stderr, "Memory allocation error in function do_create\n");
		msqlClose(SQLsock);
		close(dbh->db_fd);
		free(dbh);
		exit(1);
	}

	sprintf(query, "CREATE TABLE %s (", table);
	length = strlen(query);
	for ( i = 0; i < dbh->db_nfields; i++) {
		if ((strlen(query) != length)) {
                        strcat(query, ",");
                }

				if (fieldlow)
					strtolower(dbh->db_fields[i].db_name);

                strcat(query, dbh->db_fields[i].db_name);
                switch(dbh->db_fields[i].db_type) {
						case 'D':
                        case 'C':
                                strcat(query, " char");
                                sprintf(t," (%d)",dbh->db_fields[i].db_flen);
                                strcat(query, t);
                                strcat(query, "");
                                break;
                        case 'N':
                                if (dbh->db_fields[i].db_dec != 0) {
                                        strcat(query, " real");
                                } else {
                                        strcat(query, " int");
                                }
                                break;
                        case 'L':
                                strcat(query, " char (1)");
                                break;
                }
                if (strcmp(dbh->db_fields[i].db_name, primary) == 0) {
					strcat(query, " primary key");
				}
	}

	strcat(query, ")\n");

	if (verbose > 1) {
		printf("Sending create-clause\n");
		printf("%s\n", query);
	}

	if (msqlQuery(SQLsock, query) == -1) {
		fprintf(stderr, "Error creating table!\n");
		fprintf(stderr, "Detailed report: %s\n", msqlErrMsg);
                close(dbh->db_fd);
                free(dbh);
				free(query);
                msqlClose(SQLsock);
                exit(1);
	}

	free(query);
}

void do_inserts(int SQLsock, char *table, dbhead *dbh) {
	int		result, i, h;
	field	*fields;
	char	*query, *keys, *vals, *foo;
	u_long	key_len = 0;
	u_long	val_len = 0;

	if (verbose > 1) {
		printf("Inserting records\n");
	}

	for ( i = 0 ; i < dbh->db_nfields ; i++ ) {
		key_len += strlen(dbh->db_fields[i].db_name) + 1;
		val_len += dbh->db_fields[i].db_flen + 3;
	}

	if (!(query = (char *)malloc(26 + strlen(table) + key_len + val_len))) {
		fprintf(stderr,
			"Memory allocation error in function do_inserts (query)\n");
		msqlClose(SQLsock);
		close(dbh->db_fd);
		free(dbh);
		exit(1);
	}

	if (!(keys = (char *)malloc(key_len))) {
		fprintf(stderr,
			"Memory allocation error in function do_inserts (keys)\n");
		msqlClose(SQLsock);
		close(dbh->db_fd);
		free(query);
		free(dbh);
		exit(1);
	}

	if (!(vals = (char *)malloc(val_len))) {
		fprintf(stderr,
			"Memory allocation error in function do_inserts (vals)\n");
		msqlClose(SQLsock);
		close(dbh->db_fd);
		free(keys);
		free(dbh);
		exit(1);
	}

	for ( i = 0; i < dbh->db_records; i++) {
		if ((fields = dbf_build_record(dbh)) != (field *)DBF_ERROR) {
			result = dbf_get_record(dbh, fields,  i);
			if (result == DBF_VALID) {
				keys[0]='\0';
				vals[0]='\0';

				for (h = 0; h < dbh->db_nfields; h++) {
					if (strlen(keys) != 0) {
						strcat(keys, ",");
					}
					if (strlen(vals) != 0) {
						strcat(vals, ",");
					}
					strcat(keys, fields[h].db_name);
					if (fields[h].db_type != 'N') {
						strcat(vals, "'");
					}
					if (upper) {
					   strtoupper(fields[h].db_contents);
					}
					if (lower) {
						strtolower(fields[h].db_contents);
					}
					foo=msqlEscape(fields[h].db_contents);
					strcat(vals, foo);
					if (fields[h].db_type == 'N') {
						if (*foo == '\0') {
							strcat(vals, "NULL");
						}
					}
					free(foo);
					if (fields[h].db_type != 'N') {
						strcat(vals, "'");
					}
				}

				sprintf(query,
					"INSERT INTO %s ( %s ) VALUES ( %s )\n",
				    	table, keys, vals);

				if ((verbose > 1) && ((i % 100) == 0)) {
					printf("Inserting record %d\n", i);
				}

				if (msqlQuery(SQLsock, query) == -1) {
					fprintf(stderr,
						"Error sending INSERT in record %04d\n", i);
					fprintf(stderr,
						"Detailed report: %s\n",
						msqlErrMsg);
					if (verbose > 1) {
						fprintf(stderr, "%s", query);
					}
				}
			}
			dbf_free_record(dbh, fields);
		}
	}

	free(query);
	free(keys);
	free(vals);
}

int main(int argc, char **argv)
{
	int 		SQLsock, i;
	extern int 	optind;
	extern char	*optarg;
	char		*query;
	dbhead		*dbh;

	primary[0] = '\0';

	while ((i = getopt(argc, argv, "flucvh:p:d:t:")) != EOF) {
		switch (i) {
			case 'f':
				fieldlow=1;
				break;
			case 'v':
				verbose++;
				break;
			case 'c':
				create=1;
				break;
			case 'l':
				lower=1;
				break;
			case 'u':
				if (lower) {
					usage();
					printf("Can't use -u and -l at the same time!\n");
					exit(1);
				}
				upper=1;
				break;
			case 'h':
				host = (char *)strdup(optarg);
				break;
			case 'p':
				strncpy(primary, optarg, 11);
				break;
			case 'd':
				dbase = (char *)strdup(optarg);
				break;
			case 't':
				table = (char *)strdup(optarg);
				break;
			case ':':
				usage();
				printf("missing argument!\n");
				exit(1);
			case '?':
				usage();
				printf("unknown argument: %s\n", argv[0]);
				exit(1);
			default:
				break;
		}
	}

	argc -= optind;
	argv = &argv[optind];

	if (argc != 1) {
		usage();
		exit(1);
	}

	if (verbose > 1) {
		printf("Opening dbf-file\n");
	}

	if ((dbh = dbf_open(argv[0], O_RDONLY)) == (dbhead *)-1) {
		fprintf(stderr, "Couldn't open xbase-file %s\n", argv[0]);
		exit(1);
	}

	if (verbose) {
		printf("dbf-file: %s, mSQL-dbase: %s, mSQL-table: %s\n", argv[0],
																 dbase,
																 table);
		printf("Number of records: %ld\n", dbh->db_records);
		printf("NAME:\t\tLENGTH:\t\tTYPE:\n");
		printf("-------------------------------------\n");
		for (i = 0; i < dbh->db_nfields ; i++) {
			printf("%-12s\t%7d\t\t%5c\n",dbh->db_fields[i].db_name,
									 dbh->db_fields[i].db_flen,
									 dbh->db_fields[i].db_type);
		}
	}

	if (verbose > 1) {
		printf("Making connection to mSQL-server\n");
	}

	if ((SQLsock = msqlConnect(host)) == -1) {
		fprintf(stderr, "Couldn't get a connection with the ");
		fprintf(stderr, "designated host!\n");
		fprintf(stderr, "Detailed report: %s\n", msqlErrMsg);
		close(dbh->db_fd);
		free(dbh);
		exit(1);
	}

	if (verbose > 1) {
		printf("Selecting database\n");
	}

	if ((msqlSelectDB(SQLsock, dbase)) == -1) {
		fprintf(stderr, "Couldn't select database %s.\n", dbase);
		fprintf(stderr, "Detailed report: %s\n", msqlErrMsg);
		close(dbh->db_fd);
		free(dbh);
		msqlClose(SQLsock);
		exit(1);
	}

	if (!create) {
		if (!check_table(SQLsock, table)) {
			printf("Table does not exist!\n");
			exit(1);
		}
	} else {
		if (verbose > 1) {
			printf("Dropping original table (if one exists)\n");
		}

		if (!(query = (char *)malloc(12 + strlen(table)))) {
			printf("Memory-allocation error in main (drop)!\n");
			close(dbh->db_fd);
			free(dbh);
			msqlClose(SQLsock);
			exit(1);
		}

		sprintf(query, "DROP TABLE %s", table);
		msqlQuery(SQLsock, query);
		free(query);

/* Build a CREATE-clause
*/
		do_create(SQLsock, table, dbh);
	}

/* Build an INSERT-clause
*/
	do_inserts(SQLsock, table, dbh);

	if (verbose > 1) {
		printf("Closing up....\n");
	}

    close(dbh->db_fd);
    free(dbh);
    msqlClose(SQLsock);
	exit(0);
}
