/****************************************************************************

    PROGRAM: wmsqlrs.c

    PURPOSE: Windows version of relshow program
    
    		 This software is intended to allow the user to test that the
    		 winmsql.dll library is functioning correctly.


****************************************************************************/


#include <stdio.h>
#include <windows.h>


#include <msql.h>

/* These are some global variables for tracking the database connection.
*/
static int giDBConnect;
static BOOL gbConnected = FALSE;
static char gszConnection[100]; /* Name of the Host in MSQL.INI */
static char gszTableName[100];  /* Name of the selected table */
static char gszDatabaseName[100]; /* Name of database */
static HWND hwndFieldList;

#include "resource.h"
#include "wmsqlrs.h"


/* These are prototypes for functions used only in this source.
   All of them should be static. However some have to be global
   due to constraints in this development system.
*/
int PASCAL WinMain(HANDLE, HANDLE, LPSTR, int);
BOOL InitApplication(HANDLE);
BOOL InitInstance(HANDLE, int);
long CALLBACK __export MainWndProc(HWND, UINT, WPARAM, LPARAM);
BOOL __export CALLBACK About(HWND, unsigned, WORD, LONG);
BOOL __export CALLBACK ServerConnect(HWND, unsigned, WORD, LONG);
BOOL __export CALLBACK DatabaseSelect(HWND, unsigned, WORD, LONG);
BOOL __export CALLBACK Table(HWND, unsigned, WORD, LONG);
BOOL __export CALLBACK DatabaseSetup(HWND, unsigned, WORD, LONG);


HANDLE hInst;               /* current instance              */

/****************************************************************************

    FUNCTION: WinMain(HANDLE, HANDLE, LPSTR, int)

    PURPOSE: calls initialization function, processes message loop

    COMMENTS:

        Windows recognizes this function by name as the initial entry point
        for the program.  This function calls the application initialization
        routine, if no other instance of the program is running, and always
        calls the instance initialization routine.  It then executes a message
        retrieval and dispatch loop that is the top-level control structure
        for the remainder of execution.  The loop is terminated when a WM_QUIT
        message is received, at which time this function exits the application
        instance by returning the value passed by PostQuitMessage().

        If this function must abort before entering the message loop, it
        returns the conventional value NULL.

****************************************************************************/

int PASCAL WinMain(hInstance, hPrevInstance, lpCmdLine, nCmdShow)
HANDLE hInstance;                /* current instance         */
HANDLE hPrevInstance;            /* previous instance        */
LPSTR lpCmdLine;                 /* command line             */
int nCmdShow;                    /* show-window type (open/icon) */
{
    MSG msg;                     /* message              */

    if (!hPrevInstance)          /* Other instances of app running? */
    if (!InitApplication(hInstance)) /* Initialize shared things */
        return (FALSE);      /* Exits if unable to initialize     */

    /* Perform initializations that apply to a specific instance */

    if (!InitInstance(hInstance, nCmdShow))
        return (FALSE);

    /* Acquire and dispatch messages until a WM_QUIT message is received. */

    while (GetMessage(&msg,    /* message structure              */
        NULL,          /* handle of window receiving the message */
        NULL,          /* lowest message to examine          */
        NULL))         /* highest message to examine         */
    {
    TranslateMessage(&msg);    /* Translates virtual key codes       */
    DispatchMessage(&msg);     /* Dispatches message to window       */
    }
    return (msg.wParam);       /* Returns the value from PostQuitMessage */
}


/****************************************************************************

    FUNCTION: InitApplication(HANDLE)

    PURPOSE: Initializes window data and registers window class

    COMMENTS:

        This function is called at initialization time only if no other
        instances of the application are running.  This function performs
        initialization tasks that can be done once for any number of running
        instances.

        In this case, we initialize a window class by filling out a data
        structure of type WNDCLASS and calling the Windows RegisterClass()
        function.  Since all instances of this application use the same window
        class, we only need to do this when the first instance is initialized.


****************************************************************************/

BOOL InitApplication(hInstance)
HANDLE hInstance;                  /* current instance       */
{
    WNDCLASS  wc;

    /* Fill in window class structure with parameters that describe the       */
    /* main window.                                                           */

    wc.style = NULL;                    /* Class style(s).                    */
    wc.lpfnWndProc = MainWndProc;       /* Function to retrieve messages for  */
                                        /* windows of this class.             */
    wc.cbClsExtra = 0;                  /* No per-class extra data.           */
    wc.cbWndExtra = 0;                  /* No per-window extra data.          */
    wc.hInstance = hInstance;           /* Application that owns the class.   */
    wc.hIcon = LoadIcon(NULL, IDI_APPLICATION);
    wc.hCursor = LoadCursor(NULL, IDC_ARROW);
    wc.hbrBackground = GetStockObject(WHITE_BRUSH);
    wc.lpszMenuName =  "GenericMenu";   /* Name of menu resource in .RC file. */
    wc.lpszClassName = "GenericWClass"; /* Name used in call to CreateWindow. */

    /* Register the window class and return success/failure code. */

    return (RegisterClass(&wc));

}


/****************************************************************************

    FUNCTION:  InitInstance(HANDLE, int)

    PURPOSE:  Saves instance handle and creates main window

    COMMENTS:

        This function is called at initialization time for every instance of
        this application.  This function performs initialization tasks that
        cannot be shared by multiple instances.

        In this case, we save the instance handle in a static variable and
        create and display the main program window.

****************************************************************************/

BOOL InitInstance(hInstance, nCmdShow)
    HANDLE          hInstance;          /* Current instance identifier.       */
    int             nCmdShow;           /* Param for first ShowWindow() call. */
{
    HWND            hWnd;               /* Main window handle.                */

    /* Save the instance handle in static variable, which will be used in  */
    /* many subsequence calls from this application to Windows.            */

    hInst = hInstance;

    /* Create a main window for this application instance.  */

    hWnd = CreateWindow(
        "GenericWClass",                /* See RegisterClass() call.          */
        "Windows Mini-SQL Schema Viewer",   /* Text for window title bar.         */
        WS_OVERLAPPEDWINDOW,            /* Window style.                      */
        CW_USEDEFAULT,                  /* Default horizontal position.       */
        CW_USEDEFAULT,                  /* Default vertical position.         */
        CW_USEDEFAULT,                  /* Default width.                     */
        CW_USEDEFAULT,                  /* Default height.                    */
        NULL,                           /* Overlapped windows have no parent. */
        NULL,                           /* Use the window class menu.         */
        hInstance,                      /* This instance owns this window.    */
        NULL                            /* Pointer not needed.                */
    );

    /* If window could not be created, return "failure" */

    if (!hWnd)
        return (FALSE);

    /* Make the window visible; update its client area; and return "success" */

    ShowWindow(hWnd, nCmdShow);  /* Show the window                        */
    UpdateWindow(hWnd);          /* Sends WM_PAINT message                 */
    
    return (TRUE);               /* Returns the value from PostQuitMessage */

}

/****************************************************************************

    FUNCTION: MainWndProc(HWND, UINT, WPARAM, LPARAM)

    PURPOSE:  Processes messages

    MESSAGES:

    WM_COMMAND    - application menu (About dialog box)
    WM_DESTROY    - destroy window

    COMMENTS:

    To process the IDM_ABOUT message, call MakeProcInstance() to get the
    current instance address of the About() function.  Then call Dialog
    box which will create the box according to the information in your
    generic.rc file and turn control over to the About() function.  When
    it returns, free the intance address.

****************************************************************************/

long CALLBACK __export MainWndProc(hWnd, message, wParam, lParam)
HWND hWnd;                      /* window handle                 */
UINT message;                   /* type of message               */
WPARAM wParam;                  /* additional information        */
LPARAM lParam;                  /* additional information        */
{
	
    switch (message)
    {
        case WM_COMMAND:       /* message: command from application menu */
        	switch (wParam)
        	{
            case IDM_ABOUT:
                DialogBox(hInst,        /* current instance          */
                    "AboutBox",         /* resource to use           */
                    hWnd,               /* parent handle             */
                    About);             /* About() instance address  */
                break;
            case IDM_DATABASE_CONNECT:
               DialogBox(hInst, "DATABASE_CONNECT", hWnd, ServerConnect);
               break;
            case IDM_DATABASE_TABLE:
               DialogBox(hInst, "TABLES_DIALOG", hWnd, Table);
               /* Check to see if gszTableName has a value. If so display
                  the schema
                */
                if (gszTableName[0])
                {
                	m_field *pMField;
					m_result *pMResult;
					int iYCoord = 0;
					HDC hDC;                          /* display-context variable  */
					TEXTMETRIC textmetric;
					RECT r;
					char szDisplay[100];
	
                	pMResult = msqlListFields(giDBConnect, gszTableName);
                	if (!pMResult)
                	{
                		char szErrorMsg[160];
                		msqlErrorText(szErrorMsg);
            			MessageBox(hWnd, szErrorMsg, "Error in MSQL", MB_OK | MB_APPLMODAL);
            			break;
                	}
                	hDC = GetDC(hWnd);
                	/*
                	 * This rectangle manipulation clears the background.
                	 */
					GetClientRect(hWnd, &r);
					Rectangle(hDC, r.left, r.top, r.right, r.bottom);
					/*
					 * Use the font metrics to work out window offsets
					 */
					GetTextMetrics (hDC, &textmetric);
					iYCoord = 10; // leave a gap at the top.
					
					/*
					 * Put up the column Titles
					 */
               		r.left = 0;
               		r.top = iYCoord;
               		r.right = textmetric.tmAveCharWidth * 40;
					iYCoord += textmetric.tmExternalLeading + textmetric.tmHeight;
               		r.bottom = iYCoord;
					DrawText(hDC, "Field", 5, &r, DT_LEFT);
					r.left = r.right + 10;
					r.right = r.left + textmetric.tmAveCharWidth * 12;
					DrawText(hDC, "Type", 4, &r, DT_LEFT);
					r.left = r.right + 10;
					r.right = r.left + textmetric.tmAveCharWidth * 10;
					DrawText(hDC, "Length", 6, &r, DT_LEFT);
					r.left = r.right + 10;
					r.right = r.left + textmetric.tmAveCharWidth * 10;
					DrawText(hDC, "Not Null", 8, &r, DT_LEFT);
					r.left = r.right + 10;
					r.right = r.left + textmetric.tmAveCharWidth * 10;
					DrawText(hDC, "Key", 3, &r, DT_LEFT);

					// Leave a small gap                    
                    iYCoord += 10;
                    
                	while (pMField = msqlFetchField(pMResult))
                	{
                		r.left = 0;
                		r.top = iYCoord;
                		r.right = textmetric.tmAveCharWidth * 40;
						iYCoord += textmetric.tmExternalLeading + textmetric.tmHeight;
                		r.bottom = iYCoord;
                		
                		/* TBD - Fill in the rest of the field information */
						DrawText(hDC, pMField->name, strlen(pMField->name), &r, DT_LEFT);
						switch (pMField->type)
						{
						case INT_TYPE:	strcpy(szDisplay, "integer");	break;
						case CHAR_TYPE:	strcpy(szDisplay, "char");	break;
						case REAL_TYPE:	strcpy(szDisplay, "real");	break;
						case IDENT_TYPE:	strcpy(szDisplay, "ident");	break;
						case NULL_TYPE:	strcpy(szDisplay, "null");	break;
						default:	strcpy(szDisplay, "unknown");	break;
						}
						r.left = r.right + 10;
						r.right = r.left + textmetric.tmAveCharWidth * 12;
						DrawText(hDC, szDisplay, strlen(szDisplay), &r, DT_LEFT);
						sprintf(szDisplay, "%d", pMField->length);
						r.left = r.right + 10;
						r.right = r.left + textmetric.tmAveCharWidth * 10;
						DrawText(hDC, szDisplay, strlen(szDisplay), &r, DT_LEFT);
						if (IS_NOT_NULL(pMField->flags))
							strcpy(szDisplay, "Y");
						else
							strcpy(szDisplay, "N");
						r.left = r.right + 10;
						r.right = r.left + textmetric.tmAveCharWidth * 10;
						DrawText(hDC, szDisplay, strlen(szDisplay), &r, DT_LEFT);
						if (IS_PRI_KEY(pMField->flags))
							strcpy(szDisplay, "Y");
						else
							strcpy(szDisplay, "N");
						r.left = r.right + 10;
						r.right = r.left + textmetric.tmAveCharWidth * 10;
						DrawText(hDC, szDisplay, strlen(szDisplay), &r, DT_LEFT);
           
               		}
                	msqlFreeResult(pMResult);
                	ReleaseDC(hWnd, hDC);
                }
               break;
            case IDM_DATABASE_SETUP:
               DialogBox(hInst, "SETUP_DIALOG", hWnd, DatabaseSetup);
               break;

            case IDM_README:
            	WinExec("write readme.wri", SW_SHOW);
            	break;
            
            case ID_EXIT:
            	if (gbConnected)
            	{
            		msqlClose(giDBConnect);
            	}
            	DestroyWindow (hWnd);
		   		break;

            default:
                return (DefWindowProc(hWnd, message, wParam, lParam));
            }
            break;

        case WM_DESTROY:          /* message: window being destroyed */
            PostQuitMessage(0);
            break;

        default:                  /* Passes it on if unproccessed    */
            return (DefWindowProc(hWnd, message, wParam, lParam));
    }
    return (NULL);
}


/****************************************************************************

    FUNCTION: About(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "About" dialog box

    MESSAGES:

    WM_INITDIALOG - initialize dialog box
    WM_COMMAND    - Input received

    COMMENTS:

    No initialization is needed for this particular dialog box, but TRUE
    must be returned to Windows.

    Wait for user to click on "Ok" button, then close the dialog box.

****************************************************************************/

BOOL __export CALLBACK About(hDlg, message, wParam, lParam)
HWND hDlg;               /* window handle of the dialog box */
unsigned message;        /* type of message                 */
WORD wParam;             /* message-specific information    */
LONG lParam;
{
    switch (message)
    {
        case WM_INITDIALOG:            /* message: initialize dialog box */
            return (TRUE);

        case WM_COMMAND:               /* message: received a command */
            if (wParam == IDOK         /* "OK" box selected?          */
                || wParam == IDCANCEL) /* System menu close command?  */
            {
                EndDialog(hDlg, TRUE); /* Exits the dialog box        */
                return (TRUE);
            }
            break;
    }
    return (FALSE);               /* Didn't process a message    */
}

/****************************************************************************

    FUNCTION: ServerConnect(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "Server Connect" dialog box

    MESSAGES:

    WM_INITDIALOG - initialize dialog box
    WM_COMMAND    - Input received

    COMMENTS:

    No initialization is needed for this particular dialog box, but TRUE
    must be returned to Windows.

    Wait for user to click on "Ok" button, then close the dialog box.

****************************************************************************/

BOOL __export CALLBACK ServerConnect(hDlg, message, wParam, lParam)
HWND hDlg;               /* window handle of the dialog box */
unsigned message;        /* type of message                 */
WORD wParam;             /* message-specific information    */
LONG lParam;
{
   	char szConnection[100];
   	
    switch (message)
    {
        case WM_INITDIALOG:            /* message: initialize dialog box */
        	GetPrivateProfileString("Server", "IP", "", szConnection, sizeof(szConnection),
        		"MSQL.INI");
        	SetDlgItemText(hDlg, IDC_CONNECTION, szConnection);
            return (TRUE);

        case WM_COMMAND:               /* message: received a command */
            if (wParam == IDOK)        /* "OK" box selected?          */
            {
            	char szErrorMsg[160];
            	int iRv;
            	iRv = GetDlgItemText(hDlg, IDC_CONNECTION, szConnection, sizeof(szConnection));
            	
            	if (iRv <= 0)
            	{
            		MessageBox(hDlg, "Connection must be entered", "Error in Dialog", MB_OK | MB_APPLMODAL);
            		return(TRUE);
            	}
            	/* Disconnect from an existing connection */
            	if (gbConnected)
            	{
            		msqlClose(giDBConnect);
            		gbConnected = FALSE;
            	}
            	/*
            	 * Connect to the database.
            	 */
            	giDBConnect = msqlConnect(szConnection);
            	if (giDBConnect < 0)
            	{
            		msqlErrorText(szErrorMsg);
            		MessageBox(hDlg, szErrorMsg, "Error in MSQL", MB_OK | MB_APPLMODAL);
            	}
            	else
            	{
            		gbConnected = TRUE;
            		strcpy(gszConnection, szConnection);
            		DialogBox(hInst, "DATABASE_SELECT", hDlg, DatabaseSelect);
            		if (gbConnected)
	                	EndDialog(hDlg, TRUE); /* Exits the dialog box        */
            	}
            	return (TRUE);
            }
            else if (wParam == IDCANCEL) /* System menu close command?  */
            {
                EndDialog(hDlg, TRUE); /* Exits the dialog box        */
                return (TRUE);
            }
            break;
    }
    return (FALSE);               /* Didn't process a message    */
}

/****************************************************************************

    FUNCTION: Table(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "TABLE Selection" dialog box

    MESSAGES:

    WM_INITDIALOG - initialize dialog box
    WM_COMMAND    - Input received

    COMMENTS:

    No initialization is needed for this particular dialog box, but TRUE
    must be returned to Windows.

    Wait for user to click on "Ok" button, then close the dialog box.

****************************************************************************/

BOOL __export CALLBACK Table(hDlg, message, wParam, lParam)
HWND hDlg;               /* window handle of the dialog box */
unsigned message;        /* type of message                 */
WORD wParam;             /* message-specific information    */
LONG lParam;
{
	m_result *pMResult;
	m_row  MRow;
	char szErrorMsg[160];
	
    switch (message)
    {
    	case WM_INITDIALOG:
        	if (!gbConnected)
        	{
            	MessageBox(NULL, "Must Select a server first", "Error in Dialog", MB_OK | MB_APPLMODAL);
                EndDialog(hDlg, TRUE); /* Exits the dialog box        */
            	return(FALSE);        		
        	}
        	pMResult = msqlListTables(giDBConnect);
        	if (pMResult == NULL)
        	{
				msqlErrorText(szErrorMsg);
            	MessageBox(hDlg, szErrorMsg, "Error in MSQL", MB_OK | MB_APPLMODAL);        		
         		EndDialog(hDlg, TRUE); /* Exits the dialog box        */
            	return(FALSE);
        	}
        	while (MRow = msqlFetchRow(pMResult))
        	{
	        	SendDlgItemMessage(hDlg, IDC_TABLES,
    				CB_ADDSTRING, 0, (LPARAM) ((LPSTR) *MRow));
        	}
        	SendDlgItemMessage(hDlg, IDC_TABLES, CB_SETCURSEL, 0, 0);
        	msqlFreeResult(pMResult);
        	gszTableName[0] = '\0';
        	
            return (TRUE);

        case WM_COMMAND:               /* message: received a command */
            if (wParam == IDOK)        /* "OK" box selected?          */
            {
            	int iRv;
            	iRv = GetDlgItemText(hDlg, IDC_TABLES, gszTableName, sizeof(gszTableName));
            	if (iRv <= 0)
            	{
	            	MessageBox(NULL, "No Table Selected", "Error in Dialog", MB_OK | MB_APPLMODAL);	       
	            	return(TRUE);
            	}
                EndDialog(hDlg, TRUE); /* Exits the dialog box        */
            	return (TRUE);
            }
            else if (wParam == IDCANCEL) /* System menu close command?  */
            {
                EndDialog(hDlg, TRUE); /* Exits the dialog box        */
                return (TRUE);
            }
            break;
    }
    return (FALSE);               /* Didn't process a message    */
}

/****************************************************************************

    FUNCTION: DatabaseSelect(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "Database Selection" dialog box

    MESSAGES:

    WM_INITDIALOG - initialize dialog box
    WM_COMMAND    - Input received

    COMMENTS:

    No initialization is needed for this particular dialog box, but TRUE
    must be returned to Windows.

    Wait for user to click on "Ok" button, then close the dialog box.

****************************************************************************/

BOOL __export CALLBACK DatabaseSelect(hDlg, message, wParam, lParam)
HWND hDlg;               /* window handle of the dialog box */
unsigned message;        /* type of message                 */
WORD wParam;             /* message-specific information    */
LONG lParam;
{
	m_result *pMResult;
	m_row  MRow;
	char szErrorMsg[160];
	
    switch (message)
    {
    	case WM_INITDIALOG:
        	if (!gbConnected)
        	{
            	MessageBox(NULL, "Must Select a server first", "Error in Dialog", MB_OK | MB_APPLMODAL);
                EndDialog(hDlg, TRUE); /* Exits the dialog box        */
            	return(FALSE);        		
        	}
        	pMResult = msqlListDBs(giDBConnect);
        	if (pMResult == NULL)
        	{
				msqlErrorText(szErrorMsg);
            	MessageBox(hDlg, szErrorMsg, "Error in MSQL", MB_OK | MB_APPLMODAL);        		
         		EndDialog(hDlg, TRUE); /* Exits the dialog box        */
            	return(FALSE);
        	}
        	while (MRow = msqlFetchRow(pMResult))
        	{
	        	SendDlgItemMessage(hDlg, IDC_DATABASES,
    				CB_ADDSTRING, 0, (LPARAM) ((LPSTR) *MRow));
        	}
        	SendDlgItemMessage(hDlg, IDC_DATABASES, CB_SETCURSEL, 0, 0);
        	msqlFreeResult(pMResult);
        	
            return (TRUE);

        case WM_COMMAND:               /* message: received a command */
            if (wParam == IDOK)        /* "OK" box selected?          */
            {
            	int iRv;
            	iRv = GetDlgItemText(hDlg, IDC_DATABASES, gszDatabaseName, 
            		sizeof(gszDatabaseName));
            	if (iRv <= 0)
            	{
	            	MessageBox(NULL, "No Database Selected", "Error in Dialog", MB_OK | MB_APPLMODAL);	       
	            	return(TRUE);
            	}
            	iRv = msqlSelectDB(giDBConnect, gszDatabaseName);
            	if (iRv < 0)
            	{
            		msqlErrorText(szErrorMsg);
            		MessageBox(hDlg, szErrorMsg, "Error in MSQL", MB_OK | MB_APPLMODAL);        		
                    return(TRUE);
            	}
                EndDialog(hDlg, TRUE); /* Exits the dialog box        */
            	return (TRUE);
            }
            else if (wParam == IDCANCEL) /* System menu close command?  */
            {
            	msqlClose(giDBConnect);
            	gbConnected = FALSE;
                EndDialog(hDlg, TRUE); /* Exits the dialog box        */
                return (TRUE);
            }
            break;
    }
    return (FALSE);               /* Didn't process a message    */
}

/****************************************************************************

    FUNCTION: DatabaseSetup(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for "Configure WINMSQL.DLL" dialog box

    MESSAGES:

    WM_INITDIALOG - initialize dialog box
    WM_COMMAND    - Input received

    COMMENTS:

    Wait for user to click on "Ok" button, then close the dialog box.

****************************************************************************/

BOOL __export CALLBACK DatabaseSetup(hDlg, message, wParam, lParam)
HWND hDlg;               /* window handle of the dialog box */
unsigned message;        /* type of message                 */
WORD wParam;             /* message-specific information    */
LONG lParam;
{
   	char szConnection[100];
   	char szPort[10];
   	char szUserName[20];
   	
    switch (message)
    {
        case WM_INITDIALOG:            /* message: initialize dialog box */
        	GetPrivateProfileString("Server", "IP", "", szConnection, sizeof(szConnection),
        		"MSQL.INI");
        	SetDlgItemText(hDlg, IDC_SERVER_IP, szConnection);
        	GetPrivateProfileString("Server", "Port", "", szPort, sizeof(szPort),
        		"MSQL.INI");
        	SetDlgItemText(hDlg, IDC_SERVER_PORT, szPort);
        	GetPrivateProfileString("Server", "Username", "", szUserName, sizeof(szUserName),
        		"MSQL.INI");
        	SetDlgItemText(hDlg, IDC_SERVER_USERNAME, szUserName);
            return (TRUE);

        case WM_COMMAND:               /* message: received a command */
            if (wParam == IDOK)        /* "OK" box selected?          */
            {
	        	GetDlgItemText(hDlg, IDC_SERVER_IP, szConnection, sizeof(szConnection));
            	WritePrivateProfileString("Server", "IP", szConnection, "MSQL.INI");
        		GetDlgItemText(hDlg, IDC_SERVER_PORT, szPort, sizeof(szPort));
        		WritePrivateProfileString("Server", "Port", szPort, "MSQL.INI");
	        	GetDlgItemText(hDlg, IDC_SERVER_USERNAME, szUserName, sizeof(szUserName));
        		WritePrivateProfileString("Server", "Username", szUserName, "MSQL.INI");
                EndDialog(hDlg, TRUE); /* Exits the dialog box        */
            	return (TRUE);
            }
            else if (wParam == IDCANCEL) /* System menu close command?  */
            {
                EndDialog(hDlg, TRUE); /* Exits the dialog box        */
                return (TRUE);
            }
            break;
    }
    return (FALSE);               /* Didn't process a message    */
}