/*
 * Filename: msql-import.c
 * Project:  msql-import
 *
 * Function: This program reads the contents of a flat file
 *           and loads it into a Mini SQL table.
 *
 * Date:     November 1995
 *
 * Copyright (C) 1995 Pascal Forget.  All Rights Reserved
 *
 * PASCAL FORGET MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY
 * OF THIS SOFTWARE FOR ANY PURPOSE.  IT IS SUPPLIED "AS IS"
 * WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for any purpose and without fee is
 * hereby granted, provided that the above copyright notice
 * appear in all copies of the software.
 */

#include "msql-import.h"
#include <string.h>

#define MSQL_IMPORT_VERSION "0.0.6"
#define IMPORT_MAX_FIELDS 255
#define MAX(a,b) (a>b)a:b

#define DEBUG 1

int sock;

SafeString *query;

/*****************************************************************
 * _append_character() appends a character to str                *
 *****************************************************************/

void
append_character(SafeString *str, char c)
{
    int length = 0;
    
    if (str[0].buffer != NULL)
    {
	length = strlen(str[0].buffer);
    }
    
    set_string_capacity(str, length + 1);

    str[0].buffer[length] = c;
    str[0].buffer[length+1] = '\0';
}

/*****************************************************************
 * _append_string_buffer() appends a string buffer to str        *
 *****************************************************************/

void
append_string_buffer(SafeString *str, const char *newBuffer)
{
    if (newBuffer != NULL)
    {
	if (str[0].buffer != NULL)
	{
	    set_string_capacity(str,strlen(str[0].buffer)+strlen(newBuffer));
	}
	else
	{
	    set_string_capacity(str, strlen(newBuffer));
	}
	    
	if (str[0].buffer != NULL)
	{
	    strcat(str[0].buffer, newBuffer);
	}
	else
	{
	    strcpy(str[0].buffer, newBuffer);
	}
    }
}

/*****************************************************************
 * _copy_string_buffer() copies a string buffer in str           *
 *****************************************************************/

void
copy_string_buffer(SafeString *str, const char *newBuffer)
{
    if (newBuffer != NULL) {
	set_string_capacity(str, strlen(newBuffer));
	strcpy(str[0].buffer, newBuffer);
    }
    else
    {
	str[0].buffer = NULL;
    }
}

/*****************************************************************
 * _create_string() allocates and initializes a SafeString       *
 *****************************************************************/

SafeString *
create_string(void)
{
    SafeString *string = (SafeString *)malloc(sizeof(SafeString));
    string[0].buffer = (char *)NULL;
    string[0].capacity = 0;
    return string;
}

/*****************************************************************
 * _set_string_buffer() sets the String's buffer                 *
 *****************************************************************/

void
set_string_buffer(SafeString *str, char *newBuffer)
{
    str[0].buffer = newBuffer;

    if (str[0].buffer != NULL)
    {
	str[0].capacity = strlen(newBuffer);
    }
    else
    {
	str[0].capacity = 0;
    }
}

/*****************************************************************
 * _set_string_capacity() allocates a bigger buffer if needed    *
 *****************************************************************/

void
set_string_capacity(SafeString *str, unsigned int cap)
{
    if (str[0].capacity < cap)
    {
	if (str[0].buffer != NULL)
	{
	    str[0].buffer = (char *)realloc(str[0].buffer, cap+1);
	}
	else
	{
	    str[0].buffer = (char *)malloc(cap+1);
	    str[0].buffer[0] = '\0';
	}
	str[0].capacity = cap;
    }
}

/*****************************************************************
 * _safe_c_string_copy() makes a copy of a string and returns    *
 * the new string.                                               *
 *                                                               *
 * This function replaces strdup() because:                      *
 *      Not all Unixes implement strdup                          *
 *      Some strdup implementations crash on a NULL argument     *
 *****************************************************************/

char *
safe_c_string_copy(const char *s)
{
	char	*s1;

	if (!s)
		return(NULL);
	s1 = (char *)malloc(strlen(s) + 1);
	if (!s1)
		return(NULL);
	
	strcpy(s1,s);
	return(s1);
}

/**********************************************************************/
/* File Id:                     strparse                              */
/* Author:                      Stan Milam.                           */
/* Date Written:                20-Feb-1995.                          */
/* Description:                                                       */
/*     The str_parse() function is used to extract fields from de-    */
/*     limited ASCII records. It is designed to deal with empty fields*/
/*     in a logical manner.                                           */
/*                                                                    */
/* Arguments:                                                         */
/*     char **str       - The address of a pointer which in turn      */
/*                        points to the string being parsed. The      */
/*                        actual pointer is modified with each call to*/
/*                        point to the beginning of the next field.   */
/*     char *delimiters - The address of the string containing the    */
/*                        characters used to delimit the fields within*/
/*                        the record.                                 */
/*                                                                    */
/* Return Value:                                                      */
/*     A pointer of type char which points to the current field in the*/
/*     parsed string.  If an empty field is encountered the address   */
/*     is that of an empty string (i.e. "" ). When there are no more  */
/*     fields in the record a NULL pointer value is returned.         */
/*                                                                    */
/**********************************************************************/

char *
str_parse( char **str, char *delimiters )
{
    char *head, *tail, *rv;

    if ( *str == NULL || **str == '\0' )
        rv = NULL;
    else if ( delimiters == NULL || *delimiters == '\0' )
        rv = NULL;
    else {
        rv = head = *str;
        if (( tail = strpbrk( head, delimiters ) ) == NULL)
            *str = head + strlen(head);
        else {
            *tail = '\0';
            *str = tail + 1;
        }
    }
    return rv;
}

/*****************************************************************
 * _abort_import() closes the client, then exits the program     *
 *****************************************************************/

void
abort_import(int exitCode)
{
    if (sock > -1)
    {
	msqlClose(sock);
    }
    exit(exitCode);
}

/*****************************************************************
 * alarm() prints an error message then calls abort_import()     *
 *****************************************************************/

void
alarm_msql(void)
{
    fprintf(stderr, "msql-import error: %s\n", msqlErrMsg);
    abort_import(-1);
}

void
alarm_msg(const char *message)
{
    fprintf(stderr, "%s\n", message);
    abort_import(-1);
}

/*****************************************************************
 * _append_string() returns the string s, concatenated with the  *
 * append string.  The returned string is not garanteed to be    *
 * at the same memory location as s was.                         *
 *****************************************************************/

char *
append_string(char *s, const char *append)
{
    int newLength;
    
    if ((append!= NULL) && ((newLength = strlen(append)) > 0))
    {
	s = realloc(s, strlen(s) + newLength + 1);

	strcat(s, append);
    }

    return s;
}

/*****************************************************************
 * Put a backslash in front of single quote and parentheses      *
 * characters in the string.  This is needed for inserting       *
 * strings containing single quotes in a relational database.    *
 *****************************************************************/

#define BACKSLASH_CHAR 92

char *
backslashify_special_chars(const char *s)
{
    char *str, *ptr;
    int i, index=-2, len;
    
    if ((!s) ||
	(strlen(s) == 0) ||
	((strchr(s, '\'') == NULL) &&
	 (strchr(s, ')') == NULL) &&
	 (strchr(s, '(') == NULL)))
    {
	return safe_c_string_copy(s);
    }

    str = safe_c_string_copy(s);
    
    while ((ptr = strchr(str+index+2, '\'')) != NULL)
    {
	len = strlen(str)+1;
	str = (char *)realloc(str, len+1);
	
	index = ptr-str;

	for (i=len+1; i>=index; i--)
	{
	    str[i+1] = str[i];
	}

	str[index] = BACKSLASH_CHAR;
    }

    while ((ptr = strchr(str+index+2, '(')) != NULL)
    {
	len = strlen(str)+1;
	str = (char *)realloc(str, len+1);
	
	index = ptr-str;

	for (i=len+1; i>=index; i--)
	{
	    str[i+1] = str[i];
	}

	str[index] = BACKSLASH_CHAR;
    }

    while ((ptr = strchr(str+index+2, ')')) != NULL)
    {
	len = strlen(str)+1;
	str = (char *)realloc(str, len+1);
	
	index = ptr-str;

	for (i=len+1; i>=index; i--)
	{
	    str[i+1] = str[i];
	}

	str[index] = BACKSLASH_CHAR;
    }
    
    return str;
}

/*****************************************************************
 * _datatypes() returns an array of the field datatypes.         *
 *                                                               *
 * msql-import uses this information to determine whether or     *
 * not to enclose the value to be inserted between quotes, and   *
 * msql-import will validate the data depending on the datatype. *
 *                                                               *
 * Algorithm:                                                    *
 *                                                               *
 *   FOR EACH field IN fields                                    *
 *       find its definition and add its type to the result      *
 *****************************************************************/

int *
datatypes(const char *table, const char *fields)
{
    int fieldCount = 0;
    char *tableCopy = safe_c_string_copy(table);
    int *types = (int *)malloc((IMPORT_MAX_FIELDS+1) * sizeof(int));
    m_result *result = msqlListFields(sock, tableCopy);
    m_field *field;
    const char *token;
    char *fieldsCopy;
    char **ptr;
    int done = 0;
    int found = 0;

    fieldsCopy = safe_c_string_copy(fields);
    ptr = &fieldsCopy;

    free(tableCopy);

    if (fieldsCopy != NULL)
    {
	while (!done)
	{
	    if ((token = str_parse(ptr, ",")) != NULL)
	    {
		msqlFieldSeek(result, 0);
		found = 0;
		
		while((!found) && (field = msqlFetchField(result)))
		{
		    if (!strcmp(field->name, token))
		    {
			types[fieldCount++] = field->type;
			found = 1;
		    }
		}
		
		if (!found)
		{
		    alarm_msg("Field definition not found. Exiting.");
		}
	    }
	    else
	    {
		done = 1;
	    }
	}
    }
    else
    {
	msqlFieldSeek(result, 0);

	while ((field = msqlFetchField(result)) != NULL)
	{
	    types[fieldCount++] = field->type;
	}
    }
    
    msqlFreeResult(result);

#if 0
    if (fieldsCopy!= NULL)
    {
	free(fieldsCopy);
    }
#endif
    
    return types;
}


/*****************************************************************
 * _get_record reads one row from the data file and returns it   *
 *****************************************************************/

#define GET_REC_BLOCKSIZE 8192

char *
get_record(FILE *fp, const char record_delimiter, SafeString *record)
{
    char c;
    int i = 0;

    set_string_capacity(record, GET_REC_BLOCKSIZE);
    record[0].buffer[0] = '\0';
    
    for ( ; ; )
    {
	if (((c = getc(fp)) == EOF) || (c == record_delimiter))
	{
	    break;
	}
	else
	{
	    /* add the block to the record string */
	    append_character(record, c);
	    i++;
	}
    }

    if (i)
    {
	return (char *)record[0].buffer;
    } else
    {
	return NULL;
    }
}

/*****************************************************************
 * _import_file opens the flat file, reads it one record at a    *
 * time, and imports it into the specified Mini SQL table        *
 *****************************************************************/

void
import_file(char *table,
	    char *path,
	    char *fieldDel,
	    char *rowDel,
	    const char *fields)
{
    FILE *fp;
    int recordCount = 0; /* Number of records read from the flat file */
    int *data_types = datatypes(table, fields);
    int recs = 0; /* Number of records successfully inserted */
    SafeString *record;
    
    if ((fp = fopen(path, "r")) == NULL)
    {
	perror(path);
	abort_import(-1);
    }

    record = create_string();
    query = create_string();
    set_string_capacity(query, GET_REC_BLOCKSIZE);
    
    while (get_record(fp, rowDel[0], record) != NULL)
    {
	recs += insert_record(table,
			      record[0].buffer,
			      fields,
			      data_types,
			      recordCount++);
    }

    free(fieldDel);
    free(rowDel);
    free(table);
    free(record);
    free(query);
    free(path);
    
    if (recs > 1)
    {
	fprintf(stdout, "%i rows successfully imported.  ", recs);
    }
    else
    {
	if (recs == 1)
	{
	    fprintf(stdout, "1 row was successfully imported.  ");
	}
	else
	{
	    fprintf(stdout, "No row was successfully imported.  ");
	}
    }

    if (recordCount-recs > 1)
    {
	fprintf(stdout, "%i rows were rejected.\n\n", recordCount-recs);
    }
    else
    {
	if (recordCount-recs == 1)
	{
	    fprintf(stdout, "1 row was rejected.\n\n");
	}
	else
	{
	    fprintf(stdout, "No row was rejected.\n\n");
	}
    }
}

/*****************************************************************
 * _insert_record sends the INSERT statement to the msql server  *
 * Returns 1 if record was inserted successfully, 0 otherwise.   *
 *****************************************************************/

int
insert_record(const char *table,
	      const char *record,
	      const char *fields,
	      int *types,
	      int recordCount)
{
    char **ptr = &record;
    char *token;
    char *value;
    char c = '\t';
    char delimiter[2];
    int done = 0;
    int fieldNumber = 1;
    
    sprintf(delimiter, "%c", c);
    query[0].buffer[0] = '\0';

    copy_string_buffer(query, "INSERT INTO ");
    append_string_buffer(query, table);
    
    if ((fields == NULL) || (strlen(fields) < 2))
    {
	append_string_buffer(query, " VALUES(");
    }
    else
    {
	append_string_buffer(query, " (");
	append_string_buffer(query, fields);
 	append_string_buffer(query, ") VALUES(");
    }
    
    if((record != NULL) && (token = str_parse(ptr,delimiter)))
    {
	switch(types[0])
	{
	  case CHAR_TYPE:
	    value = backslashify_special_chars(token);
	    append_character(query, '\'');
	    append_string_buffer(query, value);
	    append_string_buffer(query,"',");
	    
	    if (value != NULL)
	    {
		free(value);
	    }
	    
	    break;

	  case NULL_TYPE:
	    append_string_buffer(query, "NULL,");
	    break;

	  default:
	    append_string_buffer(query, token);
	    append_character(query,',');
	}
    }
    else
    {
	return 0;
    }

    while (!done)
    {
	if ((token = str_parse(ptr, delimiter)) != NULL)
	{
	    switch(types[fieldNumber++])
	    {
	      case CHAR_TYPE:        
		value = backslashify_special_chars(token);
		append_character(query, '\'');
		append_string_buffer(query, value);
		append_string_buffer(query,"',");
		
		if (value!=NULL)
		{
		    free(value);
		}
		break;
		
	      case NULL_TYPE:
		append_string_buffer(query, "NULL,");
		break;
		
	      default:
		append_string_buffer(query, token);
		append_character(query, ',');
	    }
	}
	else
	{
	    done = 1;
	}
    }

    /* Remove the last comma and the record delimiter ('\n') */
    query[0].buffer[strlen(query[0].buffer)-1] = '\0';
    
    append_character(query, ')');

    if (msqlQuery(sock, query[0].buffer) == -1)
    {
	fprintf(stderr, "msql-import: could not import record %i: %s\n",
		recordCount, msqlErrMsg);
	fprintf(stdout, "Query : %s\n\n", query[0].buffer);
	return 0;
    }
    
    return 1;
}

/*****************************************************************
 *  row_length verifies that table_name exists in the database,  *
 *  and returns the row length in bytes (including control bytes)*
 *****************************************************************/

int
row_length(const char *table_name)
{
    m_result *res;
    m_field *curField;
    int len = 1;
    char *tableCopy = safe_c_string_copy(table_name);
    
    /* Verify that the table_name argument is valid */

    if (table_name == NULL)
    {
	fprintf(stderr, "msql-import in row_length(), invalid table_name "
		"(NULL).  Exiting.\n");
	exit(1);
    }
    
    res = msqlListFields(sock,tableCopy);

    free(tableCopy);
    
    if (!res)
    {
	fprintf(stderr, "msql-import error : Unable to get the fields in "
		"table %s, exiting.\n", table_name);
	exit(1);
    }
    
    while((curField = msqlFetchField(res)))
    {
	len += curField->length + 1;
    }

    msqlFreeResult(res);

    if (len<3)
    {
	fprintf(stderr, "msql-import: error getting table definition. "
		"Exiting.\n");

	abort_import(1);
    }
    
    return len;
}

char *
format_delimiter(const char *str)
{
    unsigned char nl = (unsigned char)10;
    unsigned char tab = (unsigned char)9;
    char *result = (char *)NULL;
    
    if (!str)
    {
	return (char *)NULL;
    }
    
    if (!strcmp(str, "\\n"))
    {
	result = malloc(2);
	result[0] = nl;
	result[1] = '\0';
    }
    else
    {
	if (!strcmp(str, "\\t"))
	{
	    result = malloc(2);
	    result[0] = tab;
	    result[1] = '\0';
	}
	else
	{
	    result = (char *)malloc(strlen(str) + 1);
	    strcpy(result, str);
	}
    }
    return result;
}

void
printHelp(void)
{
    const char text[] =
    {
	"  msql-import loads the contents of a an ASCII delimited flat file "
	"into a\n  MiniSQL table.  It automatically performs the type "
	"conversions, and\n  validates the data.\n\n  msql-import is invoked "
	"as follows:\n\n    msql-import host base tbl field_term "
	"rec_term datafile [field [,field...]]\n\n"
	"               host: hostname of the msql server\n"
	"               base: the database name\n"
	"                tbl: the table in which to load the data\n"
	"         field_term: the character used to delimit fields within a"
	" record\n"
	"           rec_term: the character used to delimit records\n"
	"           datafile: conatins the data to be imported\n"
	"             fields: import the data in those only (optional)\n\n"
	"  Example:\n\n    msql-import zeus db table \\t \\n /tmp/file "
	"client_id,name,address\n\n"
	"  If the fields are not specified, then all fields in the table "
	"will be filled\n  with the data contained in the flat file, in "
	"order of appearance.\n\n"
    };
	
    fprintf(stdout, "\n  msql-import %s help:\n\n", MSQL_IMPORT_VERSION);
    fprintf(stdout, text);
}

void
printUsage(void)
{
    fprintf(stderr, "Usage:\n"
	    "       msql-import host base table field_terminator "
	    "record_terminator datafile [field [, field...]]\n\n"
	    "Type 'msql-import --help' for more information\n\n");
}

void
printVersion(void)
{
    fprintf(stdout, "%s\n", MSQL_IMPORT_VERSION);
}

/*****************************************************************
 * _main() connects to the server, then imports the data         *
 *****************************************************************/

void
main(int argc, char **argv)
{
    const char *buf[] ={"--help","--info","help","info","-help","-info",0};
    const char **ptr = buf;
    const char *fields = NULL;

    sock = -1;
    
    /* Validate calling arguments */

    if (argc == 2)
    {
	if (!strcmp(argv[1], "--version"))
	{
	    printVersion();
	    return;
	}
	else
	{   
	    while(ptr) {
		if (!strcmp((char *)*ptr, argv[1]))
		{
		    printHelp();
		    return;
		}
		ptr++;
	    }
	}
    }

    if (argc == 8)
    {
	fields = safe_c_string_copy(argv[7]);
    }
    else
    {
	if (argc != 7)
	{
	    printUsage();
	    return;
	}
    }

    /* Connect to the msql server */

    if (strcmp(argv[1], "localhost") == 0) {
	sock = msqlConnect(NULL);
    } else {
	sock = msqlConnect(argv[1]);
    }

    /* Set the current database */
    
    if ((sock == -1) || (msqlSelectDB(sock, argv[2]) == -1)) {
	abort_import(1);
    }    
	
    import_file(safe_c_string_copy(argv[3]),
		safe_c_string_copy(argv[6]),
		format_delimiter(safe_c_string_copy(argv[4])),
		format_delimiter(safe_c_string_copy(argv[5])),
		fields);    
}
