/*
 * Filename: safestring.c
 * Project:  SafeString
 *
 * Function: SafeStrings are ANSI-C objects that contain strings.
 *           The string contents is be accessed and manipulated
 *           through the use of C functions.
 *
 *           No more malloc errors!  This code should be useful
 *           to all C programmers, especially beginners without
 *           much experience with pointer manipulation.  SafeStrings
 *           make string manipulations easy.
 *
 * Author:   Pascal Forget <pascal@wsc.com>
 *
 * Copyright (C) 1995-1996 Pascal Forget.  All Rights Reserved.
 *
 * PASCAL FORGET MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY
 * OF THIS SOFTWARE FOR ANY PURPOSE.  IT IS SUPPLIED "AS IS"
 * WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for any purpose and without fee is
 * hereby granted, provided that the above copyright notice
 * appear in all copies of the software.
 */

#include "safestring.h"

#define SAFESTRING_VERSION "1"
#define MAX(a,b) (a>b)a:b

#define DEBUG 1

int sock;

SafeString *query;

/*****************************************************************
 * ss_append_character() appends a character to str              *
 *****************************************************************/

void
ss_append_character(SafeString *str, char c)
{
    unsigned int length = str[0].length;

    /* Grows by 64 byte blocks, no need to realloc every time */    
    ss_set_string_capacity(str, length + 64);

    str[0].buffer[length] = c;
    str[0].buffer[length+1] = '\0';
    str[0].length++;
}

/*****************************************************************
 * ss_append_string() appends a Safe String to another one       *
 *****************************************************************/

void
ss_append_string(SafeString *str, SafeString *other_str)
{
    ss_append_c_string(str, other_str[0].buffer);
}

/*****************************************************************
 * ss_append_c_string() appends a c string buffer to str         *
 *****************************************************************/

void
ss_append_c_string(SafeString *str, const char *cstring)
{
    unsigned int cstring_length;
    
    if ((cstring != NULL) && ((cstring_length = strlen(cstring)) > 0))
    {
	if (str[0].buffer != NULL)
	{
	    ss_set_string_capacity(str, str[0].length + cstring_length);
	}
	else
	{
	    ss_set_string_capacity(str, cstring_length);
	}
	    
	if (str[0].buffer != NULL)
	{
	    strcat(str[0].buffer, cstring);
	}
	else
	{
	    strcpy(str[0].buffer, cstring);
	}

	str[0].length += cstring_length;
    }
}

/*****************************************************************
 * ss_fast_append_c_string_length() appends a c string  to str   *
 * This function is faster than append_c_string_length  because  *
 * it doen't validate the calling arguments.                     *
 *****************************************************************/

void
ss_fast_append_c_string_length(SafeString *str, const char *cstring, int length)
{
    int i;
    unsigned int old_length;
    
    if (str[0].buffer != NULL)
    {
	old_length = str[0].length;
	ss_set_string_capacity(str, old_length + length);
    }
    else
    {
	ss_set_string_capacity(str, length);
	old_length = 0;
    }
    
    if (str[0].buffer != NULL)
    {

	/* the following 5 lines are faster than strcat() */
	for (i=0; i<length; i++)
	{
	    str[0].buffer[old_length + i] = cstring[i];
	}
	str[0].buffer[old_length + i] = '\0';

    }
    else
    {
	strcpy(str[0].buffer, cstring);
    }

    str[0].length += length;
}

/*******************************************************************
 * ss_append_c_string() appends the first length characters of a c *
 * string buffer to str                                            *
 *******************************************************************/

void
ss_append_c_string_length(SafeString *str, const char *cstring, int length)
{
    int real_length;
    
    if (cstring != NULL)
    {
	real_length = strlen(cstring);
	
	if (length < real_length)
	{
	    real_length = length;
	}
	
	if (str[0].buffer != NULL)
	{
	    ss_set_string_capacity(str, str[0].length + real_length);
	}
	else
	{
	    ss_set_string_capacity(str, real_length);
	}
	    
	if (str[0].buffer != NULL)
	{
	    strncat(str[0].buffer, cstring, length);
	}
	else
	{
	    strncpy(str[0].buffer, cstring, length);
	}
	
	str[0].length += real_length;	
    }
}

/*****************************************************************
 * ss_character_at_index() returns the character at index in str *
 *****************************************************************/

char
ss_character_at_index(SafeString *str, int index)
{
    if ((str != NULL) && (str[0].length > index) && (index > -1))
    {
	return str[0].buffer[index];
    }
    else
    {
	return 0;
    }
}

/*****************************************************************
 * ss_compare_strings() is the equivalent to strcmp()            *
 *****************************************************************/

int
ss_compare_strings(SafeString *str1, SafeString *str2)
{
    return strcmp(str1[0].buffer, str2[0].buffer);
}

/******************************************************************
 * ss_copy_string() puts a SafeString's value into another string *
 ******************************************************************/

void
ss_copy_string(SafeString *to_str, SafeString *from_str)
{
    if ((to_str != NULL) && (from_str != NULL))
    {
	ss_copy_c_string(to_str, from_str[0].buffer);
    }
}

/*****************************************************************
 * ss_copy_c_string() copies a c string buffer in str            *
 *****************************************************************/

void
ss_copy_c_string(SafeString *str, const char *cstring)
{
    unsigned int cstring_length;
    
    if (cstring != NULL)
    {
	cstring_length = strlen(cstring);
	
	ss_set_string_capacity(str, cstring_length);
	strcpy(str[0].buffer, cstring);
	str[0].length = cstring_length;
    }
    else
    {
	str[0].buffer = NULL;
	str[0].length = 0;
    }
}

/*****************************************************************
 * ss_create_string() allocates and initializes a SafeString     *
 *****************************************************************/

SafeString *
ss_create_string(void)
{
    SafeString *string = (SafeString *)malloc(sizeof(SafeString));
    string[0].buffer = (char *)NULL;
    string[0].length = 0;
    string[0].capacity = 0;
    return string;
}

/*****************************************************************
 * ss_create_string_from_file() creates a SafeString and fills   *
 * it with the contents of the file at filename.                 *
 *****************************************************************/

SafeString *
ss_create_string_from_file(const char *filename)
{
    FILE *fd;
    int filesize;
    char *string;
    SafeString *result;

    if (filename == NULL) return NULL;
    
    if ((fd = fopen(filename, "rw")) == NULL)
    {
	perror("error opening file.");
	return NULL;
    }

    fseek(fd, 0L, SEEK_END);

    filesize = ftell(fd);

    fseek(fd, 0L, SEEK_SET);

    string = (char *)malloc(filesize + 1);
    
    fread(string, filesize, 1, fd);

    string[filesize] = '\0';

    fclose(fd);

    result = ss_create_string_with_c_string (string);

    safe_c_string_free(string);
    
    return result;
}

/*****************************************************************
 * ss_create_string_with_capacity() creates a string and         *
 * allocates a buffer for it                                     *
 *****************************************************************/

SafeString *
ss_create_string_with_capacity(unsigned int capacity)
{
    SafeString *string = ss_create_string();
    ss_set_string_capacity(string, capacity);
    return string;
}

/******************************************************************
 * ss_create_string_with_c_string() creates and sets a SafeString *
 ******************************************************************/

SafeString *
ss_create_string_with_c_string(const char *cstring)
{
    SafeString *string = ss_create_string();
    ss_append_c_string(string, cstring);
    return string;
}

/*****************************************************************
 * ss_destroy_string() frees a SafeString and its buffer         *
 *****************************************************************/

void
ss_destroy_string(SafeString *str)
{
    if (str[0].buffer != NULL)
    {
	free(str[0].buffer);
	str[0].buffer = NULL;
    }
    free(str);
}

/******************************************************************
 * ss_duplicate_string() creates an identical copy of a SafeString *
 ******************************************************************/

SafeString *
ss_duplicate_string(SafeString *str)
{
    SafeString *string = ss_create_string();
    ss_append_c_string(string, str[0].buffer);
    return string;
}

/*****************************************************************
 * ss_insert_c_string_at_index() inserts a c string to an        *
 * existing SafeString at index.                                 *
 *****************************************************************/

void
ss_insert_c_string_at_index(SafeString *str, const char *cstring, int index)
{
    int cstring_length;
    int new_length;
    int i;
#if 0
    char *tmp;
    SafeString *new;
    char *ptr;
#endif
    
    if ((str == NULL) || (cstring == NULL)) return;

    cstring_length = strlen (cstring);

    new_length = str[0].length + cstring_length;

#if 1  /* slower but portable */
    ss_set_string_capacity (str, new_length);

    /* Make space for the cstring */
    i = new_length;

    while (i-- > index)
    {
	str[0].buffer[i] = str[0].buffer[i - cstring_length];
    }

    /* put the cstring in place */
    for (i=0; i<cstring_length; i++)
    {
	str[0].buffer[index+i] = cstring[i];
    }
#else /* faster but bcopy not portable */

    tmp = malloc (new_length + 1);
    
    bcopy (str[0].buffer, tmp, index);
    tmp[index] = '\0';
    strcat (tmp, cstring);

    ptr = str[0].buffer;
    ptr += index;
    strcat (tmp, ptr);

    ss_set_string_value (str, tmp);
    safe_c_string_free(tmp);
#endif

    str[0].length += cstring_length;
}

/*****************************************************************
 * ss_remove_count_at_index() removes count characters at the    *
 * specified offset from the beginning of the string.            *
 *****************************************************************/

void
ss_remove_count_at_index(SafeString *str, int count, int index)
{
    int i, len;

    if ((str != NULL) && (count > -1) && (index > -1))
    {
	len = ss_string_length(str);

	if ((index + count) > len)
	{
	    count = len;
	}

	for (i=index; i<(len-index); i++)
	{
	    str[0].buffer[i] = str[0].buffer[i+1];
	}

	str[0].buffer[len-count] = '\0';
	str[0].length -= count;
    }
}

/*******************************************************************
 * ss_save_string_to_file() writes the SafeString to an ASCII file *
 *******************************************************************/

void
ss_save_string_to_file(SafeString *str, const char *filename)
{
    FILE *fd;

    fd = fopen(filename, "w+");

    if (fd == NULL)
    {
	perror ("error opening file");
	return;
    }
    
    fwrite(str[0].buffer, str[0].length, 1, fd);
    fclose(fd);
}

/*****************************************************************
 * ss_string_length() returns the number of characters in str    *
 * forces an update of the length of the string object           *
 *****************************************************************/

int
ss_string_length(SafeString *str)
{
    if ((str != NULL) && (str[0].buffer != NULL))
    {
	str[0].length = strlen(str[0].buffer);
	return str[0].length;
    }
    else
    {
	str[0].length = 0;
	return 0;
    }
}

/*****************************************************************
 * ss_strings_are_equal() returns 1 if the strings are the same  *
 *****************************************************************/

int
ss_strings_are_equal(SafeString *str1, SafeString *str2)
{
    /* Trivial case */
    if (str1 == str2) return 1;

    if (strcmp(str1[0].buffer, str2[0].buffer) == 0)
    {
	return 1;
    }
    else
    {
	return 0;
    }
}

/*****************************************************************
 * ss_set_string_capacity() allocates a bigger buffer if needed  *
 *****************************************************************/

void
ss_set_string_capacity(SafeString *str, unsigned int cap)
{
    if (str[0].capacity < cap)
    {
	if (str[0].buffer != NULL)
	{
	    str[0].buffer = (char *)realloc(str[0].buffer, cap+1);
	}
	else
	{
	    str[0].buffer = (char *)malloc(cap+1);
	    str[0].buffer[0] = '\0';
	    str[0].length = 0;
	}
	str[0].capacity = cap;
    }
}



/* String values */

const char *
ss_string_value(SafeString *str)
{
    return str[0].buffer;
}

int
ss_int_value(SafeString *str)
{
    if (str[0].buffer != NULL)
    {
	return atoi(str[0].buffer);
    }
    else
    {
	return 0;
    }
}

float
ss_float_value(SafeString *str)
{
    if (str[0].buffer != NULL)
    {
	return atof(str[0].buffer);
    }
    else
    {
	return 0;
    }
}

double
ss_double_value(SafeString *str)
{
    if (str[0].buffer != NULL)
    {
	return atof(str[0].buffer);
    }
    else
    {
	return 0;
    }
}

void
ss_set_string_value(SafeString *str, const char *new_value)
{
    ss_copy_c_string(str, new_value);
}

void
ss_set_int_value(SafeString *str, int new_value)
{
    char buf[64];

    sprintf(buf, "%i", new_value);

    ss_copy_c_string(str, buf);
}

void
ss_set_float_value(SafeString *str, float new_value)
{
    char buf[64];

    sprintf(buf, "%f", new_value);

    ss_copy_c_string(str, buf);
}


void
ss_set_double_value(SafeString *str, double new_value)
{
    char buf[64];

    sprintf(buf, "%f", new_value);

    ss_copy_c_string(str, buf);
}




/* Other goodies */

void
ss_capitalize_string(SafeString *str) /* "new york city" -> "New York City" */
{
    int i, str_len;
    char *buf;

    if (str != NULL)
    {
	str_len = ss_string_length(str);
	
	if (str_len > 0)
	{
	    buf = str[0].buffer;
	    
	    for (i=0; i<str_len; i++)
	    {
		if (((i==0) || (buf[i-1] == ' ')) && (islower(buf[i])))
		{
		    buf[i] = (int)buf[i] - 'a' + 'A';
		}
	    }
	}
    }
}

int
ss_has_case_prefix(SafeString *str, SafeString *prefix) /* case insensitive */
{
    if ((str == NULL) || (prefix == NULL))
    {
	return 0;
    } else
    {
	if (strncasecmp(ss_string_value(prefix),
			ss_string_value(str),
			ss_string_length(str)) == 0)
	{
	    return 1;
	}
	else
	{
	    return 0;
	}
    }
}

int
ss_has_prefix(SafeString *str, SafeString *prefix)
{
    if ((str == NULL) || (prefix == NULL))
    {
	return 0;
    }
    else
    {
	if (strncmp(ss_string_value(prefix),
		    ss_string_value(str),
		    ss_string_length(str)) == 0)
	{
	    return 1;
	}
	else
	{
	    return 0;
	}
    }
}

int
ss_has_suffix(SafeString *str, SafeString *suffix)
{
    const char *extStr;
    const char *string;
    int string_len;
    int suffix_len;
    
    if ((str == NULL) || (suffix == NULL))
    {
	return 0;
    }
    else
    {
	extStr = ss_string_value(suffix);
	string = ss_string_value(str);
	string_len = ss_string_length(str);
	suffix_len = ss_string_length(suffix);
	
	return (strcmp(string + string_len - suffix_len, extStr) == 0);
    }
}

SafeString *
ss_last_path_component(SafeString *str)
{
    const char *myInternalBuffer = ss_string_value(str);
    char *buffer = NULL;
    int length = ss_string_length(str);
    SafeString *lastPathComponent;
    const char *lastString;

    if (myInternalBuffer != NULL)
    {
	buffer = (char *)malloc(length + 1);
	strcpy(buffer, myInternalBuffer);
    }
    
    if ((buffer == NULL) ||
	(length < 1) ||
	((length == 1) && (buffer[0] == '/')))
    {
	return ss_create_string_with_c_string("");
    }
    else
    {	
	/* get rid of a trailing slash if there is one */
	if (buffer[length-1] == '/')
	{
	    buffer[length-1] = '\0';
	}
    
	/* find the last slash in the string buffer */
	lastString = strrchr(buffer, '/');

	/* if the string does not contain any slash, return it as is */
	if (lastString == NULL)
	{
	    safe_c_string_free(buffer);
	    return ss_duplicate_string(str);
	}
	else if (strlen(lastString) > 1)
	{
	    lastPathComponent = ss_create_string_with_c_string(lastString);

	    /* Remove leading slash if there is one */
	    if (lastString[0] == '/') {
		ss_remove_count_at_index(lastPathComponent, 1, 0);
	    }

	    lastString = NULL;
	    safe_c_string_free(buffer);
	    return lastPathComponent;
	}
    }    

    safe_c_string_free(buffer);
    
    return ss_create_string_with_c_string("");
}

SafeString *
ss_path_extension(SafeString *str)
{
    SafeString *last = ss_last_path_component(str);
    const char *buffer = ss_string_value(last);
    const char *extension;
    int length;
    SafeString *result;

    if ((buffer == NULL) ||
	((length = strlen(buffer)) < 1) ||
	((length == 1) && (buffer[0] == '.')) ||
	(buffer[length-1] == '.') ||
	((extension = strrchr(buffer, '.')) == NULL))
    {
	ss_destroy_string(last);
	return ss_create_string_with_c_string("");
    }
    else
    {
	result = ss_create_string_with_c_string(extension+1);
	ss_destroy_string(last);
	return result;
    }
}

void
ss_remove_trailing_zeroes(SafeString *str)
{
    int len = ss_string_length(str);
    int pos;
    const char *strBuf;
    
    if (len > 0)
    {
	pos = len - 1;

	strBuf = ss_string_value(str);
	
	while ((pos >= 0) && ((char)(*(strBuf+pos)) == '0'))
	{
	    pos--;
	}

	if (pos < (len-1))
	{
	    ss_remove_count_at_index(str, len-pos+1, pos+1);
	}
    }
}



/*****************************************************************
 * _fast_c_string_length_compare() is a substitute for strncmp() *
 * that is more than twice as fast.  It just tells you if the    *
 * two c strings are equal or not, and it doesn't validate the   *
 * arguments.                                                    *
 *****************************************************************/
int
ss_fast_c_string_length_compare(const char *s1, const char *s2, int length)
{
    int i;

    /* Trivial equality */
    if (s1 == s2) return 0;
    
    for (i=0; i<length; i++)
    {
	if (s1[i] != s2[i])
	{
	    return 1;
	}
    }

    return 0; /* equality */
}

/******************************************************************
 * ss_replace_substring() replaces all occurences of old with new *
 ******************************************************************/

void
ss_replace_substring(SafeString *str, const char *old, const char *new)
{
    char first = old[0];
    char *ptr;
    SafeString *copy;
    int old_length, new_length;
    unsigned int characters_left_to_process;
    unsigned int buf_length;
    
    if (old == NULL)
    {
	return;
    }

    old_length = strlen (old);
    new_length = 0;

    if (new != NULL)
    {
	new_length = strlen(new);
    }

    copy = ss_create_string_with_c_string(str[0].buffer);

    characters_left_to_process = copy[0].length;
    
    ss_set_string_value (str, "");    

    ptr = copy[0].buffer;
    
    while (ptr[0] != '\0')
    {
	/* advance to the next occurence of the first character of 'old' */
	while ((ptr[0] != first) && (ptr[0] != '\0'))
	{
	    /* Append one character to str */	    
	    buf_length = str[0].length++;
	    str[0].buffer[buf_length] = ptr[0];
	    str[0].buffer[buf_length+1] = '\0';

	    characters_left_to_process--;
	    ptr++;
	}

	if (characters_left_to_process >= old_length)
	{
	    /* Check if the token found is equal to 'old' */

	    /* if (!strncmp (old, ptr, old_length)) */
	    if (ss_fast_c_string_length_compare(old, ptr, old_length) == 0)
	    {
		if (new_length > 0)
		{
		    ss_fast_append_c_string_length (str, new, new_length);
		}
		
		ptr += old_length;
		characters_left_to_process -= old_length;
	    }
	    else
	    {
		/* Append one character to str */
		buf_length = str[0].length++;
		str[0].buffer[buf_length] = ptr[0];
		str[0].buffer[buf_length+1] = '\0';

		characters_left_to_process--;
		ptr++;
	    }
	    
	} else {
	    /* Append one character to str */
	    buf_length = str[0].length++;
	    str[0].buffer[buf_length] = ptr[0];
	    str[0].buffer[buf_length+1] = '\0';

	    /* add the characters left to process */
	    ss_append_c_string (str, ptr);

	    characters_left_to_process--;
	    ptr++;
	    break;
	}
    }
    
    ss_destroy_string(copy);
}

SafeString *
ss_string_by_deleting_last_path_component(SafeString *str)
{
    char *str_buf;
    int i, len;
    SafeString *result;

    len = ss_string_length(str);

    str_buf = (char *)malloc(len + 1);
    strcpy(str_buf, ss_string_value(str));
    
    if (len > 1)
    {
	str_buf[len-1] = '\0';
    
	for (i = len-2; i > 0; i--)
	{
	    if (str_buf[i] == '/')
	    {
		str_buf[i] = '\0';
		break;
	    }
	    str_buf[i] = '\0';
	}
    }

    if ((str_buf[0] != '/') && (str_buf[0] != '\0') && (str_buf[1] == '\0'))
    {
	safe_c_string_free(str_buf);
	return ss_create_string_with_c_string("");
    }

    result = ss_create_string_with_c_string(str_buf);
    safe_c_string_free(str_buf);
    return result;
}

SafeString *
ss_string_by_deleting_path_extension(SafeString *str)
{
    char *str_buf;
    char *ptr;
    int len;

    /* Work on a copy of the buffer */
    
    str_buf = (char *)malloc(ss_string_length(str) + 1);
    strcpy(str_buf, ss_string_value(str));

    if (str_buf != NULL)
    {
	if (ptr = strrchr(str_buf, '.'))
	{
	    ptr[0] = '\0';
	}
	
	if (((len = strlen(str_buf)) > 1) && (str_buf[len - 1] == '/'))
	{
	    str_buf[len-1] = '\0';
	}
	
	return ss_create_string_with_c_string(str_buf);
	
    }
    else
    {
	return NULL;
    }    
}


/* Trim leading and trailing blanks */

void
ss_trim_string(SafeString *str)
{
    int i, j, len;
    char *buf;
    char *ptr;
    
    if ((len = ss_string_length(str)) < 1)
    {
	return;
    }

    /* Work with a copy of the string buffer */
    
    buf = (char *)malloc(len + 1);
    strcpy(buf, ss_string_value(str));
    
    if (buf != NULL)
    {
	ptr = buf + strlen(buf) - 1;

	while (*ptr == ' ')
	{
	    *ptr = '\0';
	    ptr--;
	}

        /* get position of rightmost blank */
	for(i=0; i<len; i++)
	{
	    if (buf[i] != ' ')
	    {
		break;
	    }
	}

	if (i)
	{
	    /* move string to the left to eliminate leading blanks */
	    for (j=i; j<len; j++)
	    {
		buf[j-i] = buf[j];
	    }
	    buf[len-i] = '\0';
	}

	ss_copy_c_string(str, buf);
	safe_c_string_free(buf);
    }
}

/*****************************************************************
 * _safe_c_string_copy() makes a copy of a string and returns    *
 * the new string.                                               *
 *                                                               *
 * This function replaces strdup() because:                      *
 *      Not all Unixes implement strdup                          *
 *      Some strdup implementations crash on a NULL argument     *
 *****************************************************************/

char *
safe_c_string_copy(const char *s)
{
    char *s1;
    
    if (s == NULL)
    {
	return(NULL);
    }
    
    s1 = (char *)malloc(strlen(s) + 1);
    
    if (s1 == NULL)
    {
	return(NULL);
    }
    
    (void)strcpy(s1,s);
    
    return(s1);
}

void
safe_c_string_free(char *s)
{
    if (s != NULL)
    {
	free(s);
    }
}
