/*    imaginary.sql.iMsqlStatement
 *    from the Imaginary Java Class Library
 *    Copyright (c) 1996 George Reese
 *    created by George Reese (borg@imaginary.com) 960311
 *    mSQL implementation of the JDBC Statement interface
 */

package imaginary.sql;

import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.ResultSet;
import imaginary.sql.msql.MsqlResult;

/**
 * This class is the implementation of the JDBC draft specification
 * Statement interface.  Statement objects are used for executing SQL
 * statements.
 *
 * @see java.sql.Statement
 * @version 1.5, 96/06/05
 * @author George Reese
 */
public class iMsqlStatement implements java.sql.Statement {
  /**
   * A connection object to direct information to.
   */
  private iMsqlConnection connection;
  /**
   * A result set returned from this query or whatever.
   */
  private iMsqlResultSet result;
  /**
   * The max field size for mSQL
   * NOTE: I have no idea what this should default to
   */
  private int max_field_size = 0;
  /**
   * The max rows supported by mSQL
   */
  private int max_rows = 4096;
  /**
   * The number of seconds the driver will allow for a SQL statement to
   * execute before giving up.  The default is to wait forever (0).
   */
  private int timeout = 0;
  
  /**
   * Constructs a new iMsqlStatement object.
   * @param conn the iMsqlConnection object
   */
  public iMsqlStatement(iMsqlConnection conn) {
    connection = conn;
  }
  
  /**
   * JDBC draft specification method to execute the specified SQL
   * query and gives back a result set.
   * @see java.sql.Statement#executeQuery
   * @exception SQLException raised for any MsqlException
   * @param sql the SQL statement string
   * @return the result set from the query
   */
  public ResultSet executeQuery(String sql)
       throws SQLException {
    result = null;
    return new iMsqlResultSet(connection.executeMsql(sql));
  }

  /**
   * JDBC 0.54 specification method for executing simple UPDATE, INSERT,
   * DELETE, etc. statements which do not return ResultSet's.  The
   * return value is not particularly meaningful in mSQL since you cannot
   * get information on the number of rows affcted by such an action
   * in mSQL.
   * @see java.sql.Statement#executeUpdate
   * @exception java.sql.SQLException thrown when an error occurs executing
   * the SQL
   * @return 0
   */
  public int executeUpdate(String sql) throws SQLException {
    connection.executeMsql(sql);
    return 0;
  }
  
  /**
   * JDBC draft specification method to execute an SQL statement and return
   * true if a result set was gotten, false otherwise.  Note that
   * proper mSQL use of this method should always result in a
   * false return value.
   * @see java.sql.Statement#execute
   * @exception SQLException raised for any MsqlException encountered
   * @param sql the SQL to be executed
   * @return true if there is a result set available
   */
  public boolean execute(String sql) throws SQLException {
    MsqlResult r;
    
    r = connection.executeMsql(sql);
    if( r == null ) {
      result = null;
    }
    else {
      result = new iMsqlResultSet(r);
    }
    return (result != null);
  }

  /**
   * JDBC draft specification method to close off any open result sets
   * for this Statement.  This is a non-issue with mSQL, but portable
   * code should still be calling it.
   * @see java.sql.Statement#close
   */
  public void close() throws SQLException {
  }
  
  /**
   * JDBC 0.54 specification method for obtaining a result set frm a
   * statement object.
   * @see java.sql.Statement#getResultSet
   * @return null if no result set is available, otherwise a result set
   */
  public ResultSet getResultSet() throws SQLException {
    ResultSet r;

    r = result;    // This needs some changing if mSQL ever supports multiple 
    result = null; // result sets
    return r;
  }

  /**
   * Returns -1 always for mSQL.
   * @see java.sql.Statement#getUpdateCount
   * @return -1
   */
  public int getUpdateCount() throws SQLException {
    return -1;
  }

  /**
   * JDBC 0.54 specification for determining if any more result sets
   * are left from a database statement.  Should always return false for mSQL.
   * @see java.sql.Statement#getMoreResults
   * @return true if rows are to be gotten
   */
  public boolean getMoreResults() throws SQLException {
    return (result != null);
  }

  /**
   * JDBC draft specification method for determining the number of
   * results sets left to process for multiple result set processing.
   * Since mSQL does not support multiple result sets, this method always
   * returns -1.
   * @see java.sql.Statement#getMoreCounts
   * @see getMoreResults
   * @return -1
   */
  public int getMoreCounts() throws SQLException {
    return -1;
  }

  /**
   * JDBC draft specification method for getting the max field size
   * supported by mSQL.
   * @see java.sql.Statement#getMaxFieldSize
   * @return the value of max field size
   */
  public int getMaxFieldSize() throws SQLException {
    return max_field_size;
  }

  /**
   * JDBC draft specification method for setting the max field size.
   * @see java.sql.Statement#setMaxFieldSize
   * @param max the maximum field size
   */
  public void setMaxFieldSize(int max) throws SQLException {
    max_field_size = max;
  }

  /**
   * JDBC draft specification method for getting the max rows supported by
   * mSQL.
   * @see java.sql.Statement#getMaxRows
   * @return the maximum rows supported by mSQL
   */
  public int getMaxRows() throws SQLException {
    return max_rows;
  }

  /**
   * JDBC draft specification method for setting the max rows.
   * @see java.sql.Statement.setMaxRows
   * @param max the max rows
   */
  public void setMaxRows(int max) throws SQLException {
    max_rows = max;
  }

  /**
   * JDBC draft specification method for escape processing.
   * This boggles me.
   * @see java.sql.Statement#setEscapeProcessing
   * @param enable this does nothing right now
   */
  public void setEscapeProcessing(boolean enable)
       throws SQLException {
    throw new SQLException("mSQL does not support escape processing.");
  }

  /**
   * JDBC draft specification method for getting the query timeout, which
   * is the number of seconds the driver waits before giving up on the
   * query.
   * @see java.sql.Statement#getQueryTimeout
   * @see setQueryTimeout
   * @return the timeout value for this statement
   */
  public int getQueryTimeout() throws SQLException {
    return timeout;
  }

  /**
   * JDBC draft specification method for setting the query timeout.
   * @see java.sql.Statement#setQueryTimeout
   * @see getQueryTimeout
   * @param x the new query timeout value
   */
  public void setQueryTimeout(int x) throws SQLException {
    timeout = x;
  }

  /**
   * JDBC draft specification method for allowing one thread to cancel this
   * statement which may be running in another thread.  Right now, there
   * is nothing to cancel with mSQL JDBC.  Maybe I will make it do something
   * if I get actual processing in this class!
   * @see java.sql.Statement#cancel
   */
  public void cancel() {
  }
  
  /**
   * JDBC draft specification for getting the chain of warnings for this
   * statement.
   * @see java.sql.Statement#getWarnings
   * @return the chain of warnings
   */
  public final SQLWarning getWarnings() throws SQLException {
    return null;
  }

  /**
   * JDBC draft specification for clearing the warning chain.
   * @see java.sql.Statement#clearWarnings
   */
  public void clearWarnings() throws SQLException {
  }

  /**
   * JDBC 0.65 specification for setting cursor names.
   * mSQL does not support cursors
   */
  public void setCursorName(String unused) throws SQLException {
    throw new SQLException("mSQL does not support cursors.");
  }
}
