//
// Msql client library for JAVA
// Copyright (C) 1995 Darryl Collins <darryl@minmet.uq.oz.au>
// This software is provided "as is" without any expressed or implied warranty.
//
// This package implements a client for the mSQL server designed and
// implemented by David Hughes <bambi@hughes.com.au>
//
// $Log: Msql.java,v $
// Revision 1.1.1  1996/02/15 06:11:42  root
// Initial revision
//
//

package imaginary.sql.msql;

import java.awt.*;
import java.lang.*;
import java.io.*;
import java.net.*;
import java.util.*;


/**
 * Msql Class - implements a connection to a remote server
 * and a JAVA variant of the Msql API as defined by David Hughes
 *
 * @version 1.1.1
 * @author <a href="http://www.minmet.uq.oz.au/~darryl">Darryl Collins</a>
 */
public class Msql {

	// Communication variables
	private Socket s;
	private DataInputStream in;
	private DataOutputStream out;

	// User name and version info
	private String userName;
	private String strVersion;

	// Temporary storage for reply packets
	private String strReply;

	// TCP port
	private static final int rootPort = 1112;
	private static final int mortalPort = 4333;
	private int msqlPort = rootPort;

/**
 * Constructor
 */
	public Msql() {
	}

/**
 * Msql Constructor - establishes connection to named host
 *
 * @param msqlServer	the name of the msql server.
 */
	public Msql(String msqlServer) throws MsqlException {

		if(msqlServer != null) 	
			Connect(msqlServer);
	}

/**
 * Connect - establishes connection to named host
 *
 * @param msqlServer	the name of the msql server.
 * @param userName	user name the applet will assume.
 * @exception 		MsqlException on mSQL protocol failure.
 * @exception 		MsqlException on Unknown Host.
 * @exception 		MsqlException on Unknown Service.
 * @exception 		MsqlException on TCP-IP Protocol Error.
 * @exception 		MsqlException on Socket Error.
 * @exception 		MsqlException on IO Error.
 * @exception 		MsqlException on Security Exception.
 *
 * 	1. Open socket.
 *	2. Establish IO Streams on socket.
 *	3. Obtain version info.
 *	4. Supply user name.
 *	5. Retrieve status.
 */
	public void Connect(String msqlServer, String userName) throws MsqlException {

		try {
			// Open connection 
			s = new Socket(msqlServer, msqlPort);

			// Get Streams
			in = new DataInputStream(s.getInputStream());
			out = new DataOutputStream(s.getOutputStream());

			// Read Version Info 
			strVersion = recvString();
			
			// Send Username
			sendString(this.userName = userName);

			// Get reply
			strReply = recvString();

			if(!strReply.startsWith("-100:",0))
				throw new MsqlException("Msql connection failed");

		} catch(UnknownHostException e) {
			throw new MsqlException(
				"Unknown Host Exception during connection"
			);
		} catch(UnknownServiceException e) {
			throw new MsqlException(
				"Unknown Service Exception during connection"
			);
		} catch(ProtocolException e) {
			throw new MsqlException(
				"Protocol Exception during connection"
			);
		} catch(SocketException e) {
			throw new MsqlException(
				"Socket Exception during connection"
			);
		} catch(IOException e) {
			throw new MsqlException(
				"IO Exception during connection"
			);
		} catch(SecurityException e) {
			throw new MsqlException(
				"Security Exception during connection"
			);
		}
	}

/**
 * Connect - connect using a specified port (root=1112 or mortal=4333).
 *
 * @param msqlServer	the name of the msql server.
 * @param userName	user name the applet will assume.
 * @param rootServer	true if server is run as root.
 * @exception		MsqlException as above.
 */
	public void Connect(String msqlServer, String userName, boolean rootServer) 
							throws MsqlException {

		if(rootServer)
			msqlPort = rootPort;
		else
			msqlPort = mortalPort;

		Connect(msqlServer, userName);
	}

/**
 * Connect - establishes connection to named host
 *
 * @param msqlServer	the name of the msql server.
 * @exception		MsqlException as above.
 */
	public void Connect(String msqlServer) throws MsqlException {

		// try {
		// 	String userName = System.getProperty("user.name"));
		// 	Connect(msqlServer, userName);
		// } catch(SecurityException e) {
			Connect(msqlServer, "nobody");
		// }
	}

/**
 * Connect - connect using a specified port (root=1112 or mortal=4333).
 *
 * @param msqlServer	the name of the msql server.
 * @param rootServer	true if server is run as root.
 * @exception		MsqlException as above.
 */
	public void Connect(String msqlServer, boolean rootPort) throws MsqlException {

		// try {
		// 	String userName = System.getProperty("user.name"));
		// 	Connect(msqlServer, userName, rootPort);
		// } catch(SecurityException e) {
			Connect(msqlServer, "nobody", rootPort);
		// }
	}

/**
 * Sends a String as an mSQL packet.  The trailing linefeed
 * is added automatically by the routine.
 *
 * @param str		the string to send to the server.
 * @exception		IOException
 * @exception		SecurityException
 */
	private void sendString(String str) throws IOException, SecurityException {
		
		int i,j,k;

		j = k = str.length()+1;

		out.writeByte(k&0xff); k >>= 8;
		out.writeByte(k&0xff); k >>= 8;
		out.writeByte(k&0xff); k >>= 8;
		out.writeByte(k&0xff);
		
		for(i=0; i<j-1; i++) {
			byte b = (byte)str.charAt(i);
			out.writeByte(b);
		}

		out.writeByte((byte)'\n');
	}

/**
 * Receives a packet and converts it to a String
 * (trailing linefeed is dropped)
 *
 * @return 		the string retrieved from the socket.
 * @exception		IOException
 * @exception		SecurityException
 */
	private String recvString() throws IOException, SecurityException {

		StringBuffer str = new StringBuffer();

		int i,j,k=0;

		j = in.readByte(); k += (j&0xff);
		j = in.readByte(); k += (j&0xff)<<8;
		j = in.readByte(); k += (j&0xff)<<16;
		j = in.readByte(); k += (j&0xff)<<24;

		for(i=0; i<k; i++) {
			char c = (char)in.readByte();
			if(i < k) str.append(c);
		}

		return str.toString();
	}

/**
 * Close - close the connection to the server.
 *
 * @exception		MsqlException
 */
	public void Close() throws MsqlException {

		try {
			sendString("1");
			s.close();
		} catch(IOException e) {
			throw new MsqlException("QUIT: "+e.getMessage());
		}
	}

/**
 * SelectDB - Select the database to work with
 * 
 * @param db		the name of the database to select.
 * @exception		MsqlException if select fails. 
 */
	public void SelectDB(String db) throws MsqlException {

		try {
			sendString("2 "+db+"\n");
			strReply = recvString();
			if(strReply.startsWith("-1:",0))
				throw new MsqlException("DB_INIT: "+strReply.substring(3));

		} catch(IOException e) {
			throw new MsqlException("DB_INIT: "+e.getMessage());
		}
	}

/**
 * Send an SQL Query to the server.  
 *
 * @param s		the Query string.
 * @return		null when used with INSERT/UPDATE or similar queries which do not return data
 * @return		instance of MsqlResult when used with SELECT.
 * @exception		MsqlException when query or communication fails.
 */
	public MsqlResult Query(String s) throws MsqlException {

		int m,n;
		MsqlData table, fields;
		MsqlResult results;

		try {
			// Send the command
			sendString("3 "+s+"\n");
			
			// Retrieve the reply
			strReply = recvString();

			// Check for errors
			if(!strReply.startsWith("1:"))
				throw new MsqlException("QUERY: "+strReply.substring(2));
			// Determine number of fields
			m = strReply.indexOf(':',2);
			if(m == -1)
				return null;

			try {
				n=Integer.parseInt(strReply.substring(2,m));
			} catch(NumberFormatException e) {
				throw new MsqlException("Invalid format for number fo fields in reply");
			}

			// Receive the row data
			table = recvData(n);

			// Receive the field data
			fields = recvData(6);

			results = new MsqlResult(table, fields);
		} catch(IOException e) {
			throw new MsqlException("QUERY: "+e.getMessage());
		}

		return results;
	}

/**
 * List Databases available on server. 
 *
 * @return		MsqlData structure cointaining names of databases.
 * @exception		MsqlException if ListDB or communication fails.
 */
	public MsqlData _ListDBs() throws MsqlException {

		MsqlData result;

		try {
			sendString("4\n");
			result = recvData(1);
		} catch(IOException e) {
			throw new MsqlException("DB_LIST: "+e.getMessage());
		}

		return result;
	}

/**
 * List Databases available on server. 
 *
 * @return		array of String cointaining names of databases.
 * @exception		MsqlException if ListDB or communication fails.
 */
	public String[] ListDBs() throws MsqlException {

		MsqlData result;
		String dbList[];

		try {
			result = _ListDBs();
		} catch(IOException e) {
			throw new MsqlException("DB_LIST: "+e.getMessage());
		}

		int n = result.NumRows();

		if(n != 0) {
			dbList = new String[n];
			for(int i=0; i<n; i++)
				dbList[i] = new String((result.FetchRow())[0]);
			return dbList;
		}

		return null;
	}

/**
 * List tables in the selected database.
 *
 * @return		MsqlData structure containing names of tables.
 * @exception		MsqlException if ListTables or communication fails.
 */
	public MsqlData _ListTables() throws MsqlException {

		MsqlData result;

		try {
			sendString("5\n");
			result = recvData(1);
		} catch(IOException e) {
			throw new MsqlException("TABLE_LIST: "+e.getMessage());
		}

		return result;
	}
/**
 * List tables in the selected database.
 *
 * @return		array of String conttaining names of tables.
 * @exception		MsqlException if ListTables or communication fails.
 */
	public String[] ListTables() throws MsqlException {

		MsqlData result;
		String tableList[];

		try {
			result = _ListTables();
		} catch(IOException e) {
			throw new MsqlException("TABLE_LIST: "+e.getMessage());
		}
 
		int n = result.NumRows();

		if(n != 0) {
			tableList = new String[n];
			for(int i=0; i<n; i++) 
				tableList[i] = new String((result.FetchRow())[0]);
			return tableList;
		}
				
		return null;
	}

/**
 * List Fields in a given table.
 *
 * @param s		name of the table for which information requested.
 * @return		MsqlData structure containing field information.
 * @exception		MsqlException if ListTables or communication fails.
 */
	public MsqlData _ListFields(String s) throws MsqlException {

		MsqlData result;

		try {
			sendString("6 "+s+"\n");
			result = recvData(6);
		} catch(IOException e) {
			throw new MsqlException("FIELD_LIST: "+e.getMessage());
		}

		return result;
	}

/**
 * List Fields in a given table.
 *
 * @param s		name of the table for which information requested.
 * @return		array of MsqlFieldDesc containing field information.
 * @exception		MsqlException if ListTables or communication fails.
 */
	public MsqlFieldDesc[] ListFields(String s) throws MsqlException {

		MsqlData result;
		MsqlFieldDesc fieldList[];

		try {
			result = _ListFields(s);
		} catch(IOException e) {
			throw new MsqlException("FIELD_LIST: "+e.getMessage());
		}

		int n = result.NumRows();

		if(n != 0) {
			fieldList = new MsqlFieldDesc[n];
			for(int i=0; i<n; i++) 
				fieldList[i] = new MsqlFieldDesc(result.FetchRow());
			return fieldList;
		}

		return null;
	}
			

/**
 * Create a new database.  Included for completeness only since
 * command can only be executed if issued from UNIX domain socket.
 */
	public void CreateDB(String s) throws MsqlException {

		try {
			sendString("7 "+s+"\n");
			strReply = recvString();
			if(strReply.startsWith("-1:",0))
				throw new MsqlException("CREATE_DB: "+strReply.substring(3));

		} catch(IOException e) {
			throw new MsqlException("CREATE_DB: "+e.getMessage());
		}
	}

/**
 * Drop a  database.  Included for completeness only since
 * command can only be executed if issued from UNIX domain socket.
 */
	public void DropDB(String s) throws MsqlException {

		try {
			sendString("8 "+s+"\n");
			strReply = recvString();
			if(strReply.startsWith("-1:",0))
				throw new MsqlException("DROP_DB: "+strReply.substring(3));

		} catch(IOException e) {
			throw new MsqlException("DROP_DB: "+e.getMessage());
		}
	}

/**
 * Reload access control list.  Included for completeness only since
 * command can only be executed if issued from UNIX domain socket.
 */
	public void ReloadACL(String s) throws MsqlException {

		try {
			sendString("9\n");
			strReply = recvString();
			if(strReply.startsWith("-1:",0))
				throw new MsqlException("RELOAD_ACL: "+strReply.substring(3));

		} catch(IOException e) {
			throw new MsqlException("RELOAD_ACL: "+e.getMessage());
		}
	}

/**
 * Shutdown server.  Included for completeness only since
 * command can only be executed if issued from UNIX domain socket.
 */
	public void Shutdown() throws MsqlException {

		try {
			sendString("10\n");
			strReply = recvString();
			if(strReply.startsWith("-1:",0))
				throw new MsqlException("SHUTDOWN: "+strReply.substring(3));

		} catch(IOException e) {
			throw new MsqlException("SHUTDOWN: "+e.getMessage());
		}
	}

/**
 * Retrieve the results of a query.  Returns an instance of MsqlData.
 * Retrieves pakcets from server until an error status (-1:) or
 * success status (-100:) is received.
 */
	private MsqlData recvData(int n) throws MsqlException {

		MsqlData result = new MsqlData(n);

		boolean finished = false;

		while(!finished) {
			try {
				strReply = recvString();
			} catch(IOException e) {
				throw new MsqlException(e.getMessage());
			}

			if(strReply.startsWith("-1:",0))
				throw new MsqlException(strReply.substring(3));

			if(strReply.startsWith("-100:",0))
				finished = true;
			else
				result.addRow(strReply);
		}

		return result;
	}
}

/**
 * Object which contains a set of results returned from the server.  The results
 * might be a view which is the result of a SELECT query, or it may be a table
 * of information about a given database, table or field.  Note that the 
 * results of a SELECT query are never returned in a standalone instance of
 * MsqlData, but are always combined with a second instance of MsqlData
 * containing Field information.  The resulting combined data structure
 * is an instance of MsqlResult.
 */
class MsqlData {

	// Maximum number of rows it can cope with
	static final int maxRows = 4096;

	
	// Number of rows and columns in table
	private int nRows=0, nCols=0;

	// Data storage
	private MsqlRow strData[];

	// Cursor for data access
	private int nCursor=0; 

/**
 * Constructor - accepts number of fields as argument
 * 
 * @param n		the number of fields this instance will contain.
 */
	public MsqlData(int n) {
		nCols = n;
		strData = new MsqlRow[maxRows];
	}

/**
 * Constructor - copies an existing object
 * 
 * @param d		another MsqlData to be duplicated.
 */
	public MsqlData(MsqlData d) {

		nRows = d.nRows;
		nCols = d.nCols;

		strData = new MsqlRow[maxRows];

		for(int i=0; i<nRows; i++)
			strData[i] = new MsqlRow(d.strData[i]);
	}

/** 
 * Add a new row to the table.  Accepts a raw string from the 
 * server as an argument and parses it to identify column contents
 *
 * @param s		a string obtained from the server which represents a row of the incoming table.
 */
	public void addRow(String s) {

		int i,j,k,l;

		if(nRows >= maxRows)
			return;

		strData[nRows] = new MsqlRow(nCols);

		for(i=j=0; i<nCols; i++) {
			k = s.indexOf(':',j);
			l = Integer.parseInt(s.substring(j,k));

			if(l == -2) {
				strData[nRows].addField(i, null);
				j = k+1;
			} else {
				strData[nRows].addField(i, s.substring(k+1,k+1+l));
				j = k+l+1;
			}
		}
	
		nRows++;
	}

/**
 * Return a specific row from the table
 * 
 * @param i		the index of the row to be retrieved.
 * @return		an array of Strings representing the columns of the row.
 */
	public String [] getRow(int i) {
		
		return strData[i].rowData();
	}

/**
 * Fetch the next row from the table
 *
 * @return		an array of Strings representing the columns of the row.
 */
	public String [] FetchRow() {

		if(nCursor < nRows)
			return strData[nCursor++].rowData();
		
		return null;
	}

/**
 * Data Seek
 *
 * @param i		the index of the row to be retrieved.
 */
	public void DataSeek(int i) {

		if(i < nRows)
			nCursor = i;
	}

/**
 * Return the number of fields in the table
 * 
 * @return		the number of fields (columns) in this data set.
 */
	public int NumFields() {
		
		return nCols;
	}

/**
 * return the number of rows in the table
 * 
 * @return		the number of rows in this data set.
 */
	public int NumRows() {

		return nRows;
	}
}
