/* 
 * This is an interface to the miniSQL database for Python.
 *   Based on a prior work by Anthony Baxter
 *   Updated, fixed and extended by David Gibson working for
 *   Thawte Consulting cc, South Africa.
 *
 *   Copyright 1995 Thawte Consulting cc
 *   Portions copyright (C) 1994 Anthony Baxter.
 *
 *   Permission is hereby granted, free of charge, to any person obtaining
 *   a copy of this source file to use, copy, modify, merge, or publish it
 *   subject to the following conditions:
 *
 *   The above copyright notice and this permission notice shall be included
 *   in all copies or in any new file that contains a substantial portion of
 *   this file.
 *
 *   THE  AUTHOR  MAKES  NO  REPRESENTATIONS ABOUT  THE  SUITABILITY  OF
 *   THE  SOFTWARE FOR  ANY  PURPOSE.  IT IS  PROVIDED  "AS IS"  WITHOUT
 *   EXPRESS OR  IMPLIED WARRANTY.  THE AUTHOR DISCLAIMS  ALL WARRANTIES
 *   WITH  REGARD TO  THIS  SOFTWARE, INCLUDING  ALL IMPLIED  WARRANTIES
 *   OF   MERCHANTABILITY,  FITNESS   FOR  A   PARTICULAR  PURPOSE   AND
 *   NON-INFRINGEMENT  OF THIRD  PARTY  RIGHTS. IN  NO  EVENT SHALL  THE
 *   AUTHOR  BE LIABLE  TO  YOU  OR ANY  OTHER  PARTY  FOR ANY  SPECIAL,
 *   INDIRECT,  OR  CONSEQUENTIAL  DAMAGES  OR  ANY  DAMAGES  WHATSOEVER
 *   WHETHER IN AN  ACTION OF CONTRACT, NEGLIGENCE,  STRICT LIABILITY OR
 *   ANY OTHER  ACTION ARISING OUT OF  OR IN CONNECTION WITH  THE USE OR
 *   PERFORMANCE OF THIS SOFTWARE.
 *
 * $Id: msqlmodule.c,v 1.5 1995/12/26 12:29:25 davidg Exp davidg $

******************************************************

Modified by David Gibson December 1995

- listdbs and listtables now return a list of strings
- new Python naming conventions introduced
- queries now return data types in native Python format (String,Float,Int)
- solved spurious 'function requires at least one argument' error: old
	getargs would not handle optional arguments.  ParseTuple is being used now.
	(so method table has got 1's in now)
	(old Parse routine still needed for subscript handling)
- msqlFreeResult now called after query!
- assignment to subscript trapped correctly.  Ditto len()
- added DbType to the module dictionary
- mSQL.error object introduced

 */

/*#define DEBUG_MSQLMOD*/

#include "allobjects.h"
#include "modsupport.h"
#include <sys/types.h>
#include <sys/stat.h>

#include "msql.h"

char   PROGNAME[] = "python";
static PyObject *pythonify_res();
static PyObject *pythonify_single_res();
static PyObject *pythonify_lf_res();

typedef struct {
    PyObject_HEAD
    int handle;
    int valid;
} msqlobject;

staticforward PyTypeObject MsqlType;
static PyObject * mSQLError;

#define is_msqlobject(v) ((v)->ob_type == &MsqlType)

static PyObject * 
msqlmod_connect(self,args)
PyObject *self,*args;
{
    char *dbname = NULL;
    msqlobject *n;
    int newhandle;

    if(!PyArg_ParseTuple(args,"|s",&dbname)) {
        PyErr_SetString(mSQLError,"connect has one optional arg, a database host name");
        return NULL;
    }

    newhandle = msqlConnect(dbname);

    if(newhandle == -1) {
        PyErr_SetString(mSQLError,msqlErrMsg); 
        return NULL;
    } else {
        n=PyObject_NEW(msqlobject,&MsqlType);
        if(!n)
            return NULL;
        n->valid=1;
        n->handle = newhandle;
        return((PyObject *)n);
    }
}
static PyObject *
msqlobj_selectdb(self,args)
msqlobject *self;
PyObject *args;
{
    char *dbname;
    if(!PyArg_ParseTuple(args,"s",&dbname)) {
        PyErr_SetString(mSQLError,"selectdb has one arg, a database name"); 
        return NULL;
    }
    if(msqlSelectDB(self->handle,dbname)==-1) {
        PyErr_SetString(mSQLError,msqlErrMsg);
        return NULL;
    } 
    Py_INCREF(Py_None);
    return(Py_None);
}   

static PyObject * 
msqlobj_listdbs(self,args)
msqlobject *self;
PyObject *args;
{
    m_result *res;
    PyObject *resobj;

    if(!PyArg_ParseTuple(args,"")) {
        PyErr_SetString(mSQLError,"listdbs takes no args");
        return NULL;
    }
    if((res=msqlListDBs(self->handle))==NULL) {
        Py_INCREF(Py_None);
        return(Py_None);
    }
    resobj=pythonify_single_res(res);
    return(resobj);
}

static PyObject * 
msqlobj_listtables(self,args)
msqlobject *self;
PyObject *args;
{
    m_result *res;
    PyObject *resobj;

    if(!PyArg_ParseTuple(args,"")) {
        PyErr_SetString(mSQLError,"listtables takes no args");
        return NULL;
    }
    if((res=msqlListTables(self->handle))==NULL) {
        Py_INCREF(Py_None);
        return(Py_None);
    }
    resobj=pythonify_single_res(res);
    return(resobj);
}

static PyObject *
msqlobj_listfields(self,args)
msqlobject *self;
PyObject *args;
{
    char *tname;
    m_result *res;
    PyObject *resobj;

    if(!PyArg_ParseTuple(args,"s",&tname)) {
        PyErr_SetString(mSQLError,"listfields takes one arg, a table name");
        return NULL;
    }
    if((res=msqlListFields(self->handle,tname))==NULL) {
        Py_INCREF(Py_None);
        return(Py_None);
    }
    resobj=pythonify_lf_res(res);
    return(resobj);
}

static PyObject * 
msqlobj_query_helper(self,query)
msqlobject *self;
char * query;
{
    m_result *res;
    PyObject *resobj;

    if(msqlQuery(self->handle,query)==-1) {
        PyErr_SetString(mSQLError,msqlErrMsg); 
        return NULL;
    }
    res=msqlStoreResult();
    if(!res) {
        Py_INCREF(Py_None);
        return(Py_None);
    }
    resobj=pythonify_res(res);
	msqlFreeResult(res);
    return(resobj);
}

static PyObject * 
msqlobj_query(self,args)
msqlobject *self;
PyObject *args;
{
    char *query;

    if(!PyArg_ParseTuple(args,"s",&query)) {
        PyErr_SetString(mSQLError,"query has one arg, a query string");
        return NULL;
    }
	return msqlobj_query_helper(self, query);
}

/*
 * Take an mSQL m_result, turn it into a list of tuples.
 */
static PyObject *
pythonify_res(res)
m_result *res;
{
    PyObject *reslist,*rowtuple, *fieldobj;
    m_row thisrow;
    m_field *tf;
    int i,n;

#ifdef DEBUG_MSQLMOD
    printf("data ready, %d rows of %d fields\n",msqlNumRows(res),msqlNumFields(res));
#endif
    reslist=PyList_New(0);
    n=msqlNumFields(res);

    while(thisrow=msqlFetchRow(res)) {
        rowtuple=PyTuple_New(n);
        msqlFieldSeek(res, 0);
        for (i=0;i<n;i++) {

            tf=msqlFetchField(res);

            if (thisrow[i])
                switch (tf->type) {
                    case INT_TYPE: 
                        fieldobj = PyInt_FromLong(atol(thisrow[i]));
                        break;
                    case CHAR_TYPE: 
                        fieldobj = PyString_FromString(thisrow[i]);
                        break;
                    case REAL_TYPE: 
                        fieldobj = PyFloat_FromDouble(atof(thisrow[i]));
                        break;
                    default: 
                        fieldobj = NULL;
                        break;
                }
            else {
                Py_INCREF(Py_None);
                fieldobj = Py_None;
            }

            if (fieldobj == NULL) {
                Py_DECREF(rowtuple);
                Py_DECREF(reslist);
                PyErr_SetString(mSQLError,"mSQL database returned bad value"); 
                return NULL;
            }

            PyTuple_SetItem(rowtuple,i,fieldobj);
        }
        PyList_Append(reslist,rowtuple);
        Py_DECREF(rowtuple);
    }
    return(reslist);
}

/*
 * Take an mSQL m_result, turn it into a list of strings.
 */
static PyObject *
pythonify_single_res(res)
m_result *res;
{
    PyObject *reslist,*rowtuple, *str;
    m_row thisrow;
    int i,n;

#ifdef DEBUG_MSQLMOD
    printf("data ready, %d rows of %d fields\n",msqlNumRows(res),msqlNumFields(res));
#endif
    reslist=PyList_New(0);
    n=msqlNumFields(res);
     if (n != 1) {
        PyErr_SetString(mSQLError,"expected mSQL to return singletons"); 
        return NULL;
    }

    while(thisrow=msqlFetchRow(res)) {
          str=PyString_FromString(thisrow[0]);
          PyList_Append(reslist,str);
    }
    return(reslist);
}

/*
 * Take an mSQL m_result, return a list of tuples of the FetchField data.
 */
static PyObject *
pythonify_lf_res(res)
m_result *res;
{
    PyObject *reslist, *thistuple, *str;
    int i,n;
    char *type,flags[14];
    m_field *tf;
    
#ifdef DEBUG_MSQLMOD
    printf("data ready, %d fields\n", msqlNumFields(res));
#endif
    reslist=PyList_New(0);
    n=msqlNumFields(res);
    for(i=0;i<n;i++) {
        tf=msqlFetchField(res);
        switch(tf->type) {
            case INT_TYPE: type="int"; break;
            case CHAR_TYPE: type="char"; break;
            case REAL_TYPE: type="real"; break;
            default: type="????"; break;
        }
        if (IS_PRI_KEY(tf->flags)) 
            strcpy(flags,"pri");
        else 
            flags[0]=0;
        if (IS_NOT_NULL(tf->flags)) 
            if(flags[0])
                strcat(flags," notnull");
            else
                strcpy(flags,"notnull");
        else 
            flags[0]=0;

        thistuple=Py_BuildValue("(sssis)",tf->name,tf->table,type,tf->length,
                                                                flags);
        PyList_Append(reslist,thistuple);
        Py_DECREF(thistuple);
    }
    return(reslist);
}

static struct PyMethodDef msqlobj_methods[] = {
    {"selectdb",(PyCFunction)msqlobj_selectdb, 1},
    {"query",(PyCFunction)msqlobj_query, 1},
    {"listdbs",(PyCFunction)msqlobj_listdbs, 1},
    {"listtables",(PyCFunction)msqlobj_listtables, 1},
    {"listfields",(PyCFunction)msqlobj_listfields, 1},
    {NULL, NULL}           /* sentinel */
};

static PyObject *
msqlobj_getattr(ms, name)
        msqlobject *ms;
        char *name;
{
        return Py_FindMethod(msqlobj_methods, (PyObject *)ms, name);
}
static void     
msqlobj_dealloc(m)
        register msqlobject *m;
{
        if ( m->valid )
          msqlClose(m->handle);
        PyMem_DEL(m);
}       

int
msqlobj_print(sp, fp, flags)
msqlobject *sp;
FILE *fp;
int flags;
{
    fprintf(fp,"mSQL handle");
    return 0;
}

static int
msqlobj_len(self,subs)
PyObject *self, *subs;
{
    PyErr_SetString(mSQLError,"can't take length of an mSQL handle");
    return -1;
}
static PyObject *
msqlobj_subscript(self,subs)
PyObject *self, *subs;
{
	char * query;

    if(!PyArg_Parse(subs,"s",&query)) {
        PyErr_SetString(mSQLError,"subscript expects a query string");
        return NULL;
    }
    return msqlobj_query_helper(self,query);
}

static int
msqlobj_ass_sub(self,subs,val)
PyObject *self, *subs,*val;
{
    PyErr_SetString(mSQLError,"can't assign to an mSQL handle (use query insert)");
    return -1;     /* -1 is error code in interpreter main loop! (ceval.c) */
}

static PyMappingMethods msql_as_mapping = {
        (inquiry)msqlobj_len,            /*length*/
        (binaryfunc)msqlobj_subscript,      /*subscript*/
        (objobjargproc)msqlobj_ass_sub,     /*assign subscript*/
};              


static PyTypeObject MsqlType = {
        PyObject_HEAD_INIT(&PyType_Type)
        0,
        "msqlobject",
        sizeof(msqlobject),
        0,
        (destructor)msqlobj_dealloc,  /*tp_dealloc*/
        0,                            /*tp_print*/
        (getattrfunc)msqlobj_getattr, /*tp_getattr*/
        0,                            /*tp_setattr*/
        0,                            /*tp_compare*/
        0,                            /*tp_repr*/
        0,                            /*tp_as_number*/
        0,                            /*tp_as_sequence*/
        &msql_as_mapping,             /*tp_as_mapping*/
};      


static struct PyMethodDef msql_methods[] = {
    {"connect",msqlmod_connect, 1},
    {NULL,NULL}
};


void
initmSQL()
{
    PyObject *module, *dict;
    char *err;

    module = Py_InitModule("mSQL", msql_methods);
    dict = PyModule_GetDict(module);

	if (PyDict_SetItemString(dict, "DbType", (PyObject*)&MsqlType) != 0)
		Py_FatalError("Cannot add to mSQL dictionary");

	mSQLError = PyString_FromString("mSQL.error");
	if (PyDict_SetItemString(dict, "error", mSQLError) != 0)
		Py_FatalError("Cannot add to mSQL dictionary");
}
