/*
 *	tclSQL.c	- Source for sql commands
 *
 * Copyright (c) 1994   Brad Pepers
 *
 * Permission to use, copy, and distribute for non-commercial purposes,
 * is hereby granted without fee, providing that the above copyright
 * notice appear in all copies and that both the copyright notice and this
 * permission notice appear in supporting documentation.
 *
 * This software is provided "as is" without any expressed or implied
 * warranty.
 *
 * $Id: tclSQL.c,v 1.6 1994/08/25 00:28:07 pepers Exp $
 *
 * $Log: tclSQL.c,v $
 * Revision 1.6  1994/08/25  00:28:07  pepers
 * Changed for mSQL version 0.2
 *
 * Revision 1.5  1994/07/28  23:18:05  pepers
 * Added rows and cols commands
 *
 * Revision 1.4  1994/07/28  22:44:43  pepers
 * Changed around how commands work
 *
 * Revision 1.3  1994/07/28  02:04:54  pepers
 * Changed close command to disconnect
 *
 * Revision 1.2  1994/07/27  22:43:27  pepers
 * Fixed log files
 *
 * Revision 1.1  1994/07/27  22:34:48  pepers
 * First version of mSQL to Tcl/Tk interface
 *
*/

#include "tclSQL.h"

char RCS_ID[] = "$Id: tclSQL.c,v 1.6 1994/08/25 00:28:07 pepers Exp $";

m_result *query_result = NULL;

void MSQL_Init(Tcl_Interp *interp)
{
  Tcl_CreateCommand(interp, "msql", MSQL_Cmd, 0, 0);
}

void MSQL_Cleanup(void)
{
  msqlClose();
}

int MSQL_Cmd(ClientData clientData, Tcl_Interp *interp, int argc,
	    char **argv)
{
  if (argc < 2) {
    Tcl_AppendResult(interp, argv[0], ": missing command name.", 0);
    return TCL_ERROR;
  }

  if (strcmp(argv[1], "connect") == 0)
    return MSQL_Cmd_Connect(clientData, interp, argc, argv);
  else if (strcmp(argv[1], "connections") == 0)
    return MSQL_Cmd_Connections(clientData, interp, argc, argv);

  Tcl_AppendResult(interp, argv[0], ": unsupported command.", 0);
  return TCL_ERROR;
}

int MSQL_Cmd_Connect(ClientData clientData, Tcl_Interp *interp, int argc,
	    char **argv)
{
  char *name = NULL;
  char *host = NULL;
  int socket;
  
  if (argc > 4 || *argv[2] == '\0') {
    Tcl_AppendResult(interp, "usage: sql connect name ?host?.", 0);
    return TCL_ERROR;
  }

  name = argv[2];

  if (argc > 3 && *argv[3] != '\0')
    host = argv[3];

  socket = msqlConnect(host);
  if (socket == -1) {
    Tcl_AppendResult(interp, argv[0], ": ", msqlErrMsg, 0);
    return TCL_ERROR;
  }

  Tcl_CreateCommand(interp, name, MSQL_Conn_Cmd, (ClientData)socket, 0);
  return TCL_OK;
}

int MSQL_Cmd_Connections(ClientData clientData, Tcl_Interp *interp, int argc,
	    char **argv)
{
  return TCL_OK;
}

int MSQL_Conn_Cmd(ClientData clientData, Tcl_Interp *interp, int argc,
	    char **argv)
{
  if (argc < 2) {
    Tcl_AppendResult(interp, argv[0], ": missing command name.", 0);
    return TCL_ERROR;
  }

  if (strcmp(argv[1], "get") == 0)
    return MSQL_Cmd_Get((int)clientData, interp, argc, argv);
  else if (strcmp(argv[1], "set") == 0)
    return MSQL_Cmd_Set((int)clientData, interp, argc, argv);
  else if (strcmp(argv[1], "query") == 0)
    return MSQL_Cmd_Query((int)clientData, interp, argc, argv);
  else if (strcmp(argv[1], "disconnect") == 0) {
    return MSQL_Cmd_Disconnect((int)clientData, interp, argc, argv);
  }

  Tcl_AppendResult(interp, argv[0], ": unsupported command.", 0);
  return TCL_ERROR;
}

int MSQL_Cmd_Disconnect(int socket, Tcl_Interp *interp, int argc,
	    char **argv)
{
  if (argc != 2) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " disconnect.", 0);
    return TCL_ERROR;
  }

  if (msqlClose(socket) == -1) {
    Tcl_AppendResult(interp, argv[0], ": ", msqlErrMsg, 0);
    return TCL_ERROR;
  }

  return Tcl_DeleteCommand(interp, argv[0]);
}

int MSQL_Cmd_Set(int socket, Tcl_Interp *interp, int argc,
	    char **argv)
{
  if (argc < 3) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " set ???.", 0);
    return TCL_ERROR;
  }

  if (strcmp(argv[2], "database") == 0)
    return MSQL_Set_DB(socket, interp, argc, argv);
  else if (strcmp(argv[2], "row") == 0)
    return MSQL_Set_Row(socket, interp, argc, argv);

  Tcl_AppendResult(interp, argv[0], ": unknown set name ", argv[2], ".", 0);
  return TCL_ERROR;
}

int MSQL_Set_DB(int socket, Tcl_Interp *interp, int argc,
	       char **argv)
{
  if (argc != 4) {
    Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2],
		     ": missing database name", 0);
    return TCL_ERROR;
  }

  if (msqlSelectDB(socket, argv[3]) == -1) {
    Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2],
		     ": ", msqlErrMsg, 0);
    return TCL_ERROR;
  }

  return TCL_OK;
}

int MSQL_Set_Row(int socket, Tcl_Interp *interp, int argc,
	       char **argv)
{
  if (argc != 4) {
    Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2],
		     ": missing row number", 0);
    return TCL_ERROR;
  }

  if (query_result == NULL) {
    Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2],
		     ": no current query", 0);
    return TCL_ERROR;
  }

  msqlDataSeek(query_result, atoi(argv[3]));
  
  return TCL_OK;
}

int MSQL_Cmd_Get(int socket, Tcl_Interp *interp, int argc,
	    char **argv)
{
  if (argc < 3) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " get ???.", 0);
    return TCL_ERROR;
  }

  if (strcmp(argv[2], "databases") == 0)
    return MSQL_Get_DBs(socket, interp, argc, argv);
  else if (strcmp(argv[2], "tables") == 0)
    return MSQL_Get_Tables(socket, interp, argc, argv);
  else if (strcmp(argv[2], "fields") == 0)
    return MSQL_Get_Fields(socket, interp, argc, argv);
  else if (strcmp(argv[2], "rest") == 0)
    return MSQL_Get_Rest(socket, interp, argc, argv);
  else if (strcmp(argv[2], "next") == 0)
    return MSQL_Get_Next(socket, interp, argc, argv);
  else if (strcmp(argv[2], "rows") == 0)
    return MSQL_Get_Rows(socket, interp, argc, argv);
  else if (strcmp(argv[2], "cols") == 0)
    return MSQL_Get_Cols(socket, interp, argc, argv);

  Tcl_AppendResult(interp, argv[0], ": unknown get name ", argv[2], ".", 0);
  return TCL_ERROR;
}

void free_query_result(void)
{
  if (query_result != NULL) {
    msqlFreeResult(query_result);
    query_result = NULL;
  }
}

int MSQL_Get_DBs(int socket, Tcl_Interp *interp, int argc,
	       char **argv)
{
  Tcl_DString results;
  m_result *result;
  int num_cols;
  m_row row;
  int pos;

  if (argc != 3) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " get databases.", 0);
    return TCL_ERROR;
  }

  result = msqlListDBs(socket);

  if (result == NULL) {
    if (msqlErrMsg[0] != '\0') {
      Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2],
		       ": ", msqlErrMsg, 0);
      return TCL_ERROR;
    }
    return TCL_OK;
  }

  Tcl_DStringInit(&results);

  num_cols = 1;
  while ((row = msqlFetchRow(result)) != NULL) {
    Tcl_DStringStartSublist(&results);

    for (pos = 0; pos < num_cols; pos++)
      Tcl_DStringAppendElement(&results, row[pos]);

    Tcl_DStringEndSublist(&results);
  }

  msqlFreeResult(result);

  Tcl_DStringResult(interp, &results);
  return TCL_OK;
}

int MSQL_Get_Tables(int socket, Tcl_Interp *interp, int argc,
	       char **argv)
{
  Tcl_DString results;
  m_result *result;
  int num_cols;
  m_row row;
  int pos;

  if (argc != 3) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " get tables.", 0);
    return TCL_ERROR;
  }

  result = msqlListTables(socket);

  if (result == NULL) {
    if (msqlErrMsg[0] != '\0') {
      Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2],
		       ": ", msqlErrMsg, 0);
      return TCL_ERROR;
    }
    return TCL_OK;
  }

  Tcl_DStringInit(&results);

  num_cols = 1;
  while ((row = msqlFetchRow(result)) != NULL) {
    Tcl_DStringStartSublist(&results);

    for (pos = 0; pos < num_cols; pos++)
      Tcl_DStringAppendElement(&results, row[pos]);

    Tcl_DStringEndSublist(&results);
  }

  msqlFreeResult(result);

  Tcl_DStringResult(interp, &results);
  return TCL_OK;
}

int MSQL_Get_Fields(int socket, Tcl_Interp *interp, int argc,
	       char **argv)
{
  Tcl_DString results;
  m_result *result;
  m_field *field;
  char buf[40];

  if (argc != 4) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " get fields <table>.", 0);
    return TCL_ERROR;
  }

  result = msqlListFields(socket, argv[3]);

  if (result == NULL) {
    if (msqlErrMsg[0] != '\0') {
      Tcl_AppendResult(interp, argv[0], " ", argv[1], " ", argv[2],
		       ": ", msqlErrMsg, 0);
      return TCL_ERROR;
    }
    return TCL_OK;
  }

  Tcl_DStringInit(&results);

  while ((field = msqlFetchField(result)) != NULL) {
    Tcl_DStringStartSublist(&results);
    
    Tcl_DStringAppendElement(&results, field->name);

    switch (field->type) {
    case INT_TYPE:
      Tcl_DStringAppendElement(&results, "INT");
      break;
    case REAL_TYPE:
      Tcl_DStringAppendElement(&results, "REAL");
      break;
    case CHAR_TYPE:
      sprintf(buf, "CHAR(%d)", field->length);
      Tcl_DStringAppendElement(&results, buf);
      break;
    }

    sprintf(buf, "%d", field->flags);
    Tcl_DStringAppendElement(&results, buf);
    
    Tcl_DStringEndSublist(&results);
  }

  msqlFreeResult(result);

  Tcl_DStringResult(interp, &results);
  return TCL_OK;
}

int MSQL_Get_Rest(int socket, Tcl_Interp *interp, int argc,
	    char **argv)
{
  Tcl_DString results;
  int num_cols;
  m_row row;
  int pos;

  if (query_result == NULL) {
    return TCL_OK;
  }

  Tcl_DStringInit(&results);

  num_cols = msqlNumFields(query_result);
  while (row = msqlFetchRow(query_result)) {
    Tcl_DStringStartSublist(&results);

    for (pos = 0; pos < num_cols; pos++)
      Tcl_DStringAppendElement(&results, row[pos]);

    Tcl_DStringEndSublist(&results);
  }

  Tcl_DStringResult(interp, &results);
  return TCL_OK;
}

int MSQL_Get_Next(int socket, Tcl_Interp *interp, int argc,
	    char **argv)
{
  Tcl_DString results;
  int num_cols;
  m_row row;
  int pos;

  if (query_result == NULL) {
    return TCL_OK;
  }

  num_cols = msqlNumFields(query_result);
  row = msqlFetchRow(query_result);
  
  if (row == NULL) {
    return TCL_OK;
  }

  Tcl_DStringInit(&results);

  for (pos = 0; pos < num_cols; pos++)
    Tcl_DStringAppendElement(&results, row[pos]);

  Tcl_DStringResult(interp, &results);
  return TCL_OK;
}

int MSQL_Get_Rows(int socket, Tcl_Interp *interp, int argc,
	    char **argv)
{
  char buf[80];

  if (argc != 3) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " get rows.", 0);
    return TCL_ERROR;
  }

  if (query_result == NULL) {
    Tcl_AppendResult(interp, "0", 0);
    return TCL_OK;
  }

  sprintf(buf, "%d", msqlNumRows(query_result));
  Tcl_AppendResult(interp, buf, 0);
  return TCL_OK;
}

int MSQL_Get_Cols(int socket, Tcl_Interp *interp, int argc,
	    char **argv)
{
  char buf[80];

  if (argc != 3) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " get cols.", 0);
    return TCL_ERROR;
  }

  if (query_result == NULL) {
    Tcl_AppendResult(interp, "0", 0);
    return TCL_OK;
  }

  sprintf(buf, "%d", msqlNumFields(query_result));
  Tcl_AppendResult(interp, buf, 0);
  return TCL_OK;
}

int MSQL_Cmd_Query(int socket, Tcl_Interp *interp, int argc,
	    char **argv)
{
  if (argc != 3) {
    Tcl_AppendResult(interp, "usage: ", argv[0], " query <query>.", 0);
    return TCL_ERROR;
  }

  if (msqlQuery(socket, argv[2]) == -1) {
    Tcl_AppendResult(interp, argv[0], " ", argv[1], ": ", msqlErrMsg, 0);
    return TCL_ERROR;
  }

  free_query_result();
  query_result = msqlStoreResult(socket);

  return TCL_OK;
}
